package com.redhat.installer.installation.processpanel;

import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.util.AbstractUIProcessHandler;
import com.izforge.izpack.util.Debug;
import com.redhat.installer.installation.util.TomcatUtils;
import org.apache.commons.lang.SystemUtils;
import org.w3c.dom.*;
import org.xml.sax.SAXException;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.xpath.XPathExpressionException;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;

public class TomcatAddUser {
    private static AbstractUIProcessHandler mHandler;
    private static AutomatedInstallData idata;
    private static String algorithm;
    private static String keyLength;
    private static String handlerClassName;

    /**
     * Adds the user defined in the installation to tomcat-users.xml
     *
     * @param handler supplied by internal izpack library. Used to output to panel / console
     * @param args    User information. args[0]: username args[1]: password args[2]: role1,role2 etc
     * @return true upon successful addition of roles and user to the existing tomcat-users.xml
     */
    public static boolean run(AbstractUIProcessHandler handler, String[] args) {
        idata = AutomatedInstallData.getInstance();
        mHandler = handler;
        setHandlerInformation();
        return configureTomcatUsersFile(args);
    }

    private static boolean configureTomcatUsersFile(String[] args) {
        try {
            Path tomcatUsersPath = Paths.get(idata.getInstallPath() + "/conf/tomcat-users.xml");
            Document xml = TomcatUtils.getXmlDocument(tomcatUsersPath);
            xml.getDocumentElement().appendChild(xml.createComment(idata.langpack.getString("tomcat.installer.comment")));
            addUser(xml, args);
            TomcatUtils.writeXmlDocumentToFile(xml, tomcatUsersPath);
        } catch (ParserConfigurationException | IOException | SAXException | TransformerException | XPathExpressionException e) {
            ProcessPanelHelper.printToPanel(mHandler, idata.langpack.getString("tomcat.userfile.adduser.failed"), true);
            ProcessPanelHelper.printExceptionToLog(e.getStackTrace());
            return false;
        }
        ProcessPanelHelper.printToPanel(mHandler, String.format(idata.langpack.getString("tomcat.userfile.adduser.success"), args[0]), false);
        return true;
    }

    /**
     * Adds the user element to tomcat-users.xml
     *
     * @param xml      input DOM Document. Typically a parsed tomcat-users.xml
     * @param userInfo array containing the username, password, and rolelist of the user to add
     */
    private static void addUser(Document xml, String[] userInfo) {
        String username = userInfo[0];
        String password = userInfo[1];
        String rolelist = userInfo[2];

        Element newUser = xml.createElement("user");
        newUser.setAttribute("username", username);
        newUser.setAttribute("password", password);
        newUser.setAttribute("roles", rolelist);
        xml.getDocumentElement().appendChild(newUser);
    }

    /**
     * Executes tomcat's digest.sh / digest.bat to create a hashed password value. In conjunction with the modifications
     * to the UserDatabase realm, this fixes RHDM-322
     *
     * @param s
     * @return
     */
    private static String getPasswordHash(String s) {
        Path installPath = Paths.get(idata.getInstallPath());
        Path executable;
        boolean isWindows = SystemUtils.IS_OS_WINDOWS;
        if (isWindows) {
            executable = installPath.resolve("bin/digest.bat");
        } else {
            executable = installPath.resolve("bin/digest.sh");
        }
        ProcessBuilder pb = new ProcessBuilder(executable.toString(), "-a",
                algorithm,
                "-i",
                "10000",
                "-s",
                "16",
                "-k",
                keyLength,
                "-h",
                handlerClassName,
                s);

        if (isWindows){
            pb.environment().put("CATALINA_HOME", idata.getInstallPath());
        }

        String hash = s;
        try {
            if (Files.exists(executable)) {
                Process p = pb.start();
                p.waitFor();
                BufferedReader br = new BufferedReader(new InputStreamReader(p.getInputStream()));
                hash = br.lines().map(line -> line.split(":")[1]).findFirst().get();
            }
        } catch (InterruptedException | IOException e) {
            Debug.log("Failed to create hash of user password. Falling back to plaintext.");
            return s;
        }
        return hash;
    }

    private static void setHandlerInformation() {
        Path serverXmlPath = Paths.get(idata.getInstallPath(), "conf", "server.xml");
        try {
            Document serverXml = TomcatUtils.getXmlDocument(serverXmlPath);
            Node userDatabaseRealm = TomcatUtils.findNodeWithAttribute(serverXml.getElementsByTagName("Realm"),"className", "org.apache.catalina.realm.UserDatabaseRealm");
            NodeList userDatabaseChildren = userDatabaseRealm.getChildNodes();
            for (int i = 0; i < userDatabaseChildren.getLength(); i++){
                Node child = userDatabaseChildren.item(i);
                if (child.getNodeName().equals("CredentialHandler")){
                    algorithm = child.hasAttributes() ? child.getAttributes().getNamedItem("algorithm").getNodeValue() : "PBKDF2WithHmacSHA512";
                    keyLength = child.hasAttributes() ? child.getAttributes().getNamedItem("keyLength").getNodeValue() : "256";
                    handlerClassName = child.hasAttributes() ? child.getAttributes().getNamedItem("className").getNodeValue() : "org.apache.catalina.realm.SecretKeyCredentialHandler";
                    return;
                }
            }
        } catch (SAXException | ParserConfigurationException | IOException e) {
            e.printStackTrace();
        }
        algorithm = "PBKDF2WithHmacSHA512";
        keyLength = "256";
        handlerClassName = "org.apache.catalina.realm.SecretKeyCredentialHandler";
    }
}
