/*
 * Copyright 2018 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.optaweb.employeerostering.gwtui.client.viewport.rotation;

import java.time.Duration;
import java.time.LocalDateTime;
import javax.inject.Inject;
import javax.inject.Named;

import com.google.gwt.safehtml.shared.SafeHtmlBuilder;
import elemental2.dom.HTMLElement;
import elemental2.dom.MouseEvent;
import org.jboss.errai.ui.shared.api.annotations.DataField;
import org.jboss.errai.ui.shared.api.annotations.EventHandler;
import org.jboss.errai.ui.shared.api.annotations.ForEvent;
import org.jboss.errai.ui.shared.api.annotations.Templated;
import org.optaweb.employeerostering.gwtui.client.viewport.grid.GridObject;
import org.optaweb.employeerostering.gwtui.client.viewport.grid.Lane;
import org.optaweb.employeerostering.gwtui.client.viewport.impl.AbstractHasTimeslotGridObject;
import org.optaweb.employeerostering.gwtui.client.viewport.impl.Draggability;
import org.optaweb.employeerostering.gwtui.client.viewport.impl.Resizability;
import org.optaweb.employeerostering.shared.common.HasTimeslot;
import org.optaweb.employeerostering.shared.rotation.view.ShiftTemplateView;

@Templated
public class ShiftTemplateGridObject extends AbstractHasTimeslotGridObject<RotationMetadata> implements GridObject<LocalDateTime, RotationMetadata> {

    private ShiftTemplateModel model;
    private ShiftTemplateView shiftTemplateView;

    @Inject
    @DataField("label")
    @Named("span")
    private HTMLElement label;

    @Inject
    private Draggability<LocalDateTime, RotationMetadata> draggability;
    @Inject
    private Resizability<LocalDateTime, RotationMetadata> resizability;

    public void withShiftTemplateModel(ShiftTemplateModel model) {
        this.model = model;
        ShiftTemplateView newShift = new ShiftTemplateView();
        newShift.setId(model.getShiftTemplateView().getId());
        newShift.setRotationEmployeeId(model.getShiftTemplateView().getRotationEmployeeId());
        this.shiftTemplateView = newShift;

        if (getLane() != null) {
            label.innerHTML = (newShift.getRotationEmployeeId() != null)
                    ? new SafeHtmlBuilder().appendEscaped(getLane().getMetadata()
                                                                   .getEmployeeIdToEmployeeMap()
                                                                   .get(newShift.getRotationEmployeeId())
                                                                   .getName()).toSafeHtml().asString()
                    : "Unassigned";
            updatePositionInLane();
        }
    }

    @Override
    public void setStartPositionInScaleUnits(LocalDateTime newStartPosition) {
        updateStartDateTimeWithoutRefresh(newStartPosition);
        model.refreshTwin(this);
    }

    @Override
    public void setEndPositionInScaleUnits(LocalDateTime newEndPosition) {
        updateEndDateTimeWithoutRefresh(newEndPosition);
        model.refreshTwin(this);
    }

    @Override
    public void init(Lane<LocalDateTime, RotationMetadata> lane) {
        updatePositionInLane();
        if (shiftTemplateView != null) {
            label.innerHTML = (shiftTemplateView.getRotationEmployeeId() != null)
                    ? new SafeHtmlBuilder().appendEscaped(getLane().getMetadata()
                                                                   .getEmployeeIdToEmployeeMap()
                                                                   .get(shiftTemplateView.getRotationEmployeeId())
                                                                   .getName()).toSafeHtml().asString()
                    : "Unassigned";
        }
        draggability.applyFor(this, lane.getScale());
        resizability.applyFor(this, lane.getScale());
    }

    private void updatePositionInLane() {
        Long daysInRotation = getDaysInRotation();
        LocalDateTime baseDate = (model.isLaterTwin(this)) ? getLane().getScale().getStartInScaleUnits() : getLane().getScale().getStartInScaleUnits()
                                                                                                                    .minusDays(daysInRotation);
        updateStartDateTimeWithoutRefresh(baseDate.plus(model.getShiftTemplateView().getDurationBetweenRotationStartAndTemplateStart()));
        updateEndDateTimeWithoutRefresh(getStartPositionInScaleUnits().plus(model.getShiftTemplateView().getShiftTemplateDuration()));
    }

    @Override
    public Long getId() {
        return shiftTemplateView.getId();
    }

    protected void updateStartDateTimeWithoutRefresh(LocalDateTime newStartDateTime) {
        shiftTemplateView.setDurationBetweenRotationStartAndTemplateStart(Duration.between(getLane().getScale().getStartInScaleUnits(),
                                                                                           newStartDateTime));
    }

    protected void updateEndDateTimeWithoutRefresh(LocalDateTime newEndDateTime) {
        shiftTemplateView.setShiftTemplateDuration(Duration.between(getStartPositionInScaleUnits(),
                                                                    newEndDateTime));
    }
    
    protected Long getDaysInRotation() {
        return Duration.between(getLane().getScale().getStartInScaleUnits(),
                                getLane().getScale().getEndInScaleUnits()).getSeconds() / 60 / 60 / 24;
    }

    protected void reposition() {
        if (getLane() != null) {
            getLane().positionGridObject(this);
        }
    }

    @EventHandler("root")
    private void onClick(@ForEvent("click") MouseEvent e) {
        if (e.shiftKey) {
            getLane().removeGridObject(model);
        }
    }

    @Override
    protected HasTimeslot getTimeslot() {
        return shiftTemplateView;
    }

    @Override
    public void save() {
        // Nothing to save; save is done in batch for Rotation
    }

}
