/*
 * IzPack - Copyright 2001-2008 Julien Ponge, All Rights Reserved.
 *
 * http://izpack.org/
 * http://izpack.codehaus.org/
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.izforge.izpack.panels;

import com.izforge.izpack.gui.*;
import com.izforge.izpack.gui.patternfly.PatternflyProgressBarUI;
import com.izforge.izpack.installer.InstallData;
import com.izforge.izpack.installer.InstallerFrame;
import com.izforge.izpack.installer.IzPanel;
import com.izforge.izpack.util.AbstractUIProgressHandler;

import javax.swing.*;
import java.awt.*;
import java.io.File;
import java.util.*;
import java.util.List;

/**
 * The install panel class. Launches the actual installation job.
 *
 * @author Julien Ponge
 */
public class InstallPanel extends IzPanel implements AbstractUIProgressHandler {

    private static final long serialVersionUID = 3257282547959410992L;

    private final String INSTALL_FINISHED = "InstallPanel.gui.finished";
    /**
     * The operation label .
     */
    protected JLabel packOpLabel;

    /**
     * The operation label prefix
     */
    protected String packOpLabelPrefix;

    /**
     * The operation label .
     */
    protected JLabel overallOpLabel;

    /**
     * The icon used.
     */
    protected String iconName = "preferences";

    /**
     * The pack progress bar.
     */
    protected JProgressBar packProgressBar;

    /**
     * List of all pack progress bars
     */
    protected List<PackProgressPanel> packProgressPanels;
    private int curPackNo = 0; // the current pack we're on, for accessing the list

    /**
     * Are we using multi progress bars?
     */
    private boolean useMultiProgress;

    /**
     * The progress bar.
     */
    protected JProgressBar overallProgressBar;

    /**
     * True if the installation has been done.
     */
    private volatile boolean validated = false;

    /**
     * How many packs we are going to install.
     */
    private int noOfPacks = 0;

    /**
     * The constructor.
     *
     * @param parent The parent window.
     * @param idata  The installation data.
     */
    public InstallPanel(InstallerFrame parent, InstallData idata) {
        super(parent, idata, new GridBagLayout());
        add(createTitle(), GridBagConstraintsFactory.getTitleConstraints());

        GridBagConstraints frameConstraints = GridBagConstraintsFactory.getBasicConstraints();
        frameConstraints.gridy = 1;
        frameConstraints.fill = GridBagConstraints.HORIZONTAL;
        add(createInstallingPanel(), frameConstraints);
        frameConstraints.gridy++;

        useMultiProgress = idata.guiPrefs.modifier.containsKey("useMultiProgressBar") &&
                "yes".equals(idata.guiPrefs.modifier.get("useMultiProgressBar"));
        // if the user indicates that they want multi progress bars, we do this!
        if (useMultiProgress) {
            add(createMultiProgressBarPanel(), frameConstraints);
        } else {
            add(createProgressBarPanel(), frameConstraints);
        }
        frameConstraints.gridy++;
        frameConstraints.weighty = 1;
        frameConstraints.weightx = 1;
        frameConstraints.fill = GridBagConstraints.BOTH;
        add(Box.createGlue(), frameConstraints);
    }

    private JPanel createProgressBarPanel() {
        JPanel panel = new JPanel(new GridBagLayout());
        GridBagConstraints constraints = GridBagConstraintsFactory.getBasicNoInsetsConstraints();
        constraints.fill = GridBagConstraints.HORIZONTAL;

        packProgressBar = new JProgressBar();
        packProgressBar.setStringPainted(true);
        packProgressBar.setString(parent.langpack.getString("InstallPanel.begin"));
        packProgressBar.setValue(0);
        panel.add(packProgressBar, constraints);
        // make sure there is some space between the progress bars
        panel.add(IzPanelLayout.createVerticalStrut(5));
        overallOpLabel = LabelFactory.create(parent.langpack.getString("InstallPanel.progress"),
                parent.icons.getImageIcon(iconName), LEADING);
        constraints.gridy++;
        panel.add(this.overallOpLabel, constraints);

        overallProgressBar = new JProgressBar();
        overallProgressBar.setStringPainted(true);
        if (noOfPacks == 1) {
            overallProgressBar.setIndeterminate(true);
        }
        overallProgressBar.setString("");
        overallProgressBar.setValue(0);
        constraints.gridy++;
        panel.add(this.overallProgressBar, constraints);
        return panel;
    }

    private JPanel createMultiProgressBarPanel() {
        JPanel panel = new JPanel(new GridBagLayout());
        GridBagConstraints constraints = GridBagConstraintsFactory.getBasicNoInsetsConstraints();
        constraints.fill = GridBagConstraints.BOTH;
        noOfPacks = idata.availablePacks.size();
        packProgressPanels = new ArrayList<PackProgressPanel>();

        for (int i = 0; i < noOfPacks; i++) {
            constraints.insets = new Insets(3, 0, 0, 0);
            PackProgressPanel toAdd = new PackProgressPanel();
            packProgressPanels.add(toAdd);
            panel.add(toAdd, constraints);
            constraints.gridy++;
            toAdd.setVisible(false);
        }
        constraints.weightx = 1;
        constraints.weighty = 1;
        panel.add(Box.createGlue(), constraints);
        return panel;
    }

    private JLabel createTitle() {
        return LabelFactory.createTitleLabel(idata.langpack.getString("InstallPanel.title"), !parent.hasBackground);
    }

    private JPanel createInstallingPanel() {
        JPanel panel = new JPanel(new GridBagLayout());
        packOpLabelPrefix = idata.langpack.getString("InstallPanel.currentlyInstalling") + "%s";
        packOpLabel = LabelFactory.create(String.format(packOpLabelPrefix, ""), LEADING);
        packOpLabel.setFont(FontResources.getOpenSansRegular());

        GridBagConstraints constraints = GridBagConstraintsFactory.getBasicNoInsetsConstraints();
        constraints.weightx = 1.0;
        constraints.fill = GridBagConstraints.BOTH;
        panel.add(packOpLabel, constraints);
        return panel;
    }

    /**
     * Indicates wether the panel has been validated or not.
     *
     * @return The validation state.
     */
    public boolean isValidated() {
        return this.validated;
    }

    /**
     * The unpacker starts.
     */
    public void startAction(String name, int noOfJobs) {
        this.noOfPacks = noOfJobs;
        Iterator<?> iter = idata.panels.iterator();
        IzPanel panel;

        while (iter.hasNext()) {
            panel = (IzPanel) iter.next();
            if ((panel.getSummaryBody()) != null) {
                panel.setView();
            }
        }
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                parent.blockGUI();

                // figure out how many packs there are to install
                if (!useMultiProgress) {
                    overallProgressBar.setMinimum(0);
                    overallProgressBar.setMaximum(noOfPacks);
                    overallProgressBar.setString("0 / " + Integer.toString(noOfPacks));
                }
            }
        });
    }

    /**
     * An error was encountered.
     *
     * @param error The error text.
     */
    public void emitError(String title, String error) {
        this.packOpLabel.setText(error);
        idata.installSuccess = false;
        super.emitError(title, error);
    }

    /**
     * The unpacker stops.
     * Set all previous panels as "viewed by summary panel"
     */
    public void stopAction() {
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                parent.releaseGUI();
                parent.lockPrevButton();

                // With custom actions it is possible, that the current value
                // is not max - 1. Therefore we use always max for both
                // progress bars to signal finish state.
                // Tom: assumption is, if we got here, there weren't any failures. so we can mark everything as done ;)

                if (useMultiProgress) {
                    for (PackProgressPanel panel : packProgressPanels) {
                    }
                    packOpLabel.setText(String.format("<html>%s</html>", parent.langpack.getString(INSTALL_FINISHED)));
                } else {
                    overallProgressBar.setValue(overallProgressBar.getMaximum());
                    int ppbMax = packProgressBar.getMaximum();
                    if (ppbMax < 1) {
                        ppbMax = 1;
                        packProgressBar.setMaximum(ppbMax);
                    }
                    packProgressBar.setValue(ppbMax);

                    packProgressBar.setString(parent.langpack.getString(INSTALL_FINISHED));
                    packProgressBar.setEnabled(false);
                    String no_of_packs = Integer.toString(noOfPacks);
                    if (noOfPacks == 1) {
                        overallProgressBar.setIndeterminate(false);
                    }
                    overallProgressBar.setString(no_of_packs + " / " + no_of_packs);
                    overallProgressBar.setEnabled(false);
                    packOpLabel.setText(String.format(packOpLabelPrefix, " "));
                    packOpLabel.setEnabled(false);
                }
                idata.canClose = true;
                validated = true;
                if (idata.panels.indexOf(this) != (idata.panels.size() - 1)) {
                    parent.unlockNextButton();
                }
            }
        });
    }

    public String wrapText(String packName, int size) {
        return "<html><div style=\"width:"+size+"px;\">"+packName+"</div></html>";
    }

    /**
     * Normal progress indicator.
     *
     * @param val The progression value.
     * @param msg The progression message.
     */
    public void progress(final int val, final String msg) {
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                if (useMultiProgress) {
                    if (!(curPackNo >= packProgressPanels.size())) {
                        PackProgressPanel panel = packProgressPanels.get(curPackNo);
                        JProgressBar bar = panel.getProgressBar();
                        bar.setValue(val + 1);
                        JLabel percentage = panel.getPercentLabel();
                        int percentComplete = (int) (bar.getPercentComplete() * 100);
                        percentage.setForeground(Color.BLACK);
                        percentage.setText(percentComplete + "%");
                        if (percentComplete == 100) {
                            panel.setComplete();
                        }
                    }
                } else {
                    packProgressBar.setValue(val + 1);
                }
                String installPath = idata.getInstallPath()+ File.separator;
                String finalMsg = "";
                if (msg.startsWith(installPath))
                    finalMsg = msg.replace(installPath, "");
                packOpLabel.setText(String.format(packOpLabelPrefix, finalMsg));
            }
        });

    }

    /**
     * Pack changing.
     *
     * @param packName The pack name.
     * @param stepno   The number of the pack.
     * @param max      The new maximum progress.
     */
    public void nextStep(final String packName, final int stepno, final int max) {
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                if (useMultiProgress) {
                    curPackNo = stepno - 1;
                    if (!(curPackNo >= packProgressPanels.size())) {
                        PackProgressPanel panel = packProgressPanels.get(curPackNo);
                        panel.getPackLabel().setText(wrapText(packName, 150));

                        JProgressBar bar = panel.getProgressBar();
                        if (max == 0) { //Hack to get a 0/0 progress bar to get filled up
                            panel.setComplete();
                        }
                        bar.setMaximum(max);
                        bar.setValue(0);
                    }

                } else {
                    packProgressBar.setValue(0);
                    packProgressBar.setMinimum(0);
                    packProgressBar.setMaximum(max);
                    packProgressBar.setString(packName);
                    overallProgressBar.setValue(stepno - 1);
                    overallProgressBar.setString(Integer.toString(stepno) + " / "
                            + Integer.toString(noOfPacks));
                }

            }
        });
    }

    /**
     * {@inheritDoc}
     */
    public void setSubStepNo(final int no_of_substeps) {
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                if (useMultiProgress) {
                    PackProgressPanel panel = packProgressPanels.get(curPackNo);
                    panel.getProgressBar().setMaximum(no_of_substeps);
                } else {
                    packProgressBar.setMaximum(no_of_substeps);
                }
            }
        });
    }

    /**
     * Called when the panel becomes active.
     */
    public void panelActivate() {

        noOfPacks = idata.selectedPacks.size();

        if (useMultiProgress) {

            for (int i = 0; i < noOfPacks; i++) {

                packProgressPanels.get(i).setVisible(true);

            }
        } else {
            packProgressBar.setVisible(true);
        }
        // We clip the panel
        Dimension dim = parent.getCenterPanelSize();
        dim.width -= (dim.width / 4);
        dim.height = 150;
        setMinimumSize(dim);
        setMaximumSize(dim);
        setPreferredSize(dim);
        parent.lockNextButton();
        parent.lockPrevButton();
        parent.install(this);

    }

    private class PackProgressPanel extends JPanel {
        JLabel packLabel;
        JProgressBar progressBar;
        JLabel percentLabel;
        JLabel finishLabel;

        public PackProgressPanel() {
            super();
            super.setLayout(new GridBagLayout());
            this.setOpaque(false);
            addPackLabel();
            addProgressBar();
            addFinishLabel();
            addPercentLabel();
        }

        private void addFinishLabel() {
            JLabel finishLabel = new JLabel(FontResources.checkmarkCharacter);
            finishLabel.setFont(FontResources.getFontAwesome().deriveFont(16.0f));
            finishLabel.setForeground(UiResources.checkMarkGreen);
            GridBagConstraints gbc = new GridBagConstraints();
            gbc.gridx = 1;
            gbc.weightx = .65;
            gbc.gridy = 0;
            gbc.anchor = GridBagConstraints.LINE_START;
            gbc.fill = GridBagConstraints.BOTH;
            finishLabel.setVisible(false);
            this.add(finishLabel, gbc);
            this.finishLabel = finishLabel;
        }

        public JLabel getPackLabel() {
            return this.packLabel;
        }

        public void addPackLabel() {
            String waitingLabel = idata.langpack.getString("InstallPanel.waiting");
            waitingLabel = waitingLabel != null ? waitingLabel : "Waiting...";
            JLabel packLabel = new JLabel(waitingLabel);
            packLabel.setFont(FontResources.getOpenSansRegular());
            GridBagConstraints gbc = new GridBagConstraints();
            gbc.gridx = 0;
            gbc.weightx = .5;
            gbc.gridy = 0;
            gbc.anchor = GridBagConstraints.LINE_START;
            gbc.fill = GridBagConstraints.BOTH;
            this.add(packLabel, gbc);
            this.packLabel = packLabel;
        }

        public void addProgressBar() {
            JProgressBar bar = new JProgressBar();

            bar.setUI(new PatternflyProgressBarUI());
            GridBagConstraints gbc = new GridBagConstraints();
            gbc.gridx = 1;
            gbc.gridy = 0;
            gbc.weightx = 5;
            gbc.anchor = GridBagConstraints.LINE_START;
            gbc.fill = GridBagConstraints.HORIZONTAL;
            this.add(bar, gbc);
            this.progressBar = bar;
        }

        public JProgressBar getProgressBar() {
            return this.progressBar;
        }

        public JLabel getPercentLabel() {
            return this.percentLabel;
        }

        public void addPercentLabel() {
            JLabel percentLabel = new JLabel();
            percentLabel.setFont(FontResources.getOpenSansRegular().deriveFont(12.0f));
            percentLabel.setBorder(BorderFactory.createEmptyBorder(0, 5, 0, 0));
            percentLabel.setText("00%");
            percentLabel.setForeground(new Color(0xFFFFFFFF));
            GridBagConstraints gbc = new GridBagConstraints();
            gbc.gridx = 2;
            gbc.gridy = 0;
            gbc.fill = GridBagConstraints.HORIZONTAL;
            gbc.anchor = GridBagConstraints.LINE_START;
            this.add(percentLabel, gbc);
            this.percentLabel = percentLabel;
        }

        public void setComplete() {
            this.progressBar.setVisible(false);
            this.percentLabel.setVisible(false);
            this.finishLabel.setVisible(true);
        }
    }
}
