/**
 * Copyright (C) 2012 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jboss.dashboard.ui;

import org.apache.commons.collections.CollectionUtils;
import org.jboss.dashboard.ui.components.DashboardHandler;
import org.jboss.dashboard.ui.controller.RequestContext;
import org.jboss.dashboard.ui.panel.DashboardDriver;
import org.jboss.dashboard.kpi.KPI;
import org.jboss.dashboard.provider.DataProvider;
import org.jboss.dashboard.provider.DataProperty;
import org.jboss.dashboard.ui.components.DashboardFilterProperty;
import org.jboss.dashboard.domain.Interval;
import org.jboss.dashboard.domain.CompositeInterval;
import org.jboss.dashboard.domain.label.LabelInterval;
import org.jboss.dashboard.domain.date.DateInterval;
import org.jboss.dashboard.domain.numeric.NumericInterval;
import org.jboss.dashboard.commons.events.Publisher;
import java.util.*;

import org.jboss.dashboard.ui.panel.AjaxRefreshManager;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.jboss.dashboard.ui.panel.PanelDriver;
import org.jboss.dashboard.workspace.Panel;
import org.jboss.dashboard.workspace.Section;

/**
 * A dashboard.
 */
public class Dashboard {

    /** Logger */
    protected static transient org.slf4j.Logger log = org.slf4j.LoggerFactory.getLogger(Dashboard.class);

    /**
     * Dashboard section.
     */
    protected Long sectionDbid;

    /**
     * The dashboard filter.
     */
    protected DashboardFilter dashboardFilter;

    /**
     * Parent dashboard for Drill-down.
     */
    protected Dashboard parent;

    /**
     * For child dashboards, the set of property ids used in a parent's drill-down operation
     */
    protected Set<String> drillDownIds = new HashSet<String>();

    /**
     * Dashboard listeners
     */
    protected transient Publisher listeners;

    public Dashboard() {
        sectionDbid = null;
        parent = null;
        listeners = new Publisher();
        setDashboardFilter(new DashboardFilter());
    }

    public int hashCode() {
        return new HashCodeBuilder().append(sectionDbid).toHashCode();
    }

    public boolean equals(Object obj) {
        try {
            if (obj == null) return false;
            if (obj == this) return true;
            if (sectionDbid == null) return false;

            Dashboard other = (Dashboard) obj;
            return sectionDbid.equals(other.sectionDbid);
        }
        catch (ClassCastException e) {
            return false;
        }
    }

    public String toString() {
        return new ToStringBuilder(this).append("sectionDbid", sectionDbid).toString();
    }

    public void addListener(DashboardListener listener) {
        listeners.subscribe(listener);
    }

    public Dashboard getParent() {
        return parent;
    }

    // Parent dashboard must be set using applyDrillDown to avoid infinite parent loop.
    private void setParent(Dashboard parent) {
        this.parent = parent;
    }

    public DashboardFilter getDashboardFilter() {
        return dashboardFilter;
    }

    public void setDashboardFilter(DashboardFilter dashboardFilter) {
        this.dashboardFilter = dashboardFilter;
        if (dashboardFilter != null) this.dashboardFilter.setDashboard(this);
    }

    public Section getSection() {
        try {
            if (sectionDbid == null) return null;
            return UIServices.lookup().getSectionsManager().getSectionByDbId(sectionDbid);
        } catch (Throwable e) {
            log.error("Error getting section: " + sectionDbid, e);
            return null;
        }
    }

    public void setSection(Section newSection) {
        sectionDbid = newSection.getDbid();
    }

    /**
     * Check if a given panel belongs to this dashboard.
     */
    public boolean belongsToDashboard(Panel panel) {
        return panel.getSection().equals(getSection());
    }

    public Set<DataProvider> getDataProviders() {
        Set<DataProvider> results = new HashSet<DataProvider>();
        Section section = getSection();
        if (section == null) return results;

        for (Panel panel : section.getPanels()) {
            KPI kpi = DashboardHandler.lookup().getKPI(panel);

            // The KPI is null if the panel is not assigned to a region.
            if (kpi != null) results.add(kpi.getDataProvider());
        }
        return results;
    }

    /**
     * Search for a data property into the data providers.
     * @return The first data property found or null.
     */
    public DataProperty getDataPropertyById(String propertyId) {
        // Search for the property both in this dashboard and in its parents.
        Dashboard dashboard = this;
        while (dashboard != null) {
            for (DataProvider provider : dashboard.getDataProviders()) {
                try {
                    DataProperty p = provider.getDataSet().getPropertyById(propertyId);
                    if (p != null) return p;
                } catch (Exception e) {
                    log.error("Dashboard provider dataset load: " + provider.getCode(), e);
                }
            }
            // If the property is not found look at the parent dashboard (if exists).
            dashboard = dashboard.getParent();
        }
        return null;
    }

    /**
     * Init the dashboard with the KPIs in the section.
     * <p>All the KPIs and its related data sets are refreshed. The dashboard filter is cleared as well.
     */
    public void init() {
        // Clear the current dashboard filter.
        dashboardFilter.init();

        // Clear drill-down if any.
        parent = null;
        drillDownIds.clear();

        // Populate the dashboard with KPIs data for the first time.
        refresh();
    }

    /**
     * Refresh all the KPI data sets for this dashboard. The current dashboard filter is preserved.
     */
    public void refresh() {
        try {
            // A section instance is required.
            if (sectionDbid == null) return;

            // Reload all the data providers referenced by this dashboard.
            for (DataProvider dataProvider : getDataProviders()) {
                dataProvider.refreshDataSet();
            }

            // Preserve the current filter after refresh (if any)
            if (dashboardFilter.getPropertyIds().length > 0) {
                filter();
            }
            // Refresh all the dashboard panels.
            refreshPanels(null);
        } catch (Exception e) {
            throw new RuntimeException("Filter error after refresh the dashboard.", e);
        }
    }

    public boolean filter(String propertyId, Interval interval, int allowMode) throws Exception {
        // Get min, max and allowed values from interval.
        Object minValue = null; Object maxValue = null; Collection values = null;
        boolean minValueIncluded = true; boolean maxValueIncluded = true;
        if (interval instanceof NumericInterval) {
            minValue = ((NumericInterval)interval).getMinValue();
            minValueIncluded = ((NumericInterval)interval).isMinValueIncluded();
            maxValue = ((NumericInterval)interval).getMaxValue();
            maxValueIncluded = ((NumericInterval)interval).isMaxValueIncluded();
        }

        if (interval instanceof DateInterval) {
            minValue = ((DateInterval)interval).getMinDate();
            minValueIncluded = ((DateInterval)interval).isMinDateIncluded();
            maxValue = ((DateInterval)interval).getMaxDate();
            maxValueIncluded = ((DateInterval)interval).isMaxDateIncluded();
        }

        if (interval instanceof LabelInterval) {
            if (dashboardFilter.getPropertyIds().length == 0) values = Arrays.asList(interval);
            else values = Arrays.asList(((LabelInterval) interval).getLabel());
        }

        if (interval instanceof CompositeInterval) {
            minValue = ((CompositeInterval)interval).getMinValue();
            minValueIncluded = ((CompositeInterval)interval).isMinValueIncluded();
            maxValue = ((CompositeInterval)interval).getMaxValue();
            maxValueIncluded = ((CompositeInterval)interval).isMaxValueIncluded();
            if (minValue == null && maxValue == null) {
                if (dashboardFilter.getPropertyIds().length == 0) values = Arrays.asList(interval);
                else values = new HashSet(interval.getValues(interval.getDomain().getProperty()));
            }
        }

        DashboardFilter filterRequest = new DashboardFilter();
        filterRequest.addProperty(propertyId, minValue, minValueIncluded, maxValue, maxValueIncluded, values, allowMode);
        return filter(filterRequest);
    }

    public boolean filter(DashboardFilter filterRequest) throws Exception {
        Dashboard drillDownTarget = null;
        boolean filterRequired = false;

        for (String propertyId : filterRequest.getPropertyIds()) {
            Object minValue = filterRequest.getPropertyMinValue(propertyId);
            boolean minValueIncluded = filterRequest.minValueIncluded(propertyId);
            Object maxValue = filterRequest.getPropertyMaxValue(propertyId);
            boolean maxValueIncluded = filterRequest.maxValueIncluded(propertyId);
            Collection allowedValues = filterRequest.getPropertyAllowedValues(propertyId);
            int allowMode = filterRequest.getPropertyAllowMode(propertyId);;

            // Get the filter property configuration.
            DashboardFilterProperty dashboardFilterProperty = dashboardFilter.getPropertyInFilterComponents(propertyId);

            // Apply drill-down.
            if (dashboardFilterProperty != null && dashboardFilterProperty.isDrillDownEnabled()) {
                Dashboard targetDashboard = dashboardFilterProperty.getDrillDownDashboard();
                if (targetDashboard.drillDown(this, propertyId)) {
                    DashboardFilter targetFilter = targetDashboard.getDashboardFilter();

                    if (targetDashboard != drillDownTarget) {
                        drillDownTarget = targetDashboard;
                        targetFilter.removeAllProperty();
                    }
                    targetFilter.addProperty(propertyId, minValue, minValueIncluded, maxValue, maxValueIncluded, allowedValues, allowMode);
                }
            }
            // Apply filter only.
            else {
                dashboardFilter.addProperty(propertyId, minValue, minValueIncluded, maxValue, maxValueIncluded, allowedValues, allowMode);
                filterRequired = true;
            }
        }
        // Filter the child
        if (drillDownTarget != null) {
            Dashboard parent = this;
            DashboardFilter childFilter = drillDownTarget.getDashboardFilter();
            while (parent != null) {
                // Inherit the filters set on its ancestors
                childFilter.merge(parent.getDashboardFilter());
                parent = parent.getParent();
            }
            drillDownTarget.filter();
            return true;
        }
        // Filter the current dashboard if requested
        if (filterRequired) {
            filter();
            refreshPanels(dashboardFilter.getPropertyIds());
        }
        return false;
    }

    public boolean unfilter(String propertyId) throws Exception {
        if (dashboardFilter.containsProperty(propertyId)) {
            dashboardFilter.removeProperty(propertyId);
            Dashboard parent = drillUp();
            if (parent != null) {
                parent.filter();
                return true;
            } else {
                filter();
                refreshPanels(new String[]{propertyId});
            }
        }
        return false;
    }

    public boolean unfilter() throws Exception {
        String[] propIds = dashboardFilter.getPropertyIds();
        dashboardFilter.init();
        Dashboard parent = drillUp();
        if (parent != null) {
            parent.filter();
            return true;
        } else {
            refreshPanels(propIds);
            filter();
            return false;
        }
    }

    protected void filter() throws Exception {
        for (Object o : getDataProviders()) {
            DataProvider provider = (DataProvider) o;
            provider.filterDataSet(dashboardFilter);
        }
    }

    protected boolean drillDown(Dashboard parent, String parentPropertyId) {
        if (parent == null || parentPropertyId == null) {
            return false;
        }

        // Check if this dashboard is already in the drill down chain.
        if (!isDrillDownAllowed(parent)) {
            log.warn("Loop detected on drill-down between dashboards. Please review your filter's configuration and avoid dashboards drill-down loops.");
            return false;
        }

        // Set the parent.
        setParent(parent);
        drillDownIds.add(parentPropertyId);

        // Set drill down page.
        NavigationManager.lookup().setCurrentSection(getSection());
        listeners.notifyEvent(DashboardListener.EVENT_DRILL_DOWN, new Object[]{parent, this});
        return true;
    }

    protected Dashboard drillUp() {
        if (drillDownIds.isEmpty()) {
            return null;
        }

        List<String> filterIds = Arrays.asList(getDashboardFilter().getPropertyIds());
        Dashboard parent = this;
        Dashboard firstParent = null;
        while (!parent.drillDownIds.isEmpty()) {
            if (CollectionUtils.intersection(filterIds, parent.drillDownIds).isEmpty()) {
                // Drill-up detected
                firstParent = parent.getParent();
            }
            parent = parent.getParent();
        }
        // No drill-up detected
        if (firstParent == null) {
            return null;
        }

        // Return back to the parent dashboard.
        NavigationManager.lookup().setCurrentSection(firstParent.getSection());
        listeners.notifyEvent(DashboardListener.EVENT_DRILL_UP, new Object[] {firstParent, this});
        return firstParent;
    }

    /**
     * Check if drill down from the specified page to this one is allowed.
     * @param fromParent The parent dasboard we are coming.
     */
    public boolean isDrillDownAllowed(Dashboard fromParent) {
        if (fromParent == null) return true;
        if (this.equals(fromParent)) return false;

        return isDrillDownAllowed(fromParent.getParent());
    }

    /**
     * Refresh those dashboard panels that are related with any of the properties specified.
     * @param propertySet If null then refresh all the dashboard panels.
     */
    protected void refreshPanels(String[] propertySet) throws Exception {
        AjaxRefreshManager ajaxMgr = AjaxRefreshManager.lookup();
        List<Long> panelIdsToRefresh = ajaxMgr.getPanelIdsToRefresh();
        panelIdsToRefresh.clear();

        // Inspect all the dashboard's panels.
        RequestContext requestCtx = RequestContext.lookup();
        if (requestCtx != null) {
            Panel currentPanel = RequestContext.lookup().getActivePanel();
            for (Panel panel : getSection().getPanels()) {

                // Leave out non dashboard related panels.
                PanelDriver driver = panel.getProvider().getDriver();
                if (!(driver instanceof DashboardDriver)) {
                    continue;
                }
                // Don't refresh the active panel as it's being updated already along the execution of this request.
                Long panelId = panel.getPanelId();
                if (currentPanel != null && currentPanel.getPanelId().equals(panelId)) {
                    continue;
                }
                // Don't refresh panels that are not displaying any dashboard data.
                Set<String> propRefs = ((DashboardDriver) driver).getPropertiesReferenced(panel);
                if (propRefs.isEmpty()) {
                    continue;
                }
                // Mark panel as refreshable.
                if (propertySet == null) {
                    panelIdsToRefresh.add(panelId);
                } else {
                    for (String propertyId : propertySet) {
                        if (!panelIdsToRefresh.contains(panelId) && propRefs.contains(propertyId)) {
                            panelIdsToRefresh.add(panelId);
                        }
                    }
                }
            }
        }
    }
}
