/*
 * Copyright 2011 JBoss, by Red Hat, Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.errai.databinding.client.test;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.jboss.errai.databinding.client.BindableProxy;
import org.jboss.errai.databinding.client.InvalidPropertyExpressionException;
import org.jboss.errai.databinding.client.MockHandler;
import org.jboss.errai.databinding.client.TestModel;
import org.jboss.errai.databinding.client.TestModelWithList;
import org.jboss.errai.databinding.client.TestModelWithListWidget;
import org.jboss.errai.databinding.client.api.Convert;
import org.jboss.errai.databinding.client.api.DataBinder;
import org.jboss.errai.databinding.client.api.InitialState;
import org.jboss.errai.databinding.client.api.PropertyChangeEvent;
import org.jboss.errai.databinding.client.api.PropertyChangeHandler;
import org.jboss.errai.ioc.client.test.AbstractErraiIOCTest;
import org.jboss.errai.marshalling.client.api.MarshallerFramework;
import org.junit.Test;

import com.google.gwt.user.client.ui.TextBox;

/**
 * Tests the functionality provided by the {@link DataBinder} API for property change events.
 * 
 * @author Christian Sadilek <csadilek@redhat.com>
 */
@SuppressWarnings("unchecked")
public class PropertyChangeHandlerIntegrationTest extends AbstractErraiIOCTest {

  @Override
  public String getModuleName() {
    return "org.jboss.errai.databinding.DataBindingTestModule";
  }

  @Override
  protected void gwtSetUp() throws Exception {
    super.gwtSetUp();
    Convert.deregisterDefaultConverters();
    MarshallerFramework.initializeDefaultSessionProvider();
  }

  @Test
  public void testPropertyChangeHandling() {
    MockHandler handler = new MockHandler();

    TextBox textBox = new TextBox();
    DataBinder<TestModel> binder = DataBinder.forType(TestModel.class).bind(textBox, "value");
    binder.addPropertyChangeHandler(handler);

    textBox.setValue("UI change", true);
    assertEquals("Model not properly updated", "UI change", binder.getModel().getValue());
    assertEquals("Should have received exactly one property change event", 1, handler.getEvents().size());
    assertEquals("Wrong property name in event", "value", handler.getEvents().get(0).getPropertyName());
    assertEquals("Wrong property value in event", "UI change", handler.getEvents().get(0).getNewValue());
    assertNull("Previous value should have been null", handler.getEvents().get(0).getOldValue());
    assertEquals("Wrong event source", binder.getModel(), handler.getEvents().get(0).getSource());

    // This should not cause additional events to be fired
    binder.setModel(new TestModel(), InitialState.FROM_MODEL);

    binder.getModel().setValue("model change");
    assertEquals("Widget not properly updated", "model change", textBox.getText());
    assertEquals("Should have received exactly two property change events", 2, handler.getEvents().size());
    assertEquals("Wrong property name in event", "value", handler.getEvents().get(1).getPropertyName());
    assertEquals("Wrong property value in event", "model change", handler.getEvents().get(1).getNewValue());
    assertEquals("Wrong previous value in event", null, handler.getEvents().get(1).getOldValue());
    assertEquals("Wrong event source", binder.getModel(), handler.getEvents().get(1).getSource());
  }

  @Test
  public void testPropertyChangeHandlingWithPropertyChain() {
    MockHandler handler = new MockHandler();

    TextBox textBox = new TextBox();
    DataBinder<TestModel> binder = DataBinder.forType(TestModel.class).bind(textBox, "child.child.value");
    binder.addPropertyChangeHandler("child.child.value", handler);

    textBox.setValue("UI change", true);
    assertEquals("Model not properly updated", "UI change", binder.getModel().getChild().getChild().getValue());
    assertEquals("Should have received exactly one property change event", 1, handler.getEvents().size());
    assertEquals("Wrong property name in event", "value", handler.getEvents().get(0).getPropertyName());
    assertEquals("Wrong property value in event", "UI change", handler.getEvents().get(0).getNewValue());
    assertNull("Previous value should have been null", handler.getEvents().get(0).getOldValue());
    assertEquals("Wrong event source", binder.getModel().getChild().getChild(),
        handler.getEvents().get(0).getSource());
    
    binder.getModel().getChild().getChild().setValue("model change");
    assertEquals("Widget not properly updated", "model change", textBox.getText());
    assertEquals("Should have received exactly two property change events", 2, handler.getEvents().size());
    assertEquals("Wrong property name in event", "value", handler.getEvents().get(1).getPropertyName());
    assertEquals("Wrong property value in event", "model change", handler.getEvents().get(1).getNewValue());
    assertEquals("Wrong previous value in event", "UI change", handler.getEvents().get(1).getOldValue());
    assertEquals("Wrong event source", binder.getModel().getChild().getChild(), 
        handler.getEvents().get(1).getSource());

    binder.removePropertyChangeHandler("child.child.value", handler);
    textBox.setValue("UI change 2", true);
    assertEquals("Should have received no additional event", 2, handler.getEvents().size());
  }
  
  @Test
  public void testPropertyChangeHandlingWithPropertyChainAndRootInstanceChange() {
    MockHandler childHandler = new MockHandler();
    MockHandler valueHandler = new MockHandler();

    TextBox textBox = new TextBox();
    DataBinder<TestModel> binder = DataBinder.forType(TestModel.class).bind(textBox, "child.value");
    binder.addPropertyChangeHandler("child", childHandler);
    binder.addPropertyChangeHandler("child.value", valueHandler);
    
    TestModel oldChild = binder.getModel().getChild();
    TestModel newChild = new TestModel("model change");
    binder.getModel().setChild(newChild);
    
    assertEquals("Widget not properly updated", "model change", textBox.getText());
    assertEquals("Should have received exactly one property change event", 1, childHandler.getEvents().size());
    assertEquals("Wrong property name in event", "child", childHandler.getEvents().get(0).getPropertyName());
    assertEquals("Wrong property value in event", newChild, childHandler.getEvents().get(0).getNewValue());
    assertEquals("Wrong previous value in event", oldChild, childHandler.getEvents().get(0).getOldValue());
    assertEquals("Wrong event source", binder.getModel(), childHandler.getEvents().get(0).getSource());
    
    assertEquals("Should have received exactly one property change event", 1, valueHandler.getEvents().size());
    assertEquals("Wrong property name in event", "value", valueHandler.getEvents().get(0).getPropertyName());
    assertEquals("Wrong property value in event", "model change", valueHandler.getEvents().get(0).getNewValue());
    assertEquals("Wrong previous value in event", null, valueHandler.getEvents().get(0).getOldValue());
    assertEquals("Wrong event source", binder.getModel().getChild(), valueHandler.getEvents().get(0).getSource());
  }
  
  @Test
  public void testPropertyChangeHandlingWithPropertyChainAndRootInstanceChangeOfTwoLevels() {
    MockHandler childHandler = new MockHandler();
    MockHandler valueHandler = new MockHandler();

    TextBox textBox = new TextBox();
    DataBinder<TestModel> binder = DataBinder.forType(TestModel.class).bind(textBox, "child.child.value");
    binder.addPropertyChangeHandler("child", childHandler);
    binder.addPropertyChangeHandler("child.child.value", valueHandler);
    
    TestModel oldChild = binder.getModel().getChild();
    TestModel newChild = new TestModel();
    newChild.setChild(new TestModel("model change"));
    binder.getModel().setChild(newChild);
    
    assertEquals("Widget not properly updated", "model change", textBox.getText());
    assertEquals("Should have received exactly one property change event", 1, childHandler.getEvents().size());
    assertEquals("Wrong property name in event", "child", childHandler.getEvents().get(0).getPropertyName());
    assertEquals("Wrong property value in event", newChild, childHandler.getEvents().get(0).getNewValue());
    assertEquals("Wrong previous value in event", oldChild, childHandler.getEvents().get(0).getOldValue());
    assertEquals("Wrong event source", binder.getModel(), childHandler.getEvents().get(0).getSource());
    
    assertEquals("Should have received exactly one property change event", 1, valueHandler.getEvents().size());
    assertEquals("Wrong property name in event", "value", valueHandler.getEvents().get(0).getPropertyName());
    assertEquals("Wrong property value in event", "model change", valueHandler.getEvents().get(0).getNewValue());
    assertEquals("Wrong previous value in event", null, valueHandler.getEvents().get(0).getOldValue());
    assertEquals("Wrong event source", binder.getModel().getChild().getChild(), valueHandler.getEvents().get(0).getSource());
  }
  @Test
  public void testPropertyChangeHandlingWithWildcardAndPropertyChain() {
    MockHandler handler = new MockHandler();

    TextBox textBox = new TextBox();
    DataBinder<TestModel> binder = DataBinder.forType(TestModel.class).bind(textBox, "child.child.value");
    binder.addPropertyChangeHandler("child.child.*", handler);

    textBox.setValue("UI change", true);
    assertEquals("Model not properly updated", "UI change", binder.getModel().getChild().getChild().getValue());
    assertEquals("Should have received exactly one property change event", 1, handler.getEvents().size());
    assertEquals("Wrong property name in event", "value", handler.getEvents().get(0).getPropertyName());
    assertEquals("Wrong property value in event", "UI change", handler.getEvents().get(0).getNewValue());
    assertNull("Previous value should have been null", handler.getEvents().get(0).getOldValue());
    assertEquals("Wrong event source", binder.getModel().getChild().getChild(),
        handler.getEvents().get(0).getSource());

    // This should not cause additional events to be fired
    binder.setModel(new TestModel(), InitialState.FROM_MODEL);

    binder.getModel().getChild().getChild().setValue("model change");
    assertEquals("Widget not properly updated", "model change", textBox.getText());
    assertEquals("Should have received exactly two property change events", 2, handler.getEvents().size());
    assertEquals("Wrong property name in event", "value", handler.getEvents().get(1).getPropertyName());
    assertEquals("Wrong property value in event", "model change", handler.getEvents().get(1).getNewValue());
    assertNull("Previous value should have been null", handler.getEvents().get(1).getOldValue());
    assertEquals("Wrong event source", binder.getModel().getChild().getChild(), handler.getEvents().get(1).getSource());
  }

  @Test
  public void testPropertyChangeHandlingOfBoundList() {
    MockHandler handler = new MockHandler();

    TestModelWithListWidget widget = new TestModelWithListWidget();
    
    DataBinder<TestModelWithList> binder = DataBinder.forType(TestModelWithList.class).bind(widget, "list");
    binder.getModel().setList(null);
    binder.addPropertyChangeHandler(handler);

    List<String> list = new ArrayList<String>();
    widget.setValue(list, true);
    assertEquals("Model not properly updated", list, binder.getModel().getList());
    assertEquals("Should have received exactly one property change event", 1, handler.getEvents().size());
    assertEquals("Wrong property name in event", "list", handler.getEvents().get(0).getPropertyName());
    assertEquals("Wrong property value in event", list, handler.getEvents().get(0).getNewValue());
    assertNull("Previous value should have been null", handler.getEvents().get(0).getOldValue());
    assertEquals("Wrong event source", binder.getModel(), handler.getEvents().get(0).getSource());

    list = new ArrayList<String>(Arrays.asList("1"));
    binder.getModel().setList(list);
    assertEquals("Widget not properly updated", list, widget.getValue());
    assertEquals("Should have received exactly two property change events", 2, handler.getEvents().size());
    assertEquals("Wrong property name in event", "list", handler.getEvents().get(1).getPropertyName());
    assertEquals("Wrong property value in event", Arrays.asList("1"), handler.getEvents().get(1).getNewValue());
    assertEquals("Wrong event source", binder.getModel(), handler.getEvents().get(1).getSource());

    list = binder.getModel().getList();
    list.add("2");
    assertEquals("Should have received exactly three property change events", 3, handler.getEvents().size());
    assertEquals("Wrong property name in event", "list", handler.getEvents().get(2).getPropertyName());
    assertEquals("Wrong old property value in event", Arrays.asList("1"), handler.getEvents().get(2).getOldValue());
    assertEquals("Wrong property value in event", Arrays.asList("1", "2"), handler.getEvents().get(2).getNewValue());
    assertEquals("Wrong event source", binder.getModel(), handler.getEvents().get(2).getSource());

    list.remove(1);
    assertEquals("Should have received exactly four property change events", 4, handler.getEvents().size());
    assertEquals("Wrong property name in event", "list", handler.getEvents().get(3).getPropertyName());
    assertEquals("Wrong old property value in event", Arrays.asList("1", "2"), handler.getEvents().get(3).getOldValue());
    assertEquals("Wrong property value in event", Arrays.asList("1"), handler.getEvents().get(3).getNewValue());
    assertEquals("Wrong event source", binder.getModel(), handler.getEvents().get(3).getSource());
  }
  
  @Test
  public void testCascadingPropertyChangeHandlingSetBindingBeforeHandler() {
    MockHandler handler = new MockHandler();

    TextBox textBox = new TextBox();
    DataBinder<TestModel> binder = DataBinder.forType(TestModel.class).bind(textBox, "child.child.value");
    binder.addPropertyChangeHandler("**", handler);

    textBox.setValue("UI change", true);
    assertEquals("Model not properly updated", "UI change", binder.getModel().getChild().getChild().getValue());
    assertEquals("Should have received exactly one property change event", 1, handler.getEvents().size());
    assertEquals("Wrong property name in event", "value", handler.getEvents().get(0).getPropertyName());
    assertEquals("Wrong property value in event", "UI change", handler.getEvents().get(0).getNewValue());
    assertNull("Previous value should have been null", handler.getEvents().get(0).getOldValue());
    assertEquals("Wrong event source", binder.getModel().getChild().getChild(),
        handler.getEvents().get(0).getSource());

    // This should not cause additional events to be fired
    binder.setModel(new TestModel(), InitialState.FROM_MODEL);

    binder.getModel().getChild().getChild().setValue("model change");
    assertEquals("Widget not properly updated", "model change", textBox.getText());
    assertEquals("Should have received exactly two property change events", 2, handler.getEvents().size());
    assertEquals("Wrong property name in event", "value", handler.getEvents().get(1).getPropertyName());
    assertEquals("Wrong property value in event", "model change", handler.getEvents().get(1).getNewValue());
    assertNull("Previous value should have been null", handler.getEvents().get(1).getOldValue());
    assertEquals("Wrong event source", binder.getModel().getChild().getChild(), handler.getEvents().get(1).getSource());
  }

  @Test
  public void testCascadingPropertyChangeHandlingSetHandlerBeforeBinding() {
    MockHandler handler = new MockHandler();

    TextBox textBox = new TextBox();
    DataBinder<TestModel> binder = DataBinder.forType(TestModel.class);
    binder.addPropertyChangeHandler("**", handler);
    binder.bind(textBox, "child.child.value");

    textBox.setValue("UI change", true);
    assertEquals("Model not properly updated", "UI change", binder.getModel().getChild().getChild().getValue());
    assertEquals("Should have received exactly one property change event", 1, handler.getEvents().size());
    assertEquals("Wrong property name in event", "value", handler.getEvents().get(0).getPropertyName());
    assertEquals("Wrong property value in event", "UI change", handler.getEvents().get(0).getNewValue());
    assertNull("Previous value should have been null", handler.getEvents().get(0).getOldValue());
    assertEquals("Wrong event source", binder.getModel().getChild().getChild(),
        handler.getEvents().get(0).getSource());

    // This should not cause additional events to be fired
    binder.setModel(new TestModel(), InitialState.FROM_MODEL);

    binder.getModel().getChild().getChild().setValue("model change");
    assertEquals("Widget not properly updated", "model change", textBox.getText());
    assertEquals("Should have received exactly two property change events", 2, handler.getEvents().size());
    assertEquals("Wrong property name in event", "value", handler.getEvents().get(1).getPropertyName());
    assertEquals("Wrong property value in event", "model change", handler.getEvents().get(1).getNewValue());
    assertNull("Previous value should have been null", handler.getEvents().get(1).getOldValue());
    assertEquals("Wrong event source", binder.getModel().getChild().getChild(), handler.getEvents().get(1).getSource());
  }
  
  @Test
  public void testCascadingPropertyChangeHandlingWithPropertyChain() {
    MockHandler handler = new MockHandler();

    TextBox textBox = new TextBox();
    DataBinder<TestModel> binder = DataBinder.forType(TestModel.class).bind(textBox, "child.child.value");
    binder.addPropertyChangeHandler("child.**", handler);

    textBox.setValue("UI change", true);
    assertEquals("Model not properly updated", "UI change", binder.getModel().getChild().getChild().getValue());
    assertEquals("Should have received exactly one property change event", 1, handler.getEvents().size());
    assertEquals("Wrong property name in event", "value", handler.getEvents().get(0).getPropertyName());
    assertEquals("Wrong property value in event", "UI change", handler.getEvents().get(0).getNewValue());
    assertNull("Previous value should have been null", handler.getEvents().get(0).getOldValue());
    assertEquals("Wrong event source", binder.getModel().getChild().getChild(),
        handler.getEvents().get(0).getSource());

    // This should not cause additional events to be fired
    binder.setModel(new TestModel(), InitialState.FROM_MODEL);

    binder.getModel().getChild().getChild().setValue("model change");
    assertEquals("Widget not properly updated", "model change", textBox.getText());
    assertEquals("Should have received exactly two property change events", 2, handler.getEvents().size());
    assertEquals("Wrong property name in event", "value", handler.getEvents().get(1).getPropertyName());
    assertEquals("Wrong property value in event", "model change", handler.getEvents().get(1).getNewValue());
    assertNull("Previous value should have been null", handler.getEvents().get(1).getOldValue());
    assertEquals("Wrong event source", binder.getModel().getChild().getChild(), handler.getEvents().get(1).getSource());
  }

  @Test
  public void testBinderRetainsPropertyChangeHandlersAfterModelChange() {
    MockHandler handler = new MockHandler();

    TextBox textBox = new TextBox();
    DataBinder<TestModel> binder = DataBinder.forType(TestModel.class).bind(textBox, "value");
    binder.addPropertyChangeHandler(handler);
    binder.addPropertyChangeHandler("value", handler);
    binder.setModel(new TestModel());

    textBox.setValue("UI change", true);
    assertEquals("Model not properly updated", "UI change", binder.getModel().getValue());
    assertEquals("Should have received exactly one property change event", 2, handler.getEvents().size());

    binder.getModel().setValue("model change");
    assertEquals("Widget not properly updated", "model change", textBox.getText());
    assertEquals("Should have received exactly two property change events", 4, handler.getEvents().size());
  }

  @Test
  public void testBinderRetainsPropertyChangeHandlersWithPropertyChainAfterModelChange() {
    MockHandler handler = new MockHandler();

    TextBox textBox = new TextBox();
    DataBinder<TestModel> binder = DataBinder.forType(TestModel.class).bind(textBox, "child.child.value");
    binder.addPropertyChangeHandler("child.child.value", handler);
    binder.setModel(new TestModel());

    textBox.setValue("UI change", true);
    assertEquals("Model not properly updated", "UI change", binder.getModel().getChild().getChild().getValue());
    assertEquals("Should have received exactly one property change event", 1, handler.getEvents().size());

    binder.getModel().getChild().getChild().setValue("model change");
    assertEquals("Widget not properly updated", "model change", textBox.getText());
    assertEquals("Should have received exactly two property change events", 2, handler.getEvents().size());
  }
  
  @Test
  public void testBinderRetainsCascadingPropertyChangeHandlerAfterModelChange() {
    MockHandler handler = new MockHandler();

    TextBox textBox = new TextBox();
    DataBinder<TestModel> binder = DataBinder.forType(TestModel.class).bind(textBox, "child.child.value");
    binder.addPropertyChangeHandler("**", handler);
    binder.setModel(new TestModel());

    textBox.setValue("UI change", true);
    assertEquals("Model not properly updated", "UI change", binder.getModel().getChild().getChild().getValue());
    assertEquals("Should have received exactly one property change event", 1, handler.getEvents().size());

    binder.getModel().getChild().getChild().setValue("model change");
    assertEquals("Widget not properly updated", "model change", textBox.getText());
    assertEquals("Should have received exactly two property change events", 2, handler.getEvents().size());
  }

  @Test
  public void testPropertyChangeEventsAreFiredDuringStateSync() {
    MockHandler handler = new MockHandler();

    TextBox textBox = new TextBox();
    textBox.setValue("UI change");

    DataBinder<TestModel> binder = DataBinder.forType(TestModel.class, InitialState.FROM_UI);
    binder.addPropertyChangeHandler(handler);
    binder.bind(textBox, "value");

    assertEquals("Model not properly updated", "UI change", binder.getModel().getValue());
    assertEquals("Should have received exactly one property change event", 1, handler.getEvents().size());
    assertEquals("Wrong property name in event", "value", handler.getEvents().get(0).getPropertyName());
    assertEquals("Wrong property value in event", "UI change", handler.getEvents().get(0).getNewValue());
  }

  @Test
  public void testNewValueIsSetBeforePropertyChangeEventIsFired() {
    TextBox textBox = new TextBox();
    final DataBinder<TestModel> binder = DataBinder.forType(TestModel.class).bind(textBox, "value");
    binder.getModel().setValue("Old Value");
    class MyHandler implements PropertyChangeHandler<String> {
      String observedValueWhenEventFired;

      @Override
      public void onPropertyChange(PropertyChangeEvent<String> event) {
        observedValueWhenEventFired = binder.getModel().getValue();
      }
    }
    MyHandler handler = new MyHandler();
    binder.addPropertyChangeHandler(handler);

    textBox.setValue("New Value", true);
    assertEquals("New Value", handler.observedValueWhenEventFired);

    binder.getModel().setValue("New New Value");
    assertEquals("New New Value", handler.observedValueWhenEventFired);
  }

  @Test
  public void testPropertyChangeHandlingWithWildcardDoesNotCascade() {
    MockHandler handler = new MockHandler();

    TextBox textBox = new TextBox();
    DataBinder<TestModel> binder = DataBinder.forType(TestModel.class).bind(textBox, "child.child.value");
    binder.addPropertyChangeHandler("child.*", handler);

    textBox.setValue("UI change", true);
    assertEquals("Model not properly updated", "UI change", binder.getModel().getChild().getChild().getValue());
    assertEquals("Should have received no property change events", 0, handler.getEvents().size());

    binder.getModel().getChild().getChild().setValue("model change");
    assertEquals("Widget not properly updated", "model change", textBox.getText());
    assertEquals("Should have received no property change events", 0, handler.getEvents().size());
  }

  @Test
  public void testRemovingOfCascadedPropertyChangeHandling() {
    MockHandler handler = new MockHandler();

    TextBox textBox = new TextBox();
    DataBinder<TestModel> binder = DataBinder.forType(TestModel.class).bind(textBox, "child.child.value");
    binder.addPropertyChangeHandler("child.**", handler);
    binder.addPropertyChangeHandler("child.child.*", handler);

    textBox.setValue("UI change", true);
    assertEquals("Should have received exactly two property change events", 2, handler.getEvents().size());

    // Remove the binders
    binder.removePropertyChangeHandler("child.**", handler);
    binder.removePropertyChangeHandler("child.child.*", handler);

    textBox.setValue("Second UI change", true);
    assertEquals("Should have received no additional event", 2, handler.getEvents().size());
  }

  @Test  
  public void testWildcardFailsIfNotTheEndOfExpression() {
    DataBinder<TestModel> binder = DataBinder.forType(TestModel.class);
    try {
      binder.addPropertyChangeHandler("child.*.xx", new MockHandler());
      fail("Expected InvalidPropertyExpressionException");
    }
    catch(InvalidPropertyExpressionException e) {
      // expected
    }
  }

  @Test  
  public void testDoubleWildcardFailsIfNotTheEndOfExpression() {
    DataBinder<TestModel> binder = DataBinder.forType(TestModel.class);
    try {
      binder.addPropertyChangeHandler("**.xx", new MockHandler());
      fail("Expected InvalidPropertyExpressionException");
    }
    catch(InvalidPropertyExpressionException e) {
      // expected
    }
  }
  
  @Test
  @SuppressWarnings("rawtypes")
  public void testUpdateWidgetsInChangeHandlerDoesNotCauseRecursion() {
    final DataBinder<TestModel> binder = DataBinder.forType(TestModel.class);
    final List<PropertyChangeEvent> events = new ArrayList<PropertyChangeEvent>();
    
    binder.addPropertyChangeHandler("value", new PropertyChangeHandler() {
      @Override
      public void onPropertyChange(PropertyChangeEvent event) {        
        ((BindableProxy) binder.getModel()).updateWidgets();
        events.add(event);
      }
    });
    binder.getModel().setValue("value");
    assertEquals("Should have received exactly one event", 1, events.size());
  }
}