package com.redhat.installer.tests.validator;

import com.izforge.izpack.installer.DataValidator;
import com.redhat.installer.tests.DataValidatorTester;
import com.redhat.installer.util.ValidatorConstants;
import com.redhat.installer.validator.EapExistsValidator;
import org.junit.Before;
import org.junit.Test;

import static com.redhat.installer.tests.TestUtils.*;
import static junit.framework.TestCase.assertEquals;

/**
 * Tests for the EapExistsValidator in installer-commons
 * Created by thauser on 1/29/14.
 */
public class EapExistsValidatorTest extends DataValidatorTester {

    @Before
    public void setUp() throws Exception {
        dv = new EapExistsValidator();
        idata.setVariable("INSTALL_PATH", INSTALL_PATH);
        idata.setVariable("INSTALL_SUBPATH", INSTALL_SUBPATH);
    }

    /* Case: the given install path doesn't exist at all */
    @Test
    public void testNoEapExists() throws Exception {
        assertOk();
        assertEquals("true", idata.getVariable("eap.needs.install"));
        assertEquals(ValidatorConstants.eap, idata.getVariable(ValidatorConstants.existingProduct));
    }

    /* Case: directory exists, but product.conf doesn't exist. */
    @Test
    public void testNoProductConf() throws Exception {
        createMockInstallDir();
        assertError();
        assertEquals(String.format("The required product.conf file at: %s is missing or malformed. Please supply either an empty directory or an EAP installation.", baseTestDir
                + ValidatorConstants.productConfLoc), dv.getErrorMessageId());
        assertEquals("", idata.getVariable(ValidatorConstants.existingProduct));
    }

    /* Case: Valid EAP installation */
    @Test
    public void testEAP() throws Exception {
        createMockInstallDir();
        createProductConf("eap");
        createEAPScripts();
        createVersionTxt("6.1.1.GA");
        assertOk();
        assertEquals("", idata.getVariable(ValidatorConstants.existingLayers));
        assertEquals(ValidatorConstants.eap, idata.getVariable(ValidatorConstants.existingProduct));
    }

    /* Case:
        missing version.txt file
     */
    @Test
    public void testNoVersionTxt() throws Exception {
        createMockInstallDir();
        createProductConf("eap");
        createEAPScripts();
        assertError();
        assertEquals("", idata.getVariable(ValidatorConstants.existingLayers));
        assertEquals(ValidatorConstants.eap, idata.getVariable(ValidatorConstants.existingProduct));
        assertEquals("true", idata.getVariable("installer.has.standalone"));
        assertEquals("true", idata.getVariable("installer.has.domain"));
    }

    /* Case:
        version.txt contains incompatible version
     */
    @Test
    public void testWrongVersionTxt() throws Exception {
        createMockInstallDir();
        createProductConf("eap");
        createEAPScripts();
        createVersionTxt("6.1.0.GA");
        assertError();
        assertEquals("", idata.getVariable(ValidatorConstants.existingLayers));
        assertEquals(ValidatorConstants.eap, idata.getVariable(ValidatorConstants.existingProduct));
        assertEquals("true", idata.getVariable("installer.has.standalone"));
        assertEquals("true", idata.getVariable("installer.has.domain"));
        assertEquals("Invalid EAP version found in version.txt (6.1.0.GA)", dv.getErrorMessageId());
    }

    /* Case:
        missing bin/standalone.sh and standalone/configuration/standalone.xml
     */
    @Test
    public void testNoStandaloneScripts() throws Exception {
        createMockInstallDir();
        createProductConf("eap");
        createNewEmptyFile("/domain/configuration/domain.xml");
        createNewEmptyFile("/domain/configuration/host.xml");
        createNewEmptyFile("/bin/domain.sh");
        createVersionTxt("6.1.1.GA");
        assertWarning();
        assertEquals("Standalone configuration and standalone.sh are missing. Is this really an EAP installation?", dv.getErrorMessageId());
        // JUnit complains that the results aren't equal, but in the test log if you choose
        // to examine the results, it states that they're identical. a little weird.
        //assertEquals("", idata.getVariable("installer.has.standalone"));
    }

    /*  Case:
           missing bin/domain.sh , domain/configuration/host.xml and domain/configuration/domain.xml
     */
    @Test
    public void testNoDomainScripts() throws Exception {
        createMockInstallDir();
        createProductConf("eap");
        createVersionTxt("6.1.1.GA");
        createNewEmptyFile("/standalone/configuration/standalone.xml");
        createNewEmptyFile("/bin/standalone.sh");
        assertWarning();
        assertEquals("Domain configuration and domain.sh are missing. Is this really an EAP installation?", dv.getErrorMessageId());
        // JUnit complains that the results aren't equal, but in the test log if you choose
        // to examine the results, it states that they're identical. a little weird.
        //assertEquals("", idata.getVariable("installer.has.domain"));
    }

    /* Case:
        neither standalone.sh/.xml or domain.sh/.xml + host.xml exist.
     */
    @Test
    public void testNoScripts() throws Exception {
        createProductConf("eap");
        createVersionTxt("6.1.1.GA");
        assertError();
        assertEquals("No server start scripts could be found. Neither domain.sh nor standalone.sh exist in this path.", dv.getErrorMessageId());
    }

    /* Case:
        product.conf contains non-eap platform and layers.conf doesn't exist.
     */
    @Test
    public void testSOANoLayersConf() throws Exception {
        createProductConf("soa");
        assertError();
        assertEquals(ValidatorConstants.soa,idata.getVariable(ValidatorConstants.existingProduct));
        assertEquals("", idata.getVariable(ValidatorConstants.existingLayers));
        assertEquals(String.format("The required layers.conf file at: %s is missing or malformed. Please supply either an empty directory or an EAP installation.", baseTestDir
                + ValidatorConstants.layersConfLoc), dv.getErrorMessageId());
    }

    /* Case:
        product.conf is malformed
     */
    @Test
    public void testEAPBadProductConf() throws Exception {
        createProductConf("");
        assertError();
        assertEquals("", idata.getVariable(ValidatorConstants.existingProduct));
        assertEquals(String.format("The required product.conf file at: %s is missing or malformed. Please supply either an empty directory or an EAP installation.", baseTestDir
                + ValidatorConstants.productConfLoc), dv.getErrorMessageId());
    }

    /* Case:
        non-eap installation layers.conf is malformed
     */
    @Test
    public void testSOABadLayersConf() throws Exception {
        createMockInstallDir();
        createProductConf("soa");
        createLayersConf("");
        assertError();
        assertEquals("", idata.getVariable(ValidatorConstants.existingLayers));
        assertEquals(String.format("The required layers.conf file at: %s is missing or malformed. Please supply either an empty directory or an EAP installation.", baseTestDir
                + ValidatorConstants.layersConfLoc), dv.getErrorMessageId());
    }

    /* Case:
        test comment-skipping of the file reading method
     */
    @Test
    public void testCommentedLineSkip() throws Exception {
        createMockInstallDir();
        createNewFileWithContent("/bin/product.conf", "#this is a comment", "#another comment", "slot=eap");
        createEAPScripts();
        createVersionTxt("6.1.1.GA");
        assertOk();
        assertEquals(ValidatorConstants.eap, idata.getVariable(ValidatorConstants.existingProduct));
    }
}
