package com.redhat.installer.tests.util;

import com.redhat.installer.tests.TestUtils;
import com.redhat.installer.util.KeystoreUtils;
import org.apache.commons.io.FileUtils;
import org.junit.*;

import java.io.File;

import static com.redhat.installer.tests.TestUtils.*;
import static junit.framework.TestCase.assertEquals;
import static junit.framework.TestCase.assertFalse;
import static junit.framework.TestCase.assertTrue;


/**
 * Created by thauser on 2/4/14.
 */
public class KeystoreUtilsTest {
    public static final String absoluteMockPath = System.getProperty("user.dir") + TestUtils.mockKeystorePath;
    @After
    public void tearDown() throws Exception {
        FileUtils.deleteDirectory(TestUtils.INSTALL_PATH_FILE);
    }

    @Test
    public void isValidKeystoreTestValidKeystore() throws Exception{
        createMockKeystore();
        assertEquals(0, KeystoreUtils.isValidKeystore(TestUtils.mockKeystorePath, mockKeystorePassword.toCharArray()));
    }

    @Test
    public void isValidKeystoreTestWrongPassword() throws Exception {
        createMockKeystore();
        assertEquals(1, KeystoreUtils.isValidKeystore(TestUtils.mockKeystorePath, "incorrectpassword".toCharArray()));
    }

    @Test
    public void isValidKeystoreTestFileNotExist() throws Exception {
        assertEquals(2, KeystoreUtils.isValidKeystore(TestUtils.mockKeystorePath, mockKeystorePassword.toCharArray()));
    }

    @Test
    public void isValidKeystoreTestNonJKSKeystore() throws Exception {
        createMockKeystore("jceks");
        assertEquals(0, KeystoreUtils.isValidKeystore(TestUtils.mockKeystorePath, mockKeystorePassword.toCharArray()));
    }

    @Test
    public void testValidFileUriPath() throws Exception {
        createMockKeystore();
        assertEquals(0, KeystoreUtils.isValidKeystore("file://"+System.getProperty("user.dir")+"/"+TestUtils.mockKeystorePath, TestUtils.mockKeystorePassword.toCharArray()));
    }

    @Test
    public void testInvalidFileUriPath() throws Exception {
        assertEquals(2, KeystoreUtils.isValidKeystore("file://"+System.getProperty("user.dir")+"/"+TestUtils.mockKeystorePath, TestUtils.mockKeystorePassword.toCharArray()));
    }

    @Test
    public void testMalformedFileUriPath() throws Exception {
        assertEquals(2, KeystoreUtils.isValidKeystore(TestUtils.mockKeystorePath, TestUtils.mockKeystorePassword.toCharArray()));
    }

    // since we also allow straight up File paths, this never gets caught. So we should be ok here
    @Test
    public void testNotAbsoluteFileUriPath() throws Exception {
        assertEquals(2, KeystoreUtils.isValidKeystore("file://"+TestUtils.mockKeystorePath, TestUtils.mockKeystorePassword.toCharArray()));
    }

    @Test
    public void testNoEncodingCharsUrlPath() throws Exception {
        assertEquals(4, KeystoreUtils.isValidKeystore("http://www.google.com/needs^encoding", TestUtils.mockKeystorePassword.toCharArray()));
    }

    @Test
    public void testEncodedInvalidUrlPath() throws Exception {
        assertEquals(2, KeystoreUtils.isValidKeystore("http://www.google.com/needs%5Eencoding", TestUtils.mockKeystorePassword.toCharArray()));
    }

    @Test
    public void isValidReadableFileTestValidFile() throws Exception {
        String filename = "testfile.txt";
        String path = TestUtils.baseTestDir + "/" + filename;
        createNewEmptyFile(filename);
        assertTrue(KeystoreUtils.isValidReadableFile(path));
    }

    @Test
    public void isValidReadableFileTestFileNotExist() throws Exception {
        String path = TestUtils.baseTestDir+"/non-existing-file.txt";
        assertFalse(KeystoreUtils.isValidReadableFile(path));
    }

    @Test
    public void isValidReadableFileTestDirectory() throws Exception {
        createMockInstallDir();
        assertFalse(KeystoreUtils.isValidReadableFile(TestUtils.baseTestDir));
    }

    @Test
    public void isValidReadableFileTestFileNotReadable() throws Exception{
        String filename = "/unreadable-file.txt";
        String path = TestUtils.baseTestDir + filename;
        createNewUnreadableFile("/unreadable-file.txt");
        assertFalse(KeystoreUtils.isValidReadableFile(path));
        revertUnreadableFile("/unreadable-file.txt");
    }

    @Test
    public void isValidAccessibleUrlTestValidUrl() throws Exception {
        assertTrue(KeystoreUtils.isValidAccessibleUrl("http://www.google.com"));
    }

    @Test
    public void isValidAccessibleUrlTestInvalidUrl() throws Exception {
        assertFalse(KeystoreUtils.isValidAccessibleUrl("http://www.thisshouldntexisttt.com/thefilehere.jar"));
    }
    //throws IllegalArgumentException?
    //TODO: fix code to make this test succeed.
/*    @Test
    public void isValidAccessibleUrlTestInvalidProtocol() throws Exception {
        assertFalse(KeystoreUtils.isValidAccessibleUrl("http::::://www.google.com"));
    }*/
}
