package com.redhat.installer.validator;
import java.util.Properties;

import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.installer.DataValidator;
import com.redhat.installer.util.KeystoreUtils;

/**
 *  Validates the security domain JSSE element panel.
 *  If user selects to add a JSSE element, then this validator will 
 *  check that either situation is true:
 *  1) securitydomain.keystore.password != null && securitydomain.keystore.url != null
 *      OR
 *  2) securitydomain.truststore.password != null && securitydomain.truststore.url != null
 *        If both of these are false, the user is prompted to change their input.
 * @author fcanas
 */
public class JSSEValidator implements DataValidator
{
    String error;
    
    public Status validateData(AutomatedInstallData idata) {
        Properties variables = idata.getVariables();
        String isKeyStore = (String) variables.get("securityDomainJsseAddKeystore");
        String isTrustStore = (String) variables.get("securityDomainJsseAddTruststore");
        String keyPwd = (String)variables.get("securitydomain.jsse.keystore.password");
        String keyUrl = (String)variables.get("securitydomain.jsse.keystore.url");
        String storePwd = (String)variables.get("securitydomain.jsse.truststore.password");
        String storeUrl = (String)variables.get("securitydomain.jsse.truststore.url");
        Status keyStoreStatus = Status.OK;
        Status trustStoreStatus = Status.OK;
        
        if (idata.getVariable("securityDomainAddJsse").contains("false")) {
            return Status.OK;
        }
        
        // If user selected the keystore, validate their keystore input.
        if (isKeyStore.contains("true")) {
            switch(validateKeystore(keyPwd, keyUrl, idata)){
                case 0:
                    keyStoreStatus = Status.OK;
                    break;
                case 1:
	            	error = idata.langpack.getString("securitydomain.jsse.keystore.passincorrect");
                    keyStoreStatus = Status.WARNING;
                    break;
                case 2:
	            	error = idata.langpack.getString("securitydomain.jsse.keystore.inaccessible");
                    keyStoreStatus = Status.WARNING;
                    break;
				case 3:
					error = idata.langpack.getString("securitydomain.jsse.keystore.invalid");
					keyStoreStatus = Status.WARNING;
                    break;
                case 4:
                    error = idata.langpack.getString("securitydomain.jsse.keystore.encoding");
                    keyStoreStatus = Status.ERROR;
                    break;
                case 5:
                    error = idata.langpack.getString("securitydomain.jsse.keystore.absolute");
                    keyStoreStatus = Status.ERROR;
                    break;
                case 6:
                    error = idata.langpack.getString("securitydomain.jsse.keystore.reqs");
                    keyStoreStatus = Status.ERROR;
                    break;
            }
        }
        // If user selected the truststore, validate their truststore input.
        if (isTrustStore.contains("true")) {
            switch (validateKeystore(storePwd, storeUrl, idata)) {
                case 0:
                    trustStoreStatus = Status.OK;
                    break;
                case 1:
            	error = idata.langpack.getString("securitydomain.jsse.truststore.passincorrect");
                    trustStoreStatus = Status.WARNING;
                    break;
                case 2:
            	error = idata.langpack.getString("securitydomain.jsse.truststore.inaccessible");
                    trustStoreStatus = Status.WARNING;
                    break;
                case 3:
					error = idata.langpack.getString("securitydomain.jsse.keystore.invalid");
					trustStoreStatus = Status.WARNING;
					break;
				case 4:
					error = idata.langpack.getString("securitydomain.jsse.truststore.encoding");
                    trustStoreStatus = Status.ERROR;
                    break;
                case 5:
                    error = idata.langpack.getString("securitydomain.jsse.truststore.absolute");
                    trustStoreStatus = Status.WARNING;
                    break;
                case 6:
                    error = idata.langpack.getString("securitydomain.jsse.truststore.reqs");
                    trustStoreStatus = Status.ERROR;
                    break;
            }
        }
        
        if (!(isTrustStore.contains("true") || isKeyStore.contains("true")))
        {
	        error = idata.langpack.getString("securitydomain.jsse.requirements");
	        return Status.ERROR;
        }
        
        if (keyStoreStatus != Status.OK){
            return keyStoreStatus;
        }

        if (trustStoreStatus != Status.OK){
            return trustStoreStatus;
        }
        return Status.OK;
    }
    
    /**
     * Returns true if any string in the parameter list is either null or empty
     * @param values
     * @return
     */
    private boolean nullOrEmpty(String ... values)
    {
    	boolean anyFailure = false;
    	for (String value : values){
    		if (value == null || value.isEmpty()){
    			anyFailure = true;
    		}
    	}
        return anyFailure;
    }

    private int validateKeystore(String keyPwd, String keyUrl, AutomatedInstallData idata){
        if (nullOrEmpty(keyPwd, keyUrl)){
            error = idata.langpack.getString("securitydomain.jsse.keystore.reqs");
            return 6;
        }
        return KeystoreUtils.isValidKeystore(keyUrl, keyPwd.toCharArray());
    }

    public String getErrorMessageId() 
    {
        return error;
    }

    public String getWarningMessageId() 
    {
        return error;
    }

    public boolean getDefaultAnswer() 
    {
        return false;
    }

}

