package com.redhat.installer.util;

import com.izforge.izpack.util.Debug;
import com.izforge.izpack.util.Log;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.*;
import java.security.*;
import java.security.cert.CertificateException;

public class KeystoreUtils {
	/**
	 * This method checks that a given keystore<br/>
	 * a) exists (file) or is accessible (remote)<br/>
	 * b) the provided password is correct for the keystore<br/>
	 * Valid return values:<br/>
	 * 0 : everything is fine <br/>
	 * 1 : authentication failure<br/>
	 * 2 : file doesn't exist / hostname not accessible
     * 3 : the JVM doesn't have a provider that can read the keystore
     * 4 : the given URL contains characters that require encoding and does not do this correctly
     * 5 : the given URI is not absolute
	 * @param keystoreUrl
	 * @param pwd
	 * @return
	 */
	
	public static int isValidKeystore(String keystoreUrl, char[] pwd) {
		boolean notFile = false;

		if ((notFile = !isValidReadableFile(keystoreUrl)) && !isValidAccessibleUrl(keystoreUrl)){
			//setError("The given location doesn't seem to be valid.");
			return 2;
		}

        int status = 0;
		KeyStore ks = null;
        for (String algorithm : Security.getAlgorithms("KeyStore")) {
            try {
                ks = KeyStore.getInstance(algorithm);
                if (notFile){
                    ks.load(new URI(keystoreUrl).toURL().openStream(), pwd);
                } else {
                    ks.load(new FileInputStream(new File(keystoreUrl)), pwd);
                }
                // found one that worked. short circuit
                status = 0;
                break;
            } catch (IllegalArgumentException iae){
                // assume that this exception indicates that the URI is not absolute. Indeed, docs: http://docs.oracle.com/javase/7/docs/api/java/net/URI.html
                // indicate that URI.toURL() will throws this only in this case.
                // currently, previous checks seem to make this impossible. still accounted for in usages though
                return 5;
                // it is safe to short circuit here, since an invalid URI will never load successfully.
            } catch (NoSuchAlgorithmException e) {
                // may occur if the user is using a bizarre JRE that doesn't include JavaKeyStore format
                //e.printStackTrace();
                status = 3;
            } catch (CertificateException e) {
                //e.printStackTrace();
                status = 3;
            } catch (FileNotFoundException e) {
                // If connection is legit, but file is not accessible/doesn't exist at remote location.
                status = 2;
            } catch (IOException e) {
                //setError("Keystore password is incorrect.");
                //e.printStackTrace();
                status = 1;
                // one of the IOExceptions is UnrecoverableKeyException, which is not all that specific for us.
                // i had originally had in a check, but i realized it was of little value
/*			Class<? extends Throwable> cause = e.getCause().getClass();
            if (cause.isAssignableFrom(UnrecoverableKeyException.class)){
				// assume password is wrong?
				setError("Keystore password is incorrect.");
				return false;
			} else {
				e.printStackTrace();
			}*/

            } catch (KeyStoreException e) {
                // this shouldn't really happen ever
                // if this is thrown, again, the JRE doesn't include a Provider that can provide a KeyStore instance of type "JKS", which
                // means that either the JRE is very old or is non-standard in some critical way
                status = 3;
            } catch (URISyntaxException e) {
                // some values weren't encoded on in the URI, so we give a better message to users
                status = 4;
            }
        }
        return status;
	}

	
	/**
	 * Determines if a given file exists, can be read, and isn't a directory
	 * TODO: This could probably be moved to another util class to be utilized by other classes
	 * @param file
	 * @return
	 */
	public static boolean isValidReadableFile(String file) {
		File check = new File(file);
		if (check.exists() && check.canRead() && !check.isDirectory()){
			return true;
		} else {
			return false;	
		}
	}

	/**
	 * Determines if a given URL is both valid and accessible
	 * TODO: this could probably be moved to another util class and utilized elsewhere
	 * @param url
	 * @return
	 */
	public static boolean isValidAccessibleUrl(String url) {
		URL check;
		try {
			check = new URL(url);
            URLConnection conn = check.openConnection();
            conn.connect();
		} catch (MalformedURLException e1) {
            // bad URL
			return false;
		} catch (IOException e) {
            // the connection couldn't be established
            return false;
        }
        return true;
	}
}
