package com.redhat.installer.action;

import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.installer.PanelAction;
import com.izforge.izpack.installer.PanelActionConfiguration;
import com.izforge.izpack.util.AbstractUIHandler;
import com.redhat.installer.util.JBossJDBCConstants;
import com.redhat.installer.util.JDBCConnectionUtils;
import com.redhat.installer.util.PreExistingConfigurationConstants;
import org.apache.commons.io.FileUtils;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;
import org.jsoup.select.Elements;

import java.io.File;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

/**
 * Note that this class can only search default descriptors.
 */
public class SetPreExistingDefaults implements PanelAction {
    private HashSet<File> moduleSubdirs = new HashSet<File>();

    private Set<String> securityDomainNames = new HashSet<String>();

    private Set<String> ldapRealmNames = new HashSet<String>();

    private Set<String> ldapConnNames = new HashSet<String>();

    private Set<String> sslCertificates = new HashSet<String>();

    private AutomatedInstallData idata;

    @Override
    public void executeAction(AutomatedInstallData idata, AbstractUIHandler handler) {
        // only if validation is passed will we reach here. Thus, we know the
        // path is valid.
        this.idata = idata;
        boolean needInstall = Boolean.parseBoolean(idata.getVariable("eap.needs.install"));

        resetDefaults();
        if (needInstall) {
            // nothing to do, thankfully
            return;
        } else {
            // for every xml descriptor, we need to find out the existing information
            for (String descriptor : PreExistingConfigurationConstants.standaloneDescriptors) {
                String path = idata.getVariable("INSTALL_PATH") + "/" + idata.getVariable("INSTALL_SUBPATH") + "/standalone/configuration/" + descriptor;
                File descFile = new File(path);
                if (descFile.exists()) {
                    findDefaultInfo(descriptor, descFile);
                }
            }
            for (String descriptor : PreExistingConfigurationConstants.domainDescriptors) {
                String path = idata.getVariable("INSTALL_PATH") + "/" + idata.getVariable("INSTALL_SUBPATH") + "/domain/configuration/" + descriptor;
                File descFile = new File(path);
                if (descFile.exists()) {
                    findDefaultInfo(descriptor, descFile);
                }
            }
        }
    }

    /**
     * This needs to revert all variables set anywhere in this class, since the user can specify a new directory
     * between each invocation
     */
    private void resetDefaults() {
        idata.setVariable("vault.keystoreloc", idata.getVariable("vault.keystoreloc.default"));
        idata.setVariable("vault.encrdir", idata.getVariable("vault.encrdir.default"));
        idata.setVariable("vault.preexisting", "false");
        idata.setVariable("securitydomain.preexisting.names", "other,jboss-web-policy,jboss-ejb-policy,");
        idata.setVariable("ldap.preexisting.realm.names", "");
        idata.setVariable("ldap.preexisting.conn.names", "");
        idata.setVariable("jdbc.driver.preexisting", "false");
        for (String xml : PreExistingConfigurationConstants.standaloneDescriptors) {
            idata.setVariable(xml + ".vault.preexisting", "false");
            for (String extension : PreExistingConfigurationConstants.collidingExtensions) {
                idata.setVariable(xml + "." + extension + ".extension.exists", "false");
            }
            for (String subsystem : PreExistingConfigurationConstants.collidingSubsystems) {
                idata.setVariable(xml + "." + subsystem + ".subsystem.exists", "false");
            }
            for (String infinispan : PreExistingConfigurationConstants.collidingInfinispan) {
                idata.setVariable(xml + "." + infinispan + ".infinispan.exists", "false");
            }
        }
        // different possible subdirs of where modules can be found
        moduleSubdirs.clear();
        AutomatedInstallData idata = AutomatedInstallData.getInstance();
        String installPath = idata.getInstallPath() + "/" + idata.getVariable("INSTALL_SUBPATH");
        moduleSubdirs.add(new File(installPath + PreExistingConfigurationConstants.modulesPath));
        moduleSubdirs.add(new File(installPath + PreExistingConfigurationConstants.baseModulesPath));
        moduleSubdirs.add(new File(installPath + PreExistingConfigurationConstants.soaModulesPath));
        moduleSubdirs.add(new File(installPath + PreExistingConfigurationConstants.srampModulesPath));
        moduleSubdirs.add(new File(installPath + PreExistingConfigurationConstants.dvModulesPath));
        moduleSubdirs.add(new File(installPath + PreExistingConfigurationConstants.brmsModulesPath));
        moduleSubdirs.add(new File(installPath + PreExistingConfigurationConstants.bpmsModulesPath));
    }


    /**
     * if the user defined multiple different vaults in different descriptors, this method will have trouble.
     *
     * @param xml
     * @param doc
     */
    private void setVaultDefaults(String xml, Document doc) {
        if (!doc.select("vault").isEmpty()) {
            idata.setVariable("vault.preexisting", "true");
            idata.setVariable(xml + ".vault.preexisting", "true");
            // all of the vault options must exist, or the standalone.xml is malformed
            idata.setVariable("vault.keystoreloc", doc.select("vault > vault-option[name=KEYSTORE_URL]").first().attr("value"));
            idata.setVariable("vault.alias", doc.select("vault > vault-option[name=KEYSTORE_ALIAS]").first().attr("value"));
            idata.setVariable("vault.salt", doc.select("vault > vault-option[name=SALT]").first().attr("value"));
            idata.setVariable("vault.itercount", doc.select("vault > vault-option[name=ITERATION_COUNT]").first().attr("value"));
            idata.setVariable("vault.encrdir", doc.select("vault > vault-option[name=ENC_FILE_DIR]").first().attr("value"));
        }
    }

    /**
     * This method finds all of the declared JDBC drivers per file, and matches them against the driver
     * classes that the installer can use to execute the SQL / configure components correctly with. It saves
     * drivers it finds in idata under the scheme: jdbc.preexisting.driver."xml"."number".{name,class,jar}
     * Additionally, the jdbc.driver.preexisting variable is set to true, for use in conditions and panels.
     * @param xml
     * @param doc
*/
    private void setJDBCDefaults(String xml, Document doc) {
        // get all of the defined drivers that aren't named h2, because that's there by default
        Elements drivers = doc.select("drivers > driver[name!=h2]");

        for (Element driver : drivers) {
            int foundCount = 0;
            String driverName = driver.attr("name");
            String driverModule = driver.attr("module");
            String driverClass = null;
            String driverJarPath = null;
            //jdbcDriverModules.add(driver.attr("module"));
            URL[] moduleJarUrls = JDBCConnectionUtils.convertToUrlArray(parseJarLocationFromModule(driverModule));
            for (URL jar : moduleJarUrls){
                driverClass = findExistingDriverClass(jar);
                try {
                    driverJarPath = (driverClass != null) ? new File(jar.toURI()).getPath() : null;
                } catch (URISyntaxException e) {
                    // this can't occur, since the convertToUrlArray would have caught it.
                    e.printStackTrace();
                }

            }

            // we have found a usable driver jar!
            if (driverClass != null && driverJarPath != null){
                foundCount++;
                String driverNameInternal = JBossJDBCConstants.classnameToJDBCMap.get(driverClass); //Ensure
                idata.setVariable("jdbc.driver.preexisting", "true");
                idata.setVariable("jdbc.preexisting.driver."+xml+"."+foundCount+".name", driverName);
                idata.setVariable("jdbc.preexisting.driver."+xml+"."+foundCount+".name.internal", driverNameInternal);
                idata.setVariable("jdbc.preexisting.driver." + xml + "." + foundCount + ".classname", driverClass);
                idata.setVariable("jdbc.preexisting.driver." + xml + "." + foundCount + ".jar", driverJarPath);
                idata.setVariable("jdbc.driver." + xml + ".found.count", String.valueOf(foundCount));

            }
        }
    }

    /**
     * This class makes sure that no .cli scripts that will cause installation failure if ran together are ran together, except for the
     * ones adding hornetq configuration (this needs to have user input, so it is in a validator instead)
     * @param xml
     * @param doc
     */
    private void ensureNoCliCollision(String xml, Document doc) {
        // check extensions
        //NodeList extensions = doc.getElementsByTagName("extension");
        for (String module : PreExistingConfigurationConstants.collidingExtensions) {
            // for every problematic module, try to find it in the descriptor:
            // <extensions>
            // ...
            // <extension module="module"/>
            // ...
            // <extensions>
            Elements elements = doc.select(String.format("extensions extension[module=%s]", module));
            if (!elements.isEmpty()) {
                idata.setVariable(xml + "." + module + ".extension.exists", "true");
            }
        }

        for (String xmlns : PreExistingConfigurationConstants.collidingSubsystems) {
            Elements elements = doc.select(String.format("profile subsystem[xmlns=%s]", xmlns));
            if (!elements.isEmpty()) {
                idata.setVariable(xml + "." + xmlns + ".subsystem.exists", "true");
            }
        }

        for (String name : PreExistingConfigurationConstants.collidingInfinispan) {
            Elements elements = doc.select(String.format("cache-container[name=%s]", name));
            if (!elements.isEmpty()) {
                idata.setVariable(xml + "." + name + ".infinispan.exists", "true");
            }
        }
    }

    /**
     * General method to find out pre-existing information given the name of the profile and a path to the file.
     * @param xml
     * @param descriptor
     */
    private void findDefaultInfo(String xml, File descriptor) {
        try {
            Document doc = Jsoup.parse(descriptor, "UTF-8", "");
            ensureNoCliCollision(xml, doc);
            setVaultDefaults(xml, doc);
            setLdapDefaults(doc);
            setSecurityDomainDefaults(doc);
            setSSLDefaults(xml, doc);
            setJDBCDefaults(xml, doc);

        } catch (Exception e) {
            e.printStackTrace();
        }

    }

    /**
     * This simple method gets a list of pre-existing securitydomain names, and saves them to idata, delimited by spaces.
     * The contents of the variable should be used in the validation of the security domain so that no collision is possible.
     * @param doc
     */
    private void setSecurityDomainDefaults(Document doc) {
        // securitydomain.preexisting.names
        Elements securityDomains = doc.select("security-domains > security-domain");

        for (Element domain : securityDomains) {
            securityDomainNames.add(domain.attr("name"));
        }
    }

    /**
     * Same as setSecurityDomainDefaults, but for ssl certs
     * TODO: it seems like the <ssl> element may be unique. need to double check the schema for this fact and modify the
     * check to be more efficient perhaps
     * @param xml
     * @param doc
     */
    private void setSSLDefaults(String xml, Document doc) {
        Elements sslIdentities = doc.select("server-identities > ssl");

        for (Element ssl : sslIdentities) {
            sslCertificates.add(ssl.select("keystore").attr("path"));
        }

    }

    /**
     * Same as setSecurityDomainDefaults, but for ldap connections and realms.
     * @param doc
     */
    private void setLdapDefaults(Document doc) {
        Elements ldapRealms = doc.select("security-realm > authentication > ldap");
        Elements ldapConnections = doc.select("outbound-connections > ldap");

        for (Element realm : ldapRealms) {
            ldapRealmNames.add(realm.attr("name"));
        }

        for (Element conn : ldapConnections) {
            ldapConnNames.add(conn.attr("name"));
        }
    }

    /**
     * This method runs through all of the jars in the parameter, and
     * attempts to load every supported driver class the installer knows about.
     * If successful, the name of the FIRST driver class successfully loaded is returned.
     * @param preExistingJDBCJars
     * @return
     */
    private String findExistingDriverClass(URL... preExistingJDBCJars) {
        for (String driverClassname : JBossJDBCConstants.classnameList) {
            Class<?> driver = JDBCConnectionUtils.findDriverClass(driverClassname, preExistingJDBCJars);
            if (driver != null) {
                // TODO: see if it's premature to short circuit the search here.
                return driverClassname;
            }
        }
        return null; // found nothing
    }

    /**
     * Attempts to find a JDBC driver jar referenced by the existing standalone*.xml descriptors and domain.xml
     * under the modules directory. Note that this only works if the driver has been installed as a core module (and we don't
     * really care about it if it hasn't)
     * @return
     */
    private Object[] parseJarLocationFromModule(String module) {
        ArrayList<File> fileList = new ArrayList<File>(1);

            for (File moduleSubdir : moduleSubdirs) {
            File searchDir = new File(moduleSubdir + "/" + module.replace(".", "/"), "/main/");
                if (searchDir.exists()) {
                Iterator<File> jars = FileUtils.iterateFiles(searchDir, new String[]{"jar"}, false);
                    while (jars.hasNext()) {
                        fileList.add(jars.next());
                    }
                }
            }
        ArrayList<String> resultList = new ArrayList<String>();
        for (File file : fileList) {
            resultList.add(file.getPath());
         }
        return resultList.toArray();
    }

    @Override
    public void initialize(PanelActionConfiguration configuration) {
        // unneeded
    }

}
