/*
Copyright 2018 The Knative Authors

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package names

import (
	"strings"
	"testing"

	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"

	"knative.dev/pkg/kmeta"
	v1 "knative.dev/serving/pkg/apis/serving/v1"
)

func TestNamer(t *testing.T) {
	tests := []struct {
		name string
		rev  *v1.Revision
		f    func(kmeta.Accessor) string
		want string
	}{{
		name: "Deployment too long",
		rev: &v1.Revision{
			ObjectMeta: metav1.ObjectMeta{
				Name: strings.Repeat("f", 63),
			},
		},
		f:    Deployment,
		want: "ffffffffffffffffffff105d7597f637e83cc711605ac3ea4957-deployment",
	}, {
		name: "Deployment long enough",
		rev: &v1.Revision{
			ObjectMeta: metav1.ObjectMeta{
				Name: strings.Repeat("f", 52),
			},
		},
		f:    Deployment,
		want: strings.Repeat("f", 52) + "-deployment",
	}, {
		name: "Deployment",
		rev: &v1.Revision{
			ObjectMeta: metav1.ObjectMeta{
				Name: "foo",
			},
		},
		f:    Deployment,
		want: "foo-deployment",
	}, {
		name: "ImageCache, barely fits",
		rev: &v1.Revision{
			ObjectMeta: metav1.ObjectMeta{
				Name: strings.Repeat("u", 57),
			},
		},
		f:    ImageCache,
		want: strings.Repeat("u", 57) + "-cache",
	}, {
		name: "ImageCache, already too long",
		rev: &v1.Revision{
			ObjectMeta: metav1.ObjectMeta{
				Name: strings.Repeat("u", 63),
			},
		},
		f:    ImageCache,
		want: "uuuuuuuuuuuuuuuuuuuuuuuuuca47ad1ce8479df271ec0d23653ce256-cache",
	}, {
		name: "ImageCache",
		rev: &v1.Revision{
			ObjectMeta: metav1.ObjectMeta{
				Name: "foo",
			},
		},
		f:    ImageCache,
		want: "foo-cache",
	}, {
		name: "PA",
		rev: &v1.Revision{
			ObjectMeta: metav1.ObjectMeta{
				Name: "baz",
			},
		},
		f:    PA,
		want: "baz",
	}}

	for _, test := range tests {
		t.Run(test.name, func(t *testing.T) {
			got := test.f(test.rev)
			if got != test.want {
				t.Errorf("%s() = %v, wanted %v", test.name, got, test.want)
			}
		})
	}
}
