package tls

import (
	"github.com/openshift/installer/pkg/asset"
	"github.com/openshift/installer/pkg/asset/installconfig"
	awstypes "github.com/openshift/installer/pkg/types/aws"
)

// CloudProviderCABundle is the asset the generates the CA bundle for
// trusting communication with the cloud provider. This bundle is used
// by the machine-config-operator on the bootstrap node.
type CloudProviderCABundle struct {
	File *asset.File
}

var _ asset.WritableAsset = (*CloudProviderCABundle)(nil)

// Dependencies returns the dependency of the CA bundle.
func (a *CloudProviderCABundle) Dependencies() []asset.Asset {
	return []asset.Asset{
		&installconfig.InstallConfig{},
	}
}

// Generate generates the CA bundle based on its dependencies.
func (a *CloudProviderCABundle) Generate(deps asset.Parents) error {
	ic := &installconfig.InstallConfig{}
	deps.Get(ic)

	if ic.Config.AdditionalTrustBundle == "" {
		return nil
	}
	if ic.Config.Platform.Name() != awstypes.Name {
		return nil
	}
	if !awstypes.C2SRegions.Has(ic.Config.Platform.AWS.Region) {
		return nil
	}

	a.File = &asset.File{
		Filename: assetFilePath("cloud-ca-cert.pem"),
		Data:     []byte(ic.Config.AdditionalTrustBundle),
	}

	return nil
}

// Name returns the human-friendly name of the asset.
func (a *CloudProviderCABundle) Name() string {
	return "Cloud Provider CA Bundle"
}

// Files returns the files generated by the asset.
func (a *CloudProviderCABundle) Files() []*asset.File {
	if a.File == nil {
		return nil
	}
	return []*asset.File{a.File}
}

// Load is a no-op because TLS assets are not written to disk.
func (a *CloudProviderCABundle) Load(asset.FileFetcher) (bool, error) {
	return false, nil
}
