// Copyright Project Contour Authors
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// +build e2e

package gateway

import (
	"context"
	"fmt"
	"testing"

	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/onsi/gomega/gexec"
	"github.com/projectcontour/contour/pkg/config"
	"github.com/projectcontour/contour/test/e2e"
	"github.com/stretchr/testify/require"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	gatewayv1alpha1 "sigs.k8s.io/gateway-api/apis/v1alpha1"
)

var f = e2e.NewFramework(false)

func TestGatewayAPI(t *testing.T) {
	RegisterFailHandler(Fail)
	RunSpecs(t, "Gateway API tests")
}

var _ = BeforeSuite(func() {
	require.NoError(f.T(), f.Deployment.EnsureResourcesForLocalContour())
})

var _ = AfterSuite(func() {
	f.DeleteNamespace(f.Deployment.Namespace.Name, true)
	gexec.CleanupBuildArtifacts()
})

var _ = Describe("Gateway API", func() {
	var (
		contourCmd            *gexec.Session
		contourConfig         *config.Parameters
		contourConfigFile     string
		additionalContourArgs []string
	)

	// Creates specified gateway in namespace and runs namespaced test
	// body. Modifies contour config to point to gateway.
	testWithGateway := func(gw *gatewayv1alpha1.Gateway, body e2e.NamespacedTestBody) e2e.NamespacedTestBody {
		return func(namespace string) {
			Context(fmt.Sprintf("with gateway %s/%s", namespace, gw.Name), func() {
				BeforeEach(func() {
					// Ensure gateway created in this test's namespace.
					gw.Namespace = namespace
					// Update contour config to point to specified gateway.
					contourConfig.GatewayConfig = &config.GatewayParameters{
						Namespace: namespace,
						Name:      gw.Name,
					}
					require.NoError(f.T(), f.Client.Create(context.TODO(), gw))
				})
				AfterEach(func() {
					require.NoError(f.T(), f.Client.Delete(context.TODO(), gw))
				})

				body(namespace)
			})
		}
	}

	BeforeEach(func() {
		// Contour config file contents, can be modified in nested
		// BeforeEach.
		contourConfig = &config.Parameters{}

		// Default contour serve command line arguments can be appended to in
		// nested BeforeEach.
		additionalContourArgs = []string{}
	})

	// JustBeforeEach is called after each of the nested BeforeEach are
	// called, so it is a final setup step before running a test.
	// A nested BeforeEach may have modified Contour config, so we wait
	// until here to start Contour.
	JustBeforeEach(func() {
		var err error
		contourCmd, contourConfigFile, err = f.Deployment.StartLocalContour(contourConfig, additionalContourArgs...)
		require.NoError(f.T(), err)

		// Wait for Envoy to be healthy.
		require.NoError(f.T(), f.Deployment.WaitForEnvoyDaemonSetUpdated())
	})

	AfterEach(func() {
		require.NoError(f.T(), f.Deployment.StopLocalContour(contourCmd, contourConfigFile))
	})

	Describe("HTTPRoute: Insecure (Non-TLS) Gateway", func() {
		testWithHTTPGateway := func(body e2e.NamespacedTestBody) e2e.NamespacedTestBody {
			gw := &gatewayv1alpha1.Gateway{
				ObjectMeta: metav1.ObjectMeta{
					Name: "http",
				},
				Spec: gatewayv1alpha1.GatewaySpec{
					GatewayClassName: "contour-class",
					Listeners: []gatewayv1alpha1.Listener{
						{
							Protocol: gatewayv1alpha1.HTTPProtocolType,
							Port:     gatewayv1alpha1.PortNumber(80),
							Routes: gatewayv1alpha1.RouteBindingSelector{
								Kind: "HTTPRoute",
								Namespaces: &gatewayv1alpha1.RouteNamespaces{
									From: routeSelectTypePtr(gatewayv1alpha1.RouteSelectAll),
								},
								Selector: &metav1.LabelSelector{
									MatchLabels: map[string]string{"app": "filter"},
								},
							},
						},
					},
				},
			}
			return testWithGateway(gw, body)
		}

		f.NamespacedTest("001-path-condition-match", testWithHTTPGateway(testGatewayPathConditionMatch))

		f.NamespacedTest("002-header-condition-match", testWithHTTPGateway(testGatewayHeaderConditionMatch))

		f.NamespacedTest("003-invalid-forward-to", testWithHTTPGateway(testInvalidForwardTo))

		f.NamespacedTest("005-request-header-modifier-forward-to", testWithHTTPGateway(testRequestHeaderModifierForwardTo))

		f.NamespacedTest("005-request-header-modifier-rule", testWithHTTPGateway(testRequestHeaderModifierRule))

		f.NamespacedTest("006-host-rewrite", testWithHTTPGateway(testHostRewrite))

		f.NamespacedTest("007-gateway-allow-type", testWithHTTPGateway(testGatewayAllowType))
	})

	Describe("HTTPRoute: TLS Gateway", func() {
		testWithHTTPSGateway := func(hostname string, body e2e.NamespacedTestBody) e2e.NamespacedTestBody {
			gw := &gatewayv1alpha1.Gateway{
				ObjectMeta: metav1.ObjectMeta{
					Name: "https",
				},
				Spec: gatewayv1alpha1.GatewaySpec{
					GatewayClassName: "contour-class",
					Listeners: []gatewayv1alpha1.Listener{
						{
							Protocol: gatewayv1alpha1.HTTPProtocolType,
							Port:     gatewayv1alpha1.PortNumber(80),
							Routes: gatewayv1alpha1.RouteBindingSelector{
								Kind: "HTTPRoute",
								Namespaces: &gatewayv1alpha1.RouteNamespaces{
									From: routeSelectTypePtr(gatewayv1alpha1.RouteSelectAll),
								},
								Selector: &metav1.LabelSelector{
									MatchLabels: map[string]string{"type": "insecure"},
								},
							},
						},
						{
							Protocol: gatewayv1alpha1.HTTPSProtocolType,
							Port:     gatewayv1alpha1.PortNumber(443),
							TLS: &gatewayv1alpha1.GatewayTLSConfig{
								CertificateRef: &gatewayv1alpha1.LocalObjectReference{
									Group: "core",
									Kind:  "Secret",
									Name:  "tlscert",
								},
							},
							Routes: gatewayv1alpha1.RouteBindingSelector{
								Kind: "HTTPRoute",
								Namespaces: &gatewayv1alpha1.RouteNamespaces{
									From: routeSelectTypePtr(gatewayv1alpha1.RouteSelectAll),
								},
								Selector: &metav1.LabelSelector{
									MatchLabels: map[string]string{"type": "secure"},
								},
							},
						},
					},
				},
			}
			return testWithGateway(gw, func(namespace string) {
				Context(fmt.Sprintf("with TLS secret %s/tlscert for hostname %s", namespace, hostname), func() {
					BeforeEach(func() {
						f.Certs.CreateSelfSignedCert(namespace, "tlscert", "tlscert", hostname)
					})

					body(namespace)
				})
			})
		}

		f.NamespacedTest("004-httproute-tls-gateway", testWithHTTPSGateway("tls-gateway.projectcontour.io", testTLSGateway))

		f.NamespacedTest("009-tls-wildcard-host", testWithHTTPSGateway("*.wildcardhost.gateway.projectcontour.io", testTLSWildcardHost))
	})

	Describe("TLSRoute: Gateway", func() {
		gw := &gatewayv1alpha1.Gateway{
			ObjectMeta: metav1.ObjectMeta{
				Name: "tls",
			},
			Spec: gatewayv1alpha1.GatewaySpec{
				GatewayClassName: "contour-class",
				Listeners: []gatewayv1alpha1.Listener{
					{
						Protocol: gatewayv1alpha1.TLSProtocolType,
						Port:     gatewayv1alpha1.PortNumber(443),
						Routes: gatewayv1alpha1.RouteBindingSelector{
							Kind: "TLSRoute",
							Namespaces: &gatewayv1alpha1.RouteNamespaces{
								From: routeSelectTypePtr(gatewayv1alpha1.RouteSelectAll),
							},
						},
					},
				},
			},
		}

		f.NamespacedTest("008-tlsroute", testWithGateway(gw, testTLSRoutePassthrough))
	})

	Describe("TLSRoute Gateway: Mode: Passthrough", func() {
		gw := &gatewayv1alpha1.Gateway{
			ObjectMeta: metav1.ObjectMeta{
				Name: "tls-passthrough",
			},
			Spec: gatewayv1alpha1.GatewaySpec{
				GatewayClassName: "contour-class",
				Listeners: []gatewayv1alpha1.Listener{
					{
						Protocol: gatewayv1alpha1.TLSProtocolType,
						Port:     gatewayv1alpha1.PortNumber(443),
						TLS: &gatewayv1alpha1.GatewayTLSConfig{
							Mode: tlsModeTypePtr(gatewayv1alpha1.TLSModePassthrough),
						},
						Routes: gatewayv1alpha1.RouteBindingSelector{
							Kind: "TLSRoute",
							Namespaces: &gatewayv1alpha1.RouteNamespaces{
								From: routeSelectTypePtr(gatewayv1alpha1.RouteSelectAll),
							},
						},
					},
				},
			},
		}

		f.NamespacedTest("008-tlsroute-mode-passthrough", testWithGateway(gw, testTLSRoutePassthrough))
	})
})

func stringPtr(s string) *string {
	return &s
}

func portNumPtr(port int) *gatewayv1alpha1.PortNumber {
	pn := gatewayv1alpha1.PortNumber(port)
	return &pn
}

func routeSelectTypePtr(val gatewayv1alpha1.RouteSelectType) *gatewayv1alpha1.RouteSelectType {
	return &val
}

func pathMatchTypePtr(val gatewayv1alpha1.PathMatchType) *gatewayv1alpha1.PathMatchType {
	return &val
}

func headerMatchTypePtr(val gatewayv1alpha1.HeaderMatchType) *gatewayv1alpha1.HeaderMatchType {
	return &val
}

func gatewayAllowTypePtr(val gatewayv1alpha1.GatewayAllowType) *gatewayv1alpha1.GatewayAllowType {
	return &val
}

func tlsModeTypePtr(mode gatewayv1alpha1.TLSModeType) *gatewayv1alpha1.TLSModeType {
	return &mode
}

// httpRouteAdmitted returns true if the route has a .status.conditions
// entry of "Admitted: true".
func httpRouteAdmitted(route *gatewayv1alpha1.HTTPRoute) bool {
	if route == nil {
		return false
	}

	for _, gw := range route.Status.Gateways {
		for _, cond := range gw.Conditions {
			if cond.Type == string(gatewayv1alpha1.ConditionRouteAdmitted) && cond.Status == metav1.ConditionTrue {
				return true
			}
		}
	}

	return false
}

// tlsRouteAdmitted returns true if the route has a .status.conditions
// entry of "Admitted: true".
func tlsRouteAdmitted(route *gatewayv1alpha1.TLSRoute) bool {
	if route == nil {
		return false
	}

	for _, gw := range route.Status.Gateways {
		for _, cond := range gw.Conditions {
			if cond.Type == string(gatewayv1alpha1.ConditionRouteAdmitted) && cond.Status == metav1.ConditionTrue {
				return true
			}
		}
	}

	return false
}
