// +build tools

package main

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"path/filepath"
	"sort"
	"strings"
	"text/template"
)

func main() {
	if len(os.Args) != 5 {
		log.Fatalln("exactly 5 arguments must be provided")
	}
	argsWithoutProg := os.Args[1:]

	pkg := argsWithoutProg[0]
	srcPath, err := filepath.Abs(argsWithoutProg[1])
	log.Println("srcPath: ", srcPath)
	if err != nil {
		log.Fatalln("failed to load absolute path for the source")
	}
	dstPath, err := filepath.Abs(argsWithoutProg[2])
	log.Println("dstPath: ", dstPath)
	if err != nil {
		log.Fatalln("failed to load absolute path for the source")
	}

	architecture := argsWithoutProg[3]
	log.Println("architecture: ", architecture)

	srcData, err := ioutil.ReadFile(srcPath)
	if err != nil {
		log.Fatalln(err)
	}

	var m metadata
	if err := json.Unmarshal(srcData, &m); err != nil {
		log.Fatalln(fmt.Errorf("failed to unmarshal source: %v", err))
	}

	var regions []string
	for arch, images := range m.Architectures {
		if arch == architecture {
			regions = make([]string, 0, len(images.Images.AWS.Regions))
			for region := range images.Images.AWS.Regions {
				regions = append(regions, region)
			}
		}
	}
	sort.Strings(regions)

	tinput := struct {
		Architecture string
		Pkg          string
		Regions      []string
	}{Architecture: strings.ToUpper(architecture), Pkg: pkg, Regions: regions}

	t := template.Must(template.New(fmt.Sprintf("ami_regions_%s", architecture)).Parse(tmpl))
	buf := &bytes.Buffer{}
	if err := t.Execute(buf, tinput); err != nil {
		log.Fatalln(fmt.Errorf("failed to execute the template: %v", err))
	}

	if err := ioutil.WriteFile(dstPath, buf.Bytes(), 0664); err != nil {
		log.Fatalln(err)
	}
}

// use the rhco-stream.json to get AMIs per arch
type metadata struct {
	Architectures map[string]struct {
		Images struct {
			AWS struct {
				Regions map[string]struct {
					Release string `json:"release"`
					Image   string `json:"image"`
				} `json:"regions"`
			} `json:"aws"`
		} `json:"images"`
	} `json:"architectures"`
}

var tmpl = `// Code generated by ami_regions_generate.go; DO NOT EDIT.

package {{ .Pkg }}

// AMIRegions{{ .Architecture }} is a list of regions where the RHEL CoreOS is published for architecture {{ .Architecture }}.
var AMIRegions{{ .Architecture }} = []string{
{{- range $region := .Regions}}
	"{{ $region }}",
{{- end}}
}
`
