// Copyright 2012-2018 The Rust Project Developers. See the COPYRIGHT
// file at the top-level directory of this distribution and at
// http://rust-lang.org/COPYRIGHT.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

// NOTE: The following code was generated by "scripts/unicode.py", do not edit directly

#![allow(missing_docs, non_upper_case_globals, non_snake_case)]

/// The version of [Unicode](http://www.unicode.org/)
/// that this version of unicode-segmentation is based on.
pub const UNICODE_VERSION: (u64, u64, u64) = (9, 0, 0);

pub mod util {
    #[inline]
    pub fn bsearch_range_table(c: char, r: &'static [(char,char)]) -> bool {
        use core::cmp::Ordering::{Equal, Less, Greater};
        r.binary_search_by(|&(lo,hi)| {
            if lo <= c && c <= hi { Equal }
            else if hi < c { Less }
            else { Greater }
        }).is_ok()
    }

    #[inline]
    fn is_alphabetic(c: char) -> bool {
        match c {
            'a' ... 'z' | 'A' ... 'Z' => true,
            c if c > '' => super::derived_property::Alphabetic(c),
            _ => false,
        }
    }

    #[inline]
    fn is_numeric(c: char) -> bool {
        match c {
            '0' ... '9' => true,
            c if c > '' => super::general_category::N(c),
            _ => false,
        }
    }

    #[inline]
    pub fn is_alphanumeric(c: char) -> bool {
        is_alphabetic(c) || is_numeric(c)
    }
}

mod general_category {
    const N_table: &'static [(char, char)] = &[
        ('\u{30}', '\u{39}'), ('\u{b2}', '\u{b3}'), ('\u{b9}', '\u{b9}'), ('\u{bc}', '\u{be}'),
        ('\u{660}', '\u{669}'), ('\u{6f0}', '\u{6f9}'), ('\u{7c0}', '\u{7c9}'), ('\u{966}',
        '\u{96f}'), ('\u{9e6}', '\u{9ef}'), ('\u{9f4}', '\u{9f9}'), ('\u{a66}', '\u{a6f}'),
        ('\u{ae6}', '\u{aef}'), ('\u{b66}', '\u{b6f}'), ('\u{b72}', '\u{b77}'), ('\u{be6}',
        '\u{bf2}'), ('\u{c66}', '\u{c6f}'), ('\u{c78}', '\u{c7e}'), ('\u{ce6}', '\u{cef}'),
        ('\u{d58}', '\u{d5e}'), ('\u{d66}', '\u{d78}'), ('\u{de6}', '\u{def}'), ('\u{e50}',
        '\u{e59}'), ('\u{ed0}', '\u{ed9}'), ('\u{f20}', '\u{f33}'), ('\u{1040}', '\u{1049}'),
        ('\u{1090}', '\u{1099}'), ('\u{1369}', '\u{137c}'), ('\u{16ee}', '\u{16f0}'), ('\u{17e0}',
        '\u{17e9}'), ('\u{17f0}', '\u{17f9}'), ('\u{1810}', '\u{1819}'), ('\u{1946}', '\u{194f}'),
        ('\u{19d0}', '\u{19da}'), ('\u{1a80}', '\u{1a89}'), ('\u{1a90}', '\u{1a99}'), ('\u{1b50}',
        '\u{1b59}'), ('\u{1bb0}', '\u{1bb9}'), ('\u{1c40}', '\u{1c49}'), ('\u{1c50}', '\u{1c59}'),
        ('\u{2070}', '\u{2070}'), ('\u{2074}', '\u{2079}'), ('\u{2080}', '\u{2089}'), ('\u{2150}',
        '\u{2182}'), ('\u{2185}', '\u{2189}'), ('\u{2460}', '\u{249b}'), ('\u{24ea}', '\u{24ff}'),
        ('\u{2776}', '\u{2793}'), ('\u{2cfd}', '\u{2cfd}'), ('\u{3007}', '\u{3007}'), ('\u{3021}',
        '\u{3029}'), ('\u{3038}', '\u{303a}'), ('\u{3192}', '\u{3195}'), ('\u{3220}', '\u{3229}'),
        ('\u{3248}', '\u{324f}'), ('\u{3251}', '\u{325f}'), ('\u{3280}', '\u{3289}'), ('\u{32b1}',
        '\u{32bf}'), ('\u{a620}', '\u{a629}'), ('\u{a6e6}', '\u{a6ef}'), ('\u{a830}', '\u{a835}'),
        ('\u{a8d0}', '\u{a8d9}'), ('\u{a900}', '\u{a909}'), ('\u{a9d0}', '\u{a9d9}'), ('\u{a9f0}',
        '\u{a9f9}'), ('\u{aa50}', '\u{aa59}'), ('\u{abf0}', '\u{abf9}'), ('\u{ff10}', '\u{ff19}'),
        ('\u{10107}', '\u{10133}'), ('\u{10140}', '\u{10178}'), ('\u{1018a}', '\u{1018b}'),
        ('\u{102e1}', '\u{102fb}'), ('\u{10320}', '\u{10323}'), ('\u{10341}', '\u{10341}'),
        ('\u{1034a}', '\u{1034a}'), ('\u{103d1}', '\u{103d5}'), ('\u{104a0}', '\u{104a9}'),
        ('\u{10858}', '\u{1085f}'), ('\u{10879}', '\u{1087f}'), ('\u{108a7}', '\u{108af}'),
        ('\u{108fb}', '\u{108ff}'), ('\u{10916}', '\u{1091b}'), ('\u{109bc}', '\u{109bd}'),
        ('\u{109c0}', '\u{109cf}'), ('\u{109d2}', '\u{109ff}'), ('\u{10a40}', '\u{10a47}'),
        ('\u{10a7d}', '\u{10a7e}'), ('\u{10a9d}', '\u{10a9f}'), ('\u{10aeb}', '\u{10aef}'),
        ('\u{10b58}', '\u{10b5f}'), ('\u{10b78}', '\u{10b7f}'), ('\u{10ba9}', '\u{10baf}'),
        ('\u{10cfa}', '\u{10cff}'), ('\u{10e60}', '\u{10e7e}'), ('\u{11052}', '\u{1106f}'),
        ('\u{110f0}', '\u{110f9}'), ('\u{11136}', '\u{1113f}'), ('\u{111d0}', '\u{111d9}'),
        ('\u{111e1}', '\u{111f4}'), ('\u{112f0}', '\u{112f9}'), ('\u{11450}', '\u{11459}'),
        ('\u{114d0}', '\u{114d9}'), ('\u{11650}', '\u{11659}'), ('\u{116c0}', '\u{116c9}'),
        ('\u{11730}', '\u{1173b}'), ('\u{118e0}', '\u{118f2}'), ('\u{11c50}', '\u{11c6c}'),
        ('\u{12400}', '\u{1246e}'), ('\u{16a60}', '\u{16a69}'), ('\u{16b50}', '\u{16b59}'),
        ('\u{16b5b}', '\u{16b61}'), ('\u{1d360}', '\u{1d371}'), ('\u{1d7ce}', '\u{1d7ff}'),
        ('\u{1e8c7}', '\u{1e8cf}'), ('\u{1e950}', '\u{1e959}'), ('\u{1f100}', '\u{1f10c}')
    ];

    #[inline]
    pub fn N(c: char) -> bool {
        super::util::bsearch_range_table(c, N_table)
    }

}

mod derived_property {
    const Alphabetic_table: &'static [(char, char)] = &[
        ('\u{41}', '\u{5a}'), ('\u{61}', '\u{7a}'), ('\u{aa}', '\u{aa}'), ('\u{b5}', '\u{b5}'),
        ('\u{ba}', '\u{ba}'), ('\u{c0}', '\u{d6}'), ('\u{d8}', '\u{f6}'), ('\u{f8}', '\u{2c1}'),
        ('\u{2c6}', '\u{2d1}'), ('\u{2e0}', '\u{2e4}'), ('\u{2ec}', '\u{2ec}'), ('\u{2ee}',
        '\u{2ee}'), ('\u{345}', '\u{345}'), ('\u{370}', '\u{374}'), ('\u{376}', '\u{377}'),
        ('\u{37a}', '\u{37d}'), ('\u{37f}', '\u{37f}'), ('\u{386}', '\u{386}'), ('\u{388}',
        '\u{38a}'), ('\u{38c}', '\u{38c}'), ('\u{38e}', '\u{3a1}'), ('\u{3a3}', '\u{3f5}'),
        ('\u{3f7}', '\u{481}'), ('\u{48a}', '\u{52f}'), ('\u{531}', '\u{556}'), ('\u{559}',
        '\u{559}'), ('\u{561}', '\u{587}'), ('\u{5b0}', '\u{5bd}'), ('\u{5bf}', '\u{5bf}'),
        ('\u{5c1}', '\u{5c2}'), ('\u{5c4}', '\u{5c5}'), ('\u{5c7}', '\u{5c7}'), ('\u{5d0}',
        '\u{5ea}'), ('\u{5f0}', '\u{5f2}'), ('\u{610}', '\u{61a}'), ('\u{620}', '\u{657}'),
        ('\u{659}', '\u{65f}'), ('\u{66e}', '\u{6d3}'), ('\u{6d5}', '\u{6dc}'), ('\u{6e1}',
        '\u{6e8}'), ('\u{6ed}', '\u{6ef}'), ('\u{6fa}', '\u{6fc}'), ('\u{6ff}', '\u{6ff}'),
        ('\u{710}', '\u{73f}'), ('\u{74d}', '\u{7b1}'), ('\u{7ca}', '\u{7ea}'), ('\u{7f4}',
        '\u{7f5}'), ('\u{7fa}', '\u{7fa}'), ('\u{800}', '\u{817}'), ('\u{81a}', '\u{82c}'),
        ('\u{840}', '\u{858}'), ('\u{8a0}', '\u{8b4}'), ('\u{8b6}', '\u{8bd}'), ('\u{8d4}',
        '\u{8df}'), ('\u{8e3}', '\u{8e9}'), ('\u{8f0}', '\u{93b}'), ('\u{93d}', '\u{94c}'),
        ('\u{94e}', '\u{950}'), ('\u{955}', '\u{963}'), ('\u{971}', '\u{983}'), ('\u{985}',
        '\u{98c}'), ('\u{98f}', '\u{990}'), ('\u{993}', '\u{9a8}'), ('\u{9aa}', '\u{9b0}'),
        ('\u{9b2}', '\u{9b2}'), ('\u{9b6}', '\u{9b9}'), ('\u{9bd}', '\u{9c4}'), ('\u{9c7}',
        '\u{9c8}'), ('\u{9cb}', '\u{9cc}'), ('\u{9ce}', '\u{9ce}'), ('\u{9d7}', '\u{9d7}'),
        ('\u{9dc}', '\u{9dd}'), ('\u{9df}', '\u{9e3}'), ('\u{9f0}', '\u{9f1}'), ('\u{a01}',
        '\u{a03}'), ('\u{a05}', '\u{a0a}'), ('\u{a0f}', '\u{a10}'), ('\u{a13}', '\u{a28}'),
        ('\u{a2a}', '\u{a30}'), ('\u{a32}', '\u{a33}'), ('\u{a35}', '\u{a36}'), ('\u{a38}',
        '\u{a39}'), ('\u{a3e}', '\u{a42}'), ('\u{a47}', '\u{a48}'), ('\u{a4b}', '\u{a4c}'),
        ('\u{a51}', '\u{a51}'), ('\u{a59}', '\u{a5c}'), ('\u{a5e}', '\u{a5e}'), ('\u{a70}',
        '\u{a75}'), ('\u{a81}', '\u{a83}'), ('\u{a85}', '\u{a8d}'), ('\u{a8f}', '\u{a91}'),
        ('\u{a93}', '\u{aa8}'), ('\u{aaa}', '\u{ab0}'), ('\u{ab2}', '\u{ab3}'), ('\u{ab5}',
        '\u{ab9}'), ('\u{abd}', '\u{ac5}'), ('\u{ac7}', '\u{ac9}'), ('\u{acb}', '\u{acc}'),
        ('\u{ad0}', '\u{ad0}'), ('\u{ae0}', '\u{ae3}'), ('\u{af9}', '\u{af9}'), ('\u{b01}',
        '\u{b03}'), ('\u{b05}', '\u{b0c}'), ('\u{b0f}', '\u{b10}'), ('\u{b13}', '\u{b28}'),
        ('\u{b2a}', '\u{b30}'), ('\u{b32}', '\u{b33}'), ('\u{b35}', '\u{b39}'), ('\u{b3d}',
        '\u{b44}'), ('\u{b47}', '\u{b48}'), ('\u{b4b}', '\u{b4c}'), ('\u{b56}', '\u{b57}'),
        ('\u{b5c}', '\u{b5d}'), ('\u{b5f}', '\u{b63}'), ('\u{b71}', '\u{b71}'), ('\u{b82}',
        '\u{b83}'), ('\u{b85}', '\u{b8a}'), ('\u{b8e}', '\u{b90}'), ('\u{b92}', '\u{b95}'),
        ('\u{b99}', '\u{b9a}'), ('\u{b9c}', '\u{b9c}'), ('\u{b9e}', '\u{b9f}'), ('\u{ba3}',
        '\u{ba4}'), ('\u{ba8}', '\u{baa}'), ('\u{bae}', '\u{bb9}'), ('\u{bbe}', '\u{bc2}'),
        ('\u{bc6}', '\u{bc8}'), ('\u{bca}', '\u{bcc}'), ('\u{bd0}', '\u{bd0}'), ('\u{bd7}',
        '\u{bd7}'), ('\u{c00}', '\u{c03}'), ('\u{c05}', '\u{c0c}'), ('\u{c0e}', '\u{c10}'),
        ('\u{c12}', '\u{c28}'), ('\u{c2a}', '\u{c39}'), ('\u{c3d}', '\u{c44}'), ('\u{c46}',
        '\u{c48}'), ('\u{c4a}', '\u{c4c}'), ('\u{c55}', '\u{c56}'), ('\u{c58}', '\u{c5a}'),
        ('\u{c60}', '\u{c63}'), ('\u{c80}', '\u{c83}'), ('\u{c85}', '\u{c8c}'), ('\u{c8e}',
        '\u{c90}'), ('\u{c92}', '\u{ca8}'), ('\u{caa}', '\u{cb3}'), ('\u{cb5}', '\u{cb9}'),
        ('\u{cbd}', '\u{cc4}'), ('\u{cc6}', '\u{cc8}'), ('\u{cca}', '\u{ccc}'), ('\u{cd5}',
        '\u{cd6}'), ('\u{cde}', '\u{cde}'), ('\u{ce0}', '\u{ce3}'), ('\u{cf1}', '\u{cf2}'),
        ('\u{d01}', '\u{d03}'), ('\u{d05}', '\u{d0c}'), ('\u{d0e}', '\u{d10}'), ('\u{d12}',
        '\u{d3a}'), ('\u{d3d}', '\u{d44}'), ('\u{d46}', '\u{d48}'), ('\u{d4a}', '\u{d4c}'),
        ('\u{d4e}', '\u{d4e}'), ('\u{d54}', '\u{d57}'), ('\u{d5f}', '\u{d63}'), ('\u{d7a}',
        '\u{d7f}'), ('\u{d82}', '\u{d83}'), ('\u{d85}', '\u{d96}'), ('\u{d9a}', '\u{db1}'),
        ('\u{db3}', '\u{dbb}'), ('\u{dbd}', '\u{dbd}'), ('\u{dc0}', '\u{dc6}'), ('\u{dcf}',
        '\u{dd4}'), ('\u{dd6}', '\u{dd6}'), ('\u{dd8}', '\u{ddf}'), ('\u{df2}', '\u{df3}'),
        ('\u{e01}', '\u{e3a}'), ('\u{e40}', '\u{e46}'), ('\u{e4d}', '\u{e4d}'), ('\u{e81}',
        '\u{e82}'), ('\u{e84}', '\u{e84}'), ('\u{e87}', '\u{e88}'), ('\u{e8a}', '\u{e8a}'),
        ('\u{e8d}', '\u{e8d}'), ('\u{e94}', '\u{e97}'), ('\u{e99}', '\u{e9f}'), ('\u{ea1}',
        '\u{ea3}'), ('\u{ea5}', '\u{ea5}'), ('\u{ea7}', '\u{ea7}'), ('\u{eaa}', '\u{eab}'),
        ('\u{ead}', '\u{eb9}'), ('\u{ebb}', '\u{ebd}'), ('\u{ec0}', '\u{ec4}'), ('\u{ec6}',
        '\u{ec6}'), ('\u{ecd}', '\u{ecd}'), ('\u{edc}', '\u{edf}'), ('\u{f00}', '\u{f00}'),
        ('\u{f40}', '\u{f47}'), ('\u{f49}', '\u{f6c}'), ('\u{f71}', '\u{f81}'), ('\u{f88}',
        '\u{f97}'), ('\u{f99}', '\u{fbc}'), ('\u{1000}', '\u{1036}'), ('\u{1038}', '\u{1038}'),
        ('\u{103b}', '\u{103f}'), ('\u{1050}', '\u{1062}'), ('\u{1065}', '\u{1068}'), ('\u{106e}',
        '\u{1086}'), ('\u{108e}', '\u{108e}'), ('\u{109c}', '\u{109d}'), ('\u{10a0}', '\u{10c5}'),
        ('\u{10c7}', '\u{10c7}'), ('\u{10cd}', '\u{10cd}'), ('\u{10d0}', '\u{10fa}'), ('\u{10fc}',
        '\u{1248}'), ('\u{124a}', '\u{124d}'), ('\u{1250}', '\u{1256}'), ('\u{1258}', '\u{1258}'),
        ('\u{125a}', '\u{125d}'), ('\u{1260}', '\u{1288}'), ('\u{128a}', '\u{128d}'), ('\u{1290}',
        '\u{12b0}'), ('\u{12b2}', '\u{12b5}'), ('\u{12b8}', '\u{12be}'), ('\u{12c0}', '\u{12c0}'),
        ('\u{12c2}', '\u{12c5}'), ('\u{12c8}', '\u{12d6}'), ('\u{12d8}', '\u{1310}'), ('\u{1312}',
        '\u{1315}'), ('\u{1318}', '\u{135a}'), ('\u{135f}', '\u{135f}'), ('\u{1380}', '\u{138f}'),
        ('\u{13a0}', '\u{13f5}'), ('\u{13f8}', '\u{13fd}'), ('\u{1401}', '\u{166c}'), ('\u{166f}',
        '\u{167f}'), ('\u{1681}', '\u{169a}'), ('\u{16a0}', '\u{16ea}'), ('\u{16ee}', '\u{16f8}'),
        ('\u{1700}', '\u{170c}'), ('\u{170e}', '\u{1713}'), ('\u{1720}', '\u{1733}'), ('\u{1740}',
        '\u{1753}'), ('\u{1760}', '\u{176c}'), ('\u{176e}', '\u{1770}'), ('\u{1772}', '\u{1773}'),
        ('\u{1780}', '\u{17b3}'), ('\u{17b6}', '\u{17c8}'), ('\u{17d7}', '\u{17d7}'), ('\u{17dc}',
        '\u{17dc}'), ('\u{1820}', '\u{1877}'), ('\u{1880}', '\u{18aa}'), ('\u{18b0}', '\u{18f5}'),
        ('\u{1900}', '\u{191e}'), ('\u{1920}', '\u{192b}'), ('\u{1930}', '\u{1938}'), ('\u{1950}',
        '\u{196d}'), ('\u{1970}', '\u{1974}'), ('\u{1980}', '\u{19ab}'), ('\u{19b0}', '\u{19c9}'),
        ('\u{1a00}', '\u{1a1b}'), ('\u{1a20}', '\u{1a5e}'), ('\u{1a61}', '\u{1a74}'), ('\u{1aa7}',
        '\u{1aa7}'), ('\u{1b00}', '\u{1b33}'), ('\u{1b35}', '\u{1b43}'), ('\u{1b45}', '\u{1b4b}'),
        ('\u{1b80}', '\u{1ba9}'), ('\u{1bac}', '\u{1baf}'), ('\u{1bba}', '\u{1be5}'), ('\u{1be7}',
        '\u{1bf1}'), ('\u{1c00}', '\u{1c35}'), ('\u{1c4d}', '\u{1c4f}'), ('\u{1c5a}', '\u{1c7d}'),
        ('\u{1c80}', '\u{1c88}'), ('\u{1ce9}', '\u{1cec}'), ('\u{1cee}', '\u{1cf3}'), ('\u{1cf5}',
        '\u{1cf6}'), ('\u{1d00}', '\u{1dbf}'), ('\u{1de7}', '\u{1df4}'), ('\u{1e00}', '\u{1f15}'),
        ('\u{1f18}', '\u{1f1d}'), ('\u{1f20}', '\u{1f45}'), ('\u{1f48}', '\u{1f4d}'), ('\u{1f50}',
        '\u{1f57}'), ('\u{1f59}', '\u{1f59}'), ('\u{1f5b}', '\u{1f5b}'), ('\u{1f5d}', '\u{1f5d}'),
        ('\u{1f5f}', '\u{1f7d}'), ('\u{1f80}', '\u{1fb4}'), ('\u{1fb6}', '\u{1fbc}'), ('\u{1fbe}',
        '\u{1fbe}'), ('\u{1fc2}', '\u{1fc4}'), ('\u{1fc6}', '\u{1fcc}'), ('\u{1fd0}', '\u{1fd3}'),
        ('\u{1fd6}', '\u{1fdb}'), ('\u{1fe0}', '\u{1fec}'), ('\u{1ff2}', '\u{1ff4}'), ('\u{1ff6}',
        '\u{1ffc}'), ('\u{2071}', '\u{2071}'), ('\u{207f}', '\u{207f}'), ('\u{2090}', '\u{209c}'),
        ('\u{2102}', '\u{2102}'), ('\u{2107}', '\u{2107}'), ('\u{210a}', '\u{2113}'), ('\u{2115}',
        '\u{2115}'), ('\u{2119}', '\u{211d}'), ('\u{2124}', '\u{2124}'), ('\u{2126}', '\u{2126}'),
        ('\u{2128}', '\u{2128}'), ('\u{212a}', '\u{212d}'), ('\u{212f}', '\u{2139}'), ('\u{213c}',
        '\u{213f}'), ('\u{2145}', '\u{2149}'), ('\u{214e}', '\u{214e}'), ('\u{2160}', '\u{2188}'),
        ('\u{24b6}', '\u{24e9}'), ('\u{2c00}', '\u{2c2e}'), ('\u{2c30}', '\u{2c5e}'), ('\u{2c60}',
        '\u{2ce4}'), ('\u{2ceb}', '\u{2cee}'), ('\u{2cf2}', '\u{2cf3}'), ('\u{2d00}', '\u{2d25}'),
        ('\u{2d27}', '\u{2d27}'), ('\u{2d2d}', '\u{2d2d}'), ('\u{2d30}', '\u{2d67}'), ('\u{2d6f}',
        '\u{2d6f}'), ('\u{2d80}', '\u{2d96}'), ('\u{2da0}', '\u{2da6}'), ('\u{2da8}', '\u{2dae}'),
        ('\u{2db0}', '\u{2db6}'), ('\u{2db8}', '\u{2dbe}'), ('\u{2dc0}', '\u{2dc6}'), ('\u{2dc8}',
        '\u{2dce}'), ('\u{2dd0}', '\u{2dd6}'), ('\u{2dd8}', '\u{2dde}'), ('\u{2de0}', '\u{2dff}'),
        ('\u{2e2f}', '\u{2e2f}'), ('\u{3005}', '\u{3007}'), ('\u{3021}', '\u{3029}'), ('\u{3031}',
        '\u{3035}'), ('\u{3038}', '\u{303c}'), ('\u{3041}', '\u{3096}'), ('\u{309d}', '\u{309f}'),
        ('\u{30a1}', '\u{30fa}'), ('\u{30fc}', '\u{30ff}'), ('\u{3105}', '\u{312d}'), ('\u{3131}',
        '\u{318e}'), ('\u{31a0}', '\u{31ba}'), ('\u{31f0}', '\u{31ff}'), ('\u{3400}', '\u{4db5}'),
        ('\u{4e00}', '\u{9fd5}'), ('\u{a000}', '\u{a48c}'), ('\u{a4d0}', '\u{a4fd}'), ('\u{a500}',
        '\u{a60c}'), ('\u{a610}', '\u{a61f}'), ('\u{a62a}', '\u{a62b}'), ('\u{a640}', '\u{a66e}'),
        ('\u{a674}', '\u{a67b}'), ('\u{a67f}', '\u{a6ef}'), ('\u{a717}', '\u{a71f}'), ('\u{a722}',
        '\u{a788}'), ('\u{a78b}', '\u{a7ae}'), ('\u{a7b0}', '\u{a7b7}'), ('\u{a7f7}', '\u{a801}'),
        ('\u{a803}', '\u{a805}'), ('\u{a807}', '\u{a80a}'), ('\u{a80c}', '\u{a827}'), ('\u{a840}',
        '\u{a873}'), ('\u{a880}', '\u{a8c3}'), ('\u{a8c5}', '\u{a8c5}'), ('\u{a8f2}', '\u{a8f7}'),
        ('\u{a8fb}', '\u{a8fb}'), ('\u{a8fd}', '\u{a8fd}'), ('\u{a90a}', '\u{a92a}'), ('\u{a930}',
        '\u{a952}'), ('\u{a960}', '\u{a97c}'), ('\u{a980}', '\u{a9b2}'), ('\u{a9b4}', '\u{a9bf}'),
        ('\u{a9cf}', '\u{a9cf}'), ('\u{a9e0}', '\u{a9e4}'), ('\u{a9e6}', '\u{a9ef}'), ('\u{a9fa}',
        '\u{a9fe}'), ('\u{aa00}', '\u{aa36}'), ('\u{aa40}', '\u{aa4d}'), ('\u{aa60}', '\u{aa76}'),
        ('\u{aa7a}', '\u{aa7a}'), ('\u{aa7e}', '\u{aabe}'), ('\u{aac0}', '\u{aac0}'), ('\u{aac2}',
        '\u{aac2}'), ('\u{aadb}', '\u{aadd}'), ('\u{aae0}', '\u{aaef}'), ('\u{aaf2}', '\u{aaf5}'),
        ('\u{ab01}', '\u{ab06}'), ('\u{ab09}', '\u{ab0e}'), ('\u{ab11}', '\u{ab16}'), ('\u{ab20}',
        '\u{ab26}'), ('\u{ab28}', '\u{ab2e}'), ('\u{ab30}', '\u{ab5a}'), ('\u{ab5c}', '\u{ab65}'),
        ('\u{ab70}', '\u{abea}'), ('\u{ac00}', '\u{d7a3}'), ('\u{d7b0}', '\u{d7c6}'), ('\u{d7cb}',
        '\u{d7fb}'), ('\u{f900}', '\u{fa6d}'), ('\u{fa70}', '\u{fad9}'), ('\u{fb00}', '\u{fb06}'),
        ('\u{fb13}', '\u{fb17}'), ('\u{fb1d}', '\u{fb28}'), ('\u{fb2a}', '\u{fb36}'), ('\u{fb38}',
        '\u{fb3c}'), ('\u{fb3e}', '\u{fb3e}'), ('\u{fb40}', '\u{fb41}'), ('\u{fb43}', '\u{fb44}'),
        ('\u{fb46}', '\u{fbb1}'), ('\u{fbd3}', '\u{fd3d}'), ('\u{fd50}', '\u{fd8f}'), ('\u{fd92}',
        '\u{fdc7}'), ('\u{fdf0}', '\u{fdfb}'), ('\u{fe70}', '\u{fe74}'), ('\u{fe76}', '\u{fefc}'),
        ('\u{ff21}', '\u{ff3a}'), ('\u{ff41}', '\u{ff5a}'), ('\u{ff66}', '\u{ffbe}'), ('\u{ffc2}',
        '\u{ffc7}'), ('\u{ffca}', '\u{ffcf}'), ('\u{ffd2}', '\u{ffd7}'), ('\u{ffda}', '\u{ffdc}'),
        ('\u{10000}', '\u{1000b}'), ('\u{1000d}', '\u{10026}'), ('\u{10028}', '\u{1003a}'),
        ('\u{1003c}', '\u{1003d}'), ('\u{1003f}', '\u{1004d}'), ('\u{10050}', '\u{1005d}'),
        ('\u{10080}', '\u{100fa}'), ('\u{10140}', '\u{10174}'), ('\u{10280}', '\u{1029c}'),
        ('\u{102a0}', '\u{102d0}'), ('\u{10300}', '\u{1031f}'), ('\u{10330}', '\u{1034a}'),
        ('\u{10350}', '\u{1037a}'), ('\u{10380}', '\u{1039d}'), ('\u{103a0}', '\u{103c3}'),
        ('\u{103c8}', '\u{103cf}'), ('\u{103d1}', '\u{103d5}'), ('\u{10400}', '\u{1049d}'),
        ('\u{104b0}', '\u{104d3}'), ('\u{104d8}', '\u{104fb}'), ('\u{10500}', '\u{10527}'),
        ('\u{10530}', '\u{10563}'), ('\u{10600}', '\u{10736}'), ('\u{10740}', '\u{10755}'),
        ('\u{10760}', '\u{10767}'), ('\u{10800}', '\u{10805}'), ('\u{10808}', '\u{10808}'),
        ('\u{1080a}', '\u{10835}'), ('\u{10837}', '\u{10838}'), ('\u{1083c}', '\u{1083c}'),
        ('\u{1083f}', '\u{10855}'), ('\u{10860}', '\u{10876}'), ('\u{10880}', '\u{1089e}'),
        ('\u{108e0}', '\u{108f2}'), ('\u{108f4}', '\u{108f5}'), ('\u{10900}', '\u{10915}'),
        ('\u{10920}', '\u{10939}'), ('\u{10980}', '\u{109b7}'), ('\u{109be}', '\u{109bf}'),
        ('\u{10a00}', '\u{10a03}'), ('\u{10a05}', '\u{10a06}'), ('\u{10a0c}', '\u{10a13}'),
        ('\u{10a15}', '\u{10a17}'), ('\u{10a19}', '\u{10a33}'), ('\u{10a60}', '\u{10a7c}'),
        ('\u{10a80}', '\u{10a9c}'), ('\u{10ac0}', '\u{10ac7}'), ('\u{10ac9}', '\u{10ae4}'),
        ('\u{10b00}', '\u{10b35}'), ('\u{10b40}', '\u{10b55}'), ('\u{10b60}', '\u{10b72}'),
        ('\u{10b80}', '\u{10b91}'), ('\u{10c00}', '\u{10c48}'), ('\u{10c80}', '\u{10cb2}'),
        ('\u{10cc0}', '\u{10cf2}'), ('\u{11000}', '\u{11045}'), ('\u{11082}', '\u{110b8}'),
        ('\u{110d0}', '\u{110e8}'), ('\u{11100}', '\u{11132}'), ('\u{11150}', '\u{11172}'),
        ('\u{11176}', '\u{11176}'), ('\u{11180}', '\u{111bf}'), ('\u{111c1}', '\u{111c4}'),
        ('\u{111da}', '\u{111da}'), ('\u{111dc}', '\u{111dc}'), ('\u{11200}', '\u{11211}'),
        ('\u{11213}', '\u{11234}'), ('\u{11237}', '\u{11237}'), ('\u{1123e}', '\u{1123e}'),
        ('\u{11280}', '\u{11286}'), ('\u{11288}', '\u{11288}'), ('\u{1128a}', '\u{1128d}'),
        ('\u{1128f}', '\u{1129d}'), ('\u{1129f}', '\u{112a8}'), ('\u{112b0}', '\u{112e8}'),
        ('\u{11300}', '\u{11303}'), ('\u{11305}', '\u{1130c}'), ('\u{1130f}', '\u{11310}'),
        ('\u{11313}', '\u{11328}'), ('\u{1132a}', '\u{11330}'), ('\u{11332}', '\u{11333}'),
        ('\u{11335}', '\u{11339}'), ('\u{1133d}', '\u{11344}'), ('\u{11347}', '\u{11348}'),
        ('\u{1134b}', '\u{1134c}'), ('\u{11350}', '\u{11350}'), ('\u{11357}', '\u{11357}'),
        ('\u{1135d}', '\u{11363}'), ('\u{11400}', '\u{11441}'), ('\u{11443}', '\u{11445}'),
        ('\u{11447}', '\u{1144a}'), ('\u{11480}', '\u{114c1}'), ('\u{114c4}', '\u{114c5}'),
        ('\u{114c7}', '\u{114c7}'), ('\u{11580}', '\u{115b5}'), ('\u{115b8}', '\u{115be}'),
        ('\u{115d8}', '\u{115dd}'), ('\u{11600}', '\u{1163e}'), ('\u{11640}', '\u{11640}'),
        ('\u{11644}', '\u{11644}'), ('\u{11680}', '\u{116b5}'), ('\u{11700}', '\u{11719}'),
        ('\u{1171d}', '\u{1172a}'), ('\u{118a0}', '\u{118df}'), ('\u{118ff}', '\u{118ff}'),
        ('\u{11ac0}', '\u{11af8}'), ('\u{11c00}', '\u{11c08}'), ('\u{11c0a}', '\u{11c36}'),
        ('\u{11c38}', '\u{11c3e}'), ('\u{11c40}', '\u{11c40}'), ('\u{11c72}', '\u{11c8f}'),
        ('\u{11c92}', '\u{11ca7}'), ('\u{11ca9}', '\u{11cb6}'), ('\u{12000}', '\u{12399}'),
        ('\u{12400}', '\u{1246e}'), ('\u{12480}', '\u{12543}'), ('\u{13000}', '\u{1342e}'),
        ('\u{14400}', '\u{14646}'), ('\u{16800}', '\u{16a38}'), ('\u{16a40}', '\u{16a5e}'),
        ('\u{16ad0}', '\u{16aed}'), ('\u{16b00}', '\u{16b36}'), ('\u{16b40}', '\u{16b43}'),
        ('\u{16b63}', '\u{16b77}'), ('\u{16b7d}', '\u{16b8f}'), ('\u{16f00}', '\u{16f44}'),
        ('\u{16f50}', '\u{16f7e}'), ('\u{16f93}', '\u{16f9f}'), ('\u{16fe0}', '\u{16fe0}'),
        ('\u{17000}', '\u{187ec}'), ('\u{18800}', '\u{18af2}'), ('\u{1b000}', '\u{1b001}'),
        ('\u{1bc00}', '\u{1bc6a}'), ('\u{1bc70}', '\u{1bc7c}'), ('\u{1bc80}', '\u{1bc88}'),
        ('\u{1bc90}', '\u{1bc99}'), ('\u{1bc9e}', '\u{1bc9e}'), ('\u{1d400}', '\u{1d454}'),
        ('\u{1d456}', '\u{1d49c}'), ('\u{1d49e}', '\u{1d49f}'), ('\u{1d4a2}', '\u{1d4a2}'),
        ('\u{1d4a5}', '\u{1d4a6}'), ('\u{1d4a9}', '\u{1d4ac}'), ('\u{1d4ae}', '\u{1d4b9}'),
        ('\u{1d4bb}', '\u{1d4bb}'), ('\u{1d4bd}', '\u{1d4c3}'), ('\u{1d4c5}', '\u{1d505}'),
        ('\u{1d507}', '\u{1d50a}'), ('\u{1d50d}', '\u{1d514}'), ('\u{1d516}', '\u{1d51c}'),
        ('\u{1d51e}', '\u{1d539}'), ('\u{1d53b}', '\u{1d53e}'), ('\u{1d540}', '\u{1d544}'),
        ('\u{1d546}', '\u{1d546}'), ('\u{1d54a}', '\u{1d550}'), ('\u{1d552}', '\u{1d6a5}'),
        ('\u{1d6a8}', '\u{1d6c0}'), ('\u{1d6c2}', '\u{1d6da}'), ('\u{1d6dc}', '\u{1d6fa}'),
        ('\u{1d6fc}', '\u{1d714}'), ('\u{1d716}', '\u{1d734}'), ('\u{1d736}', '\u{1d74e}'),
        ('\u{1d750}', '\u{1d76e}'), ('\u{1d770}', '\u{1d788}'), ('\u{1d78a}', '\u{1d7a8}'),
        ('\u{1d7aa}', '\u{1d7c2}'), ('\u{1d7c4}', '\u{1d7cb}'), ('\u{1e000}', '\u{1e006}'),
        ('\u{1e008}', '\u{1e018}'), ('\u{1e01b}', '\u{1e021}'), ('\u{1e023}', '\u{1e024}'),
        ('\u{1e026}', '\u{1e02a}'), ('\u{1e800}', '\u{1e8c4}'), ('\u{1e900}', '\u{1e943}'),
        ('\u{1e947}', '\u{1e947}'), ('\u{1ee00}', '\u{1ee03}'), ('\u{1ee05}', '\u{1ee1f}'),
        ('\u{1ee21}', '\u{1ee22}'), ('\u{1ee24}', '\u{1ee24}'), ('\u{1ee27}', '\u{1ee27}'),
        ('\u{1ee29}', '\u{1ee32}'), ('\u{1ee34}', '\u{1ee37}'), ('\u{1ee39}', '\u{1ee39}'),
        ('\u{1ee3b}', '\u{1ee3b}'), ('\u{1ee42}', '\u{1ee42}'), ('\u{1ee47}', '\u{1ee47}'),
        ('\u{1ee49}', '\u{1ee49}'), ('\u{1ee4b}', '\u{1ee4b}'), ('\u{1ee4d}', '\u{1ee4f}'),
        ('\u{1ee51}', '\u{1ee52}'), ('\u{1ee54}', '\u{1ee54}'), ('\u{1ee57}', '\u{1ee57}'),
        ('\u{1ee59}', '\u{1ee59}'), ('\u{1ee5b}', '\u{1ee5b}'), ('\u{1ee5d}', '\u{1ee5d}'),
        ('\u{1ee5f}', '\u{1ee5f}'), ('\u{1ee61}', '\u{1ee62}'), ('\u{1ee64}', '\u{1ee64}'),
        ('\u{1ee67}', '\u{1ee6a}'), ('\u{1ee6c}', '\u{1ee72}'), ('\u{1ee74}', '\u{1ee77}'),
        ('\u{1ee79}', '\u{1ee7c}'), ('\u{1ee7e}', '\u{1ee7e}'), ('\u{1ee80}', '\u{1ee89}'),
        ('\u{1ee8b}', '\u{1ee9b}'), ('\u{1eea1}', '\u{1eea3}'), ('\u{1eea5}', '\u{1eea9}'),
        ('\u{1eeab}', '\u{1eebb}'), ('\u{1f130}', '\u{1f149}'), ('\u{1f150}', '\u{1f169}'),
        ('\u{1f170}', '\u{1f189}'), ('\u{20000}', '\u{2a6d6}'), ('\u{2a700}', '\u{2b734}'),
        ('\u{2b740}', '\u{2b81d}'), ('\u{2b820}', '\u{2cea1}'), ('\u{2f800}', '\u{2fa1d}')
    ];

    #[inline]
    pub fn Alphabetic(c: char) -> bool {
        super::util::bsearch_range_table(c, Alphabetic_table)
    }

}

pub mod grapheme {
    use core::result::Result::{Ok, Err};

    pub use self::GraphemeCat::*;

    #[allow(non_camel_case_types)]
    #[derive(Clone, Copy, PartialEq, Eq)]
    pub enum GraphemeCat {
        GC_Any,
        GC_CR,
        GC_Control,
        GC_E_Base,
        GC_E_Base_GAZ,
        GC_E_Modifier,
        GC_Extend,
        GC_Glue_After_Zwj,
        GC_L,
        GC_LF,
        GC_LV,
        GC_LVT,
        GC_Prepend,
        GC_Regional_Indicator,
        GC_SpacingMark,
        GC_T,
        GC_V,
        GC_ZWJ,
    }

    fn bsearch_range_value_table(c: char, r: &'static [(char, char, GraphemeCat)]) -> GraphemeCat {
        use core::cmp::Ordering::{Equal, Less, Greater};
        match r.binary_search_by(|&(lo, hi, _)| {
            if lo <= c && c <= hi { Equal }
            else if hi < c { Less }
            else { Greater }
        }) {
            Ok(idx) => {
                let (_, _, cat) = r[idx];
                cat
            }
            Err(_) => GC_Any
        }
    }

    pub fn grapheme_category(c: char) -> GraphemeCat {
        bsearch_range_value_table(c, grapheme_cat_table)
    }

    const grapheme_cat_table: &'static [(char, char, GraphemeCat)] = &[
        ('\u{0}', '\u{9}', GC_Control), ('\u{a}', '\u{a}', GC_LF), ('\u{b}', '\u{c}', GC_Control),
        ('\u{d}', '\u{d}', GC_CR), ('\u{e}', '\u{1f}', GC_Control), ('\u{7f}', '\u{9f}',
        GC_Control), ('\u{ad}', '\u{ad}', GC_Control), ('\u{300}', '\u{36f}', GC_Extend),
        ('\u{483}', '\u{489}', GC_Extend), ('\u{591}', '\u{5bd}', GC_Extend), ('\u{5bf}', '\u{5bf}',
        GC_Extend), ('\u{5c1}', '\u{5c2}', GC_Extend), ('\u{5c4}', '\u{5c5}', GC_Extend),
        ('\u{5c7}', '\u{5c7}', GC_Extend), ('\u{600}', '\u{605}', GC_Prepend), ('\u{610}',
        '\u{61a}', GC_Extend), ('\u{61c}', '\u{61c}', GC_Control), ('\u{64b}', '\u{65f}',
        GC_Extend), ('\u{670}', '\u{670}', GC_Extend), ('\u{6d6}', '\u{6dc}', GC_Extend),
        ('\u{6dd}', '\u{6dd}', GC_Prepend), ('\u{6df}', '\u{6e4}', GC_Extend), ('\u{6e7}',
        '\u{6e8}', GC_Extend), ('\u{6ea}', '\u{6ed}', GC_Extend), ('\u{70f}', '\u{70f}',
        GC_Prepend), ('\u{711}', '\u{711}', GC_Extend), ('\u{730}', '\u{74a}', GC_Extend),
        ('\u{7a6}', '\u{7b0}', GC_Extend), ('\u{7eb}', '\u{7f3}', GC_Extend), ('\u{816}', '\u{819}',
        GC_Extend), ('\u{81b}', '\u{823}', GC_Extend), ('\u{825}', '\u{827}', GC_Extend),
        ('\u{829}', '\u{82d}', GC_Extend), ('\u{859}', '\u{85b}', GC_Extend), ('\u{8d4}', '\u{8e1}',
        GC_Extend), ('\u{8e2}', '\u{8e2}', GC_Prepend), ('\u{8e3}', '\u{902}', GC_Extend),
        ('\u{903}', '\u{903}', GC_SpacingMark), ('\u{93a}', '\u{93a}', GC_Extend), ('\u{93b}',
        '\u{93b}', GC_SpacingMark), ('\u{93c}', '\u{93c}', GC_Extend), ('\u{93e}', '\u{940}',
        GC_SpacingMark), ('\u{941}', '\u{948}', GC_Extend), ('\u{949}', '\u{94c}', GC_SpacingMark),
        ('\u{94d}', '\u{94d}', GC_Extend), ('\u{94e}', '\u{94f}', GC_SpacingMark), ('\u{951}',
        '\u{957}', GC_Extend), ('\u{962}', '\u{963}', GC_Extend), ('\u{981}', '\u{981}', GC_Extend),
        ('\u{982}', '\u{983}', GC_SpacingMark), ('\u{9bc}', '\u{9bc}', GC_Extend), ('\u{9be}',
        '\u{9be}', GC_Extend), ('\u{9bf}', '\u{9c0}', GC_SpacingMark), ('\u{9c1}', '\u{9c4}',
        GC_Extend), ('\u{9c7}', '\u{9c8}', GC_SpacingMark), ('\u{9cb}', '\u{9cc}', GC_SpacingMark),
        ('\u{9cd}', '\u{9cd}', GC_Extend), ('\u{9d7}', '\u{9d7}', GC_Extend), ('\u{9e2}', '\u{9e3}',
        GC_Extend), ('\u{a01}', '\u{a02}', GC_Extend), ('\u{a03}', '\u{a03}', GC_SpacingMark),
        ('\u{a3c}', '\u{a3c}', GC_Extend), ('\u{a3e}', '\u{a40}', GC_SpacingMark), ('\u{a41}',
        '\u{a42}', GC_Extend), ('\u{a47}', '\u{a48}', GC_Extend), ('\u{a4b}', '\u{a4d}', GC_Extend),
        ('\u{a51}', '\u{a51}', GC_Extend), ('\u{a70}', '\u{a71}', GC_Extend), ('\u{a75}', '\u{a75}',
        GC_Extend), ('\u{a81}', '\u{a82}', GC_Extend), ('\u{a83}', '\u{a83}', GC_SpacingMark),
        ('\u{abc}', '\u{abc}', GC_Extend), ('\u{abe}', '\u{ac0}', GC_SpacingMark), ('\u{ac1}',
        '\u{ac5}', GC_Extend), ('\u{ac7}', '\u{ac8}', GC_Extend), ('\u{ac9}', '\u{ac9}',
        GC_SpacingMark), ('\u{acb}', '\u{acc}', GC_SpacingMark), ('\u{acd}', '\u{acd}', GC_Extend),
        ('\u{ae2}', '\u{ae3}', GC_Extend), ('\u{b01}', '\u{b01}', GC_Extend), ('\u{b02}', '\u{b03}',
        GC_SpacingMark), ('\u{b3c}', '\u{b3c}', GC_Extend), ('\u{b3e}', '\u{b3f}', GC_Extend),
        ('\u{b40}', '\u{b40}', GC_SpacingMark), ('\u{b41}', '\u{b44}', GC_Extend), ('\u{b47}',
        '\u{b48}', GC_SpacingMark), ('\u{b4b}', '\u{b4c}', GC_SpacingMark), ('\u{b4d}', '\u{b4d}',
        GC_Extend), ('\u{b56}', '\u{b57}', GC_Extend), ('\u{b62}', '\u{b63}', GC_Extend),
        ('\u{b82}', '\u{b82}', GC_Extend), ('\u{bbe}', '\u{bbe}', GC_Extend), ('\u{bbf}', '\u{bbf}',
        GC_SpacingMark), ('\u{bc0}', '\u{bc0}', GC_Extend), ('\u{bc1}', '\u{bc2}', GC_SpacingMark),
        ('\u{bc6}', '\u{bc8}', GC_SpacingMark), ('\u{bca}', '\u{bcc}', GC_SpacingMark), ('\u{bcd}',
        '\u{bcd}', GC_Extend), ('\u{bd7}', '\u{bd7}', GC_Extend), ('\u{c00}', '\u{c00}', GC_Extend),
        ('\u{c01}', '\u{c03}', GC_SpacingMark), ('\u{c3e}', '\u{c40}', GC_Extend), ('\u{c41}',
        '\u{c44}', GC_SpacingMark), ('\u{c46}', '\u{c48}', GC_Extend), ('\u{c4a}', '\u{c4d}',
        GC_Extend), ('\u{c55}', '\u{c56}', GC_Extend), ('\u{c62}', '\u{c63}', GC_Extend),
        ('\u{c81}', '\u{c81}', GC_Extend), ('\u{c82}', '\u{c83}', GC_SpacingMark), ('\u{cbc}',
        '\u{cbc}', GC_Extend), ('\u{cbe}', '\u{cbe}', GC_SpacingMark), ('\u{cbf}', '\u{cbf}',
        GC_Extend), ('\u{cc0}', '\u{cc1}', GC_SpacingMark), ('\u{cc2}', '\u{cc2}', GC_Extend),
        ('\u{cc3}', '\u{cc4}', GC_SpacingMark), ('\u{cc6}', '\u{cc6}', GC_Extend), ('\u{cc7}',
        '\u{cc8}', GC_SpacingMark), ('\u{cca}', '\u{ccb}', GC_SpacingMark), ('\u{ccc}', '\u{ccd}',
        GC_Extend), ('\u{cd5}', '\u{cd6}', GC_Extend), ('\u{ce2}', '\u{ce3}', GC_Extend),
        ('\u{d01}', '\u{d01}', GC_Extend), ('\u{d02}', '\u{d03}', GC_SpacingMark), ('\u{d3e}',
        '\u{d3e}', GC_Extend), ('\u{d3f}', '\u{d40}', GC_SpacingMark), ('\u{d41}', '\u{d44}',
        GC_Extend), ('\u{d46}', '\u{d48}', GC_SpacingMark), ('\u{d4a}', '\u{d4c}', GC_SpacingMark),
        ('\u{d4d}', '\u{d4d}', GC_Extend), ('\u{d4e}', '\u{d4e}', GC_Prepend), ('\u{d57}',
        '\u{d57}', GC_Extend), ('\u{d62}', '\u{d63}', GC_Extend), ('\u{d82}', '\u{d83}',
        GC_SpacingMark), ('\u{dca}', '\u{dca}', GC_Extend), ('\u{dcf}', '\u{dcf}', GC_Extend),
        ('\u{dd0}', '\u{dd1}', GC_SpacingMark), ('\u{dd2}', '\u{dd4}', GC_Extend), ('\u{dd6}',
        '\u{dd6}', GC_Extend), ('\u{dd8}', '\u{dde}', GC_SpacingMark), ('\u{ddf}', '\u{ddf}',
        GC_Extend), ('\u{df2}', '\u{df3}', GC_SpacingMark), ('\u{e31}', '\u{e31}', GC_Extend),
        ('\u{e33}', '\u{e33}', GC_SpacingMark), ('\u{e34}', '\u{e3a}', GC_Extend), ('\u{e47}',
        '\u{e4e}', GC_Extend), ('\u{eb1}', '\u{eb1}', GC_Extend), ('\u{eb3}', '\u{eb3}',
        GC_SpacingMark), ('\u{eb4}', '\u{eb9}', GC_Extend), ('\u{ebb}', '\u{ebc}', GC_Extend),
        ('\u{ec8}', '\u{ecd}', GC_Extend), ('\u{f18}', '\u{f19}', GC_Extend), ('\u{f35}', '\u{f35}',
        GC_Extend), ('\u{f37}', '\u{f37}', GC_Extend), ('\u{f39}', '\u{f39}', GC_Extend),
        ('\u{f3e}', '\u{f3f}', GC_SpacingMark), ('\u{f71}', '\u{f7e}', GC_Extend), ('\u{f7f}',
        '\u{f7f}', GC_SpacingMark), ('\u{f80}', '\u{f84}', GC_Extend), ('\u{f86}', '\u{f87}',
        GC_Extend), ('\u{f8d}', '\u{f97}', GC_Extend), ('\u{f99}', '\u{fbc}', GC_Extend),
        ('\u{fc6}', '\u{fc6}', GC_Extend), ('\u{102d}', '\u{1030}', GC_Extend), ('\u{1031}',
        '\u{1031}', GC_SpacingMark), ('\u{1032}', '\u{1037}', GC_Extend), ('\u{1039}', '\u{103a}',
        GC_Extend), ('\u{103b}', '\u{103c}', GC_SpacingMark), ('\u{103d}', '\u{103e}', GC_Extend),
        ('\u{1056}', '\u{1057}', GC_SpacingMark), ('\u{1058}', '\u{1059}', GC_Extend), ('\u{105e}',
        '\u{1060}', GC_Extend), ('\u{1071}', '\u{1074}', GC_Extend), ('\u{1082}', '\u{1082}',
        GC_Extend), ('\u{1084}', '\u{1084}', GC_SpacingMark), ('\u{1085}', '\u{1086}', GC_Extend),
        ('\u{108d}', '\u{108d}', GC_Extend), ('\u{109d}', '\u{109d}', GC_Extend), ('\u{1100}',
        '\u{115f}', GC_L), ('\u{1160}', '\u{11a7}', GC_V), ('\u{11a8}', '\u{11ff}', GC_T),
        ('\u{135d}', '\u{135f}', GC_Extend), ('\u{1712}', '\u{1714}', GC_Extend), ('\u{1732}',
        '\u{1734}', GC_Extend), ('\u{1752}', '\u{1753}', GC_Extend), ('\u{1772}', '\u{1773}',
        GC_Extend), ('\u{17b4}', '\u{17b5}', GC_Extend), ('\u{17b6}', '\u{17b6}', GC_SpacingMark),
        ('\u{17b7}', '\u{17bd}', GC_Extend), ('\u{17be}', '\u{17c5}', GC_SpacingMark), ('\u{17c6}',
        '\u{17c6}', GC_Extend), ('\u{17c7}', '\u{17c8}', GC_SpacingMark), ('\u{17c9}', '\u{17d3}',
        GC_Extend), ('\u{17dd}', '\u{17dd}', GC_Extend), ('\u{180b}', '\u{180d}', GC_Extend),
        ('\u{180e}', '\u{180e}', GC_Control), ('\u{1885}', '\u{1886}', GC_Extend), ('\u{18a9}',
        '\u{18a9}', GC_Extend), ('\u{1920}', '\u{1922}', GC_Extend), ('\u{1923}', '\u{1926}',
        GC_SpacingMark), ('\u{1927}', '\u{1928}', GC_Extend), ('\u{1929}', '\u{192b}',
        GC_SpacingMark), ('\u{1930}', '\u{1931}', GC_SpacingMark), ('\u{1932}', '\u{1932}',
        GC_Extend), ('\u{1933}', '\u{1938}', GC_SpacingMark), ('\u{1939}', '\u{193b}', GC_Extend),
        ('\u{1a17}', '\u{1a18}', GC_Extend), ('\u{1a19}', '\u{1a1a}', GC_SpacingMark), ('\u{1a1b}',
        '\u{1a1b}', GC_Extend), ('\u{1a55}', '\u{1a55}', GC_SpacingMark), ('\u{1a56}', '\u{1a56}',
        GC_Extend), ('\u{1a57}', '\u{1a57}', GC_SpacingMark), ('\u{1a58}', '\u{1a5e}', GC_Extend),
        ('\u{1a60}', '\u{1a60}', GC_Extend), ('\u{1a62}', '\u{1a62}', GC_Extend), ('\u{1a65}',
        '\u{1a6c}', GC_Extend), ('\u{1a6d}', '\u{1a72}', GC_SpacingMark), ('\u{1a73}', '\u{1a7c}',
        GC_Extend), ('\u{1a7f}', '\u{1a7f}', GC_Extend), ('\u{1ab0}', '\u{1abe}', GC_Extend),
        ('\u{1b00}', '\u{1b03}', GC_Extend), ('\u{1b04}', '\u{1b04}', GC_SpacingMark), ('\u{1b34}',
        '\u{1b34}', GC_Extend), ('\u{1b35}', '\u{1b35}', GC_SpacingMark), ('\u{1b36}', '\u{1b3a}',
        GC_Extend), ('\u{1b3b}', '\u{1b3b}', GC_SpacingMark), ('\u{1b3c}', '\u{1b3c}', GC_Extend),
        ('\u{1b3d}', '\u{1b41}', GC_SpacingMark), ('\u{1b42}', '\u{1b42}', GC_Extend), ('\u{1b43}',
        '\u{1b44}', GC_SpacingMark), ('\u{1b6b}', '\u{1b73}', GC_Extend), ('\u{1b80}', '\u{1b81}',
        GC_Extend), ('\u{1b82}', '\u{1b82}', GC_SpacingMark), ('\u{1ba1}', '\u{1ba1}',
        GC_SpacingMark), ('\u{1ba2}', '\u{1ba5}', GC_Extend), ('\u{1ba6}', '\u{1ba7}',
        GC_SpacingMark), ('\u{1ba8}', '\u{1ba9}', GC_Extend), ('\u{1baa}', '\u{1baa}',
        GC_SpacingMark), ('\u{1bab}', '\u{1bad}', GC_Extend), ('\u{1be6}', '\u{1be6}', GC_Extend),
        ('\u{1be7}', '\u{1be7}', GC_SpacingMark), ('\u{1be8}', '\u{1be9}', GC_Extend), ('\u{1bea}',
        '\u{1bec}', GC_SpacingMark), ('\u{1bed}', '\u{1bed}', GC_Extend), ('\u{1bee}', '\u{1bee}',
        GC_SpacingMark), ('\u{1bef}', '\u{1bf1}', GC_Extend), ('\u{1bf2}', '\u{1bf3}',
        GC_SpacingMark), ('\u{1c24}', '\u{1c2b}', GC_SpacingMark), ('\u{1c2c}', '\u{1c33}',
        GC_Extend), ('\u{1c34}', '\u{1c35}', GC_SpacingMark), ('\u{1c36}', '\u{1c37}', GC_Extend),
        ('\u{1cd0}', '\u{1cd2}', GC_Extend), ('\u{1cd4}', '\u{1ce0}', GC_Extend), ('\u{1ce1}',
        '\u{1ce1}', GC_SpacingMark), ('\u{1ce2}', '\u{1ce8}', GC_Extend), ('\u{1ced}', '\u{1ced}',
        GC_Extend), ('\u{1cf2}', '\u{1cf3}', GC_SpacingMark), ('\u{1cf4}', '\u{1cf4}', GC_Extend),
        ('\u{1cf8}', '\u{1cf9}', GC_Extend), ('\u{1dc0}', '\u{1df5}', GC_Extend), ('\u{1dfb}',
        '\u{1dff}', GC_Extend), ('\u{200b}', '\u{200b}', GC_Control), ('\u{200c}', '\u{200c}',
        GC_Extend), ('\u{200d}', '\u{200d}', GC_ZWJ), ('\u{200e}', '\u{200f}', GC_Control),
        ('\u{2028}', '\u{202e}', GC_Control), ('\u{2060}', '\u{206f}', GC_Control), ('\u{20d0}',
        '\u{20f0}', GC_Extend), ('\u{261d}', '\u{261d}', GC_E_Base), ('\u{26f9}', '\u{26f9}',
        GC_E_Base), ('\u{270a}', '\u{270d}', GC_E_Base), ('\u{2764}', '\u{2764}',
        GC_Glue_After_Zwj), ('\u{2cef}', '\u{2cf1}', GC_Extend), ('\u{2d7f}', '\u{2d7f}',
        GC_Extend), ('\u{2de0}', '\u{2dff}', GC_Extend), ('\u{302a}', '\u{302f}', GC_Extend),
        ('\u{3099}', '\u{309a}', GC_Extend), ('\u{a66f}', '\u{a672}', GC_Extend), ('\u{a674}',
        '\u{a67d}', GC_Extend), ('\u{a69e}', '\u{a69f}', GC_Extend), ('\u{a6f0}', '\u{a6f1}',
        GC_Extend), ('\u{a802}', '\u{a802}', GC_Extend), ('\u{a806}', '\u{a806}', GC_Extend),
        ('\u{a80b}', '\u{a80b}', GC_Extend), ('\u{a823}', '\u{a824}', GC_SpacingMark), ('\u{a825}',
        '\u{a826}', GC_Extend), ('\u{a827}', '\u{a827}', GC_SpacingMark), ('\u{a880}', '\u{a881}',
        GC_SpacingMark), ('\u{a8b4}', '\u{a8c3}', GC_SpacingMark), ('\u{a8c4}', '\u{a8c5}',
        GC_Extend), ('\u{a8e0}', '\u{a8f1}', GC_Extend), ('\u{a926}', '\u{a92d}', GC_Extend),
        ('\u{a947}', '\u{a951}', GC_Extend), ('\u{a952}', '\u{a953}', GC_SpacingMark), ('\u{a960}',
        '\u{a97c}', GC_L), ('\u{a980}', '\u{a982}', GC_Extend), ('\u{a983}', '\u{a983}',
        GC_SpacingMark), ('\u{a9b3}', '\u{a9b3}', GC_Extend), ('\u{a9b4}', '\u{a9b5}',
        GC_SpacingMark), ('\u{a9b6}', '\u{a9b9}', GC_Extend), ('\u{a9ba}', '\u{a9bb}',
        GC_SpacingMark), ('\u{a9bc}', '\u{a9bc}', GC_Extend), ('\u{a9bd}', '\u{a9c0}',
        GC_SpacingMark), ('\u{a9e5}', '\u{a9e5}', GC_Extend), ('\u{aa29}', '\u{aa2e}', GC_Extend),
        ('\u{aa2f}', '\u{aa30}', GC_SpacingMark), ('\u{aa31}', '\u{aa32}', GC_Extend), ('\u{aa33}',
        '\u{aa34}', GC_SpacingMark), ('\u{aa35}', '\u{aa36}', GC_Extend), ('\u{aa43}', '\u{aa43}',
        GC_Extend), ('\u{aa4c}', '\u{aa4c}', GC_Extend), ('\u{aa4d}', '\u{aa4d}', GC_SpacingMark),
        ('\u{aa7c}', '\u{aa7c}', GC_Extend), ('\u{aab0}', '\u{aab0}', GC_Extend), ('\u{aab2}',
        '\u{aab4}', GC_Extend), ('\u{aab7}', '\u{aab8}', GC_Extend), ('\u{aabe}', '\u{aabf}',
        GC_Extend), ('\u{aac1}', '\u{aac1}', GC_Extend), ('\u{aaeb}', '\u{aaeb}', GC_SpacingMark),
        ('\u{aaec}', '\u{aaed}', GC_Extend), ('\u{aaee}', '\u{aaef}', GC_SpacingMark), ('\u{aaf5}',
        '\u{aaf5}', GC_SpacingMark), ('\u{aaf6}', '\u{aaf6}', GC_Extend), ('\u{abe3}', '\u{abe4}',
        GC_SpacingMark), ('\u{abe5}', '\u{abe5}', GC_Extend), ('\u{abe6}', '\u{abe7}',
        GC_SpacingMark), ('\u{abe8}', '\u{abe8}', GC_Extend), ('\u{abe9}', '\u{abea}',
        GC_SpacingMark), ('\u{abec}', '\u{abec}', GC_SpacingMark), ('\u{abed}', '\u{abed}',
        GC_Extend), ('\u{ac00}', '\u{ac00}', GC_LV), ('\u{ac01}', '\u{ac1b}', GC_LVT), ('\u{ac1c}',
        '\u{ac1c}', GC_LV), ('\u{ac1d}', '\u{ac37}', GC_LVT), ('\u{ac38}', '\u{ac38}', GC_LV),
        ('\u{ac39}', '\u{ac53}', GC_LVT), ('\u{ac54}', '\u{ac54}', GC_LV), ('\u{ac55}', '\u{ac6f}',
        GC_LVT), ('\u{ac70}', '\u{ac70}', GC_LV), ('\u{ac71}', '\u{ac8b}', GC_LVT), ('\u{ac8c}',
        '\u{ac8c}', GC_LV), ('\u{ac8d}', '\u{aca7}', GC_LVT), ('\u{aca8}', '\u{aca8}', GC_LV),
        ('\u{aca9}', '\u{acc3}', GC_LVT), ('\u{acc4}', '\u{acc4}', GC_LV), ('\u{acc5}', '\u{acdf}',
        GC_LVT), ('\u{ace0}', '\u{ace0}', GC_LV), ('\u{ace1}', '\u{acfb}', GC_LVT), ('\u{acfc}',
        '\u{acfc}', GC_LV), ('\u{acfd}', '\u{ad17}', GC_LVT), ('\u{ad18}', '\u{ad18}', GC_LV),
        ('\u{ad19}', '\u{ad33}', GC_LVT), ('\u{ad34}', '\u{ad34}', GC_LV), ('\u{ad35}', '\u{ad4f}',
        GC_LVT), ('\u{ad50}', '\u{ad50}', GC_LV), ('\u{ad51}', '\u{ad6b}', GC_LVT), ('\u{ad6c}',
        '\u{ad6c}', GC_LV), ('\u{ad6d}', '\u{ad87}', GC_LVT), ('\u{ad88}', '\u{ad88}', GC_LV),
        ('\u{ad89}', '\u{ada3}', GC_LVT), ('\u{ada4}', '\u{ada4}', GC_LV), ('\u{ada5}', '\u{adbf}',
        GC_LVT), ('\u{adc0}', '\u{adc0}', GC_LV), ('\u{adc1}', '\u{addb}', GC_LVT), ('\u{addc}',
        '\u{addc}', GC_LV), ('\u{addd}', '\u{adf7}', GC_LVT), ('\u{adf8}', '\u{adf8}', GC_LV),
        ('\u{adf9}', '\u{ae13}', GC_LVT), ('\u{ae14}', '\u{ae14}', GC_LV), ('\u{ae15}', '\u{ae2f}',
        GC_LVT), ('\u{ae30}', '\u{ae30}', GC_LV), ('\u{ae31}', '\u{ae4b}', GC_LVT), ('\u{ae4c}',
        '\u{ae4c}', GC_LV), ('\u{ae4d}', '\u{ae67}', GC_LVT), ('\u{ae68}', '\u{ae68}', GC_LV),
        ('\u{ae69}', '\u{ae83}', GC_LVT), ('\u{ae84}', '\u{ae84}', GC_LV), ('\u{ae85}', '\u{ae9f}',
        GC_LVT), ('\u{aea0}', '\u{aea0}', GC_LV), ('\u{aea1}', '\u{aebb}', GC_LVT), ('\u{aebc}',
        '\u{aebc}', GC_LV), ('\u{aebd}', '\u{aed7}', GC_LVT), ('\u{aed8}', '\u{aed8}', GC_LV),
        ('\u{aed9}', '\u{aef3}', GC_LVT), ('\u{aef4}', '\u{aef4}', GC_LV), ('\u{aef5}', '\u{af0f}',
        GC_LVT), ('\u{af10}', '\u{af10}', GC_LV), ('\u{af11}', '\u{af2b}', GC_LVT), ('\u{af2c}',
        '\u{af2c}', GC_LV), ('\u{af2d}', '\u{af47}', GC_LVT), ('\u{af48}', '\u{af48}', GC_LV),
        ('\u{af49}', '\u{af63}', GC_LVT), ('\u{af64}', '\u{af64}', GC_LV), ('\u{af65}', '\u{af7f}',
        GC_LVT), ('\u{af80}', '\u{af80}', GC_LV), ('\u{af81}', '\u{af9b}', GC_LVT), ('\u{af9c}',
        '\u{af9c}', GC_LV), ('\u{af9d}', '\u{afb7}', GC_LVT), ('\u{afb8}', '\u{afb8}', GC_LV),
        ('\u{afb9}', '\u{afd3}', GC_LVT), ('\u{afd4}', '\u{afd4}', GC_LV), ('\u{afd5}', '\u{afef}',
        GC_LVT), ('\u{aff0}', '\u{aff0}', GC_LV), ('\u{aff1}', '\u{b00b}', GC_LVT), ('\u{b00c}',
        '\u{b00c}', GC_LV), ('\u{b00d}', '\u{b027}', GC_LVT), ('\u{b028}', '\u{b028}', GC_LV),
        ('\u{b029}', '\u{b043}', GC_LVT), ('\u{b044}', '\u{b044}', GC_LV), ('\u{b045}', '\u{b05f}',
        GC_LVT), ('\u{b060}', '\u{b060}', GC_LV), ('\u{b061}', '\u{b07b}', GC_LVT), ('\u{b07c}',
        '\u{b07c}', GC_LV), ('\u{b07d}', '\u{b097}', GC_LVT), ('\u{b098}', '\u{b098}', GC_LV),
        ('\u{b099}', '\u{b0b3}', GC_LVT), ('\u{b0b4}', '\u{b0b4}', GC_LV), ('\u{b0b5}', '\u{b0cf}',
        GC_LVT), ('\u{b0d0}', '\u{b0d0}', GC_LV), ('\u{b0d1}', '\u{b0eb}', GC_LVT), ('\u{b0ec}',
        '\u{b0ec}', GC_LV), ('\u{b0ed}', '\u{b107}', GC_LVT), ('\u{b108}', '\u{b108}', GC_LV),
        ('\u{b109}', '\u{b123}', GC_LVT), ('\u{b124}', '\u{b124}', GC_LV), ('\u{b125}', '\u{b13f}',
        GC_LVT), ('\u{b140}', '\u{b140}', GC_LV), ('\u{b141}', '\u{b15b}', GC_LVT), ('\u{b15c}',
        '\u{b15c}', GC_LV), ('\u{b15d}', '\u{b177}', GC_LVT), ('\u{b178}', '\u{b178}', GC_LV),
        ('\u{b179}', '\u{b193}', GC_LVT), ('\u{b194}', '\u{b194}', GC_LV), ('\u{b195}', '\u{b1af}',
        GC_LVT), ('\u{b1b0}', '\u{b1b0}', GC_LV), ('\u{b1b1}', '\u{b1cb}', GC_LVT), ('\u{b1cc}',
        '\u{b1cc}', GC_LV), ('\u{b1cd}', '\u{b1e7}', GC_LVT), ('\u{b1e8}', '\u{b1e8}', GC_LV),
        ('\u{b1e9}', '\u{b203}', GC_LVT), ('\u{b204}', '\u{b204}', GC_LV), ('\u{b205}', '\u{b21f}',
        GC_LVT), ('\u{b220}', '\u{b220}', GC_LV), ('\u{b221}', '\u{b23b}', GC_LVT), ('\u{b23c}',
        '\u{b23c}', GC_LV), ('\u{b23d}', '\u{b257}', GC_LVT), ('\u{b258}', '\u{b258}', GC_LV),
        ('\u{b259}', '\u{b273}', GC_LVT), ('\u{b274}', '\u{b274}', GC_LV), ('\u{b275}', '\u{b28f}',
        GC_LVT), ('\u{b290}', '\u{b290}', GC_LV), ('\u{b291}', '\u{b2ab}', GC_LVT), ('\u{b2ac}',
        '\u{b2ac}', GC_LV), ('\u{b2ad}', '\u{b2c7}', GC_LVT), ('\u{b2c8}', '\u{b2c8}', GC_LV),
        ('\u{b2c9}', '\u{b2e3}', GC_LVT), ('\u{b2e4}', '\u{b2e4}', GC_LV), ('\u{b2e5}', '\u{b2ff}',
        GC_LVT), ('\u{b300}', '\u{b300}', GC_LV), ('\u{b301}', '\u{b31b}', GC_LVT), ('\u{b31c}',
        '\u{b31c}', GC_LV), ('\u{b31d}', '\u{b337}', GC_LVT), ('\u{b338}', '\u{b338}', GC_LV),
        ('\u{b339}', '\u{b353}', GC_LVT), ('\u{b354}', '\u{b354}', GC_LV), ('\u{b355}', '\u{b36f}',
        GC_LVT), ('\u{b370}', '\u{b370}', GC_LV), ('\u{b371}', '\u{b38b}', GC_LVT), ('\u{b38c}',
        '\u{b38c}', GC_LV), ('\u{b38d}', '\u{b3a7}', GC_LVT), ('\u{b3a8}', '\u{b3a8}', GC_LV),
        ('\u{b3a9}', '\u{b3c3}', GC_LVT), ('\u{b3c4}', '\u{b3c4}', GC_LV), ('\u{b3c5}', '\u{b3df}',
        GC_LVT), ('\u{b3e0}', '\u{b3e0}', GC_LV), ('\u{b3e1}', '\u{b3fb}', GC_LVT), ('\u{b3fc}',
        '\u{b3fc}', GC_LV), ('\u{b3fd}', '\u{b417}', GC_LVT), ('\u{b418}', '\u{b418}', GC_LV),
        ('\u{b419}', '\u{b433}', GC_LVT), ('\u{b434}', '\u{b434}', GC_LV), ('\u{b435}', '\u{b44f}',
        GC_LVT), ('\u{b450}', '\u{b450}', GC_LV), ('\u{b451}', '\u{b46b}', GC_LVT), ('\u{b46c}',
        '\u{b46c}', GC_LV), ('\u{b46d}', '\u{b487}', GC_LVT), ('\u{b488}', '\u{b488}', GC_LV),
        ('\u{b489}', '\u{b4a3}', GC_LVT), ('\u{b4a4}', '\u{b4a4}', GC_LV), ('\u{b4a5}', '\u{b4bf}',
        GC_LVT), ('\u{b4c0}', '\u{b4c0}', GC_LV), ('\u{b4c1}', '\u{b4db}', GC_LVT), ('\u{b4dc}',
        '\u{b4dc}', GC_LV), ('\u{b4dd}', '\u{b4f7}', GC_LVT), ('\u{b4f8}', '\u{b4f8}', GC_LV),
        ('\u{b4f9}', '\u{b513}', GC_LVT), ('\u{b514}', '\u{b514}', GC_LV), ('\u{b515}', '\u{b52f}',
        GC_LVT), ('\u{b530}', '\u{b530}', GC_LV), ('\u{b531}', '\u{b54b}', GC_LVT), ('\u{b54c}',
        '\u{b54c}', GC_LV), ('\u{b54d}', '\u{b567}', GC_LVT), ('\u{b568}', '\u{b568}', GC_LV),
        ('\u{b569}', '\u{b583}', GC_LVT), ('\u{b584}', '\u{b584}', GC_LV), ('\u{b585}', '\u{b59f}',
        GC_LVT), ('\u{b5a0}', '\u{b5a0}', GC_LV), ('\u{b5a1}', '\u{b5bb}', GC_LVT), ('\u{b5bc}',
        '\u{b5bc}', GC_LV), ('\u{b5bd}', '\u{b5d7}', GC_LVT), ('\u{b5d8}', '\u{b5d8}', GC_LV),
        ('\u{b5d9}', '\u{b5f3}', GC_LVT), ('\u{b5f4}', '\u{b5f4}', GC_LV), ('\u{b5f5}', '\u{b60f}',
        GC_LVT), ('\u{b610}', '\u{b610}', GC_LV), ('\u{b611}', '\u{b62b}', GC_LVT), ('\u{b62c}',
        '\u{b62c}', GC_LV), ('\u{b62d}', '\u{b647}', GC_LVT), ('\u{b648}', '\u{b648}', GC_LV),
        ('\u{b649}', '\u{b663}', GC_LVT), ('\u{b664}', '\u{b664}', GC_LV), ('\u{b665}', '\u{b67f}',
        GC_LVT), ('\u{b680}', '\u{b680}', GC_LV), ('\u{b681}', '\u{b69b}', GC_LVT), ('\u{b69c}',
        '\u{b69c}', GC_LV), ('\u{b69d}', '\u{b6b7}', GC_LVT), ('\u{b6b8}', '\u{b6b8}', GC_LV),
        ('\u{b6b9}', '\u{b6d3}', GC_LVT), ('\u{b6d4}', '\u{b6d4}', GC_LV), ('\u{b6d5}', '\u{b6ef}',
        GC_LVT), ('\u{b6f0}', '\u{b6f0}', GC_LV), ('\u{b6f1}', '\u{b70b}', GC_LVT), ('\u{b70c}',
        '\u{b70c}', GC_LV), ('\u{b70d}', '\u{b727}', GC_LVT), ('\u{b728}', '\u{b728}', GC_LV),
        ('\u{b729}', '\u{b743}', GC_LVT), ('\u{b744}', '\u{b744}', GC_LV), ('\u{b745}', '\u{b75f}',
        GC_LVT), ('\u{b760}', '\u{b760}', GC_LV), ('\u{b761}', '\u{b77b}', GC_LVT), ('\u{b77c}',
        '\u{b77c}', GC_LV), ('\u{b77d}', '\u{b797}', GC_LVT), ('\u{b798}', '\u{b798}', GC_LV),
        ('\u{b799}', '\u{b7b3}', GC_LVT), ('\u{b7b4}', '\u{b7b4}', GC_LV), ('\u{b7b5}', '\u{b7cf}',
        GC_LVT), ('\u{b7d0}', '\u{b7d0}', GC_LV), ('\u{b7d1}', '\u{b7eb}', GC_LVT), ('\u{b7ec}',
        '\u{b7ec}', GC_LV), ('\u{b7ed}', '\u{b807}', GC_LVT), ('\u{b808}', '\u{b808}', GC_LV),
        ('\u{b809}', '\u{b823}', GC_LVT), ('\u{b824}', '\u{b824}', GC_LV), ('\u{b825}', '\u{b83f}',
        GC_LVT), ('\u{b840}', '\u{b840}', GC_LV), ('\u{b841}', '\u{b85b}', GC_LVT), ('\u{b85c}',
        '\u{b85c}', GC_LV), ('\u{b85d}', '\u{b877}', GC_LVT), ('\u{b878}', '\u{b878}', GC_LV),
        ('\u{b879}', '\u{b893}', GC_LVT), ('\u{b894}', '\u{b894}', GC_LV), ('\u{b895}', '\u{b8af}',
        GC_LVT), ('\u{b8b0}', '\u{b8b0}', GC_LV), ('\u{b8b1}', '\u{b8cb}', GC_LVT), ('\u{b8cc}',
        '\u{b8cc}', GC_LV), ('\u{b8cd}', '\u{b8e7}', GC_LVT), ('\u{b8e8}', '\u{b8e8}', GC_LV),
        ('\u{b8e9}', '\u{b903}', GC_LVT), ('\u{b904}', '\u{b904}', GC_LV), ('\u{b905}', '\u{b91f}',
        GC_LVT), ('\u{b920}', '\u{b920}', GC_LV), ('\u{b921}', '\u{b93b}', GC_LVT), ('\u{b93c}',
        '\u{b93c}', GC_LV), ('\u{b93d}', '\u{b957}', GC_LVT), ('\u{b958}', '\u{b958}', GC_LV),
        ('\u{b959}', '\u{b973}', GC_LVT), ('\u{b974}', '\u{b974}', GC_LV), ('\u{b975}', '\u{b98f}',
        GC_LVT), ('\u{b990}', '\u{b990}', GC_LV), ('\u{b991}', '\u{b9ab}', GC_LVT), ('\u{b9ac}',
        '\u{b9ac}', GC_LV), ('\u{b9ad}', '\u{b9c7}', GC_LVT), ('\u{b9c8}', '\u{b9c8}', GC_LV),
        ('\u{b9c9}', '\u{b9e3}', GC_LVT), ('\u{b9e4}', '\u{b9e4}', GC_LV), ('\u{b9e5}', '\u{b9ff}',
        GC_LVT), ('\u{ba00}', '\u{ba00}', GC_LV), ('\u{ba01}', '\u{ba1b}', GC_LVT), ('\u{ba1c}',
        '\u{ba1c}', GC_LV), ('\u{ba1d}', '\u{ba37}', GC_LVT), ('\u{ba38}', '\u{ba38}', GC_LV),
        ('\u{ba39}', '\u{ba53}', GC_LVT), ('\u{ba54}', '\u{ba54}', GC_LV), ('\u{ba55}', '\u{ba6f}',
        GC_LVT), ('\u{ba70}', '\u{ba70}', GC_LV), ('\u{ba71}', '\u{ba8b}', GC_LVT), ('\u{ba8c}',
        '\u{ba8c}', GC_LV), ('\u{ba8d}', '\u{baa7}', GC_LVT), ('\u{baa8}', '\u{baa8}', GC_LV),
        ('\u{baa9}', '\u{bac3}', GC_LVT), ('\u{bac4}', '\u{bac4}', GC_LV), ('\u{bac5}', '\u{badf}',
        GC_LVT), ('\u{bae0}', '\u{bae0}', GC_LV), ('\u{bae1}', '\u{bafb}', GC_LVT), ('\u{bafc}',
        '\u{bafc}', GC_LV), ('\u{bafd}', '\u{bb17}', GC_LVT), ('\u{bb18}', '\u{bb18}', GC_LV),
        ('\u{bb19}', '\u{bb33}', GC_LVT), ('\u{bb34}', '\u{bb34}', GC_LV), ('\u{bb35}', '\u{bb4f}',
        GC_LVT), ('\u{bb50}', '\u{bb50}', GC_LV), ('\u{bb51}', '\u{bb6b}', GC_LVT), ('\u{bb6c}',
        '\u{bb6c}', GC_LV), ('\u{bb6d}', '\u{bb87}', GC_LVT), ('\u{bb88}', '\u{bb88}', GC_LV),
        ('\u{bb89}', '\u{bba3}', GC_LVT), ('\u{bba4}', '\u{bba4}', GC_LV), ('\u{bba5}', '\u{bbbf}',
        GC_LVT), ('\u{bbc0}', '\u{bbc0}', GC_LV), ('\u{bbc1}', '\u{bbdb}', GC_LVT), ('\u{bbdc}',
        '\u{bbdc}', GC_LV), ('\u{bbdd}', '\u{bbf7}', GC_LVT), ('\u{bbf8}', '\u{bbf8}', GC_LV),
        ('\u{bbf9}', '\u{bc13}', GC_LVT), ('\u{bc14}', '\u{bc14}', GC_LV), ('\u{bc15}', '\u{bc2f}',
        GC_LVT), ('\u{bc30}', '\u{bc30}', GC_LV), ('\u{bc31}', '\u{bc4b}', GC_LVT), ('\u{bc4c}',
        '\u{bc4c}', GC_LV), ('\u{bc4d}', '\u{bc67}', GC_LVT), ('\u{bc68}', '\u{bc68}', GC_LV),
        ('\u{bc69}', '\u{bc83}', GC_LVT), ('\u{bc84}', '\u{bc84}', GC_LV), ('\u{bc85}', '\u{bc9f}',
        GC_LVT), ('\u{bca0}', '\u{bca0}', GC_LV), ('\u{bca1}', '\u{bcbb}', GC_LVT), ('\u{bcbc}',
        '\u{bcbc}', GC_LV), ('\u{bcbd}', '\u{bcd7}', GC_LVT), ('\u{bcd8}', '\u{bcd8}', GC_LV),
        ('\u{bcd9}', '\u{bcf3}', GC_LVT), ('\u{bcf4}', '\u{bcf4}', GC_LV), ('\u{bcf5}', '\u{bd0f}',
        GC_LVT), ('\u{bd10}', '\u{bd10}', GC_LV), ('\u{bd11}', '\u{bd2b}', GC_LVT), ('\u{bd2c}',
        '\u{bd2c}', GC_LV), ('\u{bd2d}', '\u{bd47}', GC_LVT), ('\u{bd48}', '\u{bd48}', GC_LV),
        ('\u{bd49}', '\u{bd63}', GC_LVT), ('\u{bd64}', '\u{bd64}', GC_LV), ('\u{bd65}', '\u{bd7f}',
        GC_LVT), ('\u{bd80}', '\u{bd80}', GC_LV), ('\u{bd81}', '\u{bd9b}', GC_LVT), ('\u{bd9c}',
        '\u{bd9c}', GC_LV), ('\u{bd9d}', '\u{bdb7}', GC_LVT), ('\u{bdb8}', '\u{bdb8}', GC_LV),
        ('\u{bdb9}', '\u{bdd3}', GC_LVT), ('\u{bdd4}', '\u{bdd4}', GC_LV), ('\u{bdd5}', '\u{bdef}',
        GC_LVT), ('\u{bdf0}', '\u{bdf0}', GC_LV), ('\u{bdf1}', '\u{be0b}', GC_LVT), ('\u{be0c}',
        '\u{be0c}', GC_LV), ('\u{be0d}', '\u{be27}', GC_LVT), ('\u{be28}', '\u{be28}', GC_LV),
        ('\u{be29}', '\u{be43}', GC_LVT), ('\u{be44}', '\u{be44}', GC_LV), ('\u{be45}', '\u{be5f}',
        GC_LVT), ('\u{be60}', '\u{be60}', GC_LV), ('\u{be61}', '\u{be7b}', GC_LVT), ('\u{be7c}',
        '\u{be7c}', GC_LV), ('\u{be7d}', '\u{be97}', GC_LVT), ('\u{be98}', '\u{be98}', GC_LV),
        ('\u{be99}', '\u{beb3}', GC_LVT), ('\u{beb4}', '\u{beb4}', GC_LV), ('\u{beb5}', '\u{becf}',
        GC_LVT), ('\u{bed0}', '\u{bed0}', GC_LV), ('\u{bed1}', '\u{beeb}', GC_LVT), ('\u{beec}',
        '\u{beec}', GC_LV), ('\u{beed}', '\u{bf07}', GC_LVT), ('\u{bf08}', '\u{bf08}', GC_LV),
        ('\u{bf09}', '\u{bf23}', GC_LVT), ('\u{bf24}', '\u{bf24}', GC_LV), ('\u{bf25}', '\u{bf3f}',
        GC_LVT), ('\u{bf40}', '\u{bf40}', GC_LV), ('\u{bf41}', '\u{bf5b}', GC_LVT), ('\u{bf5c}',
        '\u{bf5c}', GC_LV), ('\u{bf5d}', '\u{bf77}', GC_LVT), ('\u{bf78}', '\u{bf78}', GC_LV),
        ('\u{bf79}', '\u{bf93}', GC_LVT), ('\u{bf94}', '\u{bf94}', GC_LV), ('\u{bf95}', '\u{bfaf}',
        GC_LVT), ('\u{bfb0}', '\u{bfb0}', GC_LV), ('\u{bfb1}', '\u{bfcb}', GC_LVT), ('\u{bfcc}',
        '\u{bfcc}', GC_LV), ('\u{bfcd}', '\u{bfe7}', GC_LVT), ('\u{bfe8}', '\u{bfe8}', GC_LV),
        ('\u{bfe9}', '\u{c003}', GC_LVT), ('\u{c004}', '\u{c004}', GC_LV), ('\u{c005}', '\u{c01f}',
        GC_LVT), ('\u{c020}', '\u{c020}', GC_LV), ('\u{c021}', '\u{c03b}', GC_LVT), ('\u{c03c}',
        '\u{c03c}', GC_LV), ('\u{c03d}', '\u{c057}', GC_LVT), ('\u{c058}', '\u{c058}', GC_LV),
        ('\u{c059}', '\u{c073}', GC_LVT), ('\u{c074}', '\u{c074}', GC_LV), ('\u{c075}', '\u{c08f}',
        GC_LVT), ('\u{c090}', '\u{c090}', GC_LV), ('\u{c091}', '\u{c0ab}', GC_LVT), ('\u{c0ac}',
        '\u{c0ac}', GC_LV), ('\u{c0ad}', '\u{c0c7}', GC_LVT), ('\u{c0c8}', '\u{c0c8}', GC_LV),
        ('\u{c0c9}', '\u{c0e3}', GC_LVT), ('\u{c0e4}', '\u{c0e4}', GC_LV), ('\u{c0e5}', '\u{c0ff}',
        GC_LVT), ('\u{c100}', '\u{c100}', GC_LV), ('\u{c101}', '\u{c11b}', GC_LVT), ('\u{c11c}',
        '\u{c11c}', GC_LV), ('\u{c11d}', '\u{c137}', GC_LVT), ('\u{c138}', '\u{c138}', GC_LV),
        ('\u{c139}', '\u{c153}', GC_LVT), ('\u{c154}', '\u{c154}', GC_LV), ('\u{c155}', '\u{c16f}',
        GC_LVT), ('\u{c170}', '\u{c170}', GC_LV), ('\u{c171}', '\u{c18b}', GC_LVT), ('\u{c18c}',
        '\u{c18c}', GC_LV), ('\u{c18d}', '\u{c1a7}', GC_LVT), ('\u{c1a8}', '\u{c1a8}', GC_LV),
        ('\u{c1a9}', '\u{c1c3}', GC_LVT), ('\u{c1c4}', '\u{c1c4}', GC_LV), ('\u{c1c5}', '\u{c1df}',
        GC_LVT), ('\u{c1e0}', '\u{c1e0}', GC_LV), ('\u{c1e1}', '\u{c1fb}', GC_LVT), ('\u{c1fc}',
        '\u{c1fc}', GC_LV), ('\u{c1fd}', '\u{c217}', GC_LVT), ('\u{c218}', '\u{c218}', GC_LV),
        ('\u{c219}', '\u{c233}', GC_LVT), ('\u{c234}', '\u{c234}', GC_LV), ('\u{c235}', '\u{c24f}',
        GC_LVT), ('\u{c250}', '\u{c250}', GC_LV), ('\u{c251}', '\u{c26b}', GC_LVT), ('\u{c26c}',
        '\u{c26c}', GC_LV), ('\u{c26d}', '\u{c287}', GC_LVT), ('\u{c288}', '\u{c288}', GC_LV),
        ('\u{c289}', '\u{c2a3}', GC_LVT), ('\u{c2a4}', '\u{c2a4}', GC_LV), ('\u{c2a5}', '\u{c2bf}',
        GC_LVT), ('\u{c2c0}', '\u{c2c0}', GC_LV), ('\u{c2c1}', '\u{c2db}', GC_LVT), ('\u{c2dc}',
        '\u{c2dc}', GC_LV), ('\u{c2dd}', '\u{c2f7}', GC_LVT), ('\u{c2f8}', '\u{c2f8}', GC_LV),
        ('\u{c2f9}', '\u{c313}', GC_LVT), ('\u{c314}', '\u{c314}', GC_LV), ('\u{c315}', '\u{c32f}',
        GC_LVT), ('\u{c330}', '\u{c330}', GC_LV), ('\u{c331}', '\u{c34b}', GC_LVT), ('\u{c34c}',
        '\u{c34c}', GC_LV), ('\u{c34d}', '\u{c367}', GC_LVT), ('\u{c368}', '\u{c368}', GC_LV),
        ('\u{c369}', '\u{c383}', GC_LVT), ('\u{c384}', '\u{c384}', GC_LV), ('\u{c385}', '\u{c39f}',
        GC_LVT), ('\u{c3a0}', '\u{c3a0}', GC_LV), ('\u{c3a1}', '\u{c3bb}', GC_LVT), ('\u{c3bc}',
        '\u{c3bc}', GC_LV), ('\u{c3bd}', '\u{c3d7}', GC_LVT), ('\u{c3d8}', '\u{c3d8}', GC_LV),
        ('\u{c3d9}', '\u{c3f3}', GC_LVT), ('\u{c3f4}', '\u{c3f4}', GC_LV), ('\u{c3f5}', '\u{c40f}',
        GC_LVT), ('\u{c410}', '\u{c410}', GC_LV), ('\u{c411}', '\u{c42b}', GC_LVT), ('\u{c42c}',
        '\u{c42c}', GC_LV), ('\u{c42d}', '\u{c447}', GC_LVT), ('\u{c448}', '\u{c448}', GC_LV),
        ('\u{c449}', '\u{c463}', GC_LVT), ('\u{c464}', '\u{c464}', GC_LV), ('\u{c465}', '\u{c47f}',
        GC_LVT), ('\u{c480}', '\u{c480}', GC_LV), ('\u{c481}', '\u{c49b}', GC_LVT), ('\u{c49c}',
        '\u{c49c}', GC_LV), ('\u{c49d}', '\u{c4b7}', GC_LVT), ('\u{c4b8}', '\u{c4b8}', GC_LV),
        ('\u{c4b9}', '\u{c4d3}', GC_LVT), ('\u{c4d4}', '\u{c4d4}', GC_LV), ('\u{c4d5}', '\u{c4ef}',
        GC_LVT), ('\u{c4f0}', '\u{c4f0}', GC_LV), ('\u{c4f1}', '\u{c50b}', GC_LVT), ('\u{c50c}',
        '\u{c50c}', GC_LV), ('\u{c50d}', '\u{c527}', GC_LVT), ('\u{c528}', '\u{c528}', GC_LV),
        ('\u{c529}', '\u{c543}', GC_LVT), ('\u{c544}', '\u{c544}', GC_LV), ('\u{c545}', '\u{c55f}',
        GC_LVT), ('\u{c560}', '\u{c560}', GC_LV), ('\u{c561}', '\u{c57b}', GC_LVT), ('\u{c57c}',
        '\u{c57c}', GC_LV), ('\u{c57d}', '\u{c597}', GC_LVT), ('\u{c598}', '\u{c598}', GC_LV),
        ('\u{c599}', '\u{c5b3}', GC_LVT), ('\u{c5b4}', '\u{c5b4}', GC_LV), ('\u{c5b5}', '\u{c5cf}',
        GC_LVT), ('\u{c5d0}', '\u{c5d0}', GC_LV), ('\u{c5d1}', '\u{c5eb}', GC_LVT), ('\u{c5ec}',
        '\u{c5ec}', GC_LV), ('\u{c5ed}', '\u{c607}', GC_LVT), ('\u{c608}', '\u{c608}', GC_LV),
        ('\u{c609}', '\u{c623}', GC_LVT), ('\u{c624}', '\u{c624}', GC_LV), ('\u{c625}', '\u{c63f}',
        GC_LVT), ('\u{c640}', '\u{c640}', GC_LV), ('\u{c641}', '\u{c65b}', GC_LVT), ('\u{c65c}',
        '\u{c65c}', GC_LV), ('\u{c65d}', '\u{c677}', GC_LVT), ('\u{c678}', '\u{c678}', GC_LV),
        ('\u{c679}', '\u{c693}', GC_LVT), ('\u{c694}', '\u{c694}', GC_LV), ('\u{c695}', '\u{c6af}',
        GC_LVT), ('\u{c6b0}', '\u{c6b0}', GC_LV), ('\u{c6b1}', '\u{c6cb}', GC_LVT), ('\u{c6cc}',
        '\u{c6cc}', GC_LV), ('\u{c6cd}', '\u{c6e7}', GC_LVT), ('\u{c6e8}', '\u{c6e8}', GC_LV),
        ('\u{c6e9}', '\u{c703}', GC_LVT), ('\u{c704}', '\u{c704}', GC_LV), ('\u{c705}', '\u{c71f}',
        GC_LVT), ('\u{c720}', '\u{c720}', GC_LV), ('\u{c721}', '\u{c73b}', GC_LVT), ('\u{c73c}',
        '\u{c73c}', GC_LV), ('\u{c73d}', '\u{c757}', GC_LVT), ('\u{c758}', '\u{c758}', GC_LV),
        ('\u{c759}', '\u{c773}', GC_LVT), ('\u{c774}', '\u{c774}', GC_LV), ('\u{c775}', '\u{c78f}',
        GC_LVT), ('\u{c790}', '\u{c790}', GC_LV), ('\u{c791}', '\u{c7ab}', GC_LVT), ('\u{c7ac}',
        '\u{c7ac}', GC_LV), ('\u{c7ad}', '\u{c7c7}', GC_LVT), ('\u{c7c8}', '\u{c7c8}', GC_LV),
        ('\u{c7c9}', '\u{c7e3}', GC_LVT), ('\u{c7e4}', '\u{c7e4}', GC_LV), ('\u{c7e5}', '\u{c7ff}',
        GC_LVT), ('\u{c800}', '\u{c800}', GC_LV), ('\u{c801}', '\u{c81b}', GC_LVT), ('\u{c81c}',
        '\u{c81c}', GC_LV), ('\u{c81d}', '\u{c837}', GC_LVT), ('\u{c838}', '\u{c838}', GC_LV),
        ('\u{c839}', '\u{c853}', GC_LVT), ('\u{c854}', '\u{c854}', GC_LV), ('\u{c855}', '\u{c86f}',
        GC_LVT), ('\u{c870}', '\u{c870}', GC_LV), ('\u{c871}', '\u{c88b}', GC_LVT), ('\u{c88c}',
        '\u{c88c}', GC_LV), ('\u{c88d}', '\u{c8a7}', GC_LVT), ('\u{c8a8}', '\u{c8a8}', GC_LV),
        ('\u{c8a9}', '\u{c8c3}', GC_LVT), ('\u{c8c4}', '\u{c8c4}', GC_LV), ('\u{c8c5}', '\u{c8df}',
        GC_LVT), ('\u{c8e0}', '\u{c8e0}', GC_LV), ('\u{c8e1}', '\u{c8fb}', GC_LVT), ('\u{c8fc}',
        '\u{c8fc}', GC_LV), ('\u{c8fd}', '\u{c917}', GC_LVT), ('\u{c918}', '\u{c918}', GC_LV),
        ('\u{c919}', '\u{c933}', GC_LVT), ('\u{c934}', '\u{c934}', GC_LV), ('\u{c935}', '\u{c94f}',
        GC_LVT), ('\u{c950}', '\u{c950}', GC_LV), ('\u{c951}', '\u{c96b}', GC_LVT), ('\u{c96c}',
        '\u{c96c}', GC_LV), ('\u{c96d}', '\u{c987}', GC_LVT), ('\u{c988}', '\u{c988}', GC_LV),
        ('\u{c989}', '\u{c9a3}', GC_LVT), ('\u{c9a4}', '\u{c9a4}', GC_LV), ('\u{c9a5}', '\u{c9bf}',
        GC_LVT), ('\u{c9c0}', '\u{c9c0}', GC_LV), ('\u{c9c1}', '\u{c9db}', GC_LVT), ('\u{c9dc}',
        '\u{c9dc}', GC_LV), ('\u{c9dd}', '\u{c9f7}', GC_LVT), ('\u{c9f8}', '\u{c9f8}', GC_LV),
        ('\u{c9f9}', '\u{ca13}', GC_LVT), ('\u{ca14}', '\u{ca14}', GC_LV), ('\u{ca15}', '\u{ca2f}',
        GC_LVT), ('\u{ca30}', '\u{ca30}', GC_LV), ('\u{ca31}', '\u{ca4b}', GC_LVT), ('\u{ca4c}',
        '\u{ca4c}', GC_LV), ('\u{ca4d}', '\u{ca67}', GC_LVT), ('\u{ca68}', '\u{ca68}', GC_LV),
        ('\u{ca69}', '\u{ca83}', GC_LVT), ('\u{ca84}', '\u{ca84}', GC_LV), ('\u{ca85}', '\u{ca9f}',
        GC_LVT), ('\u{caa0}', '\u{caa0}', GC_LV), ('\u{caa1}', '\u{cabb}', GC_LVT), ('\u{cabc}',
        '\u{cabc}', GC_LV), ('\u{cabd}', '\u{cad7}', GC_LVT), ('\u{cad8}', '\u{cad8}', GC_LV),
        ('\u{cad9}', '\u{caf3}', GC_LVT), ('\u{caf4}', '\u{caf4}', GC_LV), ('\u{caf5}', '\u{cb0f}',
        GC_LVT), ('\u{cb10}', '\u{cb10}', GC_LV), ('\u{cb11}', '\u{cb2b}', GC_LVT), ('\u{cb2c}',
        '\u{cb2c}', GC_LV), ('\u{cb2d}', '\u{cb47}', GC_LVT), ('\u{cb48}', '\u{cb48}', GC_LV),
        ('\u{cb49}', '\u{cb63}', GC_LVT), ('\u{cb64}', '\u{cb64}', GC_LV), ('\u{cb65}', '\u{cb7f}',
        GC_LVT), ('\u{cb80}', '\u{cb80}', GC_LV), ('\u{cb81}', '\u{cb9b}', GC_LVT), ('\u{cb9c}',
        '\u{cb9c}', GC_LV), ('\u{cb9d}', '\u{cbb7}', GC_LVT), ('\u{cbb8}', '\u{cbb8}', GC_LV),
        ('\u{cbb9}', '\u{cbd3}', GC_LVT), ('\u{cbd4}', '\u{cbd4}', GC_LV), ('\u{cbd5}', '\u{cbef}',
        GC_LVT), ('\u{cbf0}', '\u{cbf0}', GC_LV), ('\u{cbf1}', '\u{cc0b}', GC_LVT), ('\u{cc0c}',
        '\u{cc0c}', GC_LV), ('\u{cc0d}', '\u{cc27}', GC_LVT), ('\u{cc28}', '\u{cc28}', GC_LV),
        ('\u{cc29}', '\u{cc43}', GC_LVT), ('\u{cc44}', '\u{cc44}', GC_LV), ('\u{cc45}', '\u{cc5f}',
        GC_LVT), ('\u{cc60}', '\u{cc60}', GC_LV), ('\u{cc61}', '\u{cc7b}', GC_LVT), ('\u{cc7c}',
        '\u{cc7c}', GC_LV), ('\u{cc7d}', '\u{cc97}', GC_LVT), ('\u{cc98}', '\u{cc98}', GC_LV),
        ('\u{cc99}', '\u{ccb3}', GC_LVT), ('\u{ccb4}', '\u{ccb4}', GC_LV), ('\u{ccb5}', '\u{cccf}',
        GC_LVT), ('\u{ccd0}', '\u{ccd0}', GC_LV), ('\u{ccd1}', '\u{cceb}', GC_LVT), ('\u{ccec}',
        '\u{ccec}', GC_LV), ('\u{cced}', '\u{cd07}', GC_LVT), ('\u{cd08}', '\u{cd08}', GC_LV),
        ('\u{cd09}', '\u{cd23}', GC_LVT), ('\u{cd24}', '\u{cd24}', GC_LV), ('\u{cd25}', '\u{cd3f}',
        GC_LVT), ('\u{cd40}', '\u{cd40}', GC_LV), ('\u{cd41}', '\u{cd5b}', GC_LVT), ('\u{cd5c}',
        '\u{cd5c}', GC_LV), ('\u{cd5d}', '\u{cd77}', GC_LVT), ('\u{cd78}', '\u{cd78}', GC_LV),
        ('\u{cd79}', '\u{cd93}', GC_LVT), ('\u{cd94}', '\u{cd94}', GC_LV), ('\u{cd95}', '\u{cdaf}',
        GC_LVT), ('\u{cdb0}', '\u{cdb0}', GC_LV), ('\u{cdb1}', '\u{cdcb}', GC_LVT), ('\u{cdcc}',
        '\u{cdcc}', GC_LV), ('\u{cdcd}', '\u{cde7}', GC_LVT), ('\u{cde8}', '\u{cde8}', GC_LV),
        ('\u{cde9}', '\u{ce03}', GC_LVT), ('\u{ce04}', '\u{ce04}', GC_LV), ('\u{ce05}', '\u{ce1f}',
        GC_LVT), ('\u{ce20}', '\u{ce20}', GC_LV), ('\u{ce21}', '\u{ce3b}', GC_LVT), ('\u{ce3c}',
        '\u{ce3c}', GC_LV), ('\u{ce3d}', '\u{ce57}', GC_LVT), ('\u{ce58}', '\u{ce58}', GC_LV),
        ('\u{ce59}', '\u{ce73}', GC_LVT), ('\u{ce74}', '\u{ce74}', GC_LV), ('\u{ce75}', '\u{ce8f}',
        GC_LVT), ('\u{ce90}', '\u{ce90}', GC_LV), ('\u{ce91}', '\u{ceab}', GC_LVT), ('\u{ceac}',
        '\u{ceac}', GC_LV), ('\u{cead}', '\u{cec7}', GC_LVT), ('\u{cec8}', '\u{cec8}', GC_LV),
        ('\u{cec9}', '\u{cee3}', GC_LVT), ('\u{cee4}', '\u{cee4}', GC_LV), ('\u{cee5}', '\u{ceff}',
        GC_LVT), ('\u{cf00}', '\u{cf00}', GC_LV), ('\u{cf01}', '\u{cf1b}', GC_LVT), ('\u{cf1c}',
        '\u{cf1c}', GC_LV), ('\u{cf1d}', '\u{cf37}', GC_LVT), ('\u{cf38}', '\u{cf38}', GC_LV),
        ('\u{cf39}', '\u{cf53}', GC_LVT), ('\u{cf54}', '\u{cf54}', GC_LV), ('\u{cf55}', '\u{cf6f}',
        GC_LVT), ('\u{cf70}', '\u{cf70}', GC_LV), ('\u{cf71}', '\u{cf8b}', GC_LVT), ('\u{cf8c}',
        '\u{cf8c}', GC_LV), ('\u{cf8d}', '\u{cfa7}', GC_LVT), ('\u{cfa8}', '\u{cfa8}', GC_LV),
        ('\u{cfa9}', '\u{cfc3}', GC_LVT), ('\u{cfc4}', '\u{cfc4}', GC_LV), ('\u{cfc5}', '\u{cfdf}',
        GC_LVT), ('\u{cfe0}', '\u{cfe0}', GC_LV), ('\u{cfe1}', '\u{cffb}', GC_LVT), ('\u{cffc}',
        '\u{cffc}', GC_LV), ('\u{cffd}', '\u{d017}', GC_LVT), ('\u{d018}', '\u{d018}', GC_LV),
        ('\u{d019}', '\u{d033}', GC_LVT), ('\u{d034}', '\u{d034}', GC_LV), ('\u{d035}', '\u{d04f}',
        GC_LVT), ('\u{d050}', '\u{d050}', GC_LV), ('\u{d051}', '\u{d06b}', GC_LVT), ('\u{d06c}',
        '\u{d06c}', GC_LV), ('\u{d06d}', '\u{d087}', GC_LVT), ('\u{d088}', '\u{d088}', GC_LV),
        ('\u{d089}', '\u{d0a3}', GC_LVT), ('\u{d0a4}', '\u{d0a4}', GC_LV), ('\u{d0a5}', '\u{d0bf}',
        GC_LVT), ('\u{d0c0}', '\u{d0c0}', GC_LV), ('\u{d0c1}', '\u{d0db}', GC_LVT), ('\u{d0dc}',
        '\u{d0dc}', GC_LV), ('\u{d0dd}', '\u{d0f7}', GC_LVT), ('\u{d0f8}', '\u{d0f8}', GC_LV),
        ('\u{d0f9}', '\u{d113}', GC_LVT), ('\u{d114}', '\u{d114}', GC_LV), ('\u{d115}', '\u{d12f}',
        GC_LVT), ('\u{d130}', '\u{d130}', GC_LV), ('\u{d131}', '\u{d14b}', GC_LVT), ('\u{d14c}',
        '\u{d14c}', GC_LV), ('\u{d14d}', '\u{d167}', GC_LVT), ('\u{d168}', '\u{d168}', GC_LV),
        ('\u{d169}', '\u{d183}', GC_LVT), ('\u{d184}', '\u{d184}', GC_LV), ('\u{d185}', '\u{d19f}',
        GC_LVT), ('\u{d1a0}', '\u{d1a0}', GC_LV), ('\u{d1a1}', '\u{d1bb}', GC_LVT), ('\u{d1bc}',
        '\u{d1bc}', GC_LV), ('\u{d1bd}', '\u{d1d7}', GC_LVT), ('\u{d1d8}', '\u{d1d8}', GC_LV),
        ('\u{d1d9}', '\u{d1f3}', GC_LVT), ('\u{d1f4}', '\u{d1f4}', GC_LV), ('\u{d1f5}', '\u{d20f}',
        GC_LVT), ('\u{d210}', '\u{d210}', GC_LV), ('\u{d211}', '\u{d22b}', GC_LVT), ('\u{d22c}',
        '\u{d22c}', GC_LV), ('\u{d22d}', '\u{d247}', GC_LVT), ('\u{d248}', '\u{d248}', GC_LV),
        ('\u{d249}', '\u{d263}', GC_LVT), ('\u{d264}', '\u{d264}', GC_LV), ('\u{d265}', '\u{d27f}',
        GC_LVT), ('\u{d280}', '\u{d280}', GC_LV), ('\u{d281}', '\u{d29b}', GC_LVT), ('\u{d29c}',
        '\u{d29c}', GC_LV), ('\u{d29d}', '\u{d2b7}', GC_LVT), ('\u{d2b8}', '\u{d2b8}', GC_LV),
        ('\u{d2b9}', '\u{d2d3}', GC_LVT), ('\u{d2d4}', '\u{d2d4}', GC_LV), ('\u{d2d5}', '\u{d2ef}',
        GC_LVT), ('\u{d2f0}', '\u{d2f0}', GC_LV), ('\u{d2f1}', '\u{d30b}', GC_LVT), ('\u{d30c}',
        '\u{d30c}', GC_LV), ('\u{d30d}', '\u{d327}', GC_LVT), ('\u{d328}', '\u{d328}', GC_LV),
        ('\u{d329}', '\u{d343}', GC_LVT), ('\u{d344}', '\u{d344}', GC_LV), ('\u{d345}', '\u{d35f}',
        GC_LVT), ('\u{d360}', '\u{d360}', GC_LV), ('\u{d361}', '\u{d37b}', GC_LVT), ('\u{d37c}',
        '\u{d37c}', GC_LV), ('\u{d37d}', '\u{d397}', GC_LVT), ('\u{d398}', '\u{d398}', GC_LV),
        ('\u{d399}', '\u{d3b3}', GC_LVT), ('\u{d3b4}', '\u{d3b4}', GC_LV), ('\u{d3b5}', '\u{d3cf}',
        GC_LVT), ('\u{d3d0}', '\u{d3d0}', GC_LV), ('\u{d3d1}', '\u{d3eb}', GC_LVT), ('\u{d3ec}',
        '\u{d3ec}', GC_LV), ('\u{d3ed}', '\u{d407}', GC_LVT), ('\u{d408}', '\u{d408}', GC_LV),
        ('\u{d409}', '\u{d423}', GC_LVT), ('\u{d424}', '\u{d424}', GC_LV), ('\u{d425}', '\u{d43f}',
        GC_LVT), ('\u{d440}', '\u{d440}', GC_LV), ('\u{d441}', '\u{d45b}', GC_LVT), ('\u{d45c}',
        '\u{d45c}', GC_LV), ('\u{d45d}', '\u{d477}', GC_LVT), ('\u{d478}', '\u{d478}', GC_LV),
        ('\u{d479}', '\u{d493}', GC_LVT), ('\u{d494}', '\u{d494}', GC_LV), ('\u{d495}', '\u{d4af}',
        GC_LVT), ('\u{d4b0}', '\u{d4b0}', GC_LV), ('\u{d4b1}', '\u{d4cb}', GC_LVT), ('\u{d4cc}',
        '\u{d4cc}', GC_LV), ('\u{d4cd}', '\u{d4e7}', GC_LVT), ('\u{d4e8}', '\u{d4e8}', GC_LV),
        ('\u{d4e9}', '\u{d503}', GC_LVT), ('\u{d504}', '\u{d504}', GC_LV), ('\u{d505}', '\u{d51f}',
        GC_LVT), ('\u{d520}', '\u{d520}', GC_LV), ('\u{d521}', '\u{d53b}', GC_LVT), ('\u{d53c}',
        '\u{d53c}', GC_LV), ('\u{d53d}', '\u{d557}', GC_LVT), ('\u{d558}', '\u{d558}', GC_LV),
        ('\u{d559}', '\u{d573}', GC_LVT), ('\u{d574}', '\u{d574}', GC_LV), ('\u{d575}', '\u{d58f}',
        GC_LVT), ('\u{d590}', '\u{d590}', GC_LV), ('\u{d591}', '\u{d5ab}', GC_LVT), ('\u{d5ac}',
        '\u{d5ac}', GC_LV), ('\u{d5ad}', '\u{d5c7}', GC_LVT), ('\u{d5c8}', '\u{d5c8}', GC_LV),
        ('\u{d5c9}', '\u{d5e3}', GC_LVT), ('\u{d5e4}', '\u{d5e4}', GC_LV), ('\u{d5e5}', '\u{d5ff}',
        GC_LVT), ('\u{d600}', '\u{d600}', GC_LV), ('\u{d601}', '\u{d61b}', GC_LVT), ('\u{d61c}',
        '\u{d61c}', GC_LV), ('\u{d61d}', '\u{d637}', GC_LVT), ('\u{d638}', '\u{d638}', GC_LV),
        ('\u{d639}', '\u{d653}', GC_LVT), ('\u{d654}', '\u{d654}', GC_LV), ('\u{d655}', '\u{d66f}',
        GC_LVT), ('\u{d670}', '\u{d670}', GC_LV), ('\u{d671}', '\u{d68b}', GC_LVT), ('\u{d68c}',
        '\u{d68c}', GC_LV), ('\u{d68d}', '\u{d6a7}', GC_LVT), ('\u{d6a8}', '\u{d6a8}', GC_LV),
        ('\u{d6a9}', '\u{d6c3}', GC_LVT), ('\u{d6c4}', '\u{d6c4}', GC_LV), ('\u{d6c5}', '\u{d6df}',
        GC_LVT), ('\u{d6e0}', '\u{d6e0}', GC_LV), ('\u{d6e1}', '\u{d6fb}', GC_LVT), ('\u{d6fc}',
        '\u{d6fc}', GC_LV), ('\u{d6fd}', '\u{d717}', GC_LVT), ('\u{d718}', '\u{d718}', GC_LV),
        ('\u{d719}', '\u{d733}', GC_LVT), ('\u{d734}', '\u{d734}', GC_LV), ('\u{d735}', '\u{d74f}',
        GC_LVT), ('\u{d750}', '\u{d750}', GC_LV), ('\u{d751}', '\u{d76b}', GC_LVT), ('\u{d76c}',
        '\u{d76c}', GC_LV), ('\u{d76d}', '\u{d787}', GC_LVT), ('\u{d788}', '\u{d788}', GC_LV),
        ('\u{d789}', '\u{d7a3}', GC_LVT), ('\u{d7b0}', '\u{d7c6}', GC_V), ('\u{d7cb}', '\u{d7fb}',
        GC_T), ('\u{fb1e}', '\u{fb1e}', GC_Extend), ('\u{fe00}', '\u{fe0f}', GC_Extend),
        ('\u{fe20}', '\u{fe2f}', GC_Extend), ('\u{feff}', '\u{feff}', GC_Control), ('\u{ff9e}',
        '\u{ff9f}', GC_Extend), ('\u{fff0}', '\u{fffb}', GC_Control), ('\u{101fd}', '\u{101fd}',
        GC_Extend), ('\u{102e0}', '\u{102e0}', GC_Extend), ('\u{10376}', '\u{1037a}', GC_Extend),
        ('\u{10a01}', '\u{10a03}', GC_Extend), ('\u{10a05}', '\u{10a06}', GC_Extend), ('\u{10a0c}',
        '\u{10a0f}', GC_Extend), ('\u{10a38}', '\u{10a3a}', GC_Extend), ('\u{10a3f}', '\u{10a3f}',
        GC_Extend), ('\u{10ae5}', '\u{10ae6}', GC_Extend), ('\u{11000}', '\u{11000}',
        GC_SpacingMark), ('\u{11001}', '\u{11001}', GC_Extend), ('\u{11002}', '\u{11002}',
        GC_SpacingMark), ('\u{11038}', '\u{11046}', GC_Extend), ('\u{1107f}', '\u{11081}',
        GC_Extend), ('\u{11082}', '\u{11082}', GC_SpacingMark), ('\u{110b0}', '\u{110b2}',
        GC_SpacingMark), ('\u{110b3}', '\u{110b6}', GC_Extend), ('\u{110b7}', '\u{110b8}',
        GC_SpacingMark), ('\u{110b9}', '\u{110ba}', GC_Extend), ('\u{110bd}', '\u{110bd}',
        GC_Prepend), ('\u{11100}', '\u{11102}', GC_Extend), ('\u{11127}', '\u{1112b}', GC_Extend),
        ('\u{1112c}', '\u{1112c}', GC_SpacingMark), ('\u{1112d}', '\u{11134}', GC_Extend),
        ('\u{11173}', '\u{11173}', GC_Extend), ('\u{11180}', '\u{11181}', GC_Extend), ('\u{11182}',
        '\u{11182}', GC_SpacingMark), ('\u{111b3}', '\u{111b5}', GC_SpacingMark), ('\u{111b6}',
        '\u{111be}', GC_Extend), ('\u{111bf}', '\u{111c0}', GC_SpacingMark), ('\u{111c2}',
        '\u{111c3}', GC_Prepend), ('\u{111ca}', '\u{111cc}', GC_Extend), ('\u{1122c}', '\u{1122e}',
        GC_SpacingMark), ('\u{1122f}', '\u{11231}', GC_Extend), ('\u{11232}', '\u{11233}',
        GC_SpacingMark), ('\u{11234}', '\u{11234}', GC_Extend), ('\u{11235}', '\u{11235}',
        GC_SpacingMark), ('\u{11236}', '\u{11237}', GC_Extend), ('\u{1123e}', '\u{1123e}',
        GC_Extend), ('\u{112df}', '\u{112df}', GC_Extend), ('\u{112e0}', '\u{112e2}',
        GC_SpacingMark), ('\u{112e3}', '\u{112ea}', GC_Extend), ('\u{11300}', '\u{11301}',
        GC_Extend), ('\u{11302}', '\u{11303}', GC_SpacingMark), ('\u{1133c}', '\u{1133c}',
        GC_Extend), ('\u{1133e}', '\u{1133e}', GC_Extend), ('\u{1133f}', '\u{1133f}',
        GC_SpacingMark), ('\u{11340}', '\u{11340}', GC_Extend), ('\u{11341}', '\u{11344}',
        GC_SpacingMark), ('\u{11347}', '\u{11348}', GC_SpacingMark), ('\u{1134b}', '\u{1134d}',
        GC_SpacingMark), ('\u{11357}', '\u{11357}', GC_Extend), ('\u{11362}', '\u{11363}',
        GC_SpacingMark), ('\u{11366}', '\u{1136c}', GC_Extend), ('\u{11370}', '\u{11374}',
        GC_Extend), ('\u{11435}', '\u{11437}', GC_SpacingMark), ('\u{11438}', '\u{1143f}',
        GC_Extend), ('\u{11440}', '\u{11441}', GC_SpacingMark), ('\u{11442}', '\u{11444}',
        GC_Extend), ('\u{11445}', '\u{11445}', GC_SpacingMark), ('\u{11446}', '\u{11446}',
        GC_Extend), ('\u{114b0}', '\u{114b0}', GC_Extend), ('\u{114b1}', '\u{114b2}',
        GC_SpacingMark), ('\u{114b3}', '\u{114b8}', GC_Extend), ('\u{114b9}', '\u{114b9}',
        GC_SpacingMark), ('\u{114ba}', '\u{114ba}', GC_Extend), ('\u{114bb}', '\u{114bc}',
        GC_SpacingMark), ('\u{114bd}', '\u{114bd}', GC_Extend), ('\u{114be}', '\u{114be}',
        GC_SpacingMark), ('\u{114bf}', '\u{114c0}', GC_Extend), ('\u{114c1}', '\u{114c1}',
        GC_SpacingMark), ('\u{114c2}', '\u{114c3}', GC_Extend), ('\u{115af}', '\u{115af}',
        GC_Extend), ('\u{115b0}', '\u{115b1}', GC_SpacingMark), ('\u{115b2}', '\u{115b5}',
        GC_Extend), ('\u{115b8}', '\u{115bb}', GC_SpacingMark), ('\u{115bc}', '\u{115bd}',
        GC_Extend), ('\u{115be}', '\u{115be}', GC_SpacingMark), ('\u{115bf}', '\u{115c0}',
        GC_Extend), ('\u{115dc}', '\u{115dd}', GC_Extend), ('\u{11630}', '\u{11632}',
        GC_SpacingMark), ('\u{11633}', '\u{1163a}', GC_Extend), ('\u{1163b}', '\u{1163c}',
        GC_SpacingMark), ('\u{1163d}', '\u{1163d}', GC_Extend), ('\u{1163e}', '\u{1163e}',
        GC_SpacingMark), ('\u{1163f}', '\u{11640}', GC_Extend), ('\u{116ab}', '\u{116ab}',
        GC_Extend), ('\u{116ac}', '\u{116ac}', GC_SpacingMark), ('\u{116ad}', '\u{116ad}',
        GC_Extend), ('\u{116ae}', '\u{116af}', GC_SpacingMark), ('\u{116b0}', '\u{116b5}',
        GC_Extend), ('\u{116b6}', '\u{116b6}', GC_SpacingMark), ('\u{116b7}', '\u{116b7}',
        GC_Extend), ('\u{1171d}', '\u{1171f}', GC_Extend), ('\u{11720}', '\u{11721}',
        GC_SpacingMark), ('\u{11722}', '\u{11725}', GC_Extend), ('\u{11726}', '\u{11726}',
        GC_SpacingMark), ('\u{11727}', '\u{1172b}', GC_Extend), ('\u{11c2f}', '\u{11c2f}',
        GC_SpacingMark), ('\u{11c30}', '\u{11c36}', GC_Extend), ('\u{11c38}', '\u{11c3d}',
        GC_Extend), ('\u{11c3e}', '\u{11c3e}', GC_SpacingMark), ('\u{11c3f}', '\u{11c3f}',
        GC_Extend), ('\u{11c92}', '\u{11ca7}', GC_Extend), ('\u{11ca9}', '\u{11ca9}',
        GC_SpacingMark), ('\u{11caa}', '\u{11cb0}', GC_Extend), ('\u{11cb1}', '\u{11cb1}',
        GC_SpacingMark), ('\u{11cb2}', '\u{11cb3}', GC_Extend), ('\u{11cb4}', '\u{11cb4}',
        GC_SpacingMark), ('\u{11cb5}', '\u{11cb6}', GC_Extend), ('\u{16af0}', '\u{16af4}',
        GC_Extend), ('\u{16b30}', '\u{16b36}', GC_Extend), ('\u{16f51}', '\u{16f7e}',
        GC_SpacingMark), ('\u{16f8f}', '\u{16f92}', GC_Extend), ('\u{1bc9d}', '\u{1bc9e}',
        GC_Extend), ('\u{1bca0}', '\u{1bca3}', GC_Control), ('\u{1d165}', '\u{1d165}', GC_Extend),
        ('\u{1d166}', '\u{1d166}', GC_SpacingMark), ('\u{1d167}', '\u{1d169}', GC_Extend),
        ('\u{1d16d}', '\u{1d16d}', GC_SpacingMark), ('\u{1d16e}', '\u{1d172}', GC_Extend),
        ('\u{1d173}', '\u{1d17a}', GC_Control), ('\u{1d17b}', '\u{1d182}', GC_Extend), ('\u{1d185}',
        '\u{1d18b}', GC_Extend), ('\u{1d1aa}', '\u{1d1ad}', GC_Extend), ('\u{1d242}', '\u{1d244}',
        GC_Extend), ('\u{1da00}', '\u{1da36}', GC_Extend), ('\u{1da3b}', '\u{1da6c}', GC_Extend),
        ('\u{1da75}', '\u{1da75}', GC_Extend), ('\u{1da84}', '\u{1da84}', GC_Extend), ('\u{1da9b}',
        '\u{1da9f}', GC_Extend), ('\u{1daa1}', '\u{1daaf}', GC_Extend), ('\u{1e000}', '\u{1e006}',
        GC_Extend), ('\u{1e008}', '\u{1e018}', GC_Extend), ('\u{1e01b}', '\u{1e021}', GC_Extend),
        ('\u{1e023}', '\u{1e024}', GC_Extend), ('\u{1e026}', '\u{1e02a}', GC_Extend), ('\u{1e8d0}',
        '\u{1e8d6}', GC_Extend), ('\u{1e944}', '\u{1e94a}', GC_Extend), ('\u{1f1e6}', '\u{1f1ff}',
        GC_Regional_Indicator), ('\u{1f385}', '\u{1f385}', GC_E_Base), ('\u{1f3c3}', '\u{1f3c4}',
        GC_E_Base), ('\u{1f3ca}', '\u{1f3cb}', GC_E_Base), ('\u{1f3fb}', '\u{1f3ff}',
        GC_E_Modifier), ('\u{1f442}', '\u{1f443}', GC_E_Base), ('\u{1f446}', '\u{1f450}',
        GC_E_Base), ('\u{1f466}', '\u{1f469}', GC_E_Base_GAZ), ('\u{1f46e}', '\u{1f46e}',
        GC_E_Base), ('\u{1f470}', '\u{1f478}', GC_E_Base), ('\u{1f47c}', '\u{1f47c}', GC_E_Base),
        ('\u{1f481}', '\u{1f483}', GC_E_Base), ('\u{1f485}', '\u{1f487}', GC_E_Base), ('\u{1f48b}',
        '\u{1f48b}', GC_Glue_After_Zwj), ('\u{1f4aa}', '\u{1f4aa}', GC_E_Base), ('\u{1f575}',
        '\u{1f575}', GC_E_Base), ('\u{1f57a}', '\u{1f57a}', GC_E_Base), ('\u{1f590}', '\u{1f590}',
        GC_E_Base), ('\u{1f595}', '\u{1f596}', GC_E_Base), ('\u{1f5e8}', '\u{1f5e8}',
        GC_Glue_After_Zwj), ('\u{1f645}', '\u{1f647}', GC_E_Base), ('\u{1f64b}', '\u{1f64f}',
        GC_E_Base), ('\u{1f6a3}', '\u{1f6a3}', GC_E_Base), ('\u{1f6b4}', '\u{1f6b6}', GC_E_Base),
        ('\u{1f6c0}', '\u{1f6c0}', GC_E_Base), ('\u{1f918}', '\u{1f91e}', GC_E_Base), ('\u{1f926}',
        '\u{1f926}', GC_E_Base), ('\u{1f930}', '\u{1f930}', GC_E_Base), ('\u{1f933}', '\u{1f939}',
        GC_E_Base), ('\u{1f93c}', '\u{1f93e}', GC_E_Base), ('\u{e0000}', '\u{e001f}', GC_Control),
        ('\u{e0020}', '\u{e007f}', GC_Extend), ('\u{e0080}', '\u{e00ff}', GC_Control), ('\u{e0100}',
        '\u{e01ef}', GC_Extend), ('\u{e01f0}', '\u{e0fff}', GC_Control)
    ];

}

pub mod word {
    use core::result::Result::{Ok, Err};

    pub use self::WordCat::*;

    #[allow(non_camel_case_types)]
    #[derive(Clone, Copy, PartialEq, Eq)]
    pub enum WordCat {
        WC_ALetter,
        WC_Any,
        WC_CR,
        WC_Double_Quote,
        WC_E_Base,
        WC_E_Base_GAZ,
        WC_E_Modifier,
        WC_Extend,
        WC_ExtendNumLet,
        WC_Format,
        WC_Glue_After_Zwj,
        WC_Hebrew_Letter,
        WC_Katakana,
        WC_LF,
        WC_MidLetter,
        WC_MidNum,
        WC_MidNumLet,
        WC_Newline,
        WC_Numeric,
        WC_Regional_Indicator,
        WC_Single_Quote,
        WC_ZWJ,
    }

    fn bsearch_range_value_table(c: char, r: &'static [(char, char, WordCat)]) -> WordCat {
        use core::cmp::Ordering::{Equal, Less, Greater};
        match r.binary_search_by(|&(lo, hi, _)| {
            if lo <= c && c <= hi { Equal }
            else if hi < c { Less }
            else { Greater }
        }) {
            Ok(idx) => {
                let (_, _, cat) = r[idx];
                cat
            }
            Err(_) => WC_Any
        }
    }

    pub fn word_category(c: char) -> WordCat {
        bsearch_range_value_table(c, word_cat_table)
    }

    const word_cat_table: &'static [(char, char, WordCat)] = &[
        ('\u{a}', '\u{a}', WC_LF), ('\u{b}', '\u{c}', WC_Newline), ('\u{d}', '\u{d}', WC_CR),
        ('\u{22}', '\u{22}', WC_Double_Quote), ('\u{27}', '\u{27}', WC_Single_Quote), ('\u{2c}',
        '\u{2c}', WC_MidNum), ('\u{2e}', '\u{2e}', WC_MidNumLet), ('\u{30}', '\u{39}', WC_Numeric),
        ('\u{3a}', '\u{3a}', WC_MidLetter), ('\u{3b}', '\u{3b}', WC_MidNum), ('\u{41}', '\u{5a}',
        WC_ALetter), ('\u{5f}', '\u{5f}', WC_ExtendNumLet), ('\u{61}', '\u{7a}', WC_ALetter),
        ('\u{85}', '\u{85}', WC_Newline), ('\u{aa}', '\u{aa}', WC_ALetter), ('\u{ad}', '\u{ad}',
        WC_Format), ('\u{b5}', '\u{b5}', WC_ALetter), ('\u{b7}', '\u{b7}', WC_MidLetter), ('\u{ba}',
        '\u{ba}', WC_ALetter), ('\u{c0}', '\u{d6}', WC_ALetter), ('\u{d8}', '\u{f6}', WC_ALetter),
        ('\u{f8}', '\u{2c1}', WC_ALetter), ('\u{2c6}', '\u{2d1}', WC_ALetter), ('\u{2d7}',
        '\u{2d7}', WC_MidLetter), ('\u{2e0}', '\u{2e4}', WC_ALetter), ('\u{2ec}', '\u{2ec}',
        WC_ALetter), ('\u{2ee}', '\u{2ee}', WC_ALetter), ('\u{300}', '\u{36f}', WC_Extend),
        ('\u{370}', '\u{374}', WC_ALetter), ('\u{376}', '\u{377}', WC_ALetter), ('\u{37a}',
        '\u{37d}', WC_ALetter), ('\u{37e}', '\u{37e}', WC_MidNum), ('\u{37f}', '\u{37f}',
        WC_ALetter), ('\u{386}', '\u{386}', WC_ALetter), ('\u{387}', '\u{387}', WC_MidLetter),
        ('\u{388}', '\u{38a}', WC_ALetter), ('\u{38c}', '\u{38c}', WC_ALetter), ('\u{38e}',
        '\u{3a1}', WC_ALetter), ('\u{3a3}', '\u{3f5}', WC_ALetter), ('\u{3f7}', '\u{481}',
        WC_ALetter), ('\u{483}', '\u{489}', WC_Extend), ('\u{48a}', '\u{52f}', WC_ALetter),
        ('\u{531}', '\u{556}', WC_ALetter), ('\u{559}', '\u{559}', WC_ALetter), ('\u{561}',
        '\u{587}', WC_ALetter), ('\u{589}', '\u{589}', WC_MidNum), ('\u{591}', '\u{5bd}',
        WC_Extend), ('\u{5bf}', '\u{5bf}', WC_Extend), ('\u{5c1}', '\u{5c2}', WC_Extend),
        ('\u{5c4}', '\u{5c5}', WC_Extend), ('\u{5c7}', '\u{5c7}', WC_Extend), ('\u{5d0}', '\u{5ea}',
        WC_Hebrew_Letter), ('\u{5f0}', '\u{5f2}', WC_Hebrew_Letter), ('\u{5f3}', '\u{5f3}',
        WC_ALetter), ('\u{5f4}', '\u{5f4}', WC_MidLetter), ('\u{600}', '\u{605}', WC_Format),
        ('\u{60c}', '\u{60d}', WC_MidNum), ('\u{610}', '\u{61a}', WC_Extend), ('\u{61c}', '\u{61c}',
        WC_Format), ('\u{620}', '\u{64a}', WC_ALetter), ('\u{64b}', '\u{65f}', WC_Extend),
        ('\u{660}', '\u{669}', WC_Numeric), ('\u{66b}', '\u{66b}', WC_Numeric), ('\u{66c}',
        '\u{66c}', WC_MidNum), ('\u{66e}', '\u{66f}', WC_ALetter), ('\u{670}', '\u{670}',
        WC_Extend), ('\u{671}', '\u{6d3}', WC_ALetter), ('\u{6d5}', '\u{6d5}', WC_ALetter),
        ('\u{6d6}', '\u{6dc}', WC_Extend), ('\u{6dd}', '\u{6dd}', WC_Format), ('\u{6df}', '\u{6e4}',
        WC_Extend), ('\u{6e5}', '\u{6e6}', WC_ALetter), ('\u{6e7}', '\u{6e8}', WC_Extend),
        ('\u{6ea}', '\u{6ed}', WC_Extend), ('\u{6ee}', '\u{6ef}', WC_ALetter), ('\u{6f0}',
        '\u{6f9}', WC_Numeric), ('\u{6fa}', '\u{6fc}', WC_ALetter), ('\u{6ff}', '\u{6ff}',
        WC_ALetter), ('\u{70f}', '\u{70f}', WC_Format), ('\u{710}', '\u{710}', WC_ALetter),
        ('\u{711}', '\u{711}', WC_Extend), ('\u{712}', '\u{72f}', WC_ALetter), ('\u{730}',
        '\u{74a}', WC_Extend), ('\u{74d}', '\u{7a5}', WC_ALetter), ('\u{7a6}', '\u{7b0}',
        WC_Extend), ('\u{7b1}', '\u{7b1}', WC_ALetter), ('\u{7c0}', '\u{7c9}', WC_Numeric),
        ('\u{7ca}', '\u{7ea}', WC_ALetter), ('\u{7eb}', '\u{7f3}', WC_Extend), ('\u{7f4}',
        '\u{7f5}', WC_ALetter), ('\u{7f8}', '\u{7f8}', WC_MidNum), ('\u{7fa}', '\u{7fa}',
        WC_ALetter), ('\u{800}', '\u{815}', WC_ALetter), ('\u{816}', '\u{819}', WC_Extend),
        ('\u{81a}', '\u{81a}', WC_ALetter), ('\u{81b}', '\u{823}', WC_Extend), ('\u{824}',
        '\u{824}', WC_ALetter), ('\u{825}', '\u{827}', WC_Extend), ('\u{828}', '\u{828}',
        WC_ALetter), ('\u{829}', '\u{82d}', WC_Extend), ('\u{840}', '\u{858}', WC_ALetter),
        ('\u{859}', '\u{85b}', WC_Extend), ('\u{8a0}', '\u{8b4}', WC_ALetter), ('\u{8b6}',
        '\u{8bd}', WC_ALetter), ('\u{8d4}', '\u{8e1}', WC_Extend), ('\u{8e2}', '\u{8e2}',
        WC_Format), ('\u{8e3}', '\u{903}', WC_Extend), ('\u{904}', '\u{939}', WC_ALetter),
        ('\u{93a}', '\u{93c}', WC_Extend), ('\u{93d}', '\u{93d}', WC_ALetter), ('\u{93e}',
        '\u{94f}', WC_Extend), ('\u{950}', '\u{950}', WC_ALetter), ('\u{951}', '\u{957}',
        WC_Extend), ('\u{958}', '\u{961}', WC_ALetter), ('\u{962}', '\u{963}', WC_Extend),
        ('\u{966}', '\u{96f}', WC_Numeric), ('\u{971}', '\u{980}', WC_ALetter), ('\u{981}',
        '\u{983}', WC_Extend), ('\u{985}', '\u{98c}', WC_ALetter), ('\u{98f}', '\u{990}',
        WC_ALetter), ('\u{993}', '\u{9a8}', WC_ALetter), ('\u{9aa}', '\u{9b0}', WC_ALetter),
        ('\u{9b2}', '\u{9b2}', WC_ALetter), ('\u{9b6}', '\u{9b9}', WC_ALetter), ('\u{9bc}',
        '\u{9bc}', WC_Extend), ('\u{9bd}', '\u{9bd}', WC_ALetter), ('\u{9be}', '\u{9c4}',
        WC_Extend), ('\u{9c7}', '\u{9c8}', WC_Extend), ('\u{9cb}', '\u{9cd}', WC_Extend),
        ('\u{9ce}', '\u{9ce}', WC_ALetter), ('\u{9d7}', '\u{9d7}', WC_Extend), ('\u{9dc}',
        '\u{9dd}', WC_ALetter), ('\u{9df}', '\u{9e1}', WC_ALetter), ('\u{9e2}', '\u{9e3}',
        WC_Extend), ('\u{9e6}', '\u{9ef}', WC_Numeric), ('\u{9f0}', '\u{9f1}', WC_ALetter),
        ('\u{a01}', '\u{a03}', WC_Extend), ('\u{a05}', '\u{a0a}', WC_ALetter), ('\u{a0f}',
        '\u{a10}', WC_ALetter), ('\u{a13}', '\u{a28}', WC_ALetter), ('\u{a2a}', '\u{a30}',
        WC_ALetter), ('\u{a32}', '\u{a33}', WC_ALetter), ('\u{a35}', '\u{a36}', WC_ALetter),
        ('\u{a38}', '\u{a39}', WC_ALetter), ('\u{a3c}', '\u{a3c}', WC_Extend), ('\u{a3e}',
        '\u{a42}', WC_Extend), ('\u{a47}', '\u{a48}', WC_Extend), ('\u{a4b}', '\u{a4d}', WC_Extend),
        ('\u{a51}', '\u{a51}', WC_Extend), ('\u{a59}', '\u{a5c}', WC_ALetter), ('\u{a5e}',
        '\u{a5e}', WC_ALetter), ('\u{a66}', '\u{a6f}', WC_Numeric), ('\u{a70}', '\u{a71}',
        WC_Extend), ('\u{a72}', '\u{a74}', WC_ALetter), ('\u{a75}', '\u{a75}', WC_Extend),
        ('\u{a81}', '\u{a83}', WC_Extend), ('\u{a85}', '\u{a8d}', WC_ALetter), ('\u{a8f}',
        '\u{a91}', WC_ALetter), ('\u{a93}', '\u{aa8}', WC_ALetter), ('\u{aaa}', '\u{ab0}',
        WC_ALetter), ('\u{ab2}', '\u{ab3}', WC_ALetter), ('\u{ab5}', '\u{ab9}', WC_ALetter),
        ('\u{abc}', '\u{abc}', WC_Extend), ('\u{abd}', '\u{abd}', WC_ALetter), ('\u{abe}',
        '\u{ac5}', WC_Extend), ('\u{ac7}', '\u{ac9}', WC_Extend), ('\u{acb}', '\u{acd}', WC_Extend),
        ('\u{ad0}', '\u{ad0}', WC_ALetter), ('\u{ae0}', '\u{ae1}', WC_ALetter), ('\u{ae2}',
        '\u{ae3}', WC_Extend), ('\u{ae6}', '\u{aef}', WC_Numeric), ('\u{af9}', '\u{af9}',
        WC_ALetter), ('\u{b01}', '\u{b03}', WC_Extend), ('\u{b05}', '\u{b0c}', WC_ALetter),
        ('\u{b0f}', '\u{b10}', WC_ALetter), ('\u{b13}', '\u{b28}', WC_ALetter), ('\u{b2a}',
        '\u{b30}', WC_ALetter), ('\u{b32}', '\u{b33}', WC_ALetter), ('\u{b35}', '\u{b39}',
        WC_ALetter), ('\u{b3c}', '\u{b3c}', WC_Extend), ('\u{b3d}', '\u{b3d}', WC_ALetter),
        ('\u{b3e}', '\u{b44}', WC_Extend), ('\u{b47}', '\u{b48}', WC_Extend), ('\u{b4b}', '\u{b4d}',
        WC_Extend), ('\u{b56}', '\u{b57}', WC_Extend), ('\u{b5c}', '\u{b5d}', WC_ALetter),
        ('\u{b5f}', '\u{b61}', WC_ALetter), ('\u{b62}', '\u{b63}', WC_Extend), ('\u{b66}',
        '\u{b6f}', WC_Numeric), ('\u{b71}', '\u{b71}', WC_ALetter), ('\u{b82}', '\u{b82}',
        WC_Extend), ('\u{b83}', '\u{b83}', WC_ALetter), ('\u{b85}', '\u{b8a}', WC_ALetter),
        ('\u{b8e}', '\u{b90}', WC_ALetter), ('\u{b92}', '\u{b95}', WC_ALetter), ('\u{b99}',
        '\u{b9a}', WC_ALetter), ('\u{b9c}', '\u{b9c}', WC_ALetter), ('\u{b9e}', '\u{b9f}',
        WC_ALetter), ('\u{ba3}', '\u{ba4}', WC_ALetter), ('\u{ba8}', '\u{baa}', WC_ALetter),
        ('\u{bae}', '\u{bb9}', WC_ALetter), ('\u{bbe}', '\u{bc2}', WC_Extend), ('\u{bc6}',
        '\u{bc8}', WC_Extend), ('\u{bca}', '\u{bcd}', WC_Extend), ('\u{bd0}', '\u{bd0}',
        WC_ALetter), ('\u{bd7}', '\u{bd7}', WC_Extend), ('\u{be6}', '\u{bef}', WC_Numeric),
        ('\u{c00}', '\u{c03}', WC_Extend), ('\u{c05}', '\u{c0c}', WC_ALetter), ('\u{c0e}',
        '\u{c10}', WC_ALetter), ('\u{c12}', '\u{c28}', WC_ALetter), ('\u{c2a}', '\u{c39}',
        WC_ALetter), ('\u{c3d}', '\u{c3d}', WC_ALetter), ('\u{c3e}', '\u{c44}', WC_Extend),
        ('\u{c46}', '\u{c48}', WC_Extend), ('\u{c4a}', '\u{c4d}', WC_Extend), ('\u{c55}', '\u{c56}',
        WC_Extend), ('\u{c58}', '\u{c5a}', WC_ALetter), ('\u{c60}', '\u{c61}', WC_ALetter),
        ('\u{c62}', '\u{c63}', WC_Extend), ('\u{c66}', '\u{c6f}', WC_Numeric), ('\u{c80}',
        '\u{c80}', WC_ALetter), ('\u{c81}', '\u{c83}', WC_Extend), ('\u{c85}', '\u{c8c}',
        WC_ALetter), ('\u{c8e}', '\u{c90}', WC_ALetter), ('\u{c92}', '\u{ca8}', WC_ALetter),
        ('\u{caa}', '\u{cb3}', WC_ALetter), ('\u{cb5}', '\u{cb9}', WC_ALetter), ('\u{cbc}',
        '\u{cbc}', WC_Extend), ('\u{cbd}', '\u{cbd}', WC_ALetter), ('\u{cbe}', '\u{cc4}',
        WC_Extend), ('\u{cc6}', '\u{cc8}', WC_Extend), ('\u{cca}', '\u{ccd}', WC_Extend),
        ('\u{cd5}', '\u{cd6}', WC_Extend), ('\u{cde}', '\u{cde}', WC_ALetter), ('\u{ce0}',
        '\u{ce1}', WC_ALetter), ('\u{ce2}', '\u{ce3}', WC_Extend), ('\u{ce6}', '\u{cef}',
        WC_Numeric), ('\u{cf1}', '\u{cf2}', WC_ALetter), ('\u{d01}', '\u{d03}', WC_Extend),
        ('\u{d05}', '\u{d0c}', WC_ALetter), ('\u{d0e}', '\u{d10}', WC_ALetter), ('\u{d12}',
        '\u{d3a}', WC_ALetter), ('\u{d3d}', '\u{d3d}', WC_ALetter), ('\u{d3e}', '\u{d44}',
        WC_Extend), ('\u{d46}', '\u{d48}', WC_Extend), ('\u{d4a}', '\u{d4d}', WC_Extend),
        ('\u{d4e}', '\u{d4e}', WC_ALetter), ('\u{d54}', '\u{d56}', WC_ALetter), ('\u{d57}',
        '\u{d57}', WC_Extend), ('\u{d5f}', '\u{d61}', WC_ALetter), ('\u{d62}', '\u{d63}',
        WC_Extend), ('\u{d66}', '\u{d6f}', WC_Numeric), ('\u{d7a}', '\u{d7f}', WC_ALetter),
        ('\u{d82}', '\u{d83}', WC_Extend), ('\u{d85}', '\u{d96}', WC_ALetter), ('\u{d9a}',
        '\u{db1}', WC_ALetter), ('\u{db3}', '\u{dbb}', WC_ALetter), ('\u{dbd}', '\u{dbd}',
        WC_ALetter), ('\u{dc0}', '\u{dc6}', WC_ALetter), ('\u{dca}', '\u{dca}', WC_Extend),
        ('\u{dcf}', '\u{dd4}', WC_Extend), ('\u{dd6}', '\u{dd6}', WC_Extend), ('\u{dd8}', '\u{ddf}',
        WC_Extend), ('\u{de6}', '\u{def}', WC_Numeric), ('\u{df2}', '\u{df3}', WC_Extend),
        ('\u{e31}', '\u{e31}', WC_Extend), ('\u{e34}', '\u{e3a}', WC_Extend), ('\u{e47}', '\u{e4e}',
        WC_Extend), ('\u{e50}', '\u{e59}', WC_Numeric), ('\u{eb1}', '\u{eb1}', WC_Extend),
        ('\u{eb4}', '\u{eb9}', WC_Extend), ('\u{ebb}', '\u{ebc}', WC_Extend), ('\u{ec8}', '\u{ecd}',
        WC_Extend), ('\u{ed0}', '\u{ed9}', WC_Numeric), ('\u{f00}', '\u{f00}', WC_ALetter),
        ('\u{f18}', '\u{f19}', WC_Extend), ('\u{f20}', '\u{f29}', WC_Numeric), ('\u{f35}',
        '\u{f35}', WC_Extend), ('\u{f37}', '\u{f37}', WC_Extend), ('\u{f39}', '\u{f39}', WC_Extend),
        ('\u{f3e}', '\u{f3f}', WC_Extend), ('\u{f40}', '\u{f47}', WC_ALetter), ('\u{f49}',
        '\u{f6c}', WC_ALetter), ('\u{f71}', '\u{f84}', WC_Extend), ('\u{f86}', '\u{f87}',
        WC_Extend), ('\u{f88}', '\u{f8c}', WC_ALetter), ('\u{f8d}', '\u{f97}', WC_Extend),
        ('\u{f99}', '\u{fbc}', WC_Extend), ('\u{fc6}', '\u{fc6}', WC_Extend), ('\u{102b}',
        '\u{103e}', WC_Extend), ('\u{1040}', '\u{1049}', WC_Numeric), ('\u{1056}', '\u{1059}',
        WC_Extend), ('\u{105e}', '\u{1060}', WC_Extend), ('\u{1062}', '\u{1064}', WC_Extend),
        ('\u{1067}', '\u{106d}', WC_Extend), ('\u{1071}', '\u{1074}', WC_Extend), ('\u{1082}',
        '\u{108d}', WC_Extend), ('\u{108f}', '\u{108f}', WC_Extend), ('\u{1090}', '\u{1099}',
        WC_Numeric), ('\u{109a}', '\u{109d}', WC_Extend), ('\u{10a0}', '\u{10c5}', WC_ALetter),
        ('\u{10c7}', '\u{10c7}', WC_ALetter), ('\u{10cd}', '\u{10cd}', WC_ALetter), ('\u{10d0}',
        '\u{10fa}', WC_ALetter), ('\u{10fc}', '\u{1248}', WC_ALetter), ('\u{124a}', '\u{124d}',
        WC_ALetter), ('\u{1250}', '\u{1256}', WC_ALetter), ('\u{1258}', '\u{1258}', WC_ALetter),
        ('\u{125a}', '\u{125d}', WC_ALetter), ('\u{1260}', '\u{1288}', WC_ALetter), ('\u{128a}',
        '\u{128d}', WC_ALetter), ('\u{1290}', '\u{12b0}', WC_ALetter), ('\u{12b2}', '\u{12b5}',
        WC_ALetter), ('\u{12b8}', '\u{12be}', WC_ALetter), ('\u{12c0}', '\u{12c0}', WC_ALetter),
        ('\u{12c2}', '\u{12c5}', WC_ALetter), ('\u{12c8}', '\u{12d6}', WC_ALetter), ('\u{12d8}',
        '\u{1310}', WC_ALetter), ('\u{1312}', '\u{1315}', WC_ALetter), ('\u{1318}', '\u{135a}',
        WC_ALetter), ('\u{135d}', '\u{135f}', WC_Extend), ('\u{1380}', '\u{138f}', WC_ALetter),
        ('\u{13a0}', '\u{13f5}', WC_ALetter), ('\u{13f8}', '\u{13fd}', WC_ALetter), ('\u{1401}',
        '\u{166c}', WC_ALetter), ('\u{166f}', '\u{167f}', WC_ALetter), ('\u{1681}', '\u{169a}',
        WC_ALetter), ('\u{16a0}', '\u{16ea}', WC_ALetter), ('\u{16ee}', '\u{16f8}', WC_ALetter),
        ('\u{1700}', '\u{170c}', WC_ALetter), ('\u{170e}', '\u{1711}', WC_ALetter), ('\u{1712}',
        '\u{1714}', WC_Extend), ('\u{1720}', '\u{1731}', WC_ALetter), ('\u{1732}', '\u{1734}',
        WC_Extend), ('\u{1740}', '\u{1751}', WC_ALetter), ('\u{1752}', '\u{1753}', WC_Extend),
        ('\u{1760}', '\u{176c}', WC_ALetter), ('\u{176e}', '\u{1770}', WC_ALetter), ('\u{1772}',
        '\u{1773}', WC_Extend), ('\u{17b4}', '\u{17d3}', WC_Extend), ('\u{17dd}', '\u{17dd}',
        WC_Extend), ('\u{17e0}', '\u{17e9}', WC_Numeric), ('\u{180b}', '\u{180d}', WC_Extend),
        ('\u{180e}', '\u{180e}', WC_Format), ('\u{1810}', '\u{1819}', WC_Numeric), ('\u{1820}',
        '\u{1877}', WC_ALetter), ('\u{1880}', '\u{1884}', WC_ALetter), ('\u{1885}', '\u{1886}',
        WC_Extend), ('\u{1887}', '\u{18a8}', WC_ALetter), ('\u{18a9}', '\u{18a9}', WC_Extend),
        ('\u{18aa}', '\u{18aa}', WC_ALetter), ('\u{18b0}', '\u{18f5}', WC_ALetter), ('\u{1900}',
        '\u{191e}', WC_ALetter), ('\u{1920}', '\u{192b}', WC_Extend), ('\u{1930}', '\u{193b}',
        WC_Extend), ('\u{1946}', '\u{194f}', WC_Numeric), ('\u{19d0}', '\u{19d9}', WC_Numeric),
        ('\u{1a00}', '\u{1a16}', WC_ALetter), ('\u{1a17}', '\u{1a1b}', WC_Extend), ('\u{1a55}',
        '\u{1a5e}', WC_Extend), ('\u{1a60}', '\u{1a7c}', WC_Extend), ('\u{1a7f}', '\u{1a7f}',
        WC_Extend), ('\u{1a80}', '\u{1a89}', WC_Numeric), ('\u{1a90}', '\u{1a99}', WC_Numeric),
        ('\u{1ab0}', '\u{1abe}', WC_Extend), ('\u{1b00}', '\u{1b04}', WC_Extend), ('\u{1b05}',
        '\u{1b33}', WC_ALetter), ('\u{1b34}', '\u{1b44}', WC_Extend), ('\u{1b45}', '\u{1b4b}',
        WC_ALetter), ('\u{1b50}', '\u{1b59}', WC_Numeric), ('\u{1b6b}', '\u{1b73}', WC_Extend),
        ('\u{1b80}', '\u{1b82}', WC_Extend), ('\u{1b83}', '\u{1ba0}', WC_ALetter), ('\u{1ba1}',
        '\u{1bad}', WC_Extend), ('\u{1bae}', '\u{1baf}', WC_ALetter), ('\u{1bb0}', '\u{1bb9}',
        WC_Numeric), ('\u{1bba}', '\u{1be5}', WC_ALetter), ('\u{1be6}', '\u{1bf3}', WC_Extend),
        ('\u{1c00}', '\u{1c23}', WC_ALetter), ('\u{1c24}', '\u{1c37}', WC_Extend), ('\u{1c40}',
        '\u{1c49}', WC_Numeric), ('\u{1c4d}', '\u{1c4f}', WC_ALetter), ('\u{1c50}', '\u{1c59}',
        WC_Numeric), ('\u{1c5a}', '\u{1c7d}', WC_ALetter), ('\u{1c80}', '\u{1c88}', WC_ALetter),
        ('\u{1cd0}', '\u{1cd2}', WC_Extend), ('\u{1cd4}', '\u{1ce8}', WC_Extend), ('\u{1ce9}',
        '\u{1cec}', WC_ALetter), ('\u{1ced}', '\u{1ced}', WC_Extend), ('\u{1cee}', '\u{1cf1}',
        WC_ALetter), ('\u{1cf2}', '\u{1cf4}', WC_Extend), ('\u{1cf5}', '\u{1cf6}', WC_ALetter),
        ('\u{1cf8}', '\u{1cf9}', WC_Extend), ('\u{1d00}', '\u{1dbf}', WC_ALetter), ('\u{1dc0}',
        '\u{1df5}', WC_Extend), ('\u{1dfb}', '\u{1dff}', WC_Extend), ('\u{1e00}', '\u{1f15}',
        WC_ALetter), ('\u{1f18}', '\u{1f1d}', WC_ALetter), ('\u{1f20}', '\u{1f45}', WC_ALetter),
        ('\u{1f48}', '\u{1f4d}', WC_ALetter), ('\u{1f50}', '\u{1f57}', WC_ALetter), ('\u{1f59}',
        '\u{1f59}', WC_ALetter), ('\u{1f5b}', '\u{1f5b}', WC_ALetter), ('\u{1f5d}', '\u{1f5d}',
        WC_ALetter), ('\u{1f5f}', '\u{1f7d}', WC_ALetter), ('\u{1f80}', '\u{1fb4}', WC_ALetter),
        ('\u{1fb6}', '\u{1fbc}', WC_ALetter), ('\u{1fbe}', '\u{1fbe}', WC_ALetter), ('\u{1fc2}',
        '\u{1fc4}', WC_ALetter), ('\u{1fc6}', '\u{1fcc}', WC_ALetter), ('\u{1fd0}', '\u{1fd3}',
        WC_ALetter), ('\u{1fd6}', '\u{1fdb}', WC_ALetter), ('\u{1fe0}', '\u{1fec}', WC_ALetter),
        ('\u{1ff2}', '\u{1ff4}', WC_ALetter), ('\u{1ff6}', '\u{1ffc}', WC_ALetter), ('\u{200c}',
        '\u{200c}', WC_Extend), ('\u{200d}', '\u{200d}', WC_ZWJ), ('\u{200e}', '\u{200f}',
        WC_Format), ('\u{2018}', '\u{2019}', WC_MidNumLet), ('\u{2024}', '\u{2024}', WC_MidNumLet),
        ('\u{2027}', '\u{2027}', WC_MidLetter), ('\u{2028}', '\u{2029}', WC_Newline), ('\u{202a}',
        '\u{202e}', WC_Format), ('\u{202f}', '\u{202f}', WC_ExtendNumLet), ('\u{203f}', '\u{2040}',
        WC_ExtendNumLet), ('\u{2044}', '\u{2044}', WC_MidNum), ('\u{2054}', '\u{2054}',
        WC_ExtendNumLet), ('\u{2060}', '\u{2064}', WC_Format), ('\u{2066}', '\u{206f}', WC_Format),
        ('\u{2071}', '\u{2071}', WC_ALetter), ('\u{207f}', '\u{207f}', WC_ALetter), ('\u{2090}',
        '\u{209c}', WC_ALetter), ('\u{20d0}', '\u{20f0}', WC_Extend), ('\u{2102}', '\u{2102}',
        WC_ALetter), ('\u{2107}', '\u{2107}', WC_ALetter), ('\u{210a}', '\u{2113}', WC_ALetter),
        ('\u{2115}', '\u{2115}', WC_ALetter), ('\u{2119}', '\u{211d}', WC_ALetter), ('\u{2124}',
        '\u{2124}', WC_ALetter), ('\u{2126}', '\u{2126}', WC_ALetter), ('\u{2128}', '\u{2128}',
        WC_ALetter), ('\u{212a}', '\u{212d}', WC_ALetter), ('\u{212f}', '\u{2139}', WC_ALetter),
        ('\u{213c}', '\u{213f}', WC_ALetter), ('\u{2145}', '\u{2149}', WC_ALetter), ('\u{214e}',
        '\u{214e}', WC_ALetter), ('\u{2160}', '\u{2188}', WC_ALetter), ('\u{24b6}', '\u{24e9}',
        WC_ALetter), ('\u{261d}', '\u{261d}', WC_E_Base), ('\u{26f9}', '\u{26f9}', WC_E_Base),
        ('\u{270a}', '\u{270d}', WC_E_Base), ('\u{2764}', '\u{2764}', WC_Glue_After_Zwj),
        ('\u{2c00}', '\u{2c2e}', WC_ALetter), ('\u{2c30}', '\u{2c5e}', WC_ALetter), ('\u{2c60}',
        '\u{2ce4}', WC_ALetter), ('\u{2ceb}', '\u{2cee}', WC_ALetter), ('\u{2cef}', '\u{2cf1}',
        WC_Extend), ('\u{2cf2}', '\u{2cf3}', WC_ALetter), ('\u{2d00}', '\u{2d25}', WC_ALetter),
        ('\u{2d27}', '\u{2d27}', WC_ALetter), ('\u{2d2d}', '\u{2d2d}', WC_ALetter), ('\u{2d30}',
        '\u{2d67}', WC_ALetter), ('\u{2d6f}', '\u{2d6f}', WC_ALetter), ('\u{2d7f}', '\u{2d7f}',
        WC_Extend), ('\u{2d80}', '\u{2d96}', WC_ALetter), ('\u{2da0}', '\u{2da6}', WC_ALetter),
        ('\u{2da8}', '\u{2dae}', WC_ALetter), ('\u{2db0}', '\u{2db6}', WC_ALetter), ('\u{2db8}',
        '\u{2dbe}', WC_ALetter), ('\u{2dc0}', '\u{2dc6}', WC_ALetter), ('\u{2dc8}', '\u{2dce}',
        WC_ALetter), ('\u{2dd0}', '\u{2dd6}', WC_ALetter), ('\u{2dd8}', '\u{2dde}', WC_ALetter),
        ('\u{2de0}', '\u{2dff}', WC_Extend), ('\u{2e2f}', '\u{2e2f}', WC_ALetter), ('\u{3005}',
        '\u{3005}', WC_ALetter), ('\u{302a}', '\u{302f}', WC_Extend), ('\u{3031}', '\u{3035}',
        WC_Katakana), ('\u{303b}', '\u{303c}', WC_ALetter), ('\u{3099}', '\u{309a}', WC_Extend),
        ('\u{309b}', '\u{309c}', WC_Katakana), ('\u{30a0}', '\u{30fa}', WC_Katakana), ('\u{30fc}',
        '\u{30ff}', WC_Katakana), ('\u{3105}', '\u{312d}', WC_ALetter), ('\u{3131}', '\u{318e}',
        WC_ALetter), ('\u{31a0}', '\u{31ba}', WC_ALetter), ('\u{31f0}', '\u{31ff}', WC_Katakana),
        ('\u{32d0}', '\u{32fe}', WC_Katakana), ('\u{3300}', '\u{3357}', WC_Katakana), ('\u{a000}',
        '\u{a48c}', WC_ALetter), ('\u{a4d0}', '\u{a4fd}', WC_ALetter), ('\u{a500}', '\u{a60c}',
        WC_ALetter), ('\u{a610}', '\u{a61f}', WC_ALetter), ('\u{a620}', '\u{a629}', WC_Numeric),
        ('\u{a62a}', '\u{a62b}', WC_ALetter), ('\u{a640}', '\u{a66e}', WC_ALetter), ('\u{a66f}',
        '\u{a672}', WC_Extend), ('\u{a674}', '\u{a67d}', WC_Extend), ('\u{a67f}', '\u{a69d}',
        WC_ALetter), ('\u{a69e}', '\u{a69f}', WC_Extend), ('\u{a6a0}', '\u{a6ef}', WC_ALetter),
        ('\u{a6f0}', '\u{a6f1}', WC_Extend), ('\u{a717}', '\u{a71f}', WC_ALetter), ('\u{a722}',
        '\u{a788}', WC_ALetter), ('\u{a78b}', '\u{a7ae}', WC_ALetter), ('\u{a7b0}', '\u{a7b7}',
        WC_ALetter), ('\u{a7f7}', '\u{a801}', WC_ALetter), ('\u{a802}', '\u{a802}', WC_Extend),
        ('\u{a803}', '\u{a805}', WC_ALetter), ('\u{a806}', '\u{a806}', WC_Extend), ('\u{a807}',
        '\u{a80a}', WC_ALetter), ('\u{a80b}', '\u{a80b}', WC_Extend), ('\u{a80c}', '\u{a822}',
        WC_ALetter), ('\u{a823}', '\u{a827}', WC_Extend), ('\u{a840}', '\u{a873}', WC_ALetter),
        ('\u{a880}', '\u{a881}', WC_Extend), ('\u{a882}', '\u{a8b3}', WC_ALetter), ('\u{a8b4}',
        '\u{a8c5}', WC_Extend), ('\u{a8d0}', '\u{a8d9}', WC_Numeric), ('\u{a8e0}', '\u{a8f1}',
        WC_Extend), ('\u{a8f2}', '\u{a8f7}', WC_ALetter), ('\u{a8fb}', '\u{a8fb}', WC_ALetter),
        ('\u{a8fd}', '\u{a8fd}', WC_ALetter), ('\u{a900}', '\u{a909}', WC_Numeric), ('\u{a90a}',
        '\u{a925}', WC_ALetter), ('\u{a926}', '\u{a92d}', WC_Extend), ('\u{a930}', '\u{a946}',
        WC_ALetter), ('\u{a947}', '\u{a953}', WC_Extend), ('\u{a960}', '\u{a97c}', WC_ALetter),
        ('\u{a980}', '\u{a983}', WC_Extend), ('\u{a984}', '\u{a9b2}', WC_ALetter), ('\u{a9b3}',
        '\u{a9c0}', WC_Extend), ('\u{a9cf}', '\u{a9cf}', WC_ALetter), ('\u{a9d0}', '\u{a9d9}',
        WC_Numeric), ('\u{a9e5}', '\u{a9e5}', WC_Extend), ('\u{a9f0}', '\u{a9f9}', WC_Numeric),
        ('\u{aa00}', '\u{aa28}', WC_ALetter), ('\u{aa29}', '\u{aa36}', WC_Extend), ('\u{aa40}',
        '\u{aa42}', WC_ALetter), ('\u{aa43}', '\u{aa43}', WC_Extend), ('\u{aa44}', '\u{aa4b}',
        WC_ALetter), ('\u{aa4c}', '\u{aa4d}', WC_Extend), ('\u{aa50}', '\u{aa59}', WC_Numeric),
        ('\u{aa7b}', '\u{aa7d}', WC_Extend), ('\u{aab0}', '\u{aab0}', WC_Extend), ('\u{aab2}',
        '\u{aab4}', WC_Extend), ('\u{aab7}', '\u{aab8}', WC_Extend), ('\u{aabe}', '\u{aabf}',
        WC_Extend), ('\u{aac1}', '\u{aac1}', WC_Extend), ('\u{aae0}', '\u{aaea}', WC_ALetter),
        ('\u{aaeb}', '\u{aaef}', WC_Extend), ('\u{aaf2}', '\u{aaf4}', WC_ALetter), ('\u{aaf5}',
        '\u{aaf6}', WC_Extend), ('\u{ab01}', '\u{ab06}', WC_ALetter), ('\u{ab09}', '\u{ab0e}',
        WC_ALetter), ('\u{ab11}', '\u{ab16}', WC_ALetter), ('\u{ab20}', '\u{ab26}', WC_ALetter),
        ('\u{ab28}', '\u{ab2e}', WC_ALetter), ('\u{ab30}', '\u{ab5a}', WC_ALetter), ('\u{ab5c}',
        '\u{ab65}', WC_ALetter), ('\u{ab70}', '\u{abe2}', WC_ALetter), ('\u{abe3}', '\u{abea}',
        WC_Extend), ('\u{abec}', '\u{abed}', WC_Extend), ('\u{abf0}', '\u{abf9}', WC_Numeric),
        ('\u{ac00}', '\u{d7a3}', WC_ALetter), ('\u{d7b0}', '\u{d7c6}', WC_ALetter), ('\u{d7cb}',
        '\u{d7fb}', WC_ALetter), ('\u{fb00}', '\u{fb06}', WC_ALetter), ('\u{fb13}', '\u{fb17}',
        WC_ALetter), ('\u{fb1d}', '\u{fb1d}', WC_Hebrew_Letter), ('\u{fb1e}', '\u{fb1e}',
        WC_Extend), ('\u{fb1f}', '\u{fb28}', WC_Hebrew_Letter), ('\u{fb2a}', '\u{fb36}',
        WC_Hebrew_Letter), ('\u{fb38}', '\u{fb3c}', WC_Hebrew_Letter), ('\u{fb3e}', '\u{fb3e}',
        WC_Hebrew_Letter), ('\u{fb40}', '\u{fb41}', WC_Hebrew_Letter), ('\u{fb43}', '\u{fb44}',
        WC_Hebrew_Letter), ('\u{fb46}', '\u{fb4f}', WC_Hebrew_Letter), ('\u{fb50}', '\u{fbb1}',
        WC_ALetter), ('\u{fbd3}', '\u{fd3d}', WC_ALetter), ('\u{fd50}', '\u{fd8f}', WC_ALetter),
        ('\u{fd92}', '\u{fdc7}', WC_ALetter), ('\u{fdf0}', '\u{fdfb}', WC_ALetter), ('\u{fe00}',
        '\u{fe0f}', WC_Extend), ('\u{fe10}', '\u{fe10}', WC_MidNum), ('\u{fe13}', '\u{fe13}',
        WC_MidLetter), ('\u{fe14}', '\u{fe14}', WC_MidNum), ('\u{fe20}', '\u{fe2f}', WC_Extend),
        ('\u{fe33}', '\u{fe34}', WC_ExtendNumLet), ('\u{fe4d}', '\u{fe4f}', WC_ExtendNumLet),
        ('\u{fe50}', '\u{fe50}', WC_MidNum), ('\u{fe52}', '\u{fe52}', WC_MidNumLet), ('\u{fe54}',
        '\u{fe54}', WC_MidNum), ('\u{fe55}', '\u{fe55}', WC_MidLetter), ('\u{fe70}', '\u{fe74}',
        WC_ALetter), ('\u{fe76}', '\u{fefc}', WC_ALetter), ('\u{feff}', '\u{feff}', WC_Format),
        ('\u{ff07}', '\u{ff07}', WC_MidNumLet), ('\u{ff0c}', '\u{ff0c}', WC_MidNum), ('\u{ff0e}',
        '\u{ff0e}', WC_MidNumLet), ('\u{ff1a}', '\u{ff1a}', WC_MidLetter), ('\u{ff1b}', '\u{ff1b}',
        WC_MidNum), ('\u{ff21}', '\u{ff3a}', WC_ALetter), ('\u{ff3f}', '\u{ff3f}', WC_ExtendNumLet),
        ('\u{ff41}', '\u{ff5a}', WC_ALetter), ('\u{ff66}', '\u{ff9d}', WC_Katakana), ('\u{ff9e}',
        '\u{ff9f}', WC_Extend), ('\u{ffa0}', '\u{ffbe}', WC_ALetter), ('\u{ffc2}', '\u{ffc7}',
        WC_ALetter), ('\u{ffca}', '\u{ffcf}', WC_ALetter), ('\u{ffd2}', '\u{ffd7}', WC_ALetter),
        ('\u{ffda}', '\u{ffdc}', WC_ALetter), ('\u{fff9}', '\u{fffb}', WC_Format), ('\u{10000}',
        '\u{1000b}', WC_ALetter), ('\u{1000d}', '\u{10026}', WC_ALetter), ('\u{10028}', '\u{1003a}',
        WC_ALetter), ('\u{1003c}', '\u{1003d}', WC_ALetter), ('\u{1003f}', '\u{1004d}', WC_ALetter),
        ('\u{10050}', '\u{1005d}', WC_ALetter), ('\u{10080}', '\u{100fa}', WC_ALetter),
        ('\u{10140}', '\u{10174}', WC_ALetter), ('\u{101fd}', '\u{101fd}', WC_Extend), ('\u{10280}',
        '\u{1029c}', WC_ALetter), ('\u{102a0}', '\u{102d0}', WC_ALetter), ('\u{102e0}', '\u{102e0}',
        WC_Extend), ('\u{10300}', '\u{1031f}', WC_ALetter), ('\u{10330}', '\u{1034a}', WC_ALetter),
        ('\u{10350}', '\u{10375}', WC_ALetter), ('\u{10376}', '\u{1037a}', WC_Extend), ('\u{10380}',
        '\u{1039d}', WC_ALetter), ('\u{103a0}', '\u{103c3}', WC_ALetter), ('\u{103c8}', '\u{103cf}',
        WC_ALetter), ('\u{103d1}', '\u{103d5}', WC_ALetter), ('\u{10400}', '\u{1049d}', WC_ALetter),
        ('\u{104a0}', '\u{104a9}', WC_Numeric), ('\u{104b0}', '\u{104d3}', WC_ALetter),
        ('\u{104d8}', '\u{104fb}', WC_ALetter), ('\u{10500}', '\u{10527}', WC_ALetter),
        ('\u{10530}', '\u{10563}', WC_ALetter), ('\u{10600}', '\u{10736}', WC_ALetter),
        ('\u{10740}', '\u{10755}', WC_ALetter), ('\u{10760}', '\u{10767}', WC_ALetter),
        ('\u{10800}', '\u{10805}', WC_ALetter), ('\u{10808}', '\u{10808}', WC_ALetter),
        ('\u{1080a}', '\u{10835}', WC_ALetter), ('\u{10837}', '\u{10838}', WC_ALetter),
        ('\u{1083c}', '\u{1083c}', WC_ALetter), ('\u{1083f}', '\u{10855}', WC_ALetter),
        ('\u{10860}', '\u{10876}', WC_ALetter), ('\u{10880}', '\u{1089e}', WC_ALetter),
        ('\u{108e0}', '\u{108f2}', WC_ALetter), ('\u{108f4}', '\u{108f5}', WC_ALetter),
        ('\u{10900}', '\u{10915}', WC_ALetter), ('\u{10920}', '\u{10939}', WC_ALetter),
        ('\u{10980}', '\u{109b7}', WC_ALetter), ('\u{109be}', '\u{109bf}', WC_ALetter),
        ('\u{10a00}', '\u{10a00}', WC_ALetter), ('\u{10a01}', '\u{10a03}', WC_Extend), ('\u{10a05}',
        '\u{10a06}', WC_Extend), ('\u{10a0c}', '\u{10a0f}', WC_Extend), ('\u{10a10}', '\u{10a13}',
        WC_ALetter), ('\u{10a15}', '\u{10a17}', WC_ALetter), ('\u{10a19}', '\u{10a33}', WC_ALetter),
        ('\u{10a38}', '\u{10a3a}', WC_Extend), ('\u{10a3f}', '\u{10a3f}', WC_Extend), ('\u{10a60}',
        '\u{10a7c}', WC_ALetter), ('\u{10a80}', '\u{10a9c}', WC_ALetter), ('\u{10ac0}', '\u{10ac7}',
        WC_ALetter), ('\u{10ac9}', '\u{10ae4}', WC_ALetter), ('\u{10ae5}', '\u{10ae6}', WC_Extend),
        ('\u{10b00}', '\u{10b35}', WC_ALetter), ('\u{10b40}', '\u{10b55}', WC_ALetter),
        ('\u{10b60}', '\u{10b72}', WC_ALetter), ('\u{10b80}', '\u{10b91}', WC_ALetter),
        ('\u{10c00}', '\u{10c48}', WC_ALetter), ('\u{10c80}', '\u{10cb2}', WC_ALetter),
        ('\u{10cc0}', '\u{10cf2}', WC_ALetter), ('\u{11000}', '\u{11002}', WC_Extend), ('\u{11003}',
        '\u{11037}', WC_ALetter), ('\u{11038}', '\u{11046}', WC_Extend), ('\u{11066}', '\u{1106f}',
        WC_Numeric), ('\u{1107f}', '\u{11082}', WC_Extend), ('\u{11083}', '\u{110af}', WC_ALetter),
        ('\u{110b0}', '\u{110ba}', WC_Extend), ('\u{110bd}', '\u{110bd}', WC_Format), ('\u{110d0}',
        '\u{110e8}', WC_ALetter), ('\u{110f0}', '\u{110f9}', WC_Numeric), ('\u{11100}', '\u{11102}',
        WC_Extend), ('\u{11103}', '\u{11126}', WC_ALetter), ('\u{11127}', '\u{11134}', WC_Extend),
        ('\u{11136}', '\u{1113f}', WC_Numeric), ('\u{11150}', '\u{11172}', WC_ALetter),
        ('\u{11173}', '\u{11173}', WC_Extend), ('\u{11176}', '\u{11176}', WC_ALetter), ('\u{11180}',
        '\u{11182}', WC_Extend), ('\u{11183}', '\u{111b2}', WC_ALetter), ('\u{111b3}', '\u{111c0}',
        WC_Extend), ('\u{111c1}', '\u{111c4}', WC_ALetter), ('\u{111ca}', '\u{111cc}', WC_Extend),
        ('\u{111d0}', '\u{111d9}', WC_Numeric), ('\u{111da}', '\u{111da}', WC_ALetter),
        ('\u{111dc}', '\u{111dc}', WC_ALetter), ('\u{11200}', '\u{11211}', WC_ALetter),
        ('\u{11213}', '\u{1122b}', WC_ALetter), ('\u{1122c}', '\u{11237}', WC_Extend), ('\u{1123e}',
        '\u{1123e}', WC_Extend), ('\u{11280}', '\u{11286}', WC_ALetter), ('\u{11288}', '\u{11288}',
        WC_ALetter), ('\u{1128a}', '\u{1128d}', WC_ALetter), ('\u{1128f}', '\u{1129d}', WC_ALetter),
        ('\u{1129f}', '\u{112a8}', WC_ALetter), ('\u{112b0}', '\u{112de}', WC_ALetter),
        ('\u{112df}', '\u{112ea}', WC_Extend), ('\u{112f0}', '\u{112f9}', WC_Numeric), ('\u{11300}',
        '\u{11303}', WC_Extend), ('\u{11305}', '\u{1130c}', WC_ALetter), ('\u{1130f}', '\u{11310}',
        WC_ALetter), ('\u{11313}', '\u{11328}', WC_ALetter), ('\u{1132a}', '\u{11330}', WC_ALetter),
        ('\u{11332}', '\u{11333}', WC_ALetter), ('\u{11335}', '\u{11339}', WC_ALetter),
        ('\u{1133c}', '\u{1133c}', WC_Extend), ('\u{1133d}', '\u{1133d}', WC_ALetter), ('\u{1133e}',
        '\u{11344}', WC_Extend), ('\u{11347}', '\u{11348}', WC_Extend), ('\u{1134b}', '\u{1134d}',
        WC_Extend), ('\u{11350}', '\u{11350}', WC_ALetter), ('\u{11357}', '\u{11357}', WC_Extend),
        ('\u{1135d}', '\u{11361}', WC_ALetter), ('\u{11362}', '\u{11363}', WC_Extend), ('\u{11366}',
        '\u{1136c}', WC_Extend), ('\u{11370}', '\u{11374}', WC_Extend), ('\u{11400}', '\u{11434}',
        WC_ALetter), ('\u{11435}', '\u{11446}', WC_Extend), ('\u{11447}', '\u{1144a}', WC_ALetter),
        ('\u{11450}', '\u{11459}', WC_Numeric), ('\u{11480}', '\u{114af}', WC_ALetter),
        ('\u{114b0}', '\u{114c3}', WC_Extend), ('\u{114c4}', '\u{114c5}', WC_ALetter), ('\u{114c7}',
        '\u{114c7}', WC_ALetter), ('\u{114d0}', '\u{114d9}', WC_Numeric), ('\u{11580}', '\u{115ae}',
        WC_ALetter), ('\u{115af}', '\u{115b5}', WC_Extend), ('\u{115b8}', '\u{115c0}', WC_Extend),
        ('\u{115d8}', '\u{115db}', WC_ALetter), ('\u{115dc}', '\u{115dd}', WC_Extend), ('\u{11600}',
        '\u{1162f}', WC_ALetter), ('\u{11630}', '\u{11640}', WC_Extend), ('\u{11644}', '\u{11644}',
        WC_ALetter), ('\u{11650}', '\u{11659}', WC_Numeric), ('\u{11680}', '\u{116aa}', WC_ALetter),
        ('\u{116ab}', '\u{116b7}', WC_Extend), ('\u{116c0}', '\u{116c9}', WC_Numeric), ('\u{1171d}',
        '\u{1172b}', WC_Extend), ('\u{11730}', '\u{11739}', WC_Numeric), ('\u{118a0}', '\u{118df}',
        WC_ALetter), ('\u{118e0}', '\u{118e9}', WC_Numeric), ('\u{118ff}', '\u{118ff}', WC_ALetter),
        ('\u{11ac0}', '\u{11af8}', WC_ALetter), ('\u{11c00}', '\u{11c08}', WC_ALetter),
        ('\u{11c0a}', '\u{11c2e}', WC_ALetter), ('\u{11c2f}', '\u{11c36}', WC_Extend), ('\u{11c38}',
        '\u{11c3f}', WC_Extend), ('\u{11c40}', '\u{11c40}', WC_ALetter), ('\u{11c50}', '\u{11c59}',
        WC_Numeric), ('\u{11c72}', '\u{11c8f}', WC_ALetter), ('\u{11c92}', '\u{11ca7}', WC_Extend),
        ('\u{11ca9}', '\u{11cb6}', WC_Extend), ('\u{12000}', '\u{12399}', WC_ALetter), ('\u{12400}',
        '\u{1246e}', WC_ALetter), ('\u{12480}', '\u{12543}', WC_ALetter), ('\u{13000}', '\u{1342e}',
        WC_ALetter), ('\u{14400}', '\u{14646}', WC_ALetter), ('\u{16800}', '\u{16a38}', WC_ALetter),
        ('\u{16a40}', '\u{16a5e}', WC_ALetter), ('\u{16a60}', '\u{16a69}', WC_Numeric),
        ('\u{16ad0}', '\u{16aed}', WC_ALetter), ('\u{16af0}', '\u{16af4}', WC_Extend), ('\u{16b00}',
        '\u{16b2f}', WC_ALetter), ('\u{16b30}', '\u{16b36}', WC_Extend), ('\u{16b40}', '\u{16b43}',
        WC_ALetter), ('\u{16b50}', '\u{16b59}', WC_Numeric), ('\u{16b63}', '\u{16b77}', WC_ALetter),
        ('\u{16b7d}', '\u{16b8f}', WC_ALetter), ('\u{16f00}', '\u{16f44}', WC_ALetter),
        ('\u{16f50}', '\u{16f50}', WC_ALetter), ('\u{16f51}', '\u{16f7e}', WC_Extend), ('\u{16f8f}',
        '\u{16f92}', WC_Extend), ('\u{16f93}', '\u{16f9f}', WC_ALetter), ('\u{16fe0}', '\u{16fe0}',
        WC_ALetter), ('\u{1b000}', '\u{1b000}', WC_Katakana), ('\u{1bc00}', '\u{1bc6a}',
        WC_ALetter), ('\u{1bc70}', '\u{1bc7c}', WC_ALetter), ('\u{1bc80}', '\u{1bc88}', WC_ALetter),
        ('\u{1bc90}', '\u{1bc99}', WC_ALetter), ('\u{1bc9d}', '\u{1bc9e}', WC_Extend), ('\u{1bca0}',
        '\u{1bca3}', WC_Format), ('\u{1d165}', '\u{1d169}', WC_Extend), ('\u{1d16d}', '\u{1d172}',
        WC_Extend), ('\u{1d173}', '\u{1d17a}', WC_Format), ('\u{1d17b}', '\u{1d182}', WC_Extend),
        ('\u{1d185}', '\u{1d18b}', WC_Extend), ('\u{1d1aa}', '\u{1d1ad}', WC_Extend), ('\u{1d242}',
        '\u{1d244}', WC_Extend), ('\u{1d400}', '\u{1d454}', WC_ALetter), ('\u{1d456}', '\u{1d49c}',
        WC_ALetter), ('\u{1d49e}', '\u{1d49f}', WC_ALetter), ('\u{1d4a2}', '\u{1d4a2}', WC_ALetter),
        ('\u{1d4a5}', '\u{1d4a6}', WC_ALetter), ('\u{1d4a9}', '\u{1d4ac}', WC_ALetter),
        ('\u{1d4ae}', '\u{1d4b9}', WC_ALetter), ('\u{1d4bb}', '\u{1d4bb}', WC_ALetter),
        ('\u{1d4bd}', '\u{1d4c3}', WC_ALetter), ('\u{1d4c5}', '\u{1d505}', WC_ALetter),
        ('\u{1d507}', '\u{1d50a}', WC_ALetter), ('\u{1d50d}', '\u{1d514}', WC_ALetter),
        ('\u{1d516}', '\u{1d51c}', WC_ALetter), ('\u{1d51e}', '\u{1d539}', WC_ALetter),
        ('\u{1d53b}', '\u{1d53e}', WC_ALetter), ('\u{1d540}', '\u{1d544}', WC_ALetter),
        ('\u{1d546}', '\u{1d546}', WC_ALetter), ('\u{1d54a}', '\u{1d550}', WC_ALetter),
        ('\u{1d552}', '\u{1d6a5}', WC_ALetter), ('\u{1d6a8}', '\u{1d6c0}', WC_ALetter),
        ('\u{1d6c2}', '\u{1d6da}', WC_ALetter), ('\u{1d6dc}', '\u{1d6fa}', WC_ALetter),
        ('\u{1d6fc}', '\u{1d714}', WC_ALetter), ('\u{1d716}', '\u{1d734}', WC_ALetter),
        ('\u{1d736}', '\u{1d74e}', WC_ALetter), ('\u{1d750}', '\u{1d76e}', WC_ALetter),
        ('\u{1d770}', '\u{1d788}', WC_ALetter), ('\u{1d78a}', '\u{1d7a8}', WC_ALetter),
        ('\u{1d7aa}', '\u{1d7c2}', WC_ALetter), ('\u{1d7c4}', '\u{1d7cb}', WC_ALetter),
        ('\u{1d7ce}', '\u{1d7ff}', WC_Numeric), ('\u{1da00}', '\u{1da36}', WC_Extend), ('\u{1da3b}',
        '\u{1da6c}', WC_Extend), ('\u{1da75}', '\u{1da75}', WC_Extend), ('\u{1da84}', '\u{1da84}',
        WC_Extend), ('\u{1da9b}', '\u{1da9f}', WC_Extend), ('\u{1daa1}', '\u{1daaf}', WC_Extend),
        ('\u{1e000}', '\u{1e006}', WC_Extend), ('\u{1e008}', '\u{1e018}', WC_Extend), ('\u{1e01b}',
        '\u{1e021}', WC_Extend), ('\u{1e023}', '\u{1e024}', WC_Extend), ('\u{1e026}', '\u{1e02a}',
        WC_Extend), ('\u{1e800}', '\u{1e8c4}', WC_ALetter), ('\u{1e8d0}', '\u{1e8d6}', WC_Extend),
        ('\u{1e900}', '\u{1e943}', WC_ALetter), ('\u{1e944}', '\u{1e94a}', WC_Extend), ('\u{1e950}',
        '\u{1e959}', WC_Numeric), ('\u{1ee00}', '\u{1ee03}', WC_ALetter), ('\u{1ee05}', '\u{1ee1f}',
        WC_ALetter), ('\u{1ee21}', '\u{1ee22}', WC_ALetter), ('\u{1ee24}', '\u{1ee24}', WC_ALetter),
        ('\u{1ee27}', '\u{1ee27}', WC_ALetter), ('\u{1ee29}', '\u{1ee32}', WC_ALetter),
        ('\u{1ee34}', '\u{1ee37}', WC_ALetter), ('\u{1ee39}', '\u{1ee39}', WC_ALetter),
        ('\u{1ee3b}', '\u{1ee3b}', WC_ALetter), ('\u{1ee42}', '\u{1ee42}', WC_ALetter),
        ('\u{1ee47}', '\u{1ee47}', WC_ALetter), ('\u{1ee49}', '\u{1ee49}', WC_ALetter),
        ('\u{1ee4b}', '\u{1ee4b}', WC_ALetter), ('\u{1ee4d}', '\u{1ee4f}', WC_ALetter),
        ('\u{1ee51}', '\u{1ee52}', WC_ALetter), ('\u{1ee54}', '\u{1ee54}', WC_ALetter),
        ('\u{1ee57}', '\u{1ee57}', WC_ALetter), ('\u{1ee59}', '\u{1ee59}', WC_ALetter),
        ('\u{1ee5b}', '\u{1ee5b}', WC_ALetter), ('\u{1ee5d}', '\u{1ee5d}', WC_ALetter),
        ('\u{1ee5f}', '\u{1ee5f}', WC_ALetter), ('\u{1ee61}', '\u{1ee62}', WC_ALetter),
        ('\u{1ee64}', '\u{1ee64}', WC_ALetter), ('\u{1ee67}', '\u{1ee6a}', WC_ALetter),
        ('\u{1ee6c}', '\u{1ee72}', WC_ALetter), ('\u{1ee74}', '\u{1ee77}', WC_ALetter),
        ('\u{1ee79}', '\u{1ee7c}', WC_ALetter), ('\u{1ee7e}', '\u{1ee7e}', WC_ALetter),
        ('\u{1ee80}', '\u{1ee89}', WC_ALetter), ('\u{1ee8b}', '\u{1ee9b}', WC_ALetter),
        ('\u{1eea1}', '\u{1eea3}', WC_ALetter), ('\u{1eea5}', '\u{1eea9}', WC_ALetter),
        ('\u{1eeab}', '\u{1eebb}', WC_ALetter), ('\u{1f130}', '\u{1f149}', WC_ALetter),
        ('\u{1f150}', '\u{1f169}', WC_ALetter), ('\u{1f170}', '\u{1f189}', WC_ALetter),
        ('\u{1f1e6}', '\u{1f1ff}', WC_Regional_Indicator), ('\u{1f385}', '\u{1f385}', WC_E_Base),
        ('\u{1f3c3}', '\u{1f3c4}', WC_E_Base), ('\u{1f3ca}', '\u{1f3cb}', WC_E_Base), ('\u{1f3fb}',
        '\u{1f3ff}', WC_E_Modifier), ('\u{1f442}', '\u{1f443}', WC_E_Base), ('\u{1f446}',
        '\u{1f450}', WC_E_Base), ('\u{1f466}', '\u{1f469}', WC_E_Base_GAZ), ('\u{1f46e}',
        '\u{1f46e}', WC_E_Base), ('\u{1f470}', '\u{1f478}', WC_E_Base), ('\u{1f47c}', '\u{1f47c}',
        WC_E_Base), ('\u{1f481}', '\u{1f483}', WC_E_Base), ('\u{1f485}', '\u{1f487}', WC_E_Base),
        ('\u{1f48b}', '\u{1f48b}', WC_Glue_After_Zwj), ('\u{1f4aa}', '\u{1f4aa}', WC_E_Base),
        ('\u{1f575}', '\u{1f575}', WC_E_Base), ('\u{1f57a}', '\u{1f57a}', WC_E_Base), ('\u{1f590}',
        '\u{1f590}', WC_E_Base), ('\u{1f595}', '\u{1f596}', WC_E_Base), ('\u{1f5e8}', '\u{1f5e8}',
        WC_Glue_After_Zwj), ('\u{1f645}', '\u{1f647}', WC_E_Base), ('\u{1f64b}', '\u{1f64f}',
        WC_E_Base), ('\u{1f6a3}', '\u{1f6a3}', WC_E_Base), ('\u{1f6b4}', '\u{1f6b6}', WC_E_Base),
        ('\u{1f6c0}', '\u{1f6c0}', WC_E_Base), ('\u{1f918}', '\u{1f91e}', WC_E_Base), ('\u{1f926}',
        '\u{1f926}', WC_E_Base), ('\u{1f930}', '\u{1f930}', WC_E_Base), ('\u{1f933}', '\u{1f939}',
        WC_E_Base), ('\u{1f93c}', '\u{1f93e}', WC_E_Base), ('\u{e0001}', '\u{e0001}', WC_Format),
        ('\u{e0020}', '\u{e007f}', WC_Extend), ('\u{e0100}', '\u{e01ef}', WC_Extend)
    ];

}
pub mod sentence {
    use core::result::Result::{Ok, Err};

    pub use self::SentenceCat::*;

    #[allow(non_camel_case_types)]
    #[derive(Clone, Copy, PartialEq, Eq)]
    pub enum SentenceCat {
        SC_ATerm,
        SC_Any,
        SC_CR,
        SC_Close,
        SC_Extend,
        SC_Format,
        SC_LF,
        SC_Lower,
        SC_Numeric,
        SC_OLetter,
        SC_SContinue,
        SC_STerm,
        SC_Sep,
        SC_Sp,
        SC_Upper,
    }

    fn bsearch_range_value_table(c: char, r: &'static [(char, char, SentenceCat)]) -> SentenceCat {
        use core::cmp::Ordering::{Equal, Less, Greater};
        match r.binary_search_by(|&(lo, hi, _)| {
            if lo <= c && c <= hi { Equal }
            else if hi < c { Less }
            else { Greater }
        }) {
            Ok(idx) => {
                let (_, _, cat) = r[idx];
                cat
            }
            Err(_) => SC_Any
        }
    }

    pub fn sentence_category(c: char) -> SentenceCat {
        bsearch_range_value_table(c, sentence_cat_table)
    }

    const sentence_cat_table: &'static [(char, char, SentenceCat)] = &[
        ('\u{9}', '\u{9}', SC_Sp), ('\u{a}', '\u{a}', SC_LF), ('\u{b}', '\u{c}', SC_Sp), ('\u{d}',
        '\u{d}', SC_CR), ('\u{20}', '\u{20}', SC_Sp), ('\u{21}', '\u{21}', SC_STerm), ('\u{22}',
        '\u{22}', SC_Close), ('\u{27}', '\u{29}', SC_Close), ('\u{2c}', '\u{2d}', SC_SContinue),
        ('\u{2e}', '\u{2e}', SC_ATerm), ('\u{30}', '\u{39}', SC_Numeric), ('\u{3a}', '\u{3a}',
        SC_SContinue), ('\u{3f}', '\u{3f}', SC_STerm), ('\u{41}', '\u{5a}', SC_Upper), ('\u{5b}',
        '\u{5b}', SC_Close), ('\u{5d}', '\u{5d}', SC_Close), ('\u{61}', '\u{7a}', SC_Lower),
        ('\u{7b}', '\u{7b}', SC_Close), ('\u{7d}', '\u{7d}', SC_Close), ('\u{85}', '\u{85}',
        SC_Sep), ('\u{a0}', '\u{a0}', SC_Sp), ('\u{aa}', '\u{aa}', SC_Lower), ('\u{ab}', '\u{ab}',
        SC_Close), ('\u{ad}', '\u{ad}', SC_Format), ('\u{b5}', '\u{b5}', SC_Lower), ('\u{ba}',
        '\u{ba}', SC_Lower), ('\u{bb}', '\u{bb}', SC_Close), ('\u{c0}', '\u{d6}', SC_Upper),
        ('\u{d8}', '\u{de}', SC_Upper), ('\u{df}', '\u{f6}', SC_Lower), ('\u{f8}', '\u{ff}',
        SC_Lower), ('\u{100}', '\u{100}', SC_Upper), ('\u{101}', '\u{101}', SC_Lower), ('\u{102}',
        '\u{102}', SC_Upper), ('\u{103}', '\u{103}', SC_Lower), ('\u{104}', '\u{104}', SC_Upper),
        ('\u{105}', '\u{105}', SC_Lower), ('\u{106}', '\u{106}', SC_Upper), ('\u{107}', '\u{107}',
        SC_Lower), ('\u{108}', '\u{108}', SC_Upper), ('\u{109}', '\u{109}', SC_Lower), ('\u{10a}',
        '\u{10a}', SC_Upper), ('\u{10b}', '\u{10b}', SC_Lower), ('\u{10c}', '\u{10c}', SC_Upper),
        ('\u{10d}', '\u{10d}', SC_Lower), ('\u{10e}', '\u{10e}', SC_Upper), ('\u{10f}', '\u{10f}',
        SC_Lower), ('\u{110}', '\u{110}', SC_Upper), ('\u{111}', '\u{111}', SC_Lower), ('\u{112}',
        '\u{112}', SC_Upper), ('\u{113}', '\u{113}', SC_Lower), ('\u{114}', '\u{114}', SC_Upper),
        ('\u{115}', '\u{115}', SC_Lower), ('\u{116}', '\u{116}', SC_Upper), ('\u{117}', '\u{117}',
        SC_Lower), ('\u{118}', '\u{118}', SC_Upper), ('\u{119}', '\u{119}', SC_Lower), ('\u{11a}',
        '\u{11a}', SC_Upper), ('\u{11b}', '\u{11b}', SC_Lower), ('\u{11c}', '\u{11c}', SC_Upper),
        ('\u{11d}', '\u{11d}', SC_Lower), ('\u{11e}', '\u{11e}', SC_Upper), ('\u{11f}', '\u{11f}',
        SC_Lower), ('\u{120}', '\u{120}', SC_Upper), ('\u{121}', '\u{121}', SC_Lower), ('\u{122}',
        '\u{122}', SC_Upper), ('\u{123}', '\u{123}', SC_Lower), ('\u{124}', '\u{124}', SC_Upper),
        ('\u{125}', '\u{125}', SC_Lower), ('\u{126}', '\u{126}', SC_Upper), ('\u{127}', '\u{127}',
        SC_Lower), ('\u{128}', '\u{128}', SC_Upper), ('\u{129}', '\u{129}', SC_Lower), ('\u{12a}',
        '\u{12a}', SC_Upper), ('\u{12b}', '\u{12b}', SC_Lower), ('\u{12c}', '\u{12c}', SC_Upper),
        ('\u{12d}', '\u{12d}', SC_Lower), ('\u{12e}', '\u{12e}', SC_Upper), ('\u{12f}', '\u{12f}',
        SC_Lower), ('\u{130}', '\u{130}', SC_Upper), ('\u{131}', '\u{131}', SC_Lower), ('\u{132}',
        '\u{132}', SC_Upper), ('\u{133}', '\u{133}', SC_Lower), ('\u{134}', '\u{134}', SC_Upper),
        ('\u{135}', '\u{135}', SC_Lower), ('\u{136}', '\u{136}', SC_Upper), ('\u{137}', '\u{138}',
        SC_Lower), ('\u{139}', '\u{139}', SC_Upper), ('\u{13a}', '\u{13a}', SC_Lower), ('\u{13b}',
        '\u{13b}', SC_Upper), ('\u{13c}', '\u{13c}', SC_Lower), ('\u{13d}', '\u{13d}', SC_Upper),
        ('\u{13e}', '\u{13e}', SC_Lower), ('\u{13f}', '\u{13f}', SC_Upper), ('\u{140}', '\u{140}',
        SC_Lower), ('\u{141}', '\u{141}', SC_Upper), ('\u{142}', '\u{142}', SC_Lower), ('\u{143}',
        '\u{143}', SC_Upper), ('\u{144}', '\u{144}', SC_Lower), ('\u{145}', '\u{145}', SC_Upper),
        ('\u{146}', '\u{146}', SC_Lower), ('\u{147}', '\u{147}', SC_Upper), ('\u{148}', '\u{149}',
        SC_Lower), ('\u{14a}', '\u{14a}', SC_Upper), ('\u{14b}', '\u{14b}', SC_Lower), ('\u{14c}',
        '\u{14c}', SC_Upper), ('\u{14d}', '\u{14d}', SC_Lower), ('\u{14e}', '\u{14e}', SC_Upper),
        ('\u{14f}', '\u{14f}', SC_Lower), ('\u{150}', '\u{150}', SC_Upper), ('\u{151}', '\u{151}',
        SC_Lower), ('\u{152}', '\u{152}', SC_Upper), ('\u{153}', '\u{153}', SC_Lower), ('\u{154}',
        '\u{154}', SC_Upper), ('\u{155}', '\u{155}', SC_Lower), ('\u{156}', '\u{156}', SC_Upper),
        ('\u{157}', '\u{157}', SC_Lower), ('\u{158}', '\u{158}', SC_Upper), ('\u{159}', '\u{159}',
        SC_Lower), ('\u{15a}', '\u{15a}', SC_Upper), ('\u{15b}', '\u{15b}', SC_Lower), ('\u{15c}',
        '\u{15c}', SC_Upper), ('\u{15d}', '\u{15d}', SC_Lower), ('\u{15e}', '\u{15e}', SC_Upper),
        ('\u{15f}', '\u{15f}', SC_Lower), ('\u{160}', '\u{160}', SC_Upper), ('\u{161}', '\u{161}',
        SC_Lower), ('\u{162}', '\u{162}', SC_Upper), ('\u{163}', '\u{163}', SC_Lower), ('\u{164}',
        '\u{164}', SC_Upper), ('\u{165}', '\u{165}', SC_Lower), ('\u{166}', '\u{166}', SC_Upper),
        ('\u{167}', '\u{167}', SC_Lower), ('\u{168}', '\u{168}', SC_Upper), ('\u{169}', '\u{169}',
        SC_Lower), ('\u{16a}', '\u{16a}', SC_Upper), ('\u{16b}', '\u{16b}', SC_Lower), ('\u{16c}',
        '\u{16c}', SC_Upper), ('\u{16d}', '\u{16d}', SC_Lower), ('\u{16e}', '\u{16e}', SC_Upper),
        ('\u{16f}', '\u{16f}', SC_Lower), ('\u{170}', '\u{170}', SC_Upper), ('\u{171}', '\u{171}',
        SC_Lower), ('\u{172}', '\u{172}', SC_Upper), ('\u{173}', '\u{173}', SC_Lower), ('\u{174}',
        '\u{174}', SC_Upper), ('\u{175}', '\u{175}', SC_Lower), ('\u{176}', '\u{176}', SC_Upper),
        ('\u{177}', '\u{177}', SC_Lower), ('\u{178}', '\u{179}', SC_Upper), ('\u{17a}', '\u{17a}',
        SC_Lower), ('\u{17b}', '\u{17b}', SC_Upper), ('\u{17c}', '\u{17c}', SC_Lower), ('\u{17d}',
        '\u{17d}', SC_Upper), ('\u{17e}', '\u{180}', SC_Lower), ('\u{181}', '\u{182}', SC_Upper),
        ('\u{183}', '\u{183}', SC_Lower), ('\u{184}', '\u{184}', SC_Upper), ('\u{185}', '\u{185}',
        SC_Lower), ('\u{186}', '\u{187}', SC_Upper), ('\u{188}', '\u{188}', SC_Lower), ('\u{189}',
        '\u{18b}', SC_Upper), ('\u{18c}', '\u{18d}', SC_Lower), ('\u{18e}', '\u{191}', SC_Upper),
        ('\u{192}', '\u{192}', SC_Lower), ('\u{193}', '\u{194}', SC_Upper), ('\u{195}', '\u{195}',
        SC_Lower), ('\u{196}', '\u{198}', SC_Upper), ('\u{199}', '\u{19b}', SC_Lower), ('\u{19c}',
        '\u{19d}', SC_Upper), ('\u{19e}', '\u{19e}', SC_Lower), ('\u{19f}', '\u{1a0}', SC_Upper),
        ('\u{1a1}', '\u{1a1}', SC_Lower), ('\u{1a2}', '\u{1a2}', SC_Upper), ('\u{1a3}', '\u{1a3}',
        SC_Lower), ('\u{1a4}', '\u{1a4}', SC_Upper), ('\u{1a5}', '\u{1a5}', SC_Lower), ('\u{1a6}',
        '\u{1a7}', SC_Upper), ('\u{1a8}', '\u{1a8}', SC_Lower), ('\u{1a9}', '\u{1a9}', SC_Upper),
        ('\u{1aa}', '\u{1ab}', SC_Lower), ('\u{1ac}', '\u{1ac}', SC_Upper), ('\u{1ad}', '\u{1ad}',
        SC_Lower), ('\u{1ae}', '\u{1af}', SC_Upper), ('\u{1b0}', '\u{1b0}', SC_Lower), ('\u{1b1}',
        '\u{1b3}', SC_Upper), ('\u{1b4}', '\u{1b4}', SC_Lower), ('\u{1b5}', '\u{1b5}', SC_Upper),
        ('\u{1b6}', '\u{1b6}', SC_Lower), ('\u{1b7}', '\u{1b8}', SC_Upper), ('\u{1b9}', '\u{1ba}',
        SC_Lower), ('\u{1bb}', '\u{1bb}', SC_OLetter), ('\u{1bc}', '\u{1bc}', SC_Upper), ('\u{1bd}',
        '\u{1bf}', SC_Lower), ('\u{1c0}', '\u{1c3}', SC_OLetter), ('\u{1c4}', '\u{1c5}', SC_Upper),
        ('\u{1c6}', '\u{1c6}', SC_Lower), ('\u{1c7}', '\u{1c8}', SC_Upper), ('\u{1c9}', '\u{1c9}',
        SC_Lower), ('\u{1ca}', '\u{1cb}', SC_Upper), ('\u{1cc}', '\u{1cc}', SC_Lower), ('\u{1cd}',
        '\u{1cd}', SC_Upper), ('\u{1ce}', '\u{1ce}', SC_Lower), ('\u{1cf}', '\u{1cf}', SC_Upper),
        ('\u{1d0}', '\u{1d0}', SC_Lower), ('\u{1d1}', '\u{1d1}', SC_Upper), ('\u{1d2}', '\u{1d2}',
        SC_Lower), ('\u{1d3}', '\u{1d3}', SC_Upper), ('\u{1d4}', '\u{1d4}', SC_Lower), ('\u{1d5}',
        '\u{1d5}', SC_Upper), ('\u{1d6}', '\u{1d6}', SC_Lower), ('\u{1d7}', '\u{1d7}', SC_Upper),
        ('\u{1d8}', '\u{1d8}', SC_Lower), ('\u{1d9}', '\u{1d9}', SC_Upper), ('\u{1da}', '\u{1da}',
        SC_Lower), ('\u{1db}', '\u{1db}', SC_Upper), ('\u{1dc}', '\u{1dd}', SC_Lower), ('\u{1de}',
        '\u{1de}', SC_Upper), ('\u{1df}', '\u{1df}', SC_Lower), ('\u{1e0}', '\u{1e0}', SC_Upper),
        ('\u{1e1}', '\u{1e1}', SC_Lower), ('\u{1e2}', '\u{1e2}', SC_Upper), ('\u{1e3}', '\u{1e3}',
        SC_Lower), ('\u{1e4}', '\u{1e4}', SC_Upper), ('\u{1e5}', '\u{1e5}', SC_Lower), ('\u{1e6}',
        '\u{1e6}', SC_Upper), ('\u{1e7}', '\u{1e7}', SC_Lower), ('\u{1e8}', '\u{1e8}', SC_Upper),
        ('\u{1e9}', '\u{1e9}', SC_Lower), ('\u{1ea}', '\u{1ea}', SC_Upper), ('\u{1eb}', '\u{1eb}',
        SC_Lower), ('\u{1ec}', '\u{1ec}', SC_Upper), ('\u{1ed}', '\u{1ed}', SC_Lower), ('\u{1ee}',
        '\u{1ee}', SC_Upper), ('\u{1ef}', '\u{1f0}', SC_Lower), ('\u{1f1}', '\u{1f2}', SC_Upper),
        ('\u{1f3}', '\u{1f3}', SC_Lower), ('\u{1f4}', '\u{1f4}', SC_Upper), ('\u{1f5}', '\u{1f5}',
        SC_Lower), ('\u{1f6}', '\u{1f8}', SC_Upper), ('\u{1f9}', '\u{1f9}', SC_Lower), ('\u{1fa}',
        '\u{1fa}', SC_Upper), ('\u{1fb}', '\u{1fb}', SC_Lower), ('\u{1fc}', '\u{1fc}', SC_Upper),
        ('\u{1fd}', '\u{1fd}', SC_Lower), ('\u{1fe}', '\u{1fe}', SC_Upper), ('\u{1ff}', '\u{1ff}',
        SC_Lower), ('\u{200}', '\u{200}', SC_Upper), ('\u{201}', '\u{201}', SC_Lower), ('\u{202}',
        '\u{202}', SC_Upper), ('\u{203}', '\u{203}', SC_Lower), ('\u{204}', '\u{204}', SC_Upper),
        ('\u{205}', '\u{205}', SC_Lower), ('\u{206}', '\u{206}', SC_Upper), ('\u{207}', '\u{207}',
        SC_Lower), ('\u{208}', '\u{208}', SC_Upper), ('\u{209}', '\u{209}', SC_Lower), ('\u{20a}',
        '\u{20a}', SC_Upper), ('\u{20b}', '\u{20b}', SC_Lower), ('\u{20c}', '\u{20c}', SC_Upper),
        ('\u{20d}', '\u{20d}', SC_Lower), ('\u{20e}', '\u{20e}', SC_Upper), ('\u{20f}', '\u{20f}',
        SC_Lower), ('\u{210}', '\u{210}', SC_Upper), ('\u{211}', '\u{211}', SC_Lower), ('\u{212}',
        '\u{212}', SC_Upper), ('\u{213}', '\u{213}', SC_Lower), ('\u{214}', '\u{214}', SC_Upper),
        ('\u{215}', '\u{215}', SC_Lower), ('\u{216}', '\u{216}', SC_Upper), ('\u{217}', '\u{217}',
        SC_Lower), ('\u{218}', '\u{218}', SC_Upper), ('\u{219}', '\u{219}', SC_Lower), ('\u{21a}',
        '\u{21a}', SC_Upper), ('\u{21b}', '\u{21b}', SC_Lower), ('\u{21c}', '\u{21c}', SC_Upper),
        ('\u{21d}', '\u{21d}', SC_Lower), ('\u{21e}', '\u{21e}', SC_Upper), ('\u{21f}', '\u{21f}',
        SC_Lower), ('\u{220}', '\u{220}', SC_Upper), ('\u{221}', '\u{221}', SC_Lower), ('\u{222}',
        '\u{222}', SC_Upper), ('\u{223}', '\u{223}', SC_Lower), ('\u{224}', '\u{224}', SC_Upper),
        ('\u{225}', '\u{225}', SC_Lower), ('\u{226}', '\u{226}', SC_Upper), ('\u{227}', '\u{227}',
        SC_Lower), ('\u{228}', '\u{228}', SC_Upper), ('\u{229}', '\u{229}', SC_Lower), ('\u{22a}',
        '\u{22a}', SC_Upper), ('\u{22b}', '\u{22b}', SC_Lower), ('\u{22c}', '\u{22c}', SC_Upper),
        ('\u{22d}', '\u{22d}', SC_Lower), ('\u{22e}', '\u{22e}', SC_Upper), ('\u{22f}', '\u{22f}',
        SC_Lower), ('\u{230}', '\u{230}', SC_Upper), ('\u{231}', '\u{231}', SC_Lower), ('\u{232}',
        '\u{232}', SC_Upper), ('\u{233}', '\u{239}', SC_Lower), ('\u{23a}', '\u{23b}', SC_Upper),
        ('\u{23c}', '\u{23c}', SC_Lower), ('\u{23d}', '\u{23e}', SC_Upper), ('\u{23f}', '\u{240}',
        SC_Lower), ('\u{241}', '\u{241}', SC_Upper), ('\u{242}', '\u{242}', SC_Lower), ('\u{243}',
        '\u{246}', SC_Upper), ('\u{247}', '\u{247}', SC_Lower), ('\u{248}', '\u{248}', SC_Upper),
        ('\u{249}', '\u{249}', SC_Lower), ('\u{24a}', '\u{24a}', SC_Upper), ('\u{24b}', '\u{24b}',
        SC_Lower), ('\u{24c}', '\u{24c}', SC_Upper), ('\u{24d}', '\u{24d}', SC_Lower), ('\u{24e}',
        '\u{24e}', SC_Upper), ('\u{24f}', '\u{293}', SC_Lower), ('\u{294}', '\u{294}', SC_OLetter),
        ('\u{295}', '\u{2b8}', SC_Lower), ('\u{2b9}', '\u{2bf}', SC_OLetter), ('\u{2c0}', '\u{2c1}',
        SC_Lower), ('\u{2c6}', '\u{2d1}', SC_OLetter), ('\u{2e0}', '\u{2e4}', SC_Lower), ('\u{2ec}',
        '\u{2ec}', SC_OLetter), ('\u{2ee}', '\u{2ee}', SC_OLetter), ('\u{300}', '\u{36f}',
        SC_Extend), ('\u{370}', '\u{370}', SC_Upper), ('\u{371}', '\u{371}', SC_Lower), ('\u{372}',
        '\u{372}', SC_Upper), ('\u{373}', '\u{373}', SC_Lower), ('\u{374}', '\u{374}', SC_OLetter),
        ('\u{376}', '\u{376}', SC_Upper), ('\u{377}', '\u{377}', SC_Lower), ('\u{37a}', '\u{37d}',
        SC_Lower), ('\u{37f}', '\u{37f}', SC_Upper), ('\u{386}', '\u{386}', SC_Upper), ('\u{388}',
        '\u{38a}', SC_Upper), ('\u{38c}', '\u{38c}', SC_Upper), ('\u{38e}', '\u{38f}', SC_Upper),
        ('\u{390}', '\u{390}', SC_Lower), ('\u{391}', '\u{3a1}', SC_Upper), ('\u{3a3}', '\u{3ab}',
        SC_Upper), ('\u{3ac}', '\u{3ce}', SC_Lower), ('\u{3cf}', '\u{3cf}', SC_Upper), ('\u{3d0}',
        '\u{3d1}', SC_Lower), ('\u{3d2}', '\u{3d4}', SC_Upper), ('\u{3d5}', '\u{3d7}', SC_Lower),
        ('\u{3d8}', '\u{3d8}', SC_Upper), ('\u{3d9}', '\u{3d9}', SC_Lower), ('\u{3da}', '\u{3da}',
        SC_Upper), ('\u{3db}', '\u{3db}', SC_Lower), ('\u{3dc}', '\u{3dc}', SC_Upper), ('\u{3dd}',
        '\u{3dd}', SC_Lower), ('\u{3de}', '\u{3de}', SC_Upper), ('\u{3df}', '\u{3df}', SC_Lower),
        ('\u{3e0}', '\u{3e0}', SC_Upper), ('\u{3e1}', '\u{3e1}', SC_Lower), ('\u{3e2}', '\u{3e2}',
        SC_Upper), ('\u{3e3}', '\u{3e3}', SC_Lower), ('\u{3e4}', '\u{3e4}', SC_Upper), ('\u{3e5}',
        '\u{3e5}', SC_Lower), ('\u{3e6}', '\u{3e6}', SC_Upper), ('\u{3e7}', '\u{3e7}', SC_Lower),
        ('\u{3e8}', '\u{3e8}', SC_Upper), ('\u{3e9}', '\u{3e9}', SC_Lower), ('\u{3ea}', '\u{3ea}',
        SC_Upper), ('\u{3eb}', '\u{3eb}', SC_Lower), ('\u{3ec}', '\u{3ec}', SC_Upper), ('\u{3ed}',
        '\u{3ed}', SC_Lower), ('\u{3ee}', '\u{3ee}', SC_Upper), ('\u{3ef}', '\u{3f3}', SC_Lower),
        ('\u{3f4}', '\u{3f4}', SC_Upper), ('\u{3f5}', '\u{3f5}', SC_Lower), ('\u{3f7}', '\u{3f7}',
        SC_Upper), ('\u{3f8}', '\u{3f8}', SC_Lower), ('\u{3f9}', '\u{3fa}', SC_Upper), ('\u{3fb}',
        '\u{3fc}', SC_Lower), ('\u{3fd}', '\u{42f}', SC_Upper), ('\u{430}', '\u{45f}', SC_Lower),
        ('\u{460}', '\u{460}', SC_Upper), ('\u{461}', '\u{461}', SC_Lower), ('\u{462}', '\u{462}',
        SC_Upper), ('\u{463}', '\u{463}', SC_Lower), ('\u{464}', '\u{464}', SC_Upper), ('\u{465}',
        '\u{465}', SC_Lower), ('\u{466}', '\u{466}', SC_Upper), ('\u{467}', '\u{467}', SC_Lower),
        ('\u{468}', '\u{468}', SC_Upper), ('\u{469}', '\u{469}', SC_Lower), ('\u{46a}', '\u{46a}',
        SC_Upper), ('\u{46b}', '\u{46b}', SC_Lower), ('\u{46c}', '\u{46c}', SC_Upper), ('\u{46d}',
        '\u{46d}', SC_Lower), ('\u{46e}', '\u{46e}', SC_Upper), ('\u{46f}', '\u{46f}', SC_Lower),
        ('\u{470}', '\u{470}', SC_Upper), ('\u{471}', '\u{471}', SC_Lower), ('\u{472}', '\u{472}',
        SC_Upper), ('\u{473}', '\u{473}', SC_Lower), ('\u{474}', '\u{474}', SC_Upper), ('\u{475}',
        '\u{475}', SC_Lower), ('\u{476}', '\u{476}', SC_Upper), ('\u{477}', '\u{477}', SC_Lower),
        ('\u{478}', '\u{478}', SC_Upper), ('\u{479}', '\u{479}', SC_Lower), ('\u{47a}', '\u{47a}',
        SC_Upper), ('\u{47b}', '\u{47b}', SC_Lower), ('\u{47c}', '\u{47c}', SC_Upper), ('\u{47d}',
        '\u{47d}', SC_Lower), ('\u{47e}', '\u{47e}', SC_Upper), ('\u{47f}', '\u{47f}', SC_Lower),
        ('\u{480}', '\u{480}', SC_Upper), ('\u{481}', '\u{481}', SC_Lower), ('\u{483}', '\u{489}',
        SC_Extend), ('\u{48a}', '\u{48a}', SC_Upper), ('\u{48b}', '\u{48b}', SC_Lower), ('\u{48c}',
        '\u{48c}', SC_Upper), ('\u{48d}', '\u{48d}', SC_Lower), ('\u{48e}', '\u{48e}', SC_Upper),
        ('\u{48f}', '\u{48f}', SC_Lower), ('\u{490}', '\u{490}', SC_Upper), ('\u{491}', '\u{491}',
        SC_Lower), ('\u{492}', '\u{492}', SC_Upper), ('\u{493}', '\u{493}', SC_Lower), ('\u{494}',
        '\u{494}', SC_Upper), ('\u{495}', '\u{495}', SC_Lower), ('\u{496}', '\u{496}', SC_Upper),
        ('\u{497}', '\u{497}', SC_Lower), ('\u{498}', '\u{498}', SC_Upper), ('\u{499}', '\u{499}',
        SC_Lower), ('\u{49a}', '\u{49a}', SC_Upper), ('\u{49b}', '\u{49b}', SC_Lower), ('\u{49c}',
        '\u{49c}', SC_Upper), ('\u{49d}', '\u{49d}', SC_Lower), ('\u{49e}', '\u{49e}', SC_Upper),
        ('\u{49f}', '\u{49f}', SC_Lower), ('\u{4a0}', '\u{4a0}', SC_Upper), ('\u{4a1}', '\u{4a1}',
        SC_Lower), ('\u{4a2}', '\u{4a2}', SC_Upper), ('\u{4a3}', '\u{4a3}', SC_Lower), ('\u{4a4}',
        '\u{4a4}', SC_Upper), ('\u{4a5}', '\u{4a5}', SC_Lower), ('\u{4a6}', '\u{4a6}', SC_Upper),
        ('\u{4a7}', '\u{4a7}', SC_Lower), ('\u{4a8}', '\u{4a8}', SC_Upper), ('\u{4a9}', '\u{4a9}',
        SC_Lower), ('\u{4aa}', '\u{4aa}', SC_Upper), ('\u{4ab}', '\u{4ab}', SC_Lower), ('\u{4ac}',
        '\u{4ac}', SC_Upper), ('\u{4ad}', '\u{4ad}', SC_Lower), ('\u{4ae}', '\u{4ae}', SC_Upper),
        ('\u{4af}', '\u{4af}', SC_Lower), ('\u{4b0}', '\u{4b0}', SC_Upper), ('\u{4b1}', '\u{4b1}',
        SC_Lower), ('\u{4b2}', '\u{4b2}', SC_Upper), ('\u{4b3}', '\u{4b3}', SC_Lower), ('\u{4b4}',
        '\u{4b4}', SC_Upper), ('\u{4b5}', '\u{4b5}', SC_Lower), ('\u{4b6}', '\u{4b6}', SC_Upper),
        ('\u{4b7}', '\u{4b7}', SC_Lower), ('\u{4b8}', '\u{4b8}', SC_Upper), ('\u{4b9}', '\u{4b9}',
        SC_Lower), ('\u{4ba}', '\u{4ba}', SC_Upper), ('\u{4bb}', '\u{4bb}', SC_Lower), ('\u{4bc}',
        '\u{4bc}', SC_Upper), ('\u{4bd}', '\u{4bd}', SC_Lower), ('\u{4be}', '\u{4be}', SC_Upper),
        ('\u{4bf}', '\u{4bf}', SC_Lower), ('\u{4c0}', '\u{4c1}', SC_Upper), ('\u{4c2}', '\u{4c2}',
        SC_Lower), ('\u{4c3}', '\u{4c3}', SC_Upper), ('\u{4c4}', '\u{4c4}', SC_Lower), ('\u{4c5}',
        '\u{4c5}', SC_Upper), ('\u{4c6}', '\u{4c6}', SC_Lower), ('\u{4c7}', '\u{4c7}', SC_Upper),
        ('\u{4c8}', '\u{4c8}', SC_Lower), ('\u{4c9}', '\u{4c9}', SC_Upper), ('\u{4ca}', '\u{4ca}',
        SC_Lower), ('\u{4cb}', '\u{4cb}', SC_Upper), ('\u{4cc}', '\u{4cc}', SC_Lower), ('\u{4cd}',
        '\u{4cd}', SC_Upper), ('\u{4ce}', '\u{4cf}', SC_Lower), ('\u{4d0}', '\u{4d0}', SC_Upper),
        ('\u{4d1}', '\u{4d1}', SC_Lower), ('\u{4d2}', '\u{4d2}', SC_Upper), ('\u{4d3}', '\u{4d3}',
        SC_Lower), ('\u{4d4}', '\u{4d4}', SC_Upper), ('\u{4d5}', '\u{4d5}', SC_Lower), ('\u{4d6}',
        '\u{4d6}', SC_Upper), ('\u{4d7}', '\u{4d7}', SC_Lower), ('\u{4d8}', '\u{4d8}', SC_Upper),
        ('\u{4d9}', '\u{4d9}', SC_Lower), ('\u{4da}', '\u{4da}', SC_Upper), ('\u{4db}', '\u{4db}',
        SC_Lower), ('\u{4dc}', '\u{4dc}', SC_Upper), ('\u{4dd}', '\u{4dd}', SC_Lower), ('\u{4de}',
        '\u{4de}', SC_Upper), ('\u{4df}', '\u{4df}', SC_Lower), ('\u{4e0}', '\u{4e0}', SC_Upper),
        ('\u{4e1}', '\u{4e1}', SC_Lower), ('\u{4e2}', '\u{4e2}', SC_Upper), ('\u{4e3}', '\u{4e3}',
        SC_Lower), ('\u{4e4}', '\u{4e4}', SC_Upper), ('\u{4e5}', '\u{4e5}', SC_Lower), ('\u{4e6}',
        '\u{4e6}', SC_Upper), ('\u{4e7}', '\u{4e7}', SC_Lower), ('\u{4e8}', '\u{4e8}', SC_Upper),
        ('\u{4e9}', '\u{4e9}', SC_Lower), ('\u{4ea}', '\u{4ea}', SC_Upper), ('\u{4eb}', '\u{4eb}',
        SC_Lower), ('\u{4ec}', '\u{4ec}', SC_Upper), ('\u{4ed}', '\u{4ed}', SC_Lower), ('\u{4ee}',
        '\u{4ee}', SC_Upper), ('\u{4ef}', '\u{4ef}', SC_Lower), ('\u{4f0}', '\u{4f0}', SC_Upper),
        ('\u{4f1}', '\u{4f1}', SC_Lower), ('\u{4f2}', '\u{4f2}', SC_Upper), ('\u{4f3}', '\u{4f3}',
        SC_Lower), ('\u{4f4}', '\u{4f4}', SC_Upper), ('\u{4f5}', '\u{4f5}', SC_Lower), ('\u{4f6}',
        '\u{4f6}', SC_Upper), ('\u{4f7}', '\u{4f7}', SC_Lower), ('\u{4f8}', '\u{4f8}', SC_Upper),
        ('\u{4f9}', '\u{4f9}', SC_Lower), ('\u{4fa}', '\u{4fa}', SC_Upper), ('\u{4fb}', '\u{4fb}',
        SC_Lower), ('\u{4fc}', '\u{4fc}', SC_Upper), ('\u{4fd}', '\u{4fd}', SC_Lower), ('\u{4fe}',
        '\u{4fe}', SC_Upper), ('\u{4ff}', '\u{4ff}', SC_Lower), ('\u{500}', '\u{500}', SC_Upper),
        ('\u{501}', '\u{501}', SC_Lower), ('\u{502}', '\u{502}', SC_Upper), ('\u{503}', '\u{503}',
        SC_Lower), ('\u{504}', '\u{504}', SC_Upper), ('\u{505}', '\u{505}', SC_Lower), ('\u{506}',
        '\u{506}', SC_Upper), ('\u{507}', '\u{507}', SC_Lower), ('\u{508}', '\u{508}', SC_Upper),
        ('\u{509}', '\u{509}', SC_Lower), ('\u{50a}', '\u{50a}', SC_Upper), ('\u{50b}', '\u{50b}',
        SC_Lower), ('\u{50c}', '\u{50c}', SC_Upper), ('\u{50d}', '\u{50d}', SC_Lower), ('\u{50e}',
        '\u{50e}', SC_Upper), ('\u{50f}', '\u{50f}', SC_Lower), ('\u{510}', '\u{510}', SC_Upper),
        ('\u{511}', '\u{511}', SC_Lower), ('\u{512}', '\u{512}', SC_Upper), ('\u{513}', '\u{513}',
        SC_Lower), ('\u{514}', '\u{514}', SC_Upper), ('\u{515}', '\u{515}', SC_Lower), ('\u{516}',
        '\u{516}', SC_Upper), ('\u{517}', '\u{517}', SC_Lower), ('\u{518}', '\u{518}', SC_Upper),
        ('\u{519}', '\u{519}', SC_Lower), ('\u{51a}', '\u{51a}', SC_Upper), ('\u{51b}', '\u{51b}',
        SC_Lower), ('\u{51c}', '\u{51c}', SC_Upper), ('\u{51d}', '\u{51d}', SC_Lower), ('\u{51e}',
        '\u{51e}', SC_Upper), ('\u{51f}', '\u{51f}', SC_Lower), ('\u{520}', '\u{520}', SC_Upper),
        ('\u{521}', '\u{521}', SC_Lower), ('\u{522}', '\u{522}', SC_Upper), ('\u{523}', '\u{523}',
        SC_Lower), ('\u{524}', '\u{524}', SC_Upper), ('\u{525}', '\u{525}', SC_Lower), ('\u{526}',
        '\u{526}', SC_Upper), ('\u{527}', '\u{527}', SC_Lower), ('\u{528}', '\u{528}', SC_Upper),
        ('\u{529}', '\u{529}', SC_Lower), ('\u{52a}', '\u{52a}', SC_Upper), ('\u{52b}', '\u{52b}',
        SC_Lower), ('\u{52c}', '\u{52c}', SC_Upper), ('\u{52d}', '\u{52d}', SC_Lower), ('\u{52e}',
        '\u{52e}', SC_Upper), ('\u{52f}', '\u{52f}', SC_Lower), ('\u{531}', '\u{556}', SC_Upper),
        ('\u{559}', '\u{559}', SC_OLetter), ('\u{55d}', '\u{55d}', SC_SContinue), ('\u{561}',
        '\u{587}', SC_Lower), ('\u{589}', '\u{589}', SC_STerm), ('\u{591}', '\u{5bd}', SC_Extend),
        ('\u{5bf}', '\u{5bf}', SC_Extend), ('\u{5c1}', '\u{5c2}', SC_Extend), ('\u{5c4}', '\u{5c5}',
        SC_Extend), ('\u{5c7}', '\u{5c7}', SC_Extend), ('\u{5d0}', '\u{5ea}', SC_OLetter),
        ('\u{5f0}', '\u{5f3}', SC_OLetter), ('\u{600}', '\u{605}', SC_Format), ('\u{60c}',
        '\u{60d}', SC_SContinue), ('\u{610}', '\u{61a}', SC_Extend), ('\u{61c}', '\u{61c}',
        SC_Format), ('\u{61f}', '\u{61f}', SC_STerm), ('\u{620}', '\u{64a}', SC_OLetter),
        ('\u{64b}', '\u{65f}', SC_Extend), ('\u{660}', '\u{669}', SC_Numeric), ('\u{66b}',
        '\u{66c}', SC_Numeric), ('\u{66e}', '\u{66f}', SC_OLetter), ('\u{670}', '\u{670}',
        SC_Extend), ('\u{671}', '\u{6d3}', SC_OLetter), ('\u{6d4}', '\u{6d4}', SC_STerm),
        ('\u{6d5}', '\u{6d5}', SC_OLetter), ('\u{6d6}', '\u{6dc}', SC_Extend), ('\u{6dd}',
        '\u{6dd}', SC_Format), ('\u{6df}', '\u{6e4}', SC_Extend), ('\u{6e5}', '\u{6e6}',
        SC_OLetter), ('\u{6e7}', '\u{6e8}', SC_Extend), ('\u{6ea}', '\u{6ed}', SC_Extend),
        ('\u{6ee}', '\u{6ef}', SC_OLetter), ('\u{6f0}', '\u{6f9}', SC_Numeric), ('\u{6fa}',
        '\u{6fc}', SC_OLetter), ('\u{6ff}', '\u{6ff}', SC_OLetter), ('\u{700}', '\u{702}',
        SC_STerm), ('\u{70f}', '\u{70f}', SC_Format), ('\u{710}', '\u{710}', SC_OLetter),
        ('\u{711}', '\u{711}', SC_Extend), ('\u{712}', '\u{72f}', SC_OLetter), ('\u{730}',
        '\u{74a}', SC_Extend), ('\u{74d}', '\u{7a5}', SC_OLetter), ('\u{7a6}', '\u{7b0}',
        SC_Extend), ('\u{7b1}', '\u{7b1}', SC_OLetter), ('\u{7c0}', '\u{7c9}', SC_Numeric),
        ('\u{7ca}', '\u{7ea}', SC_OLetter), ('\u{7eb}', '\u{7f3}', SC_Extend), ('\u{7f4}',
        '\u{7f5}', SC_OLetter), ('\u{7f8}', '\u{7f8}', SC_SContinue), ('\u{7f9}', '\u{7f9}',
        SC_STerm), ('\u{7fa}', '\u{7fa}', SC_OLetter), ('\u{800}', '\u{815}', SC_OLetter),
        ('\u{816}', '\u{819}', SC_Extend), ('\u{81a}', '\u{81a}', SC_OLetter), ('\u{81b}',
        '\u{823}', SC_Extend), ('\u{824}', '\u{824}', SC_OLetter), ('\u{825}', '\u{827}',
        SC_Extend), ('\u{828}', '\u{828}', SC_OLetter), ('\u{829}', '\u{82d}', SC_Extend),
        ('\u{840}', '\u{858}', SC_OLetter), ('\u{859}', '\u{85b}', SC_Extend), ('\u{8a0}',
        '\u{8b4}', SC_OLetter), ('\u{8b6}', '\u{8bd}', SC_OLetter), ('\u{8d4}', '\u{8e1}',
        SC_Extend), ('\u{8e2}', '\u{8e2}', SC_Format), ('\u{8e3}', '\u{903}', SC_Extend),
        ('\u{904}', '\u{939}', SC_OLetter), ('\u{93a}', '\u{93c}', SC_Extend), ('\u{93d}',
        '\u{93d}', SC_OLetter), ('\u{93e}', '\u{94f}', SC_Extend), ('\u{950}', '\u{950}',
        SC_OLetter), ('\u{951}', '\u{957}', SC_Extend), ('\u{958}', '\u{961}', SC_OLetter),
        ('\u{962}', '\u{963}', SC_Extend), ('\u{964}', '\u{965}', SC_STerm), ('\u{966}', '\u{96f}',
        SC_Numeric), ('\u{971}', '\u{980}', SC_OLetter), ('\u{981}', '\u{983}', SC_Extend),
        ('\u{985}', '\u{98c}', SC_OLetter), ('\u{98f}', '\u{990}', SC_OLetter), ('\u{993}',
        '\u{9a8}', SC_OLetter), ('\u{9aa}', '\u{9b0}', SC_OLetter), ('\u{9b2}', '\u{9b2}',
        SC_OLetter), ('\u{9b6}', '\u{9b9}', SC_OLetter), ('\u{9bc}', '\u{9bc}', SC_Extend),
        ('\u{9bd}', '\u{9bd}', SC_OLetter), ('\u{9be}', '\u{9c4}', SC_Extend), ('\u{9c7}',
        '\u{9c8}', SC_Extend), ('\u{9cb}', '\u{9cd}', SC_Extend), ('\u{9ce}', '\u{9ce}',
        SC_OLetter), ('\u{9d7}', '\u{9d7}', SC_Extend), ('\u{9dc}', '\u{9dd}', SC_OLetter),
        ('\u{9df}', '\u{9e1}', SC_OLetter), ('\u{9e2}', '\u{9e3}', SC_Extend), ('\u{9e6}',
        '\u{9ef}', SC_Numeric), ('\u{9f0}', '\u{9f1}', SC_OLetter), ('\u{a01}', '\u{a03}',
        SC_Extend), ('\u{a05}', '\u{a0a}', SC_OLetter), ('\u{a0f}', '\u{a10}', SC_OLetter),
        ('\u{a13}', '\u{a28}', SC_OLetter), ('\u{a2a}', '\u{a30}', SC_OLetter), ('\u{a32}',
        '\u{a33}', SC_OLetter), ('\u{a35}', '\u{a36}', SC_OLetter), ('\u{a38}', '\u{a39}',
        SC_OLetter), ('\u{a3c}', '\u{a3c}', SC_Extend), ('\u{a3e}', '\u{a42}', SC_Extend),
        ('\u{a47}', '\u{a48}', SC_Extend), ('\u{a4b}', '\u{a4d}', SC_Extend), ('\u{a51}', '\u{a51}',
        SC_Extend), ('\u{a59}', '\u{a5c}', SC_OLetter), ('\u{a5e}', '\u{a5e}', SC_OLetter),
        ('\u{a66}', '\u{a6f}', SC_Numeric), ('\u{a70}', '\u{a71}', SC_Extend), ('\u{a72}',
        '\u{a74}', SC_OLetter), ('\u{a75}', '\u{a75}', SC_Extend), ('\u{a81}', '\u{a83}',
        SC_Extend), ('\u{a85}', '\u{a8d}', SC_OLetter), ('\u{a8f}', '\u{a91}', SC_OLetter),
        ('\u{a93}', '\u{aa8}', SC_OLetter), ('\u{aaa}', '\u{ab0}', SC_OLetter), ('\u{ab2}',
        '\u{ab3}', SC_OLetter), ('\u{ab5}', '\u{ab9}', SC_OLetter), ('\u{abc}', '\u{abc}',
        SC_Extend), ('\u{abd}', '\u{abd}', SC_OLetter), ('\u{abe}', '\u{ac5}', SC_Extend),
        ('\u{ac7}', '\u{ac9}', SC_Extend), ('\u{acb}', '\u{acd}', SC_Extend), ('\u{ad0}', '\u{ad0}',
        SC_OLetter), ('\u{ae0}', '\u{ae1}', SC_OLetter), ('\u{ae2}', '\u{ae3}', SC_Extend),
        ('\u{ae6}', '\u{aef}', SC_Numeric), ('\u{af9}', '\u{af9}', SC_OLetter), ('\u{b01}',
        '\u{b03}', SC_Extend), ('\u{b05}', '\u{b0c}', SC_OLetter), ('\u{b0f}', '\u{b10}',
        SC_OLetter), ('\u{b13}', '\u{b28}', SC_OLetter), ('\u{b2a}', '\u{b30}', SC_OLetter),
        ('\u{b32}', '\u{b33}', SC_OLetter), ('\u{b35}', '\u{b39}', SC_OLetter), ('\u{b3c}',
        '\u{b3c}', SC_Extend), ('\u{b3d}', '\u{b3d}', SC_OLetter), ('\u{b3e}', '\u{b44}',
        SC_Extend), ('\u{b47}', '\u{b48}', SC_Extend), ('\u{b4b}', '\u{b4d}', SC_Extend),
        ('\u{b56}', '\u{b57}', SC_Extend), ('\u{b5c}', '\u{b5d}', SC_OLetter), ('\u{b5f}',
        '\u{b61}', SC_OLetter), ('\u{b62}', '\u{b63}', SC_Extend), ('\u{b66}', '\u{b6f}',
        SC_Numeric), ('\u{b71}', '\u{b71}', SC_OLetter), ('\u{b82}', '\u{b82}', SC_Extend),
        ('\u{b83}', '\u{b83}', SC_OLetter), ('\u{b85}', '\u{b8a}', SC_OLetter), ('\u{b8e}',
        '\u{b90}', SC_OLetter), ('\u{b92}', '\u{b95}', SC_OLetter), ('\u{b99}', '\u{b9a}',
        SC_OLetter), ('\u{b9c}', '\u{b9c}', SC_OLetter), ('\u{b9e}', '\u{b9f}', SC_OLetter),
        ('\u{ba3}', '\u{ba4}', SC_OLetter), ('\u{ba8}', '\u{baa}', SC_OLetter), ('\u{bae}',
        '\u{bb9}', SC_OLetter), ('\u{bbe}', '\u{bc2}', SC_Extend), ('\u{bc6}', '\u{bc8}',
        SC_Extend), ('\u{bca}', '\u{bcd}', SC_Extend), ('\u{bd0}', '\u{bd0}', SC_OLetter),
        ('\u{bd7}', '\u{bd7}', SC_Extend), ('\u{be6}', '\u{bef}', SC_Numeric), ('\u{c00}',
        '\u{c03}', SC_Extend), ('\u{c05}', '\u{c0c}', SC_OLetter), ('\u{c0e}', '\u{c10}',
        SC_OLetter), ('\u{c12}', '\u{c28}', SC_OLetter), ('\u{c2a}', '\u{c39}', SC_OLetter),
        ('\u{c3d}', '\u{c3d}', SC_OLetter), ('\u{c3e}', '\u{c44}', SC_Extend), ('\u{c46}',
        '\u{c48}', SC_Extend), ('\u{c4a}', '\u{c4d}', SC_Extend), ('\u{c55}', '\u{c56}', SC_Extend),
        ('\u{c58}', '\u{c5a}', SC_OLetter), ('\u{c60}', '\u{c61}', SC_OLetter), ('\u{c62}',
        '\u{c63}', SC_Extend), ('\u{c66}', '\u{c6f}', SC_Numeric), ('\u{c80}', '\u{c80}',
        SC_OLetter), ('\u{c81}', '\u{c83}', SC_Extend), ('\u{c85}', '\u{c8c}', SC_OLetter),
        ('\u{c8e}', '\u{c90}', SC_OLetter), ('\u{c92}', '\u{ca8}', SC_OLetter), ('\u{caa}',
        '\u{cb3}', SC_OLetter), ('\u{cb5}', '\u{cb9}', SC_OLetter), ('\u{cbc}', '\u{cbc}',
        SC_Extend), ('\u{cbd}', '\u{cbd}', SC_OLetter), ('\u{cbe}', '\u{cc4}', SC_Extend),
        ('\u{cc6}', '\u{cc8}', SC_Extend), ('\u{cca}', '\u{ccd}', SC_Extend), ('\u{cd5}', '\u{cd6}',
        SC_Extend), ('\u{cde}', '\u{cde}', SC_OLetter), ('\u{ce0}', '\u{ce1}', SC_OLetter),
        ('\u{ce2}', '\u{ce3}', SC_Extend), ('\u{ce6}', '\u{cef}', SC_Numeric), ('\u{cf1}',
        '\u{cf2}', SC_OLetter), ('\u{d01}', '\u{d03}', SC_Extend), ('\u{d05}', '\u{d0c}',
        SC_OLetter), ('\u{d0e}', '\u{d10}', SC_OLetter), ('\u{d12}', '\u{d3a}', SC_OLetter),
        ('\u{d3d}', '\u{d3d}', SC_OLetter), ('\u{d3e}', '\u{d44}', SC_Extend), ('\u{d46}',
        '\u{d48}', SC_Extend), ('\u{d4a}', '\u{d4d}', SC_Extend), ('\u{d4e}', '\u{d4e}',
        SC_OLetter), ('\u{d54}', '\u{d56}', SC_OLetter), ('\u{d57}', '\u{d57}', SC_Extend),
        ('\u{d5f}', '\u{d61}', SC_OLetter), ('\u{d62}', '\u{d63}', SC_Extend), ('\u{d66}',
        '\u{d6f}', SC_Numeric), ('\u{d7a}', '\u{d7f}', SC_OLetter), ('\u{d82}', '\u{d83}',
        SC_Extend), ('\u{d85}', '\u{d96}', SC_OLetter), ('\u{d9a}', '\u{db1}', SC_OLetter),
        ('\u{db3}', '\u{dbb}', SC_OLetter), ('\u{dbd}', '\u{dbd}', SC_OLetter), ('\u{dc0}',
        '\u{dc6}', SC_OLetter), ('\u{dca}', '\u{dca}', SC_Extend), ('\u{dcf}', '\u{dd4}',
        SC_Extend), ('\u{dd6}', '\u{dd6}', SC_Extend), ('\u{dd8}', '\u{ddf}', SC_Extend),
        ('\u{de6}', '\u{def}', SC_Numeric), ('\u{df2}', '\u{df3}', SC_Extend), ('\u{e01}',
        '\u{e30}', SC_OLetter), ('\u{e31}', '\u{e31}', SC_Extend), ('\u{e32}', '\u{e33}',
        SC_OLetter), ('\u{e34}', '\u{e3a}', SC_Extend), ('\u{e40}', '\u{e46}', SC_OLetter),
        ('\u{e47}', '\u{e4e}', SC_Extend), ('\u{e50}', '\u{e59}', SC_Numeric), ('\u{e81}',
        '\u{e82}', SC_OLetter), ('\u{e84}', '\u{e84}', SC_OLetter), ('\u{e87}', '\u{e88}',
        SC_OLetter), ('\u{e8a}', '\u{e8a}', SC_OLetter), ('\u{e8d}', '\u{e8d}', SC_OLetter),
        ('\u{e94}', '\u{e97}', SC_OLetter), ('\u{e99}', '\u{e9f}', SC_OLetter), ('\u{ea1}',
        '\u{ea3}', SC_OLetter), ('\u{ea5}', '\u{ea5}', SC_OLetter), ('\u{ea7}', '\u{ea7}',
        SC_OLetter), ('\u{eaa}', '\u{eab}', SC_OLetter), ('\u{ead}', '\u{eb0}', SC_OLetter),
        ('\u{eb1}', '\u{eb1}', SC_Extend), ('\u{eb2}', '\u{eb3}', SC_OLetter), ('\u{eb4}',
        '\u{eb9}', SC_Extend), ('\u{ebb}', '\u{ebc}', SC_Extend), ('\u{ebd}', '\u{ebd}',
        SC_OLetter), ('\u{ec0}', '\u{ec4}', SC_OLetter), ('\u{ec6}', '\u{ec6}', SC_OLetter),
        ('\u{ec8}', '\u{ecd}', SC_Extend), ('\u{ed0}', '\u{ed9}', SC_Numeric), ('\u{edc}',
        '\u{edf}', SC_OLetter), ('\u{f00}', '\u{f00}', SC_OLetter), ('\u{f18}', '\u{f19}',
        SC_Extend), ('\u{f20}', '\u{f29}', SC_Numeric), ('\u{f35}', '\u{f35}', SC_Extend),
        ('\u{f37}', '\u{f37}', SC_Extend), ('\u{f39}', '\u{f39}', SC_Extend), ('\u{f3a}', '\u{f3d}',
        SC_Close), ('\u{f3e}', '\u{f3f}', SC_Extend), ('\u{f40}', '\u{f47}', SC_OLetter),
        ('\u{f49}', '\u{f6c}', SC_OLetter), ('\u{f71}', '\u{f84}', SC_Extend), ('\u{f86}',
        '\u{f87}', SC_Extend), ('\u{f88}', '\u{f8c}', SC_OLetter), ('\u{f8d}', '\u{f97}',
        SC_Extend), ('\u{f99}', '\u{fbc}', SC_Extend), ('\u{fc6}', '\u{fc6}', SC_Extend),
        ('\u{1000}', '\u{102a}', SC_OLetter), ('\u{102b}', '\u{103e}', SC_Extend), ('\u{103f}',
        '\u{103f}', SC_OLetter), ('\u{1040}', '\u{1049}', SC_Numeric), ('\u{104a}', '\u{104b}',
        SC_STerm), ('\u{1050}', '\u{1055}', SC_OLetter), ('\u{1056}', '\u{1059}', SC_Extend),
        ('\u{105a}', '\u{105d}', SC_OLetter), ('\u{105e}', '\u{1060}', SC_Extend), ('\u{1061}',
        '\u{1061}', SC_OLetter), ('\u{1062}', '\u{1064}', SC_Extend), ('\u{1065}', '\u{1066}',
        SC_OLetter), ('\u{1067}', '\u{106d}', SC_Extend), ('\u{106e}', '\u{1070}', SC_OLetter),
        ('\u{1071}', '\u{1074}', SC_Extend), ('\u{1075}', '\u{1081}', SC_OLetter), ('\u{1082}',
        '\u{108d}', SC_Extend), ('\u{108e}', '\u{108e}', SC_OLetter), ('\u{108f}', '\u{108f}',
        SC_Extend), ('\u{1090}', '\u{1099}', SC_Numeric), ('\u{109a}', '\u{109d}', SC_Extend),
        ('\u{10a0}', '\u{10c5}', SC_Upper), ('\u{10c7}', '\u{10c7}', SC_Upper), ('\u{10cd}',
        '\u{10cd}', SC_Upper), ('\u{10d0}', '\u{10fa}', SC_OLetter), ('\u{10fc}', '\u{1248}',
        SC_OLetter), ('\u{124a}', '\u{124d}', SC_OLetter), ('\u{1250}', '\u{1256}', SC_OLetter),
        ('\u{1258}', '\u{1258}', SC_OLetter), ('\u{125a}', '\u{125d}', SC_OLetter), ('\u{1260}',
        '\u{1288}', SC_OLetter), ('\u{128a}', '\u{128d}', SC_OLetter), ('\u{1290}', '\u{12b0}',
        SC_OLetter), ('\u{12b2}', '\u{12b5}', SC_OLetter), ('\u{12b8}', '\u{12be}', SC_OLetter),
        ('\u{12c0}', '\u{12c0}', SC_OLetter), ('\u{12c2}', '\u{12c5}', SC_OLetter), ('\u{12c8}',
        '\u{12d6}', SC_OLetter), ('\u{12d8}', '\u{1310}', SC_OLetter), ('\u{1312}', '\u{1315}',
        SC_OLetter), ('\u{1318}', '\u{135a}', SC_OLetter), ('\u{135d}', '\u{135f}', SC_Extend),
        ('\u{1362}', '\u{1362}', SC_STerm), ('\u{1367}', '\u{1368}', SC_STerm), ('\u{1380}',
        '\u{138f}', SC_OLetter), ('\u{13a0}', '\u{13f5}', SC_Upper), ('\u{13f8}', '\u{13fd}',
        SC_Lower), ('\u{1401}', '\u{166c}', SC_OLetter), ('\u{166e}', '\u{166e}', SC_STerm),
        ('\u{166f}', '\u{167f}', SC_OLetter), ('\u{1680}', '\u{1680}', SC_Sp), ('\u{1681}',
        '\u{169a}', SC_OLetter), ('\u{169b}', '\u{169c}', SC_Close), ('\u{16a0}', '\u{16ea}',
        SC_OLetter), ('\u{16ee}', '\u{16f8}', SC_OLetter), ('\u{1700}', '\u{170c}', SC_OLetter),
        ('\u{170e}', '\u{1711}', SC_OLetter), ('\u{1712}', '\u{1714}', SC_Extend), ('\u{1720}',
        '\u{1731}', SC_OLetter), ('\u{1732}', '\u{1734}', SC_Extend), ('\u{1735}', '\u{1736}',
        SC_STerm), ('\u{1740}', '\u{1751}', SC_OLetter), ('\u{1752}', '\u{1753}', SC_Extend),
        ('\u{1760}', '\u{176c}', SC_OLetter), ('\u{176e}', '\u{1770}', SC_OLetter), ('\u{1772}',
        '\u{1773}', SC_Extend), ('\u{1780}', '\u{17b3}', SC_OLetter), ('\u{17b4}', '\u{17d3}',
        SC_Extend), ('\u{17d7}', '\u{17d7}', SC_OLetter), ('\u{17dc}', '\u{17dc}', SC_OLetter),
        ('\u{17dd}', '\u{17dd}', SC_Extend), ('\u{17e0}', '\u{17e9}', SC_Numeric), ('\u{1802}',
        '\u{1802}', SC_SContinue), ('\u{1803}', '\u{1803}', SC_STerm), ('\u{1808}', '\u{1808}',
        SC_SContinue), ('\u{1809}', '\u{1809}', SC_STerm), ('\u{180b}', '\u{180d}', SC_Extend),
        ('\u{180e}', '\u{180e}', SC_Format), ('\u{1810}', '\u{1819}', SC_Numeric), ('\u{1820}',
        '\u{1877}', SC_OLetter), ('\u{1880}', '\u{1884}', SC_OLetter), ('\u{1885}', '\u{1886}',
        SC_Extend), ('\u{1887}', '\u{18a8}', SC_OLetter), ('\u{18a9}', '\u{18a9}', SC_Extend),
        ('\u{18aa}', '\u{18aa}', SC_OLetter), ('\u{18b0}', '\u{18f5}', SC_OLetter), ('\u{1900}',
        '\u{191e}', SC_OLetter), ('\u{1920}', '\u{192b}', SC_Extend), ('\u{1930}', '\u{193b}',
        SC_Extend), ('\u{1944}', '\u{1945}', SC_STerm), ('\u{1946}', '\u{194f}', SC_Numeric),
        ('\u{1950}', '\u{196d}', SC_OLetter), ('\u{1970}', '\u{1974}', SC_OLetter), ('\u{1980}',
        '\u{19ab}', SC_OLetter), ('\u{19b0}', '\u{19c9}', SC_OLetter), ('\u{19d0}', '\u{19d9}',
        SC_Numeric), ('\u{1a00}', '\u{1a16}', SC_OLetter), ('\u{1a17}', '\u{1a1b}', SC_Extend),
        ('\u{1a20}', '\u{1a54}', SC_OLetter), ('\u{1a55}', '\u{1a5e}', SC_Extend), ('\u{1a60}',
        '\u{1a7c}', SC_Extend), ('\u{1a7f}', '\u{1a7f}', SC_Extend), ('\u{1a80}', '\u{1a89}',
        SC_Numeric), ('\u{1a90}', '\u{1a99}', SC_Numeric), ('\u{1aa7}', '\u{1aa7}', SC_OLetter),
        ('\u{1aa8}', '\u{1aab}', SC_STerm), ('\u{1ab0}', '\u{1abe}', SC_Extend), ('\u{1b00}',
        '\u{1b04}', SC_Extend), ('\u{1b05}', '\u{1b33}', SC_OLetter), ('\u{1b34}', '\u{1b44}',
        SC_Extend), ('\u{1b45}', '\u{1b4b}', SC_OLetter), ('\u{1b50}', '\u{1b59}', SC_Numeric),
        ('\u{1b5a}', '\u{1b5b}', SC_STerm), ('\u{1b5e}', '\u{1b5f}', SC_STerm), ('\u{1b6b}',
        '\u{1b73}', SC_Extend), ('\u{1b80}', '\u{1b82}', SC_Extend), ('\u{1b83}', '\u{1ba0}',
        SC_OLetter), ('\u{1ba1}', '\u{1bad}', SC_Extend), ('\u{1bae}', '\u{1baf}', SC_OLetter),
        ('\u{1bb0}', '\u{1bb9}', SC_Numeric), ('\u{1bba}', '\u{1be5}', SC_OLetter), ('\u{1be6}',
        '\u{1bf3}', SC_Extend), ('\u{1c00}', '\u{1c23}', SC_OLetter), ('\u{1c24}', '\u{1c37}',
        SC_Extend), ('\u{1c3b}', '\u{1c3c}', SC_STerm), ('\u{1c40}', '\u{1c49}', SC_Numeric),
        ('\u{1c4d}', '\u{1c4f}', SC_OLetter), ('\u{1c50}', '\u{1c59}', SC_Numeric), ('\u{1c5a}',
        '\u{1c7d}', SC_OLetter), ('\u{1c7e}', '\u{1c7f}', SC_STerm), ('\u{1c80}', '\u{1c88}',
        SC_Lower), ('\u{1cd0}', '\u{1cd2}', SC_Extend), ('\u{1cd4}', '\u{1ce8}', SC_Extend),
        ('\u{1ce9}', '\u{1cec}', SC_OLetter), ('\u{1ced}', '\u{1ced}', SC_Extend), ('\u{1cee}',
        '\u{1cf1}', SC_OLetter), ('\u{1cf2}', '\u{1cf4}', SC_Extend), ('\u{1cf5}', '\u{1cf6}',
        SC_OLetter), ('\u{1cf8}', '\u{1cf9}', SC_Extend), ('\u{1d00}', '\u{1dbf}', SC_Lower),
        ('\u{1dc0}', '\u{1df5}', SC_Extend), ('\u{1dfb}', '\u{1dff}', SC_Extend), ('\u{1e00}',
        '\u{1e00}', SC_Upper), ('\u{1e01}', '\u{1e01}', SC_Lower), ('\u{1e02}', '\u{1e02}',
        SC_Upper), ('\u{1e03}', '\u{1e03}', SC_Lower), ('\u{1e04}', '\u{1e04}', SC_Upper),
        ('\u{1e05}', '\u{1e05}', SC_Lower), ('\u{1e06}', '\u{1e06}', SC_Upper), ('\u{1e07}',
        '\u{1e07}', SC_Lower), ('\u{1e08}', '\u{1e08}', SC_Upper), ('\u{1e09}', '\u{1e09}',
        SC_Lower), ('\u{1e0a}', '\u{1e0a}', SC_Upper), ('\u{1e0b}', '\u{1e0b}', SC_Lower),
        ('\u{1e0c}', '\u{1e0c}', SC_Upper), ('\u{1e0d}', '\u{1e0d}', SC_Lower), ('\u{1e0e}',
        '\u{1e0e}', SC_Upper), ('\u{1e0f}', '\u{1e0f}', SC_Lower), ('\u{1e10}', '\u{1e10}',
        SC_Upper), ('\u{1e11}', '\u{1e11}', SC_Lower), ('\u{1e12}', '\u{1e12}', SC_Upper),
        ('\u{1e13}', '\u{1e13}', SC_Lower), ('\u{1e14}', '\u{1e14}', SC_Upper), ('\u{1e15}',
        '\u{1e15}', SC_Lower), ('\u{1e16}', '\u{1e16}', SC_Upper), ('\u{1e17}', '\u{1e17}',
        SC_Lower), ('\u{1e18}', '\u{1e18}', SC_Upper), ('\u{1e19}', '\u{1e19}', SC_Lower),
        ('\u{1e1a}', '\u{1e1a}', SC_Upper), ('\u{1e1b}', '\u{1e1b}', SC_Lower), ('\u{1e1c}',
        '\u{1e1c}', SC_Upper), ('\u{1e1d}', '\u{1e1d}', SC_Lower), ('\u{1e1e}', '\u{1e1e}',
        SC_Upper), ('\u{1e1f}', '\u{1e1f}', SC_Lower), ('\u{1e20}', '\u{1e20}', SC_Upper),
        ('\u{1e21}', '\u{1e21}', SC_Lower), ('\u{1e22}', '\u{1e22}', SC_Upper), ('\u{1e23}',
        '\u{1e23}', SC_Lower), ('\u{1e24}', '\u{1e24}', SC_Upper), ('\u{1e25}', '\u{1e25}',
        SC_Lower), ('\u{1e26}', '\u{1e26}', SC_Upper), ('\u{1e27}', '\u{1e27}', SC_Lower),
        ('\u{1e28}', '\u{1e28}', SC_Upper), ('\u{1e29}', '\u{1e29}', SC_Lower), ('\u{1e2a}',
        '\u{1e2a}', SC_Upper), ('\u{1e2b}', '\u{1e2b}', SC_Lower), ('\u{1e2c}', '\u{1e2c}',
        SC_Upper), ('\u{1e2d}', '\u{1e2d}', SC_Lower), ('\u{1e2e}', '\u{1e2e}', SC_Upper),
        ('\u{1e2f}', '\u{1e2f}', SC_Lower), ('\u{1e30}', '\u{1e30}', SC_Upper), ('\u{1e31}',
        '\u{1e31}', SC_Lower), ('\u{1e32}', '\u{1e32}', SC_Upper), ('\u{1e33}', '\u{1e33}',
        SC_Lower), ('\u{1e34}', '\u{1e34}', SC_Upper), ('\u{1e35}', '\u{1e35}', SC_Lower),
        ('\u{1e36}', '\u{1e36}', SC_Upper), ('\u{1e37}', '\u{1e37}', SC_Lower), ('\u{1e38}',
        '\u{1e38}', SC_Upper), ('\u{1e39}', '\u{1e39}', SC_Lower), ('\u{1e3a}', '\u{1e3a}',
        SC_Upper), ('\u{1e3b}', '\u{1e3b}', SC_Lower), ('\u{1e3c}', '\u{1e3c}', SC_Upper),
        ('\u{1e3d}', '\u{1e3d}', SC_Lower), ('\u{1e3e}', '\u{1e3e}', SC_Upper), ('\u{1e3f}',
        '\u{1e3f}', SC_Lower), ('\u{1e40}', '\u{1e40}', SC_Upper), ('\u{1e41}', '\u{1e41}',
        SC_Lower), ('\u{1e42}', '\u{1e42}', SC_Upper), ('\u{1e43}', '\u{1e43}', SC_Lower),
        ('\u{1e44}', '\u{1e44}', SC_Upper), ('\u{1e45}', '\u{1e45}', SC_Lower), ('\u{1e46}',
        '\u{1e46}', SC_Upper), ('\u{1e47}', '\u{1e47}', SC_Lower), ('\u{1e48}', '\u{1e48}',
        SC_Upper), ('\u{1e49}', '\u{1e49}', SC_Lower), ('\u{1e4a}', '\u{1e4a}', SC_Upper),
        ('\u{1e4b}', '\u{1e4b}', SC_Lower), ('\u{1e4c}', '\u{1e4c}', SC_Upper), ('\u{1e4d}',
        '\u{1e4d}', SC_Lower), ('\u{1e4e}', '\u{1e4e}', SC_Upper), ('\u{1e4f}', '\u{1e4f}',
        SC_Lower), ('\u{1e50}', '\u{1e50}', SC_Upper), ('\u{1e51}', '\u{1e51}', SC_Lower),
        ('\u{1e52}', '\u{1e52}', SC_Upper), ('\u{1e53}', '\u{1e53}', SC_Lower), ('\u{1e54}',
        '\u{1e54}', SC_Upper), ('\u{1e55}', '\u{1e55}', SC_Lower), ('\u{1e56}', '\u{1e56}',
        SC_Upper), ('\u{1e57}', '\u{1e57}', SC_Lower), ('\u{1e58}', '\u{1e58}', SC_Upper),
        ('\u{1e59}', '\u{1e59}', SC_Lower), ('\u{1e5a}', '\u{1e5a}', SC_Upper), ('\u{1e5b}',
        '\u{1e5b}', SC_Lower), ('\u{1e5c}', '\u{1e5c}', SC_Upper), ('\u{1e5d}', '\u{1e5d}',
        SC_Lower), ('\u{1e5e}', '\u{1e5e}', SC_Upper), ('\u{1e5f}', '\u{1e5f}', SC_Lower),
        ('\u{1e60}', '\u{1e60}', SC_Upper), ('\u{1e61}', '\u{1e61}', SC_Lower), ('\u{1e62}',
        '\u{1e62}', SC_Upper), ('\u{1e63}', '\u{1e63}', SC_Lower), ('\u{1e64}', '\u{1e64}',
        SC_Upper), ('\u{1e65}', '\u{1e65}', SC_Lower), ('\u{1e66}', '\u{1e66}', SC_Upper),
        ('\u{1e67}', '\u{1e67}', SC_Lower), ('\u{1e68}', '\u{1e68}', SC_Upper), ('\u{1e69}',
        '\u{1e69}', SC_Lower), ('\u{1e6a}', '\u{1e6a}', SC_Upper), ('\u{1e6b}', '\u{1e6b}',
        SC_Lower), ('\u{1e6c}', '\u{1e6c}', SC_Upper), ('\u{1e6d}', '\u{1e6d}', SC_Lower),
        ('\u{1e6e}', '\u{1e6e}', SC_Upper), ('\u{1e6f}', '\u{1e6f}', SC_Lower), ('\u{1e70}',
        '\u{1e70}', SC_Upper), ('\u{1e71}', '\u{1e71}', SC_Lower), ('\u{1e72}', '\u{1e72}',
        SC_Upper), ('\u{1e73}', '\u{1e73}', SC_Lower), ('\u{1e74}', '\u{1e74}', SC_Upper),
        ('\u{1e75}', '\u{1e75}', SC_Lower), ('\u{1e76}', '\u{1e76}', SC_Upper), ('\u{1e77}',
        '\u{1e77}', SC_Lower), ('\u{1e78}', '\u{1e78}', SC_Upper), ('\u{1e79}', '\u{1e79}',
        SC_Lower), ('\u{1e7a}', '\u{1e7a}', SC_Upper), ('\u{1e7b}', '\u{1e7b}', SC_Lower),
        ('\u{1e7c}', '\u{1e7c}', SC_Upper), ('\u{1e7d}', '\u{1e7d}', SC_Lower), ('\u{1e7e}',
        '\u{1e7e}', SC_Upper), ('\u{1e7f}', '\u{1e7f}', SC_Lower), ('\u{1e80}', '\u{1e80}',
        SC_Upper), ('\u{1e81}', '\u{1e81}', SC_Lower), ('\u{1e82}', '\u{1e82}', SC_Upper),
        ('\u{1e83}', '\u{1e83}', SC_Lower), ('\u{1e84}', '\u{1e84}', SC_Upper), ('\u{1e85}',
        '\u{1e85}', SC_Lower), ('\u{1e86}', '\u{1e86}', SC_Upper), ('\u{1e87}', '\u{1e87}',
        SC_Lower), ('\u{1e88}', '\u{1e88}', SC_Upper), ('\u{1e89}', '\u{1e89}', SC_Lower),
        ('\u{1e8a}', '\u{1e8a}', SC_Upper), ('\u{1e8b}', '\u{1e8b}', SC_Lower), ('\u{1e8c}',
        '\u{1e8c}', SC_Upper), ('\u{1e8d}', '\u{1e8d}', SC_Lower), ('\u{1e8e}', '\u{1e8e}',
        SC_Upper), ('\u{1e8f}', '\u{1e8f}', SC_Lower), ('\u{1e90}', '\u{1e90}', SC_Upper),
        ('\u{1e91}', '\u{1e91}', SC_Lower), ('\u{1e92}', '\u{1e92}', SC_Upper), ('\u{1e93}',
        '\u{1e93}', SC_Lower), ('\u{1e94}', '\u{1e94}', SC_Upper), ('\u{1e95}', '\u{1e9d}',
        SC_Lower), ('\u{1e9e}', '\u{1e9e}', SC_Upper), ('\u{1e9f}', '\u{1e9f}', SC_Lower),
        ('\u{1ea0}', '\u{1ea0}', SC_Upper), ('\u{1ea1}', '\u{1ea1}', SC_Lower), ('\u{1ea2}',
        '\u{1ea2}', SC_Upper), ('\u{1ea3}', '\u{1ea3}', SC_Lower), ('\u{1ea4}', '\u{1ea4}',
        SC_Upper), ('\u{1ea5}', '\u{1ea5}', SC_Lower), ('\u{1ea6}', '\u{1ea6}', SC_Upper),
        ('\u{1ea7}', '\u{1ea7}', SC_Lower), ('\u{1ea8}', '\u{1ea8}', SC_Upper), ('\u{1ea9}',
        '\u{1ea9}', SC_Lower), ('\u{1eaa}', '\u{1eaa}', SC_Upper), ('\u{1eab}', '\u{1eab}',
        SC_Lower), ('\u{1eac}', '\u{1eac}', SC_Upper), ('\u{1ead}', '\u{1ead}', SC_Lower),
        ('\u{1eae}', '\u{1eae}', SC_Upper), ('\u{1eaf}', '\u{1eaf}', SC_Lower), ('\u{1eb0}',
        '\u{1eb0}', SC_Upper), ('\u{1eb1}', '\u{1eb1}', SC_Lower), ('\u{1eb2}', '\u{1eb2}',
        SC_Upper), ('\u{1eb3}', '\u{1eb3}', SC_Lower), ('\u{1eb4}', '\u{1eb4}', SC_Upper),
        ('\u{1eb5}', '\u{1eb5}', SC_Lower), ('\u{1eb6}', '\u{1eb6}', SC_Upper), ('\u{1eb7}',
        '\u{1eb7}', SC_Lower), ('\u{1eb8}', '\u{1eb8}', SC_Upper), ('\u{1eb9}', '\u{1eb9}',
        SC_Lower), ('\u{1eba}', '\u{1eba}', SC_Upper), ('\u{1ebb}', '\u{1ebb}', SC_Lower),
        ('\u{1ebc}', '\u{1ebc}', SC_Upper), ('\u{1ebd}', '\u{1ebd}', SC_Lower), ('\u{1ebe}',
        '\u{1ebe}', SC_Upper), ('\u{1ebf}', '\u{1ebf}', SC_Lower), ('\u{1ec0}', '\u{1ec0}',
        SC_Upper), ('\u{1ec1}', '\u{1ec1}', SC_Lower), ('\u{1ec2}', '\u{1ec2}', SC_Upper),
        ('\u{1ec3}', '\u{1ec3}', SC_Lower), ('\u{1ec4}', '\u{1ec4}', SC_Upper), ('\u{1ec5}',
        '\u{1ec5}', SC_Lower), ('\u{1ec6}', '\u{1ec6}', SC_Upper), ('\u{1ec7}', '\u{1ec7}',
        SC_Lower), ('\u{1ec8}', '\u{1ec8}', SC_Upper), ('\u{1ec9}', '\u{1ec9}', SC_Lower),
        ('\u{1eca}', '\u{1eca}', SC_Upper), ('\u{1ecb}', '\u{1ecb}', SC_Lower), ('\u{1ecc}',
        '\u{1ecc}', SC_Upper), ('\u{1ecd}', '\u{1ecd}', SC_Lower), ('\u{1ece}', '\u{1ece}',
        SC_Upper), ('\u{1ecf}', '\u{1ecf}', SC_Lower), ('\u{1ed0}', '\u{1ed0}', SC_Upper),
        ('\u{1ed1}', '\u{1ed1}', SC_Lower), ('\u{1ed2}', '\u{1ed2}', SC_Upper), ('\u{1ed3}',
        '\u{1ed3}', SC_Lower), ('\u{1ed4}', '\u{1ed4}', SC_Upper), ('\u{1ed5}', '\u{1ed5}',
        SC_Lower), ('\u{1ed6}', '\u{1ed6}', SC_Upper), ('\u{1ed7}', '\u{1ed7}', SC_Lower),
        ('\u{1ed8}', '\u{1ed8}', SC_Upper), ('\u{1ed9}', '\u{1ed9}', SC_Lower), ('\u{1eda}',
        '\u{1eda}', SC_Upper), ('\u{1edb}', '\u{1edb}', SC_Lower), ('\u{1edc}', '\u{1edc}',
        SC_Upper), ('\u{1edd}', '\u{1edd}', SC_Lower), ('\u{1ede}', '\u{1ede}', SC_Upper),
        ('\u{1edf}', '\u{1edf}', SC_Lower), ('\u{1ee0}', '\u{1ee0}', SC_Upper), ('\u{1ee1}',
        '\u{1ee1}', SC_Lower), ('\u{1ee2}', '\u{1ee2}', SC_Upper), ('\u{1ee3}', '\u{1ee3}',
        SC_Lower), ('\u{1ee4}', '\u{1ee4}', SC_Upper), ('\u{1ee5}', '\u{1ee5}', SC_Lower),
        ('\u{1ee6}', '\u{1ee6}', SC_Upper), ('\u{1ee7}', '\u{1ee7}', SC_Lower), ('\u{1ee8}',
        '\u{1ee8}', SC_Upper), ('\u{1ee9}', '\u{1ee9}', SC_Lower), ('\u{1eea}', '\u{1eea}',
        SC_Upper), ('\u{1eeb}', '\u{1eeb}', SC_Lower), ('\u{1eec}', '\u{1eec}', SC_Upper),
        ('\u{1eed}', '\u{1eed}', SC_Lower), ('\u{1eee}', '\u{1eee}', SC_Upper), ('\u{1eef}',
        '\u{1eef}', SC_Lower), ('\u{1ef0}', '\u{1ef0}', SC_Upper), ('\u{1ef1}', '\u{1ef1}',
        SC_Lower), ('\u{1ef2}', '\u{1ef2}', SC_Upper), ('\u{1ef3}', '\u{1ef3}', SC_Lower),
        ('\u{1ef4}', '\u{1ef4}', SC_Upper), ('\u{1ef5}', '\u{1ef5}', SC_Lower), ('\u{1ef6}',
        '\u{1ef6}', SC_Upper), ('\u{1ef7}', '\u{1ef7}', SC_Lower), ('\u{1ef8}', '\u{1ef8}',
        SC_Upper), ('\u{1ef9}', '\u{1ef9}', SC_Lower), ('\u{1efa}', '\u{1efa}', SC_Upper),
        ('\u{1efb}', '\u{1efb}', SC_Lower), ('\u{1efc}', '\u{1efc}', SC_Upper), ('\u{1efd}',
        '\u{1efd}', SC_Lower), ('\u{1efe}', '\u{1efe}', SC_Upper), ('\u{1eff}', '\u{1f07}',
        SC_Lower), ('\u{1f08}', '\u{1f0f}', SC_Upper), ('\u{1f10}', '\u{1f15}', SC_Lower),
        ('\u{1f18}', '\u{1f1d}', SC_Upper), ('\u{1f20}', '\u{1f27}', SC_Lower), ('\u{1f28}',
        '\u{1f2f}', SC_Upper), ('\u{1f30}', '\u{1f37}', SC_Lower), ('\u{1f38}', '\u{1f3f}',
        SC_Upper), ('\u{1f40}', '\u{1f45}', SC_Lower), ('\u{1f48}', '\u{1f4d}', SC_Upper),
        ('\u{1f50}', '\u{1f57}', SC_Lower), ('\u{1f59}', '\u{1f59}', SC_Upper), ('\u{1f5b}',
        '\u{1f5b}', SC_Upper), ('\u{1f5d}', '\u{1f5d}', SC_Upper), ('\u{1f5f}', '\u{1f5f}',
        SC_Upper), ('\u{1f60}', '\u{1f67}', SC_Lower), ('\u{1f68}', '\u{1f6f}', SC_Upper),
        ('\u{1f70}', '\u{1f7d}', SC_Lower), ('\u{1f80}', '\u{1f87}', SC_Lower), ('\u{1f88}',
        '\u{1f8f}', SC_Upper), ('\u{1f90}', '\u{1f97}', SC_Lower), ('\u{1f98}', '\u{1f9f}',
        SC_Upper), ('\u{1fa0}', '\u{1fa7}', SC_Lower), ('\u{1fa8}', '\u{1faf}', SC_Upper),
        ('\u{1fb0}', '\u{1fb4}', SC_Lower), ('\u{1fb6}', '\u{1fb7}', SC_Lower), ('\u{1fb8}',
        '\u{1fbc}', SC_Upper), ('\u{1fbe}', '\u{1fbe}', SC_Lower), ('\u{1fc2}', '\u{1fc4}',
        SC_Lower), ('\u{1fc6}', '\u{1fc7}', SC_Lower), ('\u{1fc8}', '\u{1fcc}', SC_Upper),
        ('\u{1fd0}', '\u{1fd3}', SC_Lower), ('\u{1fd6}', '\u{1fd7}', SC_Lower), ('\u{1fd8}',
        '\u{1fdb}', SC_Upper), ('\u{1fe0}', '\u{1fe7}', SC_Lower), ('\u{1fe8}', '\u{1fec}',
        SC_Upper), ('\u{1ff2}', '\u{1ff4}', SC_Lower), ('\u{1ff6}', '\u{1ff7}', SC_Lower),
        ('\u{1ff8}', '\u{1ffc}', SC_Upper), ('\u{2000}', '\u{200a}', SC_Sp), ('\u{200b}',
        '\u{200b}', SC_Format), ('\u{200c}', '\u{200d}', SC_Extend), ('\u{200e}', '\u{200f}',
        SC_Format), ('\u{2013}', '\u{2014}', SC_SContinue), ('\u{2018}', '\u{201f}', SC_Close),
        ('\u{2024}', '\u{2024}', SC_ATerm), ('\u{2028}', '\u{2029}', SC_Sep), ('\u{202a}',
        '\u{202e}', SC_Format), ('\u{202f}', '\u{202f}', SC_Sp), ('\u{2039}', '\u{203a}', SC_Close),
        ('\u{203c}', '\u{203d}', SC_STerm), ('\u{2045}', '\u{2046}', SC_Close), ('\u{2047}',
        '\u{2049}', SC_STerm), ('\u{205f}', '\u{205f}', SC_Sp), ('\u{2060}', '\u{2064}', SC_Format),
        ('\u{2066}', '\u{206f}', SC_Format), ('\u{2071}', '\u{2071}', SC_Lower), ('\u{207d}',
        '\u{207e}', SC_Close), ('\u{207f}', '\u{207f}', SC_Lower), ('\u{208d}', '\u{208e}',
        SC_Close), ('\u{2090}', '\u{209c}', SC_Lower), ('\u{20d0}', '\u{20f0}', SC_Extend),
        ('\u{2102}', '\u{2102}', SC_Upper), ('\u{2107}', '\u{2107}', SC_Upper), ('\u{210a}',
        '\u{210a}', SC_Lower), ('\u{210b}', '\u{210d}', SC_Upper), ('\u{210e}', '\u{210f}',
        SC_Lower), ('\u{2110}', '\u{2112}', SC_Upper), ('\u{2113}', '\u{2113}', SC_Lower),
        ('\u{2115}', '\u{2115}', SC_Upper), ('\u{2119}', '\u{211d}', SC_Upper), ('\u{2124}',
        '\u{2124}', SC_Upper), ('\u{2126}', '\u{2126}', SC_Upper), ('\u{2128}', '\u{2128}',
        SC_Upper), ('\u{212a}', '\u{212d}', SC_Upper), ('\u{212f}', '\u{212f}', SC_Lower),
        ('\u{2130}', '\u{2133}', SC_Upper), ('\u{2134}', '\u{2134}', SC_Lower), ('\u{2135}',
        '\u{2138}', SC_OLetter), ('\u{2139}', '\u{2139}', SC_Lower), ('\u{213c}', '\u{213d}',
        SC_Lower), ('\u{213e}', '\u{213f}', SC_Upper), ('\u{2145}', '\u{2145}', SC_Upper),
        ('\u{2146}', '\u{2149}', SC_Lower), ('\u{214e}', '\u{214e}', SC_Lower), ('\u{2160}',
        '\u{216f}', SC_Upper), ('\u{2170}', '\u{217f}', SC_Lower), ('\u{2180}', '\u{2182}',
        SC_OLetter), ('\u{2183}', '\u{2183}', SC_Upper), ('\u{2184}', '\u{2184}', SC_Lower),
        ('\u{2185}', '\u{2188}', SC_OLetter), ('\u{2308}', '\u{230b}', SC_Close), ('\u{2329}',
        '\u{232a}', SC_Close), ('\u{24b6}', '\u{24cf}', SC_Upper), ('\u{24d0}', '\u{24e9}',
        SC_Lower), ('\u{275b}', '\u{2760}', SC_Close), ('\u{2768}', '\u{2775}', SC_Close),
        ('\u{27c5}', '\u{27c6}', SC_Close), ('\u{27e6}', '\u{27ef}', SC_Close), ('\u{2983}',
        '\u{2998}', SC_Close), ('\u{29d8}', '\u{29db}', SC_Close), ('\u{29fc}', '\u{29fd}',
        SC_Close), ('\u{2c00}', '\u{2c2e}', SC_Upper), ('\u{2c30}', '\u{2c5e}', SC_Lower),
        ('\u{2c60}', '\u{2c60}', SC_Upper), ('\u{2c61}', '\u{2c61}', SC_Lower), ('\u{2c62}',
        '\u{2c64}', SC_Upper), ('\u{2c65}', '\u{2c66}', SC_Lower), ('\u{2c67}', '\u{2c67}',
        SC_Upper), ('\u{2c68}', '\u{2c68}', SC_Lower), ('\u{2c69}', '\u{2c69}', SC_Upper),
        ('\u{2c6a}', '\u{2c6a}', SC_Lower), ('\u{2c6b}', '\u{2c6b}', SC_Upper), ('\u{2c6c}',
        '\u{2c6c}', SC_Lower), ('\u{2c6d}', '\u{2c70}', SC_Upper), ('\u{2c71}', '\u{2c71}',
        SC_Lower), ('\u{2c72}', '\u{2c72}', SC_Upper), ('\u{2c73}', '\u{2c74}', SC_Lower),
        ('\u{2c75}', '\u{2c75}', SC_Upper), ('\u{2c76}', '\u{2c7d}', SC_Lower), ('\u{2c7e}',
        '\u{2c80}', SC_Upper), ('\u{2c81}', '\u{2c81}', SC_Lower), ('\u{2c82}', '\u{2c82}',
        SC_Upper), ('\u{2c83}', '\u{2c83}', SC_Lower), ('\u{2c84}', '\u{2c84}', SC_Upper),
        ('\u{2c85}', '\u{2c85}', SC_Lower), ('\u{2c86}', '\u{2c86}', SC_Upper), ('\u{2c87}',
        '\u{2c87}', SC_Lower), ('\u{2c88}', '\u{2c88}', SC_Upper), ('\u{2c89}', '\u{2c89}',
        SC_Lower), ('\u{2c8a}', '\u{2c8a}', SC_Upper), ('\u{2c8b}', '\u{2c8b}', SC_Lower),
        ('\u{2c8c}', '\u{2c8c}', SC_Upper), ('\u{2c8d}', '\u{2c8d}', SC_Lower), ('\u{2c8e}',
        '\u{2c8e}', SC_Upper), ('\u{2c8f}', '\u{2c8f}', SC_Lower), ('\u{2c90}', '\u{2c90}',
        SC_Upper), ('\u{2c91}', '\u{2c91}', SC_Lower), ('\u{2c92}', '\u{2c92}', SC_Upper),
        ('\u{2c93}', '\u{2c93}', SC_Lower), ('\u{2c94}', '\u{2c94}', SC_Upper), ('\u{2c95}',
        '\u{2c95}', SC_Lower), ('\u{2c96}', '\u{2c96}', SC_Upper), ('\u{2c97}', '\u{2c97}',
        SC_Lower), ('\u{2c98}', '\u{2c98}', SC_Upper), ('\u{2c99}', '\u{2c99}', SC_Lower),
        ('\u{2c9a}', '\u{2c9a}', SC_Upper), ('\u{2c9b}', '\u{2c9b}', SC_Lower), ('\u{2c9c}',
        '\u{2c9c}', SC_Upper), ('\u{2c9d}', '\u{2c9d}', SC_Lower), ('\u{2c9e}', '\u{2c9e}',
        SC_Upper), ('\u{2c9f}', '\u{2c9f}', SC_Lower), ('\u{2ca0}', '\u{2ca0}', SC_Upper),
        ('\u{2ca1}', '\u{2ca1}', SC_Lower), ('\u{2ca2}', '\u{2ca2}', SC_Upper), ('\u{2ca3}',
        '\u{2ca3}', SC_Lower), ('\u{2ca4}', '\u{2ca4}', SC_Upper), ('\u{2ca5}', '\u{2ca5}',
        SC_Lower), ('\u{2ca6}', '\u{2ca6}', SC_Upper), ('\u{2ca7}', '\u{2ca7}', SC_Lower),
        ('\u{2ca8}', '\u{2ca8}', SC_Upper), ('\u{2ca9}', '\u{2ca9}', SC_Lower), ('\u{2caa}',
        '\u{2caa}', SC_Upper), ('\u{2cab}', '\u{2cab}', SC_Lower), ('\u{2cac}', '\u{2cac}',
        SC_Upper), ('\u{2cad}', '\u{2cad}', SC_Lower), ('\u{2cae}', '\u{2cae}', SC_Upper),
        ('\u{2caf}', '\u{2caf}', SC_Lower), ('\u{2cb0}', '\u{2cb0}', SC_Upper), ('\u{2cb1}',
        '\u{2cb1}', SC_Lower), ('\u{2cb2}', '\u{2cb2}', SC_Upper), ('\u{2cb3}', '\u{2cb3}',
        SC_Lower), ('\u{2cb4}', '\u{2cb4}', SC_Upper), ('\u{2cb5}', '\u{2cb5}', SC_Lower),
        ('\u{2cb6}', '\u{2cb6}', SC_Upper), ('\u{2cb7}', '\u{2cb7}', SC_Lower), ('\u{2cb8}',
        '\u{2cb8}', SC_Upper), ('\u{2cb9}', '\u{2cb9}', SC_Lower), ('\u{2cba}', '\u{2cba}',
        SC_Upper), ('\u{2cbb}', '\u{2cbb}', SC_Lower), ('\u{2cbc}', '\u{2cbc}', SC_Upper),
        ('\u{2cbd}', '\u{2cbd}', SC_Lower), ('\u{2cbe}', '\u{2cbe}', SC_Upper), ('\u{2cbf}',
        '\u{2cbf}', SC_Lower), ('\u{2cc0}', '\u{2cc0}', SC_Upper), ('\u{2cc1}', '\u{2cc1}',
        SC_Lower), ('\u{2cc2}', '\u{2cc2}', SC_Upper), ('\u{2cc3}', '\u{2cc3}', SC_Lower),
        ('\u{2cc4}', '\u{2cc4}', SC_Upper), ('\u{2cc5}', '\u{2cc5}', SC_Lower), ('\u{2cc6}',
        '\u{2cc6}', SC_Upper), ('\u{2cc7}', '\u{2cc7}', SC_Lower), ('\u{2cc8}', '\u{2cc8}',
        SC_Upper), ('\u{2cc9}', '\u{2cc9}', SC_Lower), ('\u{2cca}', '\u{2cca}', SC_Upper),
        ('\u{2ccb}', '\u{2ccb}', SC_Lower), ('\u{2ccc}', '\u{2ccc}', SC_Upper), ('\u{2ccd}',
        '\u{2ccd}', SC_Lower), ('\u{2cce}', '\u{2cce}', SC_Upper), ('\u{2ccf}', '\u{2ccf}',
        SC_Lower), ('\u{2cd0}', '\u{2cd0}', SC_Upper), ('\u{2cd1}', '\u{2cd1}', SC_Lower),
        ('\u{2cd2}', '\u{2cd2}', SC_Upper), ('\u{2cd3}', '\u{2cd3}', SC_Lower), ('\u{2cd4}',
        '\u{2cd4}', SC_Upper), ('\u{2cd5}', '\u{2cd5}', SC_Lower), ('\u{2cd6}', '\u{2cd6}',
        SC_Upper), ('\u{2cd7}', '\u{2cd7}', SC_Lower), ('\u{2cd8}', '\u{2cd8}', SC_Upper),
        ('\u{2cd9}', '\u{2cd9}', SC_Lower), ('\u{2cda}', '\u{2cda}', SC_Upper), ('\u{2cdb}',
        '\u{2cdb}', SC_Lower), ('\u{2cdc}', '\u{2cdc}', SC_Upper), ('\u{2cdd}', '\u{2cdd}',
        SC_Lower), ('\u{2cde}', '\u{2cde}', SC_Upper), ('\u{2cdf}', '\u{2cdf}', SC_Lower),
        ('\u{2ce0}', '\u{2ce0}', SC_Upper), ('\u{2ce1}', '\u{2ce1}', SC_Lower), ('\u{2ce2}',
        '\u{2ce2}', SC_Upper), ('\u{2ce3}', '\u{2ce4}', SC_Lower), ('\u{2ceb}', '\u{2ceb}',
        SC_Upper), ('\u{2cec}', '\u{2cec}', SC_Lower), ('\u{2ced}', '\u{2ced}', SC_Upper),
        ('\u{2cee}', '\u{2cee}', SC_Lower), ('\u{2cef}', '\u{2cf1}', SC_Extend), ('\u{2cf2}',
        '\u{2cf2}', SC_Upper), ('\u{2cf3}', '\u{2cf3}', SC_Lower), ('\u{2d00}', '\u{2d25}',
        SC_Lower), ('\u{2d27}', '\u{2d27}', SC_Lower), ('\u{2d2d}', '\u{2d2d}', SC_Lower),
        ('\u{2d30}', '\u{2d67}', SC_OLetter), ('\u{2d6f}', '\u{2d6f}', SC_OLetter), ('\u{2d7f}',
        '\u{2d7f}', SC_Extend), ('\u{2d80}', '\u{2d96}', SC_OLetter), ('\u{2da0}', '\u{2da6}',
        SC_OLetter), ('\u{2da8}', '\u{2dae}', SC_OLetter), ('\u{2db0}', '\u{2db6}', SC_OLetter),
        ('\u{2db8}', '\u{2dbe}', SC_OLetter), ('\u{2dc0}', '\u{2dc6}', SC_OLetter), ('\u{2dc8}',
        '\u{2dce}', SC_OLetter), ('\u{2dd0}', '\u{2dd6}', SC_OLetter), ('\u{2dd8}', '\u{2dde}',
        SC_OLetter), ('\u{2de0}', '\u{2dff}', SC_Extend), ('\u{2e00}', '\u{2e0d}', SC_Close),
        ('\u{2e1c}', '\u{2e1d}', SC_Close), ('\u{2e20}', '\u{2e29}', SC_Close), ('\u{2e2e}',
        '\u{2e2e}', SC_STerm), ('\u{2e2f}', '\u{2e2f}', SC_OLetter), ('\u{2e3c}', '\u{2e3c}',
        SC_STerm), ('\u{2e42}', '\u{2e42}', SC_Close), ('\u{3000}', '\u{3000}', SC_Sp), ('\u{3001}',
        '\u{3001}', SC_SContinue), ('\u{3002}', '\u{3002}', SC_STerm), ('\u{3005}', '\u{3007}',
        SC_OLetter), ('\u{3008}', '\u{3011}', SC_Close), ('\u{3014}', '\u{301b}', SC_Close),
        ('\u{301d}', '\u{301f}', SC_Close), ('\u{3021}', '\u{3029}', SC_OLetter), ('\u{302a}',
        '\u{302f}', SC_Extend), ('\u{3031}', '\u{3035}', SC_OLetter), ('\u{3038}', '\u{303c}',
        SC_OLetter), ('\u{3041}', '\u{3096}', SC_OLetter), ('\u{3099}', '\u{309a}', SC_Extend),
        ('\u{309d}', '\u{309f}', SC_OLetter), ('\u{30a1}', '\u{30fa}', SC_OLetter), ('\u{30fc}',
        '\u{30ff}', SC_OLetter), ('\u{3105}', '\u{312d}', SC_OLetter), ('\u{3131}', '\u{318e}',
        SC_OLetter), ('\u{31a0}', '\u{31ba}', SC_OLetter), ('\u{31f0}', '\u{31ff}', SC_OLetter),
        ('\u{3400}', '\u{4db5}', SC_OLetter), ('\u{4e00}', '\u{9fd5}', SC_OLetter), ('\u{a000}',
        '\u{a48c}', SC_OLetter), ('\u{a4d0}', '\u{a4fd}', SC_OLetter), ('\u{a4ff}', '\u{a4ff}',
        SC_STerm), ('\u{a500}', '\u{a60c}', SC_OLetter), ('\u{a60e}', '\u{a60f}', SC_STerm),
        ('\u{a610}', '\u{a61f}', SC_OLetter), ('\u{a620}', '\u{a629}', SC_Numeric), ('\u{a62a}',
        '\u{a62b}', SC_OLetter), ('\u{a640}', '\u{a640}', SC_Upper), ('\u{a641}', '\u{a641}',
        SC_Lower), ('\u{a642}', '\u{a642}', SC_Upper), ('\u{a643}', '\u{a643}', SC_Lower),
        ('\u{a644}', '\u{a644}', SC_Upper), ('\u{a645}', '\u{a645}', SC_Lower), ('\u{a646}',
        '\u{a646}', SC_Upper), ('\u{a647}', '\u{a647}', SC_Lower), ('\u{a648}', '\u{a648}',
        SC_Upper), ('\u{a649}', '\u{a649}', SC_Lower), ('\u{a64a}', '\u{a64a}', SC_Upper),
        ('\u{a64b}', '\u{a64b}', SC_Lower), ('\u{a64c}', '\u{a64c}', SC_Upper), ('\u{a64d}',
        '\u{a64d}', SC_Lower), ('\u{a64e}', '\u{a64e}', SC_Upper), ('\u{a64f}', '\u{a64f}',
        SC_Lower), ('\u{a650}', '\u{a650}', SC_Upper), ('\u{a651}', '\u{a651}', SC_Lower),
        ('\u{a652}', '\u{a652}', SC_Upper), ('\u{a653}', '\u{a653}', SC_Lower), ('\u{a654}',
        '\u{a654}', SC_Upper), ('\u{a655}', '\u{a655}', SC_Lower), ('\u{a656}', '\u{a656}',
        SC_Upper), ('\u{a657}', '\u{a657}', SC_Lower), ('\u{a658}', '\u{a658}', SC_Upper),
        ('\u{a659}', '\u{a659}', SC_Lower), ('\u{a65a}', '\u{a65a}', SC_Upper), ('\u{a65b}',
        '\u{a65b}', SC_Lower), ('\u{a65c}', '\u{a65c}', SC_Upper), ('\u{a65d}', '\u{a65d}',
        SC_Lower), ('\u{a65e}', '\u{a65e}', SC_Upper), ('\u{a65f}', '\u{a65f}', SC_Lower),
        ('\u{a660}', '\u{a660}', SC_Upper), ('\u{a661}', '\u{a661}', SC_Lower), ('\u{a662}',
        '\u{a662}', SC_Upper), ('\u{a663}', '\u{a663}', SC_Lower), ('\u{a664}', '\u{a664}',
        SC_Upper), ('\u{a665}', '\u{a665}', SC_Lower), ('\u{a666}', '\u{a666}', SC_Upper),
        ('\u{a667}', '\u{a667}', SC_Lower), ('\u{a668}', '\u{a668}', SC_Upper), ('\u{a669}',
        '\u{a669}', SC_Lower), ('\u{a66a}', '\u{a66a}', SC_Upper), ('\u{a66b}', '\u{a66b}',
        SC_Lower), ('\u{a66c}', '\u{a66c}', SC_Upper), ('\u{a66d}', '\u{a66d}', SC_Lower),
        ('\u{a66e}', '\u{a66e}', SC_OLetter), ('\u{a66f}', '\u{a672}', SC_Extend), ('\u{a674}',
        '\u{a67d}', SC_Extend), ('\u{a67f}', '\u{a67f}', SC_OLetter), ('\u{a680}', '\u{a680}',
        SC_Upper), ('\u{a681}', '\u{a681}', SC_Lower), ('\u{a682}', '\u{a682}', SC_Upper),
        ('\u{a683}', '\u{a683}', SC_Lower), ('\u{a684}', '\u{a684}', SC_Upper), ('\u{a685}',
        '\u{a685}', SC_Lower), ('\u{a686}', '\u{a686}', SC_Upper), ('\u{a687}', '\u{a687}',
        SC_Lower), ('\u{a688}', '\u{a688}', SC_Upper), ('\u{a689}', '\u{a689}', SC_Lower),
        ('\u{a68a}', '\u{a68a}', SC_Upper), ('\u{a68b}', '\u{a68b}', SC_Lower), ('\u{a68c}',
        '\u{a68c}', SC_Upper), ('\u{a68d}', '\u{a68d}', SC_Lower), ('\u{a68e}', '\u{a68e}',
        SC_Upper), ('\u{a68f}', '\u{a68f}', SC_Lower), ('\u{a690}', '\u{a690}', SC_Upper),
        ('\u{a691}', '\u{a691}', SC_Lower), ('\u{a692}', '\u{a692}', SC_Upper), ('\u{a693}',
        '\u{a693}', SC_Lower), ('\u{a694}', '\u{a694}', SC_Upper), ('\u{a695}', '\u{a695}',
        SC_Lower), ('\u{a696}', '\u{a696}', SC_Upper), ('\u{a697}', '\u{a697}', SC_Lower),
        ('\u{a698}', '\u{a698}', SC_Upper), ('\u{a699}', '\u{a699}', SC_Lower), ('\u{a69a}',
        '\u{a69a}', SC_Upper), ('\u{a69b}', '\u{a69d}', SC_Lower), ('\u{a69e}', '\u{a69f}',
        SC_Extend), ('\u{a6a0}', '\u{a6ef}', SC_OLetter), ('\u{a6f0}', '\u{a6f1}', SC_Extend),
        ('\u{a6f3}', '\u{a6f3}', SC_STerm), ('\u{a6f7}', '\u{a6f7}', SC_STerm), ('\u{a717}',
        '\u{a71f}', SC_OLetter), ('\u{a722}', '\u{a722}', SC_Upper), ('\u{a723}', '\u{a723}',
        SC_Lower), ('\u{a724}', '\u{a724}', SC_Upper), ('\u{a725}', '\u{a725}', SC_Lower),
        ('\u{a726}', '\u{a726}', SC_Upper), ('\u{a727}', '\u{a727}', SC_Lower), ('\u{a728}',
        '\u{a728}', SC_Upper), ('\u{a729}', '\u{a729}', SC_Lower), ('\u{a72a}', '\u{a72a}',
        SC_Upper), ('\u{a72b}', '\u{a72b}', SC_Lower), ('\u{a72c}', '\u{a72c}', SC_Upper),
        ('\u{a72d}', '\u{a72d}', SC_Lower), ('\u{a72e}', '\u{a72e}', SC_Upper), ('\u{a72f}',
        '\u{a731}', SC_Lower), ('\u{a732}', '\u{a732}', SC_Upper), ('\u{a733}', '\u{a733}',
        SC_Lower), ('\u{a734}', '\u{a734}', SC_Upper), ('\u{a735}', '\u{a735}', SC_Lower),
        ('\u{a736}', '\u{a736}', SC_Upper), ('\u{a737}', '\u{a737}', SC_Lower), ('\u{a738}',
        '\u{a738}', SC_Upper), ('\u{a739}', '\u{a739}', SC_Lower), ('\u{a73a}', '\u{a73a}',
        SC_Upper), ('\u{a73b}', '\u{a73b}', SC_Lower), ('\u{a73c}', '\u{a73c}', SC_Upper),
        ('\u{a73d}', '\u{a73d}', SC_Lower), ('\u{a73e}', '\u{a73e}', SC_Upper), ('\u{a73f}',
        '\u{a73f}', SC_Lower), ('\u{a740}', '\u{a740}', SC_Upper), ('\u{a741}', '\u{a741}',
        SC_Lower), ('\u{a742}', '\u{a742}', SC_Upper), ('\u{a743}', '\u{a743}', SC_Lower),
        ('\u{a744}', '\u{a744}', SC_Upper), ('\u{a745}', '\u{a745}', SC_Lower), ('\u{a746}',
        '\u{a746}', SC_Upper), ('\u{a747}', '\u{a747}', SC_Lower), ('\u{a748}', '\u{a748}',
        SC_Upper), ('\u{a749}', '\u{a749}', SC_Lower), ('\u{a74a}', '\u{a74a}', SC_Upper),
        ('\u{a74b}', '\u{a74b}', SC_Lower), ('\u{a74c}', '\u{a74c}', SC_Upper), ('\u{a74d}',
        '\u{a74d}', SC_Lower), ('\u{a74e}', '\u{a74e}', SC_Upper), ('\u{a74f}', '\u{a74f}',
        SC_Lower), ('\u{a750}', '\u{a750}', SC_Upper), ('\u{a751}', '\u{a751}', SC_Lower),
        ('\u{a752}', '\u{a752}', SC_Upper), ('\u{a753}', '\u{a753}', SC_Lower), ('\u{a754}',
        '\u{a754}', SC_Upper), ('\u{a755}', '\u{a755}', SC_Lower), ('\u{a756}', '\u{a756}',
        SC_Upper), ('\u{a757}', '\u{a757}', SC_Lower), ('\u{a758}', '\u{a758}', SC_Upper),
        ('\u{a759}', '\u{a759}', SC_Lower), ('\u{a75a}', '\u{a75a}', SC_Upper), ('\u{a75b}',
        '\u{a75b}', SC_Lower), ('\u{a75c}', '\u{a75c}', SC_Upper), ('\u{a75d}', '\u{a75d}',
        SC_Lower), ('\u{a75e}', '\u{a75e}', SC_Upper), ('\u{a75f}', '\u{a75f}', SC_Lower),
        ('\u{a760}', '\u{a760}', SC_Upper), ('\u{a761}', '\u{a761}', SC_Lower), ('\u{a762}',
        '\u{a762}', SC_Upper), ('\u{a763}', '\u{a763}', SC_Lower), ('\u{a764}', '\u{a764}',
        SC_Upper), ('\u{a765}', '\u{a765}', SC_Lower), ('\u{a766}', '\u{a766}', SC_Upper),
        ('\u{a767}', '\u{a767}', SC_Lower), ('\u{a768}', '\u{a768}', SC_Upper), ('\u{a769}',
        '\u{a769}', SC_Lower), ('\u{a76a}', '\u{a76a}', SC_Upper), ('\u{a76b}', '\u{a76b}',
        SC_Lower), ('\u{a76c}', '\u{a76c}', SC_Upper), ('\u{a76d}', '\u{a76d}', SC_Lower),
        ('\u{a76e}', '\u{a76e}', SC_Upper), ('\u{a76f}', '\u{a778}', SC_Lower), ('\u{a779}',
        '\u{a779}', SC_Upper), ('\u{a77a}', '\u{a77a}', SC_Lower), ('\u{a77b}', '\u{a77b}',
        SC_Upper), ('\u{a77c}', '\u{a77c}', SC_Lower), ('\u{a77d}', '\u{a77e}', SC_Upper),
        ('\u{a77f}', '\u{a77f}', SC_Lower), ('\u{a780}', '\u{a780}', SC_Upper), ('\u{a781}',
        '\u{a781}', SC_Lower), ('\u{a782}', '\u{a782}', SC_Upper), ('\u{a783}', '\u{a783}',
        SC_Lower), ('\u{a784}', '\u{a784}', SC_Upper), ('\u{a785}', '\u{a785}', SC_Lower),
        ('\u{a786}', '\u{a786}', SC_Upper), ('\u{a787}', '\u{a787}', SC_Lower), ('\u{a788}',
        '\u{a788}', SC_OLetter), ('\u{a78b}', '\u{a78b}', SC_Upper), ('\u{a78c}', '\u{a78c}',
        SC_Lower), ('\u{a78d}', '\u{a78d}', SC_Upper), ('\u{a78e}', '\u{a78e}', SC_Lower),
        ('\u{a78f}', '\u{a78f}', SC_OLetter), ('\u{a790}', '\u{a790}', SC_Upper), ('\u{a791}',
        '\u{a791}', SC_Lower), ('\u{a792}', '\u{a792}', SC_Upper), ('\u{a793}', '\u{a795}',
        SC_Lower), ('\u{a796}', '\u{a796}', SC_Upper), ('\u{a797}', '\u{a797}', SC_Lower),
        ('\u{a798}', '\u{a798}', SC_Upper), ('\u{a799}', '\u{a799}', SC_Lower), ('\u{a79a}',
        '\u{a79a}', SC_Upper), ('\u{a79b}', '\u{a79b}', SC_Lower), ('\u{a79c}', '\u{a79c}',
        SC_Upper), ('\u{a79d}', '\u{a79d}', SC_Lower), ('\u{a79e}', '\u{a79e}', SC_Upper),
        ('\u{a79f}', '\u{a79f}', SC_Lower), ('\u{a7a0}', '\u{a7a0}', SC_Upper), ('\u{a7a1}',
        '\u{a7a1}', SC_Lower), ('\u{a7a2}', '\u{a7a2}', SC_Upper), ('\u{a7a3}', '\u{a7a3}',
        SC_Lower), ('\u{a7a4}', '\u{a7a4}', SC_Upper), ('\u{a7a5}', '\u{a7a5}', SC_Lower),
        ('\u{a7a6}', '\u{a7a6}', SC_Upper), ('\u{a7a7}', '\u{a7a7}', SC_Lower), ('\u{a7a8}',
        '\u{a7a8}', SC_Upper), ('\u{a7a9}', '\u{a7a9}', SC_Lower), ('\u{a7aa}', '\u{a7ae}',
        SC_Upper), ('\u{a7b0}', '\u{a7b4}', SC_Upper), ('\u{a7b5}', '\u{a7b5}', SC_Lower),
        ('\u{a7b6}', '\u{a7b6}', SC_Upper), ('\u{a7b7}', '\u{a7b7}', SC_Lower), ('\u{a7f7}',
        '\u{a7f7}', SC_OLetter), ('\u{a7f8}', '\u{a7fa}', SC_Lower), ('\u{a7fb}', '\u{a801}',
        SC_OLetter), ('\u{a802}', '\u{a802}', SC_Extend), ('\u{a803}', '\u{a805}', SC_OLetter),
        ('\u{a806}', '\u{a806}', SC_Extend), ('\u{a807}', '\u{a80a}', SC_OLetter), ('\u{a80b}',
        '\u{a80b}', SC_Extend), ('\u{a80c}', '\u{a822}', SC_OLetter), ('\u{a823}', '\u{a827}',
        SC_Extend), ('\u{a840}', '\u{a873}', SC_OLetter), ('\u{a876}', '\u{a877}', SC_STerm),
        ('\u{a880}', '\u{a881}', SC_Extend), ('\u{a882}', '\u{a8b3}', SC_OLetter), ('\u{a8b4}',
        '\u{a8c5}', SC_Extend), ('\u{a8ce}', '\u{a8cf}', SC_STerm), ('\u{a8d0}', '\u{a8d9}',
        SC_Numeric), ('\u{a8e0}', '\u{a8f1}', SC_Extend), ('\u{a8f2}', '\u{a8f7}', SC_OLetter),
        ('\u{a8fb}', '\u{a8fb}', SC_OLetter), ('\u{a8fd}', '\u{a8fd}', SC_OLetter), ('\u{a900}',
        '\u{a909}', SC_Numeric), ('\u{a90a}', '\u{a925}', SC_OLetter), ('\u{a926}', '\u{a92d}',
        SC_Extend), ('\u{a92f}', '\u{a92f}', SC_STerm), ('\u{a930}', '\u{a946}', SC_OLetter),
        ('\u{a947}', '\u{a953}', SC_Extend), ('\u{a960}', '\u{a97c}', SC_OLetter), ('\u{a980}',
        '\u{a983}', SC_Extend), ('\u{a984}', '\u{a9b2}', SC_OLetter), ('\u{a9b3}', '\u{a9c0}',
        SC_Extend), ('\u{a9c8}', '\u{a9c9}', SC_STerm), ('\u{a9cf}', '\u{a9cf}', SC_OLetter),
        ('\u{a9d0}', '\u{a9d9}', SC_Numeric), ('\u{a9e0}', '\u{a9e4}', SC_OLetter), ('\u{a9e5}',
        '\u{a9e5}', SC_Extend), ('\u{a9e6}', '\u{a9ef}', SC_OLetter), ('\u{a9f0}', '\u{a9f9}',
        SC_Numeric), ('\u{a9fa}', '\u{a9fe}', SC_OLetter), ('\u{aa00}', '\u{aa28}', SC_OLetter),
        ('\u{aa29}', '\u{aa36}', SC_Extend), ('\u{aa40}', '\u{aa42}', SC_OLetter), ('\u{aa43}',
        '\u{aa43}', SC_Extend), ('\u{aa44}', '\u{aa4b}', SC_OLetter), ('\u{aa4c}', '\u{aa4d}',
        SC_Extend), ('\u{aa50}', '\u{aa59}', SC_Numeric), ('\u{aa5d}', '\u{aa5f}', SC_STerm),
        ('\u{aa60}', '\u{aa76}', SC_OLetter), ('\u{aa7a}', '\u{aa7a}', SC_OLetter), ('\u{aa7b}',
        '\u{aa7d}', SC_Extend), ('\u{aa7e}', '\u{aaaf}', SC_OLetter), ('\u{aab0}', '\u{aab0}',
        SC_Extend), ('\u{aab1}', '\u{aab1}', SC_OLetter), ('\u{aab2}', '\u{aab4}', SC_Extend),
        ('\u{aab5}', '\u{aab6}', SC_OLetter), ('\u{aab7}', '\u{aab8}', SC_Extend), ('\u{aab9}',
        '\u{aabd}', SC_OLetter), ('\u{aabe}', '\u{aabf}', SC_Extend), ('\u{aac0}', '\u{aac0}',
        SC_OLetter), ('\u{aac1}', '\u{aac1}', SC_Extend), ('\u{aac2}', '\u{aac2}', SC_OLetter),
        ('\u{aadb}', '\u{aadd}', SC_OLetter), ('\u{aae0}', '\u{aaea}', SC_OLetter), ('\u{aaeb}',
        '\u{aaef}', SC_Extend), ('\u{aaf0}', '\u{aaf1}', SC_STerm), ('\u{aaf2}', '\u{aaf4}',
        SC_OLetter), ('\u{aaf5}', '\u{aaf6}', SC_Extend), ('\u{ab01}', '\u{ab06}', SC_OLetter),
        ('\u{ab09}', '\u{ab0e}', SC_OLetter), ('\u{ab11}', '\u{ab16}', SC_OLetter), ('\u{ab20}',
        '\u{ab26}', SC_OLetter), ('\u{ab28}', '\u{ab2e}', SC_OLetter), ('\u{ab30}', '\u{ab5a}',
        SC_Lower), ('\u{ab5c}', '\u{ab65}', SC_Lower), ('\u{ab70}', '\u{abbf}', SC_Lower),
        ('\u{abc0}', '\u{abe2}', SC_OLetter), ('\u{abe3}', '\u{abea}', SC_Extend), ('\u{abeb}',
        '\u{abeb}', SC_STerm), ('\u{abec}', '\u{abed}', SC_Extend), ('\u{abf0}', '\u{abf9}',
        SC_Numeric), ('\u{ac00}', '\u{d7a3}', SC_OLetter), ('\u{d7b0}', '\u{d7c6}', SC_OLetter),
        ('\u{d7cb}', '\u{d7fb}', SC_OLetter), ('\u{f900}', '\u{fa6d}', SC_OLetter), ('\u{fa70}',
        '\u{fad9}', SC_OLetter), ('\u{fb00}', '\u{fb06}', SC_Lower), ('\u{fb13}', '\u{fb17}',
        SC_Lower), ('\u{fb1d}', '\u{fb1d}', SC_OLetter), ('\u{fb1e}', '\u{fb1e}', SC_Extend),
        ('\u{fb1f}', '\u{fb28}', SC_OLetter), ('\u{fb2a}', '\u{fb36}', SC_OLetter), ('\u{fb38}',
        '\u{fb3c}', SC_OLetter), ('\u{fb3e}', '\u{fb3e}', SC_OLetter), ('\u{fb40}', '\u{fb41}',
        SC_OLetter), ('\u{fb43}', '\u{fb44}', SC_OLetter), ('\u{fb46}', '\u{fbb1}', SC_OLetter),
        ('\u{fbd3}', '\u{fd3d}', SC_OLetter), ('\u{fd3e}', '\u{fd3f}', SC_Close), ('\u{fd50}',
        '\u{fd8f}', SC_OLetter), ('\u{fd92}', '\u{fdc7}', SC_OLetter), ('\u{fdf0}', '\u{fdfb}',
        SC_OLetter), ('\u{fe00}', '\u{fe0f}', SC_Extend), ('\u{fe10}', '\u{fe11}', SC_SContinue),
        ('\u{fe13}', '\u{fe13}', SC_SContinue), ('\u{fe17}', '\u{fe18}', SC_Close), ('\u{fe20}',
        '\u{fe2f}', SC_Extend), ('\u{fe31}', '\u{fe32}', SC_SContinue), ('\u{fe35}', '\u{fe44}',
        SC_Close), ('\u{fe47}', '\u{fe48}', SC_Close), ('\u{fe50}', '\u{fe51}', SC_SContinue),
        ('\u{fe52}', '\u{fe52}', SC_ATerm), ('\u{fe55}', '\u{fe55}', SC_SContinue), ('\u{fe56}',
        '\u{fe57}', SC_STerm), ('\u{fe58}', '\u{fe58}', SC_SContinue), ('\u{fe59}', '\u{fe5e}',
        SC_Close), ('\u{fe63}', '\u{fe63}', SC_SContinue), ('\u{fe70}', '\u{fe74}', SC_OLetter),
        ('\u{fe76}', '\u{fefc}', SC_OLetter), ('\u{feff}', '\u{feff}', SC_Format), ('\u{ff01}',
        '\u{ff01}', SC_STerm), ('\u{ff08}', '\u{ff09}', SC_Close), ('\u{ff0c}', '\u{ff0d}',
        SC_SContinue), ('\u{ff0e}', '\u{ff0e}', SC_ATerm), ('\u{ff1a}', '\u{ff1a}', SC_SContinue),
        ('\u{ff1f}', '\u{ff1f}', SC_STerm), ('\u{ff21}', '\u{ff3a}', SC_Upper), ('\u{ff3b}',
        '\u{ff3b}', SC_Close), ('\u{ff3d}', '\u{ff3d}', SC_Close), ('\u{ff41}', '\u{ff5a}',
        SC_Lower), ('\u{ff5b}', '\u{ff5b}', SC_Close), ('\u{ff5d}', '\u{ff5d}', SC_Close),
        ('\u{ff5f}', '\u{ff60}', SC_Close), ('\u{ff61}', '\u{ff61}', SC_STerm), ('\u{ff62}',
        '\u{ff63}', SC_Close), ('\u{ff64}', '\u{ff64}', SC_SContinue), ('\u{ff66}', '\u{ff9d}',
        SC_OLetter), ('\u{ff9e}', '\u{ff9f}', SC_Extend), ('\u{ffa0}', '\u{ffbe}', SC_OLetter),
        ('\u{ffc2}', '\u{ffc7}', SC_OLetter), ('\u{ffca}', '\u{ffcf}', SC_OLetter), ('\u{ffd2}',
        '\u{ffd7}', SC_OLetter), ('\u{ffda}', '\u{ffdc}', SC_OLetter), ('\u{fff9}', '\u{fffb}',
        SC_Format), ('\u{10000}', '\u{1000b}', SC_OLetter), ('\u{1000d}', '\u{10026}', SC_OLetter),
        ('\u{10028}', '\u{1003a}', SC_OLetter), ('\u{1003c}', '\u{1003d}', SC_OLetter),
        ('\u{1003f}', '\u{1004d}', SC_OLetter), ('\u{10050}', '\u{1005d}', SC_OLetter),
        ('\u{10080}', '\u{100fa}', SC_OLetter), ('\u{10140}', '\u{10174}', SC_OLetter),
        ('\u{101fd}', '\u{101fd}', SC_Extend), ('\u{10280}', '\u{1029c}', SC_OLetter), ('\u{102a0}',
        '\u{102d0}', SC_OLetter), ('\u{102e0}', '\u{102e0}', SC_Extend), ('\u{10300}', '\u{1031f}',
        SC_OLetter), ('\u{10330}', '\u{1034a}', SC_OLetter), ('\u{10350}', '\u{10375}', SC_OLetter),
        ('\u{10376}', '\u{1037a}', SC_Extend), ('\u{10380}', '\u{1039d}', SC_OLetter), ('\u{103a0}',
        '\u{103c3}', SC_OLetter), ('\u{103c8}', '\u{103cf}', SC_OLetter), ('\u{103d1}', '\u{103d5}',
        SC_OLetter), ('\u{10400}', '\u{10427}', SC_Upper), ('\u{10428}', '\u{1044f}', SC_Lower),
        ('\u{10450}', '\u{1049d}', SC_OLetter), ('\u{104a0}', '\u{104a9}', SC_Numeric),
        ('\u{104b0}', '\u{104d3}', SC_Upper), ('\u{104d8}', '\u{104fb}', SC_Lower), ('\u{10500}',
        '\u{10527}', SC_OLetter), ('\u{10530}', '\u{10563}', SC_OLetter), ('\u{10600}', '\u{10736}',
        SC_OLetter), ('\u{10740}', '\u{10755}', SC_OLetter), ('\u{10760}', '\u{10767}', SC_OLetter),
        ('\u{10800}', '\u{10805}', SC_OLetter), ('\u{10808}', '\u{10808}', SC_OLetter),
        ('\u{1080a}', '\u{10835}', SC_OLetter), ('\u{10837}', '\u{10838}', SC_OLetter),
        ('\u{1083c}', '\u{1083c}', SC_OLetter), ('\u{1083f}', '\u{10855}', SC_OLetter),
        ('\u{10860}', '\u{10876}', SC_OLetter), ('\u{10880}', '\u{1089e}', SC_OLetter),
        ('\u{108e0}', '\u{108f2}', SC_OLetter), ('\u{108f4}', '\u{108f5}', SC_OLetter),
        ('\u{10900}', '\u{10915}', SC_OLetter), ('\u{10920}', '\u{10939}', SC_OLetter),
        ('\u{10980}', '\u{109b7}', SC_OLetter), ('\u{109be}', '\u{109bf}', SC_OLetter),
        ('\u{10a00}', '\u{10a00}', SC_OLetter), ('\u{10a01}', '\u{10a03}', SC_Extend), ('\u{10a05}',
        '\u{10a06}', SC_Extend), ('\u{10a0c}', '\u{10a0f}', SC_Extend), ('\u{10a10}', '\u{10a13}',
        SC_OLetter), ('\u{10a15}', '\u{10a17}', SC_OLetter), ('\u{10a19}', '\u{10a33}', SC_OLetter),
        ('\u{10a38}', '\u{10a3a}', SC_Extend), ('\u{10a3f}', '\u{10a3f}', SC_Extend), ('\u{10a56}',
        '\u{10a57}', SC_STerm), ('\u{10a60}', '\u{10a7c}', SC_OLetter), ('\u{10a80}', '\u{10a9c}',
        SC_OLetter), ('\u{10ac0}', '\u{10ac7}', SC_OLetter), ('\u{10ac9}', '\u{10ae4}', SC_OLetter),
        ('\u{10ae5}', '\u{10ae6}', SC_Extend), ('\u{10b00}', '\u{10b35}', SC_OLetter), ('\u{10b40}',
        '\u{10b55}', SC_OLetter), ('\u{10b60}', '\u{10b72}', SC_OLetter), ('\u{10b80}', '\u{10b91}',
        SC_OLetter), ('\u{10c00}', '\u{10c48}', SC_OLetter), ('\u{10c80}', '\u{10cb2}', SC_Upper),
        ('\u{10cc0}', '\u{10cf2}', SC_Lower), ('\u{11000}', '\u{11002}', SC_Extend), ('\u{11003}',
        '\u{11037}', SC_OLetter), ('\u{11038}', '\u{11046}', SC_Extend), ('\u{11047}', '\u{11048}',
        SC_STerm), ('\u{11066}', '\u{1106f}', SC_Numeric), ('\u{1107f}', '\u{11082}', SC_Extend),
        ('\u{11083}', '\u{110af}', SC_OLetter), ('\u{110b0}', '\u{110ba}', SC_Extend), ('\u{110bd}',
        '\u{110bd}', SC_Format), ('\u{110be}', '\u{110c1}', SC_STerm), ('\u{110d0}', '\u{110e8}',
        SC_OLetter), ('\u{110f0}', '\u{110f9}', SC_Numeric), ('\u{11100}', '\u{11102}', SC_Extend),
        ('\u{11103}', '\u{11126}', SC_OLetter), ('\u{11127}', '\u{11134}', SC_Extend), ('\u{11136}',
        '\u{1113f}', SC_Numeric), ('\u{11141}', '\u{11143}', SC_STerm), ('\u{11150}', '\u{11172}',
        SC_OLetter), ('\u{11173}', '\u{11173}', SC_Extend), ('\u{11176}', '\u{11176}', SC_OLetter),
        ('\u{11180}', '\u{11182}', SC_Extend), ('\u{11183}', '\u{111b2}', SC_OLetter), ('\u{111b3}',
        '\u{111c0}', SC_Extend), ('\u{111c1}', '\u{111c4}', SC_OLetter), ('\u{111c5}', '\u{111c6}',
        SC_STerm), ('\u{111ca}', '\u{111cc}', SC_Extend), ('\u{111cd}', '\u{111cd}', SC_STerm),
        ('\u{111d0}', '\u{111d9}', SC_Numeric), ('\u{111da}', '\u{111da}', SC_OLetter),
        ('\u{111dc}', '\u{111dc}', SC_OLetter), ('\u{111de}', '\u{111df}', SC_STerm), ('\u{11200}',
        '\u{11211}', SC_OLetter), ('\u{11213}', '\u{1122b}', SC_OLetter), ('\u{1122c}', '\u{11237}',
        SC_Extend), ('\u{11238}', '\u{11239}', SC_STerm), ('\u{1123b}', '\u{1123c}', SC_STerm),
        ('\u{1123e}', '\u{1123e}', SC_Extend), ('\u{11280}', '\u{11286}', SC_OLetter), ('\u{11288}',
        '\u{11288}', SC_OLetter), ('\u{1128a}', '\u{1128d}', SC_OLetter), ('\u{1128f}', '\u{1129d}',
        SC_OLetter), ('\u{1129f}', '\u{112a8}', SC_OLetter), ('\u{112a9}', '\u{112a9}', SC_STerm),
        ('\u{112b0}', '\u{112de}', SC_OLetter), ('\u{112df}', '\u{112ea}', SC_Extend), ('\u{112f0}',
        '\u{112f9}', SC_Numeric), ('\u{11300}', '\u{11303}', SC_Extend), ('\u{11305}', '\u{1130c}',
        SC_OLetter), ('\u{1130f}', '\u{11310}', SC_OLetter), ('\u{11313}', '\u{11328}', SC_OLetter),
        ('\u{1132a}', '\u{11330}', SC_OLetter), ('\u{11332}', '\u{11333}', SC_OLetter),
        ('\u{11335}', '\u{11339}', SC_OLetter), ('\u{1133c}', '\u{1133c}', SC_Extend), ('\u{1133d}',
        '\u{1133d}', SC_OLetter), ('\u{1133e}', '\u{11344}', SC_Extend), ('\u{11347}', '\u{11348}',
        SC_Extend), ('\u{1134b}', '\u{1134d}', SC_Extend), ('\u{11350}', '\u{11350}', SC_OLetter),
        ('\u{11357}', '\u{11357}', SC_Extend), ('\u{1135d}', '\u{11361}', SC_OLetter), ('\u{11362}',
        '\u{11363}', SC_Extend), ('\u{11366}', '\u{1136c}', SC_Extend), ('\u{11370}', '\u{11374}',
        SC_Extend), ('\u{11400}', '\u{11434}', SC_OLetter), ('\u{11435}', '\u{11446}', SC_Extend),
        ('\u{11447}', '\u{1144a}', SC_OLetter), ('\u{1144b}', '\u{1144c}', SC_STerm), ('\u{11450}',
        '\u{11459}', SC_Numeric), ('\u{11480}', '\u{114af}', SC_OLetter), ('\u{114b0}', '\u{114c3}',
        SC_Extend), ('\u{114c4}', '\u{114c5}', SC_OLetter), ('\u{114c7}', '\u{114c7}', SC_OLetter),
        ('\u{114d0}', '\u{114d9}', SC_Numeric), ('\u{11580}', '\u{115ae}', SC_OLetter),
        ('\u{115af}', '\u{115b5}', SC_Extend), ('\u{115b8}', '\u{115c0}', SC_Extend), ('\u{115c2}',
        '\u{115c3}', SC_STerm), ('\u{115c9}', '\u{115d7}', SC_STerm), ('\u{115d8}', '\u{115db}',
        SC_OLetter), ('\u{115dc}', '\u{115dd}', SC_Extend), ('\u{11600}', '\u{1162f}', SC_OLetter),
        ('\u{11630}', '\u{11640}', SC_Extend), ('\u{11641}', '\u{11642}', SC_STerm), ('\u{11644}',
        '\u{11644}', SC_OLetter), ('\u{11650}', '\u{11659}', SC_Numeric), ('\u{11680}', '\u{116aa}',
        SC_OLetter), ('\u{116ab}', '\u{116b7}', SC_Extend), ('\u{116c0}', '\u{116c9}', SC_Numeric),
        ('\u{11700}', '\u{11719}', SC_OLetter), ('\u{1171d}', '\u{1172b}', SC_Extend), ('\u{11730}',
        '\u{11739}', SC_Numeric), ('\u{1173c}', '\u{1173e}', SC_STerm), ('\u{118a0}', '\u{118bf}',
        SC_Upper), ('\u{118c0}', '\u{118df}', SC_Lower), ('\u{118e0}', '\u{118e9}', SC_Numeric),
        ('\u{118ff}', '\u{118ff}', SC_OLetter), ('\u{11ac0}', '\u{11af8}', SC_OLetter),
        ('\u{11c00}', '\u{11c08}', SC_OLetter), ('\u{11c0a}', '\u{11c2e}', SC_OLetter),
        ('\u{11c2f}', '\u{11c36}', SC_Extend), ('\u{11c38}', '\u{11c3f}', SC_Extend), ('\u{11c40}',
        '\u{11c40}', SC_OLetter), ('\u{11c41}', '\u{11c42}', SC_STerm), ('\u{11c50}', '\u{11c59}',
        SC_Numeric), ('\u{11c72}', '\u{11c8f}', SC_OLetter), ('\u{11c92}', '\u{11ca7}', SC_Extend),
        ('\u{11ca9}', '\u{11cb6}', SC_Extend), ('\u{12000}', '\u{12399}', SC_OLetter), ('\u{12400}',
        '\u{1246e}', SC_OLetter), ('\u{12480}', '\u{12543}', SC_OLetter), ('\u{13000}', '\u{1342e}',
        SC_OLetter), ('\u{14400}', '\u{14646}', SC_OLetter), ('\u{16800}', '\u{16a38}', SC_OLetter),
        ('\u{16a40}', '\u{16a5e}', SC_OLetter), ('\u{16a60}', '\u{16a69}', SC_Numeric),
        ('\u{16a6e}', '\u{16a6f}', SC_STerm), ('\u{16ad0}', '\u{16aed}', SC_OLetter), ('\u{16af0}',
        '\u{16af4}', SC_Extend), ('\u{16af5}', '\u{16af5}', SC_STerm), ('\u{16b00}', '\u{16b2f}',
        SC_OLetter), ('\u{16b30}', '\u{16b36}', SC_Extend), ('\u{16b37}', '\u{16b38}', SC_STerm),
        ('\u{16b40}', '\u{16b43}', SC_OLetter), ('\u{16b44}', '\u{16b44}', SC_STerm), ('\u{16b50}',
        '\u{16b59}', SC_Numeric), ('\u{16b63}', '\u{16b77}', SC_OLetter), ('\u{16b7d}', '\u{16b8f}',
        SC_OLetter), ('\u{16f00}', '\u{16f44}', SC_OLetter), ('\u{16f50}', '\u{16f50}', SC_OLetter),
        ('\u{16f51}', '\u{16f7e}', SC_Extend), ('\u{16f8f}', '\u{16f92}', SC_Extend), ('\u{16f93}',
        '\u{16f9f}', SC_OLetter), ('\u{16fe0}', '\u{16fe0}', SC_OLetter), ('\u{17000}', '\u{187ec}',
        SC_OLetter), ('\u{18800}', '\u{18af2}', SC_OLetter), ('\u{1b000}', '\u{1b001}', SC_OLetter),
        ('\u{1bc00}', '\u{1bc6a}', SC_OLetter), ('\u{1bc70}', '\u{1bc7c}', SC_OLetter),
        ('\u{1bc80}', '\u{1bc88}', SC_OLetter), ('\u{1bc90}', '\u{1bc99}', SC_OLetter),
        ('\u{1bc9d}', '\u{1bc9e}', SC_Extend), ('\u{1bc9f}', '\u{1bc9f}', SC_STerm), ('\u{1bca0}',
        '\u{1bca3}', SC_Format), ('\u{1d165}', '\u{1d169}', SC_Extend), ('\u{1d16d}', '\u{1d172}',
        SC_Extend), ('\u{1d173}', '\u{1d17a}', SC_Format), ('\u{1d17b}', '\u{1d182}', SC_Extend),
        ('\u{1d185}', '\u{1d18b}', SC_Extend), ('\u{1d1aa}', '\u{1d1ad}', SC_Extend), ('\u{1d242}',
        '\u{1d244}', SC_Extend), ('\u{1d400}', '\u{1d419}', SC_Upper), ('\u{1d41a}', '\u{1d433}',
        SC_Lower), ('\u{1d434}', '\u{1d44d}', SC_Upper), ('\u{1d44e}', '\u{1d454}', SC_Lower),
        ('\u{1d456}', '\u{1d467}', SC_Lower), ('\u{1d468}', '\u{1d481}', SC_Upper), ('\u{1d482}',
        '\u{1d49b}', SC_Lower), ('\u{1d49c}', '\u{1d49c}', SC_Upper), ('\u{1d49e}', '\u{1d49f}',
        SC_Upper), ('\u{1d4a2}', '\u{1d4a2}', SC_Upper), ('\u{1d4a5}', '\u{1d4a6}', SC_Upper),
        ('\u{1d4a9}', '\u{1d4ac}', SC_Upper), ('\u{1d4ae}', '\u{1d4b5}', SC_Upper), ('\u{1d4b6}',
        '\u{1d4b9}', SC_Lower), ('\u{1d4bb}', '\u{1d4bb}', SC_Lower), ('\u{1d4bd}', '\u{1d4c3}',
        SC_Lower), ('\u{1d4c5}', '\u{1d4cf}', SC_Lower), ('\u{1d4d0}', '\u{1d4e9}', SC_Upper),
        ('\u{1d4ea}', '\u{1d503}', SC_Lower), ('\u{1d504}', '\u{1d505}', SC_Upper), ('\u{1d507}',
        '\u{1d50a}', SC_Upper), ('\u{1d50d}', '\u{1d514}', SC_Upper), ('\u{1d516}', '\u{1d51c}',
        SC_Upper), ('\u{1d51e}', '\u{1d537}', SC_Lower), ('\u{1d538}', '\u{1d539}', SC_Upper),
        ('\u{1d53b}', '\u{1d53e}', SC_Upper), ('\u{1d540}', '\u{1d544}', SC_Upper), ('\u{1d546}',
        '\u{1d546}', SC_Upper), ('\u{1d54a}', '\u{1d550}', SC_Upper), ('\u{1d552}', '\u{1d56b}',
        SC_Lower), ('\u{1d56c}', '\u{1d585}', SC_Upper), ('\u{1d586}', '\u{1d59f}', SC_Lower),
        ('\u{1d5a0}', '\u{1d5b9}', SC_Upper), ('\u{1d5ba}', '\u{1d5d3}', SC_Lower), ('\u{1d5d4}',
        '\u{1d5ed}', SC_Upper), ('\u{1d5ee}', '\u{1d607}', SC_Lower), ('\u{1d608}', '\u{1d621}',
        SC_Upper), ('\u{1d622}', '\u{1d63b}', SC_Lower), ('\u{1d63c}', '\u{1d655}', SC_Upper),
        ('\u{1d656}', '\u{1d66f}', SC_Lower), ('\u{1d670}', '\u{1d689}', SC_Upper), ('\u{1d68a}',
        '\u{1d6a5}', SC_Lower), ('\u{1d6a8}', '\u{1d6c0}', SC_Upper), ('\u{1d6c2}', '\u{1d6da}',
        SC_Lower), ('\u{1d6dc}', '\u{1d6e1}', SC_Lower), ('\u{1d6e2}', '\u{1d6fa}', SC_Upper),
        ('\u{1d6fc}', '\u{1d714}', SC_Lower), ('\u{1d716}', '\u{1d71b}', SC_Lower), ('\u{1d71c}',
        '\u{1d734}', SC_Upper), ('\u{1d736}', '\u{1d74e}', SC_Lower), ('\u{1d750}', '\u{1d755}',
        SC_Lower), ('\u{1d756}', '\u{1d76e}', SC_Upper), ('\u{1d770}', '\u{1d788}', SC_Lower),
        ('\u{1d78a}', '\u{1d78f}', SC_Lower), ('\u{1d790}', '\u{1d7a8}', SC_Upper), ('\u{1d7aa}',
        '\u{1d7c2}', SC_Lower), ('\u{1d7c4}', '\u{1d7c9}', SC_Lower), ('\u{1d7ca}', '\u{1d7ca}',
        SC_Upper), ('\u{1d7cb}', '\u{1d7cb}', SC_Lower), ('\u{1d7ce}', '\u{1d7ff}', SC_Numeric),
        ('\u{1da00}', '\u{1da36}', SC_Extend), ('\u{1da3b}', '\u{1da6c}', SC_Extend), ('\u{1da75}',
        '\u{1da75}', SC_Extend), ('\u{1da84}', '\u{1da84}', SC_Extend), ('\u{1da88}', '\u{1da88}',
        SC_STerm), ('\u{1da9b}', '\u{1da9f}', SC_Extend), ('\u{1daa1}', '\u{1daaf}', SC_Extend),
        ('\u{1e000}', '\u{1e006}', SC_Extend), ('\u{1e008}', '\u{1e018}', SC_Extend), ('\u{1e01b}',
        '\u{1e021}', SC_Extend), ('\u{1e023}', '\u{1e024}', SC_Extend), ('\u{1e026}', '\u{1e02a}',
        SC_Extend), ('\u{1e800}', '\u{1e8c4}', SC_OLetter), ('\u{1e8d0}', '\u{1e8d6}', SC_Extend),
        ('\u{1e900}', '\u{1e921}', SC_Upper), ('\u{1e922}', '\u{1e943}', SC_Lower), ('\u{1e944}',
        '\u{1e94a}', SC_Extend), ('\u{1e950}', '\u{1e959}', SC_Numeric), ('\u{1ee00}', '\u{1ee03}',
        SC_OLetter), ('\u{1ee05}', '\u{1ee1f}', SC_OLetter), ('\u{1ee21}', '\u{1ee22}', SC_OLetter),
        ('\u{1ee24}', '\u{1ee24}', SC_OLetter), ('\u{1ee27}', '\u{1ee27}', SC_OLetter),
        ('\u{1ee29}', '\u{1ee32}', SC_OLetter), ('\u{1ee34}', '\u{1ee37}', SC_OLetter),
        ('\u{1ee39}', '\u{1ee39}', SC_OLetter), ('\u{1ee3b}', '\u{1ee3b}', SC_OLetter),
        ('\u{1ee42}', '\u{1ee42}', SC_OLetter), ('\u{1ee47}', '\u{1ee47}', SC_OLetter),
        ('\u{1ee49}', '\u{1ee49}', SC_OLetter), ('\u{1ee4b}', '\u{1ee4b}', SC_OLetter),
        ('\u{1ee4d}', '\u{1ee4f}', SC_OLetter), ('\u{1ee51}', '\u{1ee52}', SC_OLetter),
        ('\u{1ee54}', '\u{1ee54}', SC_OLetter), ('\u{1ee57}', '\u{1ee57}', SC_OLetter),
        ('\u{1ee59}', '\u{1ee59}', SC_OLetter), ('\u{1ee5b}', '\u{1ee5b}', SC_OLetter),
        ('\u{1ee5d}', '\u{1ee5d}', SC_OLetter), ('\u{1ee5f}', '\u{1ee5f}', SC_OLetter),
        ('\u{1ee61}', '\u{1ee62}', SC_OLetter), ('\u{1ee64}', '\u{1ee64}', SC_OLetter),
        ('\u{1ee67}', '\u{1ee6a}', SC_OLetter), ('\u{1ee6c}', '\u{1ee72}', SC_OLetter),
        ('\u{1ee74}', '\u{1ee77}', SC_OLetter), ('\u{1ee79}', '\u{1ee7c}', SC_OLetter),
        ('\u{1ee7e}', '\u{1ee7e}', SC_OLetter), ('\u{1ee80}', '\u{1ee89}', SC_OLetter),
        ('\u{1ee8b}', '\u{1ee9b}', SC_OLetter), ('\u{1eea1}', '\u{1eea3}', SC_OLetter),
        ('\u{1eea5}', '\u{1eea9}', SC_OLetter), ('\u{1eeab}', '\u{1eebb}', SC_OLetter),
        ('\u{1f130}', '\u{1f149}', SC_Upper), ('\u{1f150}', '\u{1f169}', SC_Upper), ('\u{1f170}',
        '\u{1f189}', SC_Upper), ('\u{1f676}', '\u{1f678}', SC_Close), ('\u{20000}', '\u{2a6d6}',
        SC_OLetter), ('\u{2a700}', '\u{2b734}', SC_OLetter), ('\u{2b740}', '\u{2b81d}', SC_OLetter),
        ('\u{2b820}', '\u{2cea1}', SC_OLetter), ('\u{2f800}', '\u{2fa1d}', SC_OLetter),
        ('\u{e0001}', '\u{e0001}', SC_Format), ('\u{e0020}', '\u{e007f}', SC_Extend), ('\u{e0100}',
        '\u{e01ef}', SC_Extend)
    ];

}
