//! Ed25519: Schnorr signatures using the twisted Edwards form of Curve25519
//!
//! Described in RFC 8032: <https://tools.ietf.org/html/rfc8032>
//!
//! This module contains two convenience methods for signing and verifying
//! Ed25519 signatures which work with any signer or verifier.
//!
//! # Example (with ed25519-dalek)
//!
//! ```nobuild
//! # Sample Test case for Signing and Verifying a digital signature using the *signatory* crate
//!
//! The Signatory crates provides a unified API for accessing various crypto
//! providers for digital signatures.
//!
//! Signatory provides a thread-safe and object-safe API and implements providers
//! for many popular Rust crates, including [ed25519‑dalek], [*ring*], [secp256k1],
//! and [sodiumoxide].
//!
//! This example uses a unique private key generated by the ed25519 crate, signs a
//! piece of text using the Signer function from the signature crate, and then
//! verifies the signature using the corresponding public key.
//!
//! ```
//! [dependencies]
//! base64 = "*"
//! signatory-dalek = "0.20.0"
//! signatory = "0.20.0"
//!
//! sha2 = "0.9.0"
//! hmac = "0.8.0"
//! ```
//!
//! ```
//! use signatory_dalek::{Ed25519Signer, Ed25519Verifier};
//!
//! use signatory::{
//!     ed25519, public_key::PublicKeyed, signature::{Signature, Signer, Verifier},
//! };
//!
//! ```
//! ```
//!     #[test]
//!     fn sign_verify() {
//!         let ed25519_private_key = ed25519::Seed::generate();
//!
//!         let signer = Ed25519Signer::from(&ed25519_private_key);
//!
//!         let msg = "How are you?";
//!         let sig = Signer::sign(&signer, msg.as_bytes());
//!
//!         let pk = PublicKeyed::public_key(&signer).unwrap();
//!         let verifier = Ed25519Verifier::from(&pk);
//!         assert!(Verifier::verify(&verifier, msg.as_bytes(), &sig).is_ok());
//!     }
//! ```

mod public_key;
mod seed;

#[cfg(feature = "test-vectors")]
#[macro_use]
mod test_macros;

/// RFC 8032 Ed25519 test vectors
#[cfg(feature = "test-vectors")]
mod test_vectors;

#[cfg(feature = "test-vectors")]
pub use self::test_vectors::TEST_VECTORS;
pub use self::{
    public_key::{PublicKey, PUBLIC_KEY_SIZE},
    seed::{Seed, SEED_SIZE},
};

// Import `Signature` type from the `ed25519` crate
pub use ed25519::{Signature, SIGNATURE_LENGTH as SIGNATURE_SIZE};
