package openshift

import (
	"os"
	"path/filepath"

	"github.com/openshift/installer/pkg/asset"
	"github.com/openshift/installer/pkg/asset/templates/content"
)

const (
	azureCloudProviderSecretFileName                  = "99_azure-cloud-provider-secret.yaml.template"
	azureCloudProviderSecretGetterRoleBindingFileName = "99_azure-cloud-provider-secret-getter-rolebinding.yaml"
	azureCloudProviderSecretGetterRoleFileName        = "99_azure-cloud-provider-secret-getter-role.yaml"
)

var _ asset.WritableAsset = (*AzureCloudProviderSecret)(nil)

// AzureCloudProviderSecret is the variable to represent contents of corresponding files
type AzureCloudProviderSecret struct {
	FileList []*asset.File
}

// Dependencies returns all of the dependencies directly needed by the asset
func (t *AzureCloudProviderSecret) Dependencies() []asset.Asset {
	return []asset.Asset{}
}

// Name returns the human-friendly name of the asset.
func (t *AzureCloudProviderSecret) Name() string {
	return "AzureCloudProviderSecret"
}

// Generate generates the actual files by this asset
func (t *AzureCloudProviderSecret) Generate(parents asset.Parents) error {
	t.FileList = []*asset.File{}

	for _, fileName := range []string{
		azureCloudProviderSecretFileName,
		azureCloudProviderSecretGetterRoleBindingFileName,
		azureCloudProviderSecretGetterRoleFileName,
	} {
		data, err := content.GetOpenshiftTemplate(fileName)
		if err != nil {
			return err
		}
		t.FileList = append(t.FileList, &asset.File{
			Filename: filepath.Join(content.TemplateDir, fileName),
			Data:     []byte(data),
		})
	}
	return nil
}

// Files returns the files generated by the asset.
func (t *AzureCloudProviderSecret) Files() []*asset.File {
	return t.FileList
}

// Load returns the asset from disk.
func (t *AzureCloudProviderSecret) Load(f asset.FileFetcher) (bool, error) {
	t.FileList = []*asset.File{}

	for _, fileName := range []string{
		azureCloudProviderSecretFileName,
		azureCloudProviderSecretGetterRoleBindingFileName,
		azureCloudProviderSecretGetterRoleFileName,
	} {
		file, err := f.FetchByName(filepath.Join(content.TemplateDir, fileName))
		if err != nil {
			if os.IsNotExist(err) {
				return false, nil
			}
			return false, err
		}
		t.FileList = append(t.FileList, file)
	}

	return true, nil
}
