/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { notStrictEqual, strictEqual, ok } from 'assert';
import { CLI, Common, ReplExpect, Selectors } from '@kui-shell/test';
/** The actual split terminal via button impl; splitViaButton is the mocha test wrapper */
export function doSplitViaButton(ctx, splitCount) {
    return __awaiter(this, void 0, void 0, function* () {
        yield ctx.app.client.$(Selectors.NEW_SPLIT_BUTTON).then(_ => _.click());
        yield ReplExpect.splitCount(splitCount)(ctx.app);
        yield ctx.app.client.waitUntil(() => ctx.app.client['isActive'](Selectors.CURRENT_PROMPT_FOR_SPLIT(splitCount)), {
            timeout: CLI.waitTimeout
        });
    });
}
/** Split the terminal in the current tab by using the split button */
export function splitViaButton(splitCount) {
    it(`should split the terminal via button in the current tab and expect splitCount=${splitCount}`, () => __awaiter(this, void 0, void 0, function* () {
        return doSplitViaButton(this, splitCount).catch(Common.oops(this, true));
    }));
}
/** Split the terminal in the current tab by using the "split" command */
export function splitViaCommand(splitCount, expectErr = false, inverseColors = false, where) {
    it(`should split the terminal via command in the current tab and expect splitCount=${splitCount}`, () => __awaiter(this, void 0, void 0, function* () {
        try {
            // if we were asked to splice in at a specific position, we will
            // need to remember the split ids before... and check this
            // against the id array after (which we do, just below **)
            const splitIdsBefore = where === undefined
                ? undefined
                : yield this.app.client
                    .$$(Selectors.SPLITS)
                    .then(elements => Promise.all(elements.map(_ => _.getAttribute(Selectors.SPLIT_ID))));
            if (where) {
                console.error('before', splitIdsBefore);
            }
            const messageShouldAppearHere = where === undefined ? undefined : where.messageShouldAppearHere;
            yield CLI.commandInSplit(`split ${inverseColors ? ' --inverse' : ''} ${where !== undefined ? ` --index ${where.spliceIndex}` : ''}`, this.app, splitCount - 1)
                .then(expectErr
                ? ReplExpect.error(500)
                : ReplExpect.elsewhere(inverseColors ? 'Created a split with inverted colors' : 'Created a split', messageShouldAppearHere))
                .then(ReplExpect.splitCount(splitCount, inverseColors, messageShouldAppearHere));
            if (where !== undefined) {
                // ** now we check that the new split was spliced in at the expected location
                const splitIdsAfter = yield this.app.client
                    .$$(Selectors.SPLITS)
                    .then(elements => Promise.all(elements.map(_ => _.getAttribute(Selectors.SPLIT_ID))));
                const before = !splitIdsBefore ? [] : splitIdsBefore;
                const after = !splitIdsAfter ? [] : splitIdsAfter;
                console.error('after', splitIdsAfter);
                strictEqual(after.length, before.length + 1, 'expect one more split');
                ok(before.every(id => after.indexOf(id) >= 0), 'all previous splits still exist');
                strictEqual(before.indexOf(after[where.spliceIndex]), -1, 'new split id should not be found in before list');
            }
        }
        catch (err) {
            yield Common.oops(this, true)(err);
        }
    }));
}
/** Close the split in the current tab by using the "exit" command */
export function close(splitCount, inSplit) {
    it(`should close the split via command in the current tab and expect splitCount=${splitCount}`, () => CLI.commandInSplit('exit', this.app, inSplit)
        .then(() => ReplExpect.splitCount(splitCount)(this.app))
        .catch(Common.oops(this, true)));
}
/** Close the split in the current tab by clicking the close button */
export function closeViaButton(splitCount, inSplit) {
    it(`should close the split via button in the current tab and expect splitCount=${splitCount}`, () => this.app.client
        .$(Selectors.SPLIT_N_CLOSE(inSplit))
        .then(_ => _.click())
        .then(() => ReplExpect.splitCount(splitCount)(this.app))
        .catch(Common.oops(this, true)));
}
/** Clear the split in the current tab by clicking the clear button */
export function clearViaButton(inSplit) {
    const expectBlockCount = ReplExpect.blockCount.bind(this);
    it(`should close the split via button in the current tab and expect blockCount=1`, () => this.app.client
        .$(Selectors.SPLIT_N_CLEAR(inSplit))
        .then(_ => _.click())
        .then(() => __awaiter(this, void 0, void 0, function* () {
        yield expectBlockCount()
            .inSplit(inSplit)
            .is(1);
    }))
        .catch(Common.oops(this, true)));
}
function clickToFocus(toSplitIndex) {
    return __awaiter(this, void 0, void 0, function* () {
        console.error('1');
        yield this.app.client.$(Selectors.SPLIT_N_FOCUS(toSplitIndex)).then(_ => _.click());
        console.error('2');
        yield this.app.client.waitUntil(() => this.app.client.$(Selectors.CURRENT_PROMPT_FOR_SPLIT(toSplitIndex)).then(_ => _.isFocused()), { timeout: CLI.waitTimeout });
        console.error('3', yield this.app.client.$(Selectors.CURRENT_PROMPT_FOR_SPLIT(toSplitIndex)).then(_ => _.isFocused()));
    });
}
export function focus(toSplitIndex) {
    const clickOn = clickToFocus.bind(this);
    it(`should click to focus on split ${toSplitIndex}`, () => clickOn(toSplitIndex).catch(Common.oops(this, true)));
}
export function expectSplits(nSplits) {
    it(`should have ${nSplits} split${nSplits === 1 ? '' : 's'}`, () => ReplExpect.splitCount(nSplits)(this.app));
}
/** Click to focus the given split */
export function focusAndValidate(fromSplitIndex, toSplitIndex) {
    it(`should click to focus from split ${fromSplitIndex} to split ${toSplitIndex}`, () => __awaiter(this, void 0, void 0, function* () {
        try {
            const res1 = yield CLI.commandInSplit('split-debug', this.app, fromSplitIndex);
            const N1 = res1.count;
            const id1 = yield this.app.client.$(Selectors.OUTPUT_N(N1, fromSplitIndex)).then(_ => _.getText());
            yield clickToFocus.bind(this)(toSplitIndex);
            // last true: noFocus, since we want to do this ourselves
            const res2 = yield CLI.commandInSplit('split-debug', this.app, toSplitIndex);
            const N2 = res2.count;
            const id2 = yield this.app.client.$(Selectors.OUTPUT_N(N2, toSplitIndex)).then(_ => _.getText());
            console.error('5');
            notStrictEqual(id1, id2, 'the split identifiers should differ');
        }
        catch (err) {
            yield Common.oops(this, true)(err);
        }
    }));
}
//# sourceMappingURL=split-helpers.js.map