'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
exports.PromisePoolExecutor = void 0;
const promise_pool_error_1 = require("./promise-pool-error");
class PromisePoolExecutor {
    /**
     * Creates a new promise pool executer instance with a default concurrency of 10.
     */
    constructor() {
        this.tasks = [];
        this.items = [];
        this.errors = [];
        this.results = [];
        this.concurrency = 10;
        this.handler = () => { };
        this.errorHandler = undefined;
    }
    /**
     * Set the number of tasks to process concurrently the promise pool.
     *
     * @param {Integer} concurrency
     *
     * @returns {PromisePoolExecutor}
     */
    withConcurrency(concurrency) {
        this.concurrency = concurrency;
        return this;
    }
    /**
     * Set the items to be processed in the promise pool.
     *
     * @param {Array} items
     *
     * @returns {PromisePoolExecutor}
     */
    for(items) {
        this.items = items;
        return this;
    }
    /**
     * Set the handler that is applied to each item.
     *
     * @param {Function} action
     *
     * @returns {PromisePoolExecutor}
     */
    withHandler(action) {
        this.handler = action;
        return this;
    }
    /**
     * Set the error handler function to execute when an error occurs.
     *
     * @param {Function} handler
     *
     * @returns {PromisePoolExecutor}
     */
    handleError(handler) {
        this.errorHandler = handler;
        return this;
    }
    /**
     * Determines whether the number of active tasks is greater or equal to the concurrency limit.
     *
     * @returns {Boolean}
     */
    hasReachedConcurrencyLimit() {
        return this.activeCount() >= this.concurrency;
    }
    /**
     * Returns the number of active tasks.
     *
     * @returns {Number}
     */
    activeCount() {
        return this.tasks.length;
    }
    /**
     * Start processing the promise pool.
     *
     * @returns {Array}
     */
    async start() {
        this.validateInputs();
        return await this.process();
    }
    /**
     * Ensure valid inputs and throw otherwise.
     *
     * @throws
     */
    validateInputs() {
        if (typeof this.handler !== 'function') {
            throw new Error('The first parameter for the .process(fn) method must be a function');
        }
        if (!(typeof this.concurrency === 'number' && this.concurrency >= 1)) {
            throw new TypeError(`"concurrency" must be a number, 1 or up. Received "${this.concurrency}" (${typeof this.concurrency})`);
        }
        if (!Array.isArray(this.items)) {
            throw new TypeError(`"items" must be an array. Received ${typeof this.items}`);
        }
        if (this.errorHandler) {
            if (typeof this.errorHandler !== 'function') {
                throw new Error(`The error handler must be a function. Received ${typeof this.errorHandler}`);
            }
        }
    }
    /**
     * Starts processing the promise pool by iterating over the items
     * and running each item through the async `callback` function.
     *
     * @param {Function} callback
     *
     * @returns {Promise}
     */
    async process() {
        for (const item of this.items) {
            if (this.hasReachedConcurrencyLimit()) {
                await this.processingSlot();
            }
            this.startProcessing(item);
        }
        return this.drained();
    }
    /**
     * Creates a deferred promise and pushes the related callback to the pending
     * queue. Returns the promise which is used to wait for the callback.
     *
     * @returns {Promise}
     */
    async processingSlot() {
        return this.waitForTaskToFinish();
    }
    /**
     * Wait for one of the active tasks to finish processing.
     */
    async waitForTaskToFinish() {
        await Promise.race(this.tasks);
    }
    /**
     * Create a processing function for the given `item`.
     *
     * @param {*} item
     */
    startProcessing(item) {
        const task = this.createTaskFor(item)
            .then(result => {
            this.results.push(result);
            this.tasks.splice(this.tasks.indexOf(task), 1);
        })
            .catch(error => {
            this.tasks.splice(this.tasks.indexOf(task), 1);
            if (this.errorHandler) {
                return this.errorHandler(error, item);
            }
            this.errors.push(promise_pool_error_1.PromisePoolError.createFrom(error, item));
        });
        this.tasks.push(task);
    }
    /**
     * Ensures a returned promise for the processing of the given `item`.
     *
     * @param item
     *
     * @returns {*}
     */
    async createTaskFor(item) {
        return this.handler(item);
    }
    /**
     * Wait for all active tasks to finish. Once all the tasks finished
     * processing, returns an object containing the results and errors.
     *
     * @returns {Object}
     */
    async drained() {
        await this.drainActiveTasks();
        return {
            results: this.results,
            errors: this.errors
        };
    }
    /**
     * Wait for all of the active tasks to finish processing.
     */
    async drainActiveTasks() {
        await Promise.all(this.tasks);
    }
}
exports.PromisePoolExecutor = PromisePoolExecutor;
