"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isTab = isTab;
exports.getTabId = getTabId;
exports.getTabIds = getTabIds;
exports.getPrimaryTabId = getPrimaryTabId;
exports.isTopLevelTab = isTopLevelTab;
exports.splitFor = splitFor;
exports.pexecInCurrentTab = pexecInCurrentTab;
exports.getTab = exports.getCurrentTab = exports.sameTab = void 0;

var _capabilities = require("../core/capabilities");

var _exec = require("../repl/exec");

/*
 * Copyright 2017 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
function isTab(node) {
  return node.classList.contains('kui--tab-content');
}
/**
 * Return the unique identifier for the given tab
 *
 */


function getTabId(tab) {
  return tab.uuid;
}

function getTabIds(tab) {
  const uuid = getTabId(tab);

  if (uuid) {
    const [u1] = uuid.split(/_/);
    return u1 === uuid ? [u1] : [u1, uuid];
  } else {
    return [];
  }
}

function getPrimaryTabId(tab) {
  return getTabIds(tab)[0];
}

function isTopLevelTab(tab) {
  return getPrimaryTabId(tab) === getTabId(tab);
}

const sameTab = (tab1, tab2) => {
  return getTabId(tab1) === getTabId(tab2);
};

exports.sameTab = sameTab;

const getCurrentTab = () => {
  const tab = document.querySelector('.kui--tab-content.visible');

  if ((0, _capabilities.isHeadless)() && !tab.REPL) {
    (0, _exec.getImpl)(tab);
  }

  return tab;
};

exports.getCurrentTab = getCurrentTab;

const getTab = idx => {
  if (!idx) {
    return getCurrentTab();
  } else if (typeof idx === 'number') {
    return document.querySelector(`.kui--tab-content[data-tab-id="${idx}"]`);
  } else {
    return idx;
  }
};
/**
 * We need to find the instance of the Split that has a REPL controller
 *   - if given a `topLevelTab` that is already a Split, then we're "good to go"
 *   - if we are given an actual top-level tab (i.e. not a split), then we need to find the first Split child of that given tab
 *   - if we aren't given a tab as input, then we search from `document`
 *
 */


exports.getTab = getTab;

function splitFor(topLevelTab) {
  if (topLevelTab && !isTopLevelTab(topLevelTab) && topLevelTab.REPL) {
    // "good to go"! we were given a Split that has a REPL controller
    return topLevelTab;
  } else {
    return (topLevelTab || document).querySelector((topLevelTab ? '' : '.kui--tab-content.visible') + ' .kui--scrollback:not([data-is-minisplit])').facade;
  }
}
/**
 * Execute the given command in the current (or given) tab.
 *
 * @param isInternalCallPath This is one plugin calling another
 * @param incognito Execute the command quietly but do not display the result in the Terminal
 *
 */


function pexecInCurrentTab(command, topLevelTab, isInternalCallpath = false, incognito = false) {
  const split = splitFor(topLevelTab);

  if (split) {
    const tab = split; // "tab" is the old name for the split in the repl/exec code

    return isInternalCallpath ? split.REPL.qexec(command, undefined, undefined, {
      tab
    }) // "quiet" exec i.e. don't display the fact that we are executing a command
    : split.REPL.pexec(command, {
      tab,
      echo: !incognito,
      noHistory: true
    }); // normal exec, display the Input/Output in the UI
  } else {
    return Promise.reject(new Error('Internal Error: unable to execute in current tab, as there is no initialized scrollback in the current tab. This is probably due a race condition elsewhere in the code: trying to do the exec in parallel with the tab initialization.'));
  }
}