/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { v4 as uuid } from 'uuid';
import { isNotebook, isWatchable, eventBus, isTable, isScalarContent, isMultiModalResponse } from '@kui-shell/core';
import { isAnnouncement, isOk, isOops } from './Block/BlockModel';
export function isNotebookImpl(raw) {
    const model = raw;
    return isNotebook(model) && model.spec && Array.isArray(model.spec.splits);
}
export function snapshot(block) {
    if (!isAnnouncement(block) && (isOops(block) || isOk(block))) {
        const execOptions = Object.assign({}, block.completeEvent.execOptions, { block: undefined }, { tab: block.completeEvent.execOptions.tab ? block.completeEvent.execOptions.tab.uuid : undefined });
        const evaluatorOptions = Object.assign({}, block.completeEvent.evaluatorOptions, {
            usage: undefined,
            flags: undefined
        });
        /**
         * We excluded watch for now since a `Watchable` instance is not serializable
         * see issue: https://github.com/IBM/kui/issues/5399
         * and issue: https://github.com/IBM/kui/issues/5452
         *
         */
        const excludeWatchable = (response) => {
            if (response && isWatchable(response)) {
                return Object.assign({}, response, { watch: undefined });
            }
            else {
                return response;
            }
        };
        const tab = block.startEvent.tab
            ? block.startEvent.tab.uuid
            : block.completeEvent.tab
                ? block.completeEvent.tab.uuid
                : undefined;
        const startEvent = Object.assign({}, block.startEvent, { tab });
        const completeEvent = Object.assign({}, block.completeEvent, { execOptions, evaluatorOptions }, { tab }, { response: excludeWatchable(block.completeEvent.response) });
        return Object.assign(block, {
            isReplay: true,
            response: excludeWatchable(block.response),
            startEvent,
            completeEvent
        });
    }
}
export function allocateTab(target, tab) {
    Object.assign(target, { tab });
}
/** assign tab to the block */
export function tabAlignment(block, tab) {
    const allocateTabForTable = (table) => {
        table.body.forEach(row => {
            const onclickHome = row.onclick ? row : row.attributes.find(_ => _.onclick && _.key === 'NAME');
            if (onclickHome && onclickHome.onclick && onclickHome.onclick.startEvent && onclickHome.onclick.completeEvent) {
                allocateTab(onclickHome.onclick.startEvent, tab);
                allocateTab(onclickHome.onclick.completeEvent, tab);
            }
        });
    };
    if (isMultiModalResponse(block.completeEvent.response)) {
        block.completeEvent.response.modes.forEach(mode => {
            if (isScalarContent(mode)) {
                if (isTable(mode.content)) {
                    allocateTabForTable(mode.content);
                }
            }
        });
        block.response = block.completeEvent.response;
    }
    else if (isTable(block.completeEvent.response)) {
        allocateTabForTable(block.completeEvent.response);
        block.response = block.completeEvent.response;
    }
    allocateTab(block.completeEvent, tab);
    allocateTab(block.startEvent, tab);
    return block;
}
export class FlightRecorder {
    // eslint-disable-next-line no-useless-constructor
    constructor(tab, splits) {
        this.tab = tab;
        this.splits = splits;
    }
    recordTable(table) {
        return __awaiter(this, void 0, void 0, function* () {
            yield Promise.all(table.body.map((row) => __awaiter(this, void 0, void 0, function* () {
                if (row.onclickIdempotent) {
                    // look for onclicks in either the row, or a cell NAME
                    const onclickHome = typeof row.onclick === 'string' ? row : row.attributes.find(_ => _.onclick && /Name/i.test(_.key));
                    if (onclickHome && typeof onclickHome.onclick === 'string') {
                        const fakeTab = Object.assign({}, this.tab, { uuid: uuid() });
                        const command = onclickHome.onclick;
                        onclickHome.onclick = {};
                        const onCommandStart = (startEvent) => {
                            Object.assign(onclickHome.onclick, { startEvent });
                        };
                        const onCommandComplete = (completeEvent) => {
                            Object.assign(onclickHome.onclick, { completeEvent });
                        };
                        eventBus.onCommandStart(fakeTab.uuid, onCommandStart);
                        eventBus.onCommandComplete(fakeTab.uuid, onCommandComplete);
                        try {
                            yield fakeTab.REPL.pexec(command, { tab: fakeTab });
                        }
                        catch (err) {
                            console.error('Error recording click', command, err);
                        }
                        finally {
                            eventBus.offCommandStart(fakeTab.uuid, onCommandStart);
                            eventBus.offCommandComplete(fakeTab.uuid, onCommandComplete);
                        }
                    }
                }
            })));
        });
    }
    /**
     * Run through the rows of a Table response, issue the onclick
     * handler, and store the (start,complete) event pairs, indexed by
     * row of the table; that's a ClickSnapshot.
     *
     */
    record() {
        return __awaiter(this, void 0, void 0, function* () {
            yield Promise.all(this.splits.map(split => Promise.all(split.blocks.map((_) => __awaiter(this, void 0, void 0, function* () {
                if (isMultiModalResponse(_.completeEvent.response)) {
                    yield Promise.all(_.completeEvent.response.modes.map((mode) => __awaiter(this, void 0, void 0, function* () {
                        if (isScalarContent(mode)) {
                            if (isTable(mode.content)) {
                                yield this.recordTable(mode.content);
                            }
                        }
                    })));
                }
                else if (isTable(_.completeEvent.response)) {
                    yield this.recordTable(_.completeEvent.response);
                }
            })))));
        });
    }
}
//# sourceMappingURL=Snapshot.js.map