/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import React from 'react';
import { isAbortableResponse, isMessageWithUsageModel, isMessageWithCode, getPrimaryTabId, i18n, isCommentaryResponse, isHTML, isMarkdownResponse, isMultiModalResponse, isNavResponse, isReactResponse, isRadioTable, isRandomErrorResponse1, isRandomErrorResponse2, isTable, isMixedResponse, isXtermResponse, isUsageError } from '@kui-shell/core';
const Commentary = React.lazy(() => import('../Commentary'));
import HTMLDom from './HTMLDom'; // !! DO NOT MAKE LAZY. See https://github.com/IBM/kui/issues/6758
const XtermDom = React.lazy(() => import('./XtermDom'));
import renderTable from '../Table';
const Markdown = React.lazy(() => import('../Markdown'));
import { KuiContext } from '../../../';
const RadioTableSpi = React.lazy(() => import('../../spi/RadioTable'));
import { isError } from '../../Views/Terminal/Block/BlockModel';
const TopNavSidecar = React.lazy(() => import('../../Views/Sidecar/TopNavSidecarV2'));
const LeftNavSidecar = React.lazy(() => import('../../Views/Sidecar/LeftNavSidecarV2'));
const strings = i18n('plugin-client-common', 'errors');
/**
 * Component that renders a "ScalarResponse", which is a command
 * response that doesn't require any particularly special
 * interpretation or visualization of the inner structure --- i.e. a
 * response that is suitable for rendering directly in the Terminal.
 *
 */
export default class Scalar extends React.PureComponent {
    constructor(props) {
        super(props);
        this._onRender = this.onRender.bind(this);
        this.state = Scalar.getDerivedStateFromProps();
    }
    static getDerivedStateFromProps() {
        return {
            catastrophicError: undefined
        };
    }
    static getDerivedStateFromError(error) {
        return { catastrophicError: error };
    }
    componentDidCatch(error, errorInfo) {
        console.error('catastrophic error in Scalar', error, errorInfo);
    }
    onRender() {
        if (this.props.onRender) {
            setTimeout(() => this.props.onRender(true), 0);
        }
    }
    renderResponse(response) {
        const { tab } = this.props;
        if (typeof response === 'boolean') {
            this.onRender();
            return React.createElement(React.Fragment, null);
        }
        else if (typeof response === 'number') {
            this.onRender();
            return React.createElement("pre", null, response);
        }
        else if (isUsageError(response)) {
            // hopefully we can do away with this shortly
            this.onRender();
            if (typeof response.raw === 'string') {
                return React.createElement("pre", null, response.raw);
            }
            else if (isMessageWithUsageModel(response.raw) || isMessageWithCode(response.raw)) {
                return React.createElement("pre", null, response.raw.message);
            }
            else {
                return React.createElement(HTMLDom, { content: response.raw });
            }
        }
        else if (isXtermResponse(response)) {
            this.onRender();
            return React.createElement(XtermDom, { response: response });
        }
        else if (typeof response === 'string' || isError(response)) {
            const message = isError(response) ? response.message : response;
            // Markdown interprets escapes, so we need to double-escape
            this.onRender();
            return (React.createElement("pre", null,
                React.createElement(Markdown, { tab: tab, repl: tab.REPL, source: message.replace(/\\/g, '\\\\').replace(/\n/g, '\n\n') })));
        }
        else if (isCommentaryResponse(response)) {
            return (React.createElement("span", { className: "flex-fill flex-layout flex-align-stretch" },
                React.createElement(Commentary, Object.assign({}, response.props, { repl: tab.REPL, tabUUID: getPrimaryTabId(tab), isPartOfMiniSplit: this.props.isPartOfMiniSplit, onRender: this._onRender, willRemove: this.props.willRemove, willUpdateCommand: this.props.willUpdateCommand, willUpdateResponse: (text) => {
                        response.props.children = text;
                    } }))));
        }
        else if (isRadioTable(response)) {
            this.onRender();
            return (React.createElement(KuiContext.Consumer, null, config => React.createElement(RadioTableSpi, { table: response, title: !config.disableTableTitle, repl: tab.REPL })));
        }
        else if (isTable(response)) {
            const renderBottomToolbar = true;
            const isLargeTable = response.body.length >= 50;
            const isLargeMiniTable = this.props.isPartOfMiniSplit && response.body.length > 5;
            const renderGrid = (isLargeTable || isLargeMiniTable) &&
                (response.allowedPresentations === undefined || response.allowedPresentations.indexOf('grid') >= 0);
            return renderTable(tab, tab.REPL, response, undefined, renderBottomToolbar, renderGrid, this._onRender, this.props.isPartOfMiniSplit, this.props.isWidthConstrained);
            // ^^^ Notes: typescript doesn't like this, and i don't know why:
            // "is not assignable to type IntrinsicAttributes..."
            // <PaginatedTable {...props} />
        }
        else if (isMixedResponse(response)) {
            return (React.createElement(React.Fragment, null, response.map((part, idx) => (React.createElement(Scalar, Object.assign({}, this.props, { key: idx, response: part }))))));
        }
        else if (isReactResponse(response)) {
            this.onRender();
            return response.react;
        }
        else if (isHTML(response)) {
            // ^^^ intentionally using an "else" so that typescript double
            // checks that we've covered every case of ScalarResponse
            this.onRender();
            return React.createElement(HTMLDom, { content: response });
        }
        else if (isMarkdownResponse(response)) {
            return React.createElement(Markdown, { tab: tab, repl: tab.REPL, source: response.content, onRender: this._onRender });
        }
        else if (isRandomErrorResponse1(response)) {
            // maybe this is an error response from some random API?
            return (React.createElement(Markdown, { tab: tab, repl: tab.REPL, source: strings('randomError1', response.code), onRender: this._onRender }));
        }
        else if (isRandomErrorResponse2(response)) {
            // maybe this is an error response from some random API?
            return (React.createElement(Markdown, { tab: tab, repl: tab.REPL, source: strings('randomError2', response.errno), onRender: this._onRender }));
        }
        else if (isMultiModalResponse(response)) {
            return (React.createElement(TopNavSidecar, { uuid: tab.uuid, tab: tab, execUUID: this.props.execUUID, active: true, response: response, onRender: this.props.onRender, willChangeSize: this.props.willChangeSize, argvNoOptions: this.props.completeEvent ? this.props.completeEvent.argvNoOptions : undefined, parsedOptions: this.props.completeEvent ? this.props.completeEvent.parsedOptions : undefined }));
        }
        else if (isNavResponse(response)) {
            return (React.createElement(LeftNavSidecar, { uuid: tab.uuid, tab: tab, execUUID: this.props.execUUID, active: true, response: response, onRender: this.props.onRender, willChangeSize: this.props.willChangeSize, argvNoOptions: this.props.completeEvent ? this.props.completeEvent.argvNoOptions : undefined, parsedOptions: this.props.completeEvent ? this.props.completeEvent.parsedOptions : undefined }));
        }
        else if (isAbortableResponse(response)) {
            this.onRender();
            return this.renderResponse(response.response);
        }
        console.error('unexpected null return from Scalar:', this.props.response);
        this.onRender();
        return React.createElement("pre", { className: "oops" }, "Internal Error in command execution");
    }
    render() {
        if (this.state.catastrophicError) {
            return React.createElement("div", { className: "oops" }, this.state.catastrophicError.toString());
        }
        try {
            return this.renderResponse(this.props.response);
        }
        catch (err) {
            console.error('catastrophic error rendering Scalar', err);
            return React.createElement("pre", null, err.toString());
        }
    }
}
//# sourceMappingURL=index.js.map