/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import React from 'react';
import { i18n } from '@kui-shell/core';
import Card from '../spi/Card';
import Button from '../spi/Button';
import SimpleEditor from './Editor/SimpleEditor';
const strings = i18n('plugin-client-common');
export default class Commentary extends React.PureComponent {
    constructor(props) {
        super(props);
        this._onCancel = this.onCancel.bind(this);
        this._onRevert = this.onRevert.bind(this);
        this._onDone = this.onDone.bind(this);
        this._setEdit = this.setEdit.bind(this);
        this._onContentChange = this.onContentChange.bind(this);
        this._onSaveFromEditor = this.onSaveFromEditor.bind(this);
        this._onCancelFromEditor = this.onCancelFromEditor.bind(this);
        const textValue = this.initialTextValue();
        this.state = {
            isEdit: textValue.length === 0,
            textValue,
            lastAppliedTextValue: textValue
        };
    }
    /** update state to cancel any edits and close the editor */
    onCancel(evt) {
        this.onRevert(evt, false);
        this.removeOurselvesIfEmpty();
    }
    /** cancel button */
    cancel() {
        return (React.createElement(Button, { kind: "secondary", size: "small", className: "kui--tab-navigatable kui--commentary-button kui--commentary-cancel-button", onClick: this._onCancel }, strings('Cancel')));
    }
    /** Update state to cancel any updates, but leave editor open */
    onRevert(evt, isEdit = true) {
        if (evt) {
            // so that the event doesn't propagate to the onClick on the Card itself
            evt.stopPropagation();
        }
        this.setState(curState => {
            // switch back to the lastAppliedTextValue
            const textValue = curState.lastAppliedTextValue;
            if (this.props.willUpdateResponse) {
                this.props.willUpdateResponse(textValue);
            }
            return { isEdit, textValue };
        });
    }
    /** revert button */
    revert() {
        return (React.createElement(Button, { kind: "tertiary", size: "small", className: "kui--tab-navigatable kui--commentary-button kui--commentary-revert-button", onClick: this._onRevert }, strings('Revert')));
    }
    /** If the user clicks Done or Cancel and there is no text, remove ourselves */
    removeOurselvesIfEmpty() {
        if (this.state.textValue === '') {
            if (this.props.willRemove) {
                this.props.willRemove();
            }
            return true;
        }
        else {
            return false;
        }
    }
    /** Update state to reflect lastAppliedTextValue, and close the editor */
    onDone(evt) {
        if (evt) {
            // so that the event doesn't propagate to the onClick on the Card itself
            evt.stopPropagation();
        }
        if (!this.removeOurselvesIfEmpty()) {
            this.setState(curState => {
                this.props.willUpdateCommand(`# ${curState.textValue.replace(/\n/g, '\\n').replace(/\t/g, '\\t')}`);
                return { isEdit: false, lastAppliedTextValue: curState.textValue };
            });
        }
    }
    /** done button removes the editor  */
    done() {
        return (React.createElement(Button, { size: "small", className: "kui--tab-navigatable kui--commentary-button kui--commentary-done-button", onClick: this._onDone }, strings('Done')));
    }
    /** toolbar hosts editor actions */
    toolbar() {
        return (React.createElement("div", { className: "kui--commentary-editor-toolbar fill-container flush-right" },
            this.done(),
            "\u00A0",
            this.cancel(),
            "\u00A0",
            this.revert()));
    }
    /** Enter isEdit mode */
    setEdit() {
        this.setState({ isEdit: true });
    }
    card() {
        return (React.createElement("span", { className: "kui--commentary-card", onDoubleClick: this._setEdit },
            React.createElement(Card, Object.assign({}, this.props, { "data-is-editing": this.state.isEdit || undefined, header: this.state.isEdit && strings('Editing Comment as Markdown'), footer: this.state.isEdit && this.toolbar() }),
                this.state.textValue,
                this.state.isEdit && this.editor())));
    }
    /** Percolate `SimpleEditor` edits up to the Preview view */
    onContentChange(value) {
        this.setState({ textValue: value });
        if (this.props.willUpdateResponse) {
            this.props.willUpdateResponse(this.state.textValue);
        }
    }
    /** User has requested to save changes via keyboard shortcut, from within `SimpleEditor` */
    onSaveFromEditor(value) {
        this.onContentChange(value);
        this.onDone();
    }
    /** User has requested to cancel changes via keyboard shortcut, from within `SimpleEditor` */
    onCancelFromEditor() {
        this.onCancel();
    }
    /** @return the initial content to display, before any editing */
    initialTextValue() {
        return this.props.children || '';
    }
    editor() {
        return (React.createElement(SimpleEditor, { tabUUID: this.props.tabUUID, content: this.state.textValue, className: "kui--source-ref-editor kui--inverted-color-context", readonly: false, simple: true, onSave: this._onSaveFromEditor, onCancel: this._onCancelFromEditor, onContentChange: this._onContentChange, contentType: "markdown", scrollIntoView: this.props.isPartOfMiniSplit }));
    }
    render() {
        this.props.onRender();
        return (React.createElement("div", { className: "kui--commentary", "data-is-editing": this.state.isEdit || undefined }, this.card()));
    }
}
//# sourceMappingURL=Commentary.js.map