/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import React from 'react';
import { KeyCodes, inElectron, isOfflineClient } from '@kui-shell/core';
import { Nav, NavList, Page, PageHeader } from '@patternfly/react-core';
import KuiContext from '../context';
import NewTabButton from './NewTabButton';
import Tab from './Tab';
import SplitTerminalButton from './SplitTerminalButton';
import '../../../../web/scss/components/TopTabStripe/_index.scss';
export default class TopTabStripe extends React.PureComponent {
    componentDidMount() {
        this.addKeyboardListeners();
    }
    /**
     * Register any keyboard event listeners
     *
     */
    addKeyboardListeners() {
        if (inElectron()) {
            // switch tabs based on keyboard events
            document.addEventListener('keydown', event => {
                if (event.metaKey && event.shiftKey) {
                    // shift-command+[]: switch to previous or next
                    const whichDir = event.key;
                    if (whichDir === '[' || whichDir === ']') {
                        const newIdx = whichDir === '[' ? this.props.activeIdx - 1 : this.props.activeIdx + 1;
                        this.props.onSwitchTab(newIdx);
                    }
                    event.stopPropagation();
                    return;
                }
                if (event.ctrlKey) {
                    // ctrl+PgUp/PgDown: switch to previous or next
                    const whichDir = event.keyCode;
                    if (whichDir === KeyCodes.PAGEUP || whichDir === KeyCodes.PAGEDOWN) {
                        const newIdx = whichDir === KeyCodes.PAGEUP ? this.props.activeIdx - 1 : this.props.activeIdx + 1;
                        this.props.onSwitchTab(newIdx);
                    }
                    event.stopPropagation();
                    return;
                }
                if (event.metaKey) {
                    // meta+number: switch to tab by index
                    const whichTabStr = event.key;
                    if (/\d/.test(whichTabStr)) {
                        event.stopPropagation();
                        const whichTabIdx = parseInt(whichTabStr, 10);
                        this.props.onSwitchTab(whichTabIdx - 1);
                    }
                }
            });
        }
    }
    /** Render tabs */
    tabs() {
        return (React.createElement(React.Fragment, null,
            React.createElement(Nav, { "aria-label": "Tabs", variant: "horizontal", className: "kui--header-tabs" },
                React.createElement(NavList, { className: "kui--tab-list" }, this.props.tabs.map((tab, idx) => (React.createElement(Tab, Object.assign({}, this.props, { key: idx, idx: idx, uuid: tab.uuid, title: tab.title, closeable: this.props.tabs.length > 1, active: idx === this.props.activeIdx, onCloseTab: (idx) => this.props.onCloseTab(idx), onSwitchTab: (idx) => this.props.onSwitchTab(idx) })))))),
            !isOfflineClient() && (React.createElement("div", { className: "kui--top-tab-buttons" },
                React.createElement(NewTabButton, { onNewTab: this.props.onNewTab }),
                React.createElement(SplitTerminalButton, null)))));
    }
    headerName() {
        return (React.createElement(KuiContext.Consumer, null, config => (React.createElement("div", { prefix: "", className: "kui--header--name" }, config.productName || 'Kui'))));
    }
    header() {
        const logoProps = {
        /*      href: 'https://patternfly.org',
        onClick: () => console.log('clicked logo'),
        target: '_blank' */
        };
        return (React.createElement(PageHeader, { className: "kui--top-tab-stripe-header", logo: this.headerName(), logoProps: logoProps, topNav: this.tabs() }));
    }
    /**
     * React render handler
     *
     */
    render() {
        return (React.createElement(Page, { "aria-label": "Header", className: "kui--top-tab-stripe", header: this.header(), mainContainerId: "kui--top-tab-stripe-main" }));
    }
}
//# sourceMappingURL=index.js.map