/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import React from 'react';
import { eventBus, pexecInCurrentTab } from '@kui-shell/core';
import TabModel from './TabModel';
import TabContent from './TabContent';
import TopTabStripe from './TopTabStripe';
import '../../../web/css/static/TabContainer.scss';
export default class TabContainer extends React.PureComponent {
    constructor(props) {
        super(props);
        this._onSwitchTab = this.onSwitchTab.bind(this);
        this._onCloseTab = this.onCloseTab.bind(this);
        this._onNewTab = () => this.onNewTab();
        this._onTabReady = this.onTabReady.bind(this);
        this.state = {
            tabs: [this.newTabModel()],
            isFirstTabReady: false,
            activeIdx: 0
        };
        eventBus.on('/tab/new/request', (evt) => {
            this.onNewTab(evt);
        });
        eventBus.on('/tab/switch/request', (idx) => {
            this.onSwitchTab(idx);
        });
    }
    /** save tab state such as CWD prior to a tab switch */
    captureState() {
        try {
            this.state.tabs[this.state.activeIdx].state.capture();
        }
        catch (err) {
            console.error(err);
        }
    }
    /** restore tab state after a tab switch */
    restoreState(tabIdx) {
        this.state.tabs[tabIdx].state.restore();
    }
    /**
     * Switch Tab event: update state so that activeIdx=idx
     *
     */
    onSwitchTab(idx) {
        return __awaiter(this, void 0, void 0, function* () {
            // capture current state, restore state of the switched-to tab
            const currentTabState = this.state.tabs[this.state.activeIdx].state;
            const nextTabState = this.state.tabs[idx].state;
            yield currentTabState.switchTo(nextTabState);
            if (idx >= 0 && idx < this.state.tabs.length) {
                this.setState({
                    activeIdx: idx
                });
            }
            setTimeout(() => eventBus.emit('/tab/switch/request/done', { idx, tab: nextTabState }));
        });
    }
    /**
     * Close Tab event
     *
     */
    onCloseTab(idx) {
        return __awaiter(this, void 0, void 0, function* () {
            const residualTabs = this.state.tabs.slice(0, idx).concat(this.state.tabs.slice(idx + 1));
            if (residualTabs.length > 0) {
                const activeIdx = idx === 0 ? 0 : idx - 1;
                this.restoreState(activeIdx);
                this.setState({
                    tabs: residualTabs,
                    activeIdx
                });
            }
        });
    }
    listenForTabClose(model) {
        eventBus.onceWithTabId('/tab/close/request', model.uuid, (uuid, tab) => __awaiter(this, void 0, void 0, function* () {
            if (this.state.tabs.length === 1) {
                // then we are closing the last tab, so close the window
                tab.REPL.qexec('window close');
            }
            else {
                this.onCloseTab(this.state.tabs.findIndex(_ => _.uuid === model.uuid));
            }
        }));
    }
    newTabModel(evt = {}) {
        // !this.state means: if this is the very first tab we've ever
        // !created, *and* we were given an initial title (via
        // !this.props.title), then use that
        const model = new TabModel(undefined, evt.statusStripeDecoration, evt.background, evt.title || (!this.state && this.props.title ? this.props.title : undefined), undefined, undefined, evt.cmdline, evt.onClose, evt.exec, evt.snapshot);
        this.listenForTabClose(model);
        return model;
    }
    /**
     * New Tab event
     *
     */
    onNewTab(evt = {}) {
        // if we already have a tab with this title, and this isn't a
        // background tab, then switch to it
        if (evt.title) {
            const existingIdx = this.state.tabs.findIndex(_ => _.title === evt.title);
            if (existingIdx >= 0) {
                if (!evt.background) {
                    this.onSwitchTab(existingIdx);
                }
                return;
            }
        }
        this.captureState();
        const model = this.newTabModel(evt);
        this.setState(curState => ({
            tabs: curState.tabs.concat(model),
            activeIdx: !evt.background ? curState.tabs.length : curState.activeIdx
        }));
    }
    graft(node, uuid, key) {
        if (React.isValidElement(node)) {
            // ^^^ this check avoids tsc errors
            return React.cloneElement(node, {
                key,
                uuid
            });
        }
        else {
            return node;
        }
    }
    /** Graft the tab `uuid` */
    children(uuid) {
        if (Array.isArray(this.props.children)) {
            return this.props.children.map((child, idx) => this.graft(child, uuid, idx));
        }
        else {
            return this.graft(this.props.children, uuid);
        }
    }
    willUpdateTopTabButtons(uuid, buttons) {
        this.setState(curState => {
            const idx = curState.tabs.findIndex(_ => _.uuid === uuid);
            if (idx >= 0) {
                return {
                    tabs: curState.tabs
                        .slice(0, idx)
                        .concat([curState.tabs[idx].update(buttons)])
                        .concat(curState.tabs.slice(idx + 1))
                };
            }
        });
    }
    onTabReady(tab) {
        // "initial tab" handling
        if (!this.state.isFirstTabReady) {
            if (this.props.onTabReady) {
                this.props.onTabReady(tab);
            }
            this.setState({ isFirstTabReady: true });
        }
        // then, for all tabs: we were asked to execute a command line in
        // the new tab?
        const tabModel = this.state.tabs.find(_ => _.uuid === tab.uuid);
        if (tabModel) {
            // execute a command onReady?
            if (tabModel.initialCommandLine) {
                try {
                    const quiet = tabModel.exec && tabModel.exec === 'qexec';
                    pexecInCurrentTab(tabModel.initialCommandLine, tab, quiet);
                }
                catch (err) {
                    console.error('Error executing initial command line in new tab', err);
                }
            }
            // execute a command onClose?
            if (tabModel.onClose) {
                eventBus.on('/tab/close', (tab) => __awaiter(this, void 0, void 0, function* () {
                    try {
                        yield tab.REPL.qexec(tabModel.onClose);
                    }
                    catch (err) {
                        console.error('Error executing tab onClose handler', err);
                    }
                }));
            }
        }
    }
    /** Render the row of Tabs along the top */
    topTabStripe() {
        return (React.createElement(TopTabStripe, { tabs: this.state.tabs, onNewTab: this._onNewTab, onCloseTab: this._onCloseTab, onSwitchTab: this._onSwitchTab, activeIdx: this.state.activeIdx, topTabNames: this.props.topTabNames }));
    }
    /** Render the content of the tabs */
    tabContent() {
        return (React.createElement("div", { className: "tab-container" }, this.state.tabs.map((_, idx) => (React.createElement(TabContent, Object.assign({}, this.props, { snapshot: _.snapshot, tabTitle: _.title, key: _.uuid, uuid: _.uuid, active: idx === this.state.activeIdx, willUpdateTopTabButtons: this.willUpdateTopTabButtons.bind(this, _.uuid), onTabReady: this._onTabReady, state: _.state }), this.children(_.uuid))))));
    }
    render() {
        return (React.createElement("div", { className: "kui--full-height" },
            this.topTabStripe(),
            this.tabContent()));
    }
}
//# sourceMappingURL=TabContainer.js.map