/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { Common, CLI, ReplExpect, Selectors, Util } from '@kui-shell/test';
import { remotePodYaml, defaultModeForGet, openSidecarByList, waitForGreen, waitForRed, createNS, allocateNS, deleteNS } from '@kui-shell/plugin-kubectl/tests/lib/k8s/utils';
import { dirname } from 'path';
const ROOT = dirname(require.resolve('@kui-shell/plugin-kubectl/tests/package.json'));
describe(`kubectl replay ${process.env.MOCHA_RUN_TARGET || ''}`, function () {
    before(Common.before(this));
    after(Common.after(this));
    const ns = createNS();
    const inNamespace = `-n ${ns}`;
    const file = Util.uniqueFileForSnapshot();
    allocateNS(this, ns, 'kubectl');
    it(`should create, get and delete the sample pod from URL via kubectl, and replay`, () => __awaiter(this, void 0, void 0, function* () {
        // here comes the tests
        try {
            console.error('creating');
            yield openSidecarByList(this, `kubectl create -f ${remotePodYaml} ${inNamespace}`, 'nginx');
            console.error('deleting');
            const deleteRes = yield CLI.command(`kubectl delete -f ${remotePodYaml} ${inNamespace}`, this.app);
            const deleteSelector = yield ReplExpect.okWithCustom({ selector: Selectors.BY_NAME('nginx') })(deleteRes);
            yield waitForRed(this.app, deleteSelector);
            console.error('snapshoting');
            yield CLI.command(`snapshot ${file}`, this.app).then(ReplExpect.justOK);
            console.error('replaying');
            yield Common.refresh(this);
            yield CLI.command(`replay ${file}`, this.app);
            yield waitForRed(this.app, `${Selectors.OUTPUT_LAST} ${Selectors.BY_NAME('nginx')}`);
        }
        catch (err) {
            yield Common.oops(this, true)(err);
        }
    }));
    deleteNS(this, ns, 'kubectl');
});
describe(`kubectl replay with re-execution ${process.env.MOCHA_RUN_TARGET || ''}`, function () {
    return __awaiter(this, void 0, void 0, function* () {
        before(Common.before(this));
        after(Common.after(this));
        const ns = createNS();
        const inNamespace = `-n ${ns}`;
        const file = Util.uniqueFileForSnapshot();
        allocateNS(this, ns, 'kubectl');
        it('should replay a kubectl get pods table with re-execution ', () => __awaiter(this, void 0, void 0, function* () {
            try {
                const selector = yield CLI.command(`kubectl create -f ${remotePodYaml} ${inNamespace}`, this.app).then(ReplExpect.okWithCustom({ selector: Selectors.BY_NAME('nginx') }));
                yield waitForGreen(this.app, selector);
                yield CLI.command(`snapshot ${file} --exec`, this.app).then(ReplExpect.justOK);
                yield CLI.command(`replay ${file}`, this.app);
                yield this.app.client.waitUntil(() => __awaiter(this, void 0, void 0, function* () {
                    const errorMessage = yield this.app.client
                        .$(`${Selectors.OUTPUT_LAST}.oops[data-status-code="409"]`)
                        .then(_ => _.getText());
                    return errorMessage.includes('pods "nginx"');
                }), { timeout: CLI.waitTimeout });
            }
            catch (err) {
                yield Common.oops(this, true)(err);
            }
        }));
        deleteNS(this, ns, 'kubectl');
    });
});
describe(`kubectl replay with clicks ${process.env.MOCHA_RUN_TARGET || ''}`, function () {
    return __awaiter(this, void 0, void 0, function* () {
        before(Common.before(this));
        after(Common.after(this));
        const ns = createNS();
        const inNamespace = `-n ${ns}`;
        const file = Util.uniqueFileForSnapshot();
        allocateNS(this, ns, 'kubectl');
        it(`should replay a kubectl get pods table with re-execution using snapshot file ${file}`, () => __awaiter(this, void 0, void 0, function* () {
            try {
                const res = yield CLI.command(`kubectl create -f ${remotePodYaml} ${inNamespace}`, this.app);
                const selector = yield ReplExpect.okWithCustom({ selector: Selectors.BY_NAME('nginx') })(res);
                yield waitForGreen(this.app, selector);
                yield CLI.command(`snapshot ${file}`, this.app).then(ReplExpect.justOK);
                yield CLI.command(`replay ${file}`, this.app);
                yield this.app.client.$(`${Selectors.OUTPUT_LAST} ${Selectors.BY_NAME('nginx')}`).then(_ => _.waitForDisplayed());
                yield Util.openSidecarByClick(this, `${Selectors.OUTPUT_LAST} ${Selectors.BY_NAME('nginx')} .clickable`, 'nginx', defaultModeForGet, undefined, 1 // replayed clicks currently don't support opening in a split; see https://github.com/IBM/kui/issues/6785
                );
            }
            catch (err) {
                yield Common.oops(this, true)(err);
            }
        }));
        deleteNS(this, ns, 'kubectl');
    });
});
describe(`kubectl replay with grid table ${process.env.MOCHA_RUN_TARGET || ''}`, function () {
    return __awaiter(this, void 0, void 0, function* () {
        before(Common.before(this));
        after(Common.after(this));
        const ns = createNS();
        const inNamespace = `-n ${ns}`;
        const file = Util.uniqueFileForSnapshot();
        allocateNS(this, ns, 'kubectl');
        it(`should replay a kubectl get pods table with grid using snapshot file ${file}`, () => __awaiter(this, void 0, void 0, function* () {
            try {
                const res = yield CLI.command(`kubectl create -f ${ROOT}/data/k8s/headless ${inNamespace}`, this.app);
                const selector = yield ReplExpect.okWithCustom({ selector: Selectors.BY_NAME('nginx') })(res);
                yield waitForGreen(this.app, selector);
                yield this.app.client.$(Selectors.TABLE_SHOW_AS_GRID(res.count)).then(_ => _.click());
                yield this.app.client.$(Selectors.TABLE_AS_GRID(res.count)).then(_ => _.waitForDisplayed());
                yield CLI.command(`snapshot ${file}`, this.app).then(ReplExpect.justOK);
                yield CLI.command(`replay ${file}`, this.app);
                yield this.app.client.$(`${Selectors.OUTPUT_LAST} ${Selectors._TABLE_AS_GRID}`).then(_ => _.waitForDisplayed());
            }
            catch (err) {
                yield Common.oops(this, true)(err);
            }
        }));
        deleteNS(this, ns, 'kubectl');
    });
});
//# sourceMappingURL=replay.js.map