/// <reference types="node" />
import { Arguments, KResponse, ResourceWithMetadata, MultiModalResponse } from '@kui-shell/core';
import KubeOptions from '../../controller/kubectl/options';
import kubeuiApiVersion from '../../controller/kubectl/apiVersion';
export interface KubeStatusCondition {
    lastProbeTime?: string;
    lastUpdateTime: string;
    lastTransitionTime: string;
    status: string | boolean;
    reason?: string;
    message: string;
    type?: string;
    phase?: string;
}
interface KubeContainerStatus {
    name: string;
    containerID: string;
    restartCount: number;
    ready: boolean;
    state: any;
}
export interface KubeLoadBalancer {
    ingress: string;
}
export interface KubeStatus {
    message?: string;
    state?: string;
    startTime?: string;
    completionTime?: string;
    phase?: string;
    podName?: string;
    qosClass?: string;
    replicas?: number;
    readyReplicas?: number;
    availableReplicas?: number;
    unavailableReplicas?: number;
    updatedReplicas?: number;
    loadBalancer?: KubeLoadBalancer;
    conditions?: KubeStatusCondition[];
}
export declare class DefaultKubeStatus implements KubeStatus {
    message: any;
}
export interface WithOwnerReferences {
    ownerReferences: {
        apiVersion: string;
        kind: string;
        name: string;
    }[];
}
interface WithResourceVersion {
    resourceVersion: string;
}
export declare type KubeResourceWithResourceVersion = KubeResource<{}, KubeMetadata & Required<WithResourceVersion>>;
export declare function hasResourceVersion(resource: KubeResource): resource is KubeResourceWithResourceVersion;
export declare function sameResourceVersion(a: MultiModalResponse<KubeResource>, b: MultiModalResponse<KubeResource>): boolean;
export declare type KubeMetadata = Partial<WithOwnerReferences> & Partial<WithResourceVersion> & {
    name: string;
    uid?: string;
    namespace?: string;
    labels?: {
        [key: string]: string;
    };
    annotations?: object;
    creationTimestamp?: string;
    generation?: string;
    generateName?: string;
};
export declare type KubeResourceWithOwnerReferences = KubeResource<{}, KubeMetadata & Required<WithOwnerReferences>>;
export declare function hasSingleOwnerReference(resource: KubeResource): resource is KubeResourceWithOwnerReferences;
export declare class DefaultKubeMetadata implements KubeMetadata {
    kind: any;
    name: any;
}
interface RoleRule {
    apiGroups: string[];
    resources: string[];
    verbs: string[];
}
interface RoleRef {
    apiGroup: string;
    kind: string;
    name: string;
}
export interface WithRawData<Content = void> extends ResourceWithMetadata<Content> {
    kuiRawData?: string;
}
export declare function hasRawData(resource: ResourceWithMetadata): boolean;
/**
 * The basic Kubernetes resource
 *
 */
export declare type KubeResource<Status = KubeStatus, Metadata = KubeMetadata> = ResourceWithMetadata & WithRawData & {
    apiVersion: string;
    kind: string;
    metadata?: Metadata;
    status?: Status;
    spec?: any;
    originatingCommand: Arguments<KubeOptions>;
    isSimulacrum?: boolean;
    isKubeResource: true;
};
/** is the resource Namespaced? */
export declare function isNamespaced(resource: KubeResource): boolean;
/** is the command response a Kubernetes resource? note: excluding any ones we simulate in kubeui */
export declare function isKubeResource(entity: KResponse | ResourceWithMetadata): entity is KubeResource;
export interface WithSummary {
    summary: {
        content: string;
        contentType?: 'yaml' | 'text/markdown';
    };
}
/**
 * `KubeResourceWithSummary` allows plugins to provide their own
 * Summary. Otherwise lib/views/modes/summary will try to fetch one
 * automatically.
 *
 */
export declare type KubeResourceWithSummary<Status = KubeStatus> = KubeResource<Status> & WithSummary;
export declare function isKubeResourceWithItsOwnSummary(resource: KubeResource): resource is KubeResourceWithSummary;
/**
 * This allows us to exclude certain resource kinds from auto-summarization
 *
 */
export declare function isSummarizableKubeResource(resource: KubeResource): boolean;
/** Role */
interface Role extends KubeResource {
    rules: RoleRule[];
}
export declare function isRole(resource: KubeResource): resource is Role;
/** RoleBinding */
interface RoleBinding extends KubeResource {
    roleRef: RoleRef;
    subjects: {
        kind: string;
        name: string;
    }[];
}
export declare function isRoleBinding(resource: KubeResource): resource is RoleBinding;
/** ServiceAccount */
interface ServiceAccount extends KubeResource {
    secrets: {
        name: string;
    }[];
}
export declare function isServiceAccount(resource: KubeResource): resource is ServiceAccount;
export interface CRDResource extends KubeResource {
    spec: {
        names: {
            kind: string;
            shortnames: string[];
        };
    };
}
interface ContainerSpec {
    args: string[];
    command: string[];
    env: {
        name: string;
        value: string;
    }[];
    image: string;
    imagePullPolicy: string;
    name: string;
    resource: Record<string, any>;
    terminationMessagePath: string;
    terminationMessagePolicy: string;
    volumeMounts: {
        mountPath: string;
        name: string;
    }[];
    ports?: {
        containerPort: string;
        protocol: string;
    }[];
    workingDir: string;
}
/**
 * Kubernetes Pod resource type
 *
 */
interface PodStatus extends KubeStatus {
    containerStatuses: KubeContainerStatus[];
    hostIP: string;
    podIP: string;
}
export interface PodLikeSpec {
    _podName: string;
}
export interface Pod extends KubeResource<PodStatus> {
    apiVersion: 'v1';
    kind: 'Pod';
    spec: PodLikeSpec & {
        nodeName: string;
        nominatedNodeName?: string;
        readinessGates?: {
            conditionType: string;
        }[];
        containers: ContainerSpec[];
    };
}
/**
 * @return whether the given resource is an instance of a Pod
 *
 */
export declare function isPod(resource: KubeResource): resource is Pod;
/**
 * Kubernetes Namespace resource type
 *
 */
export interface Namespace extends KubeResource {
    apiVersion: 'v1';
    kind: 'Namespace';
}
/**
 * @return whether the given resource is an instance of a Namespace
 *
 */
export declare function isNamespace(resource: KubeResource): resource is Namespace;
interface ContainerTemplate<Labels extends Record<string, any>> {
    metadata: {
        creationTimestamp: string;
        labels: Labels;
    };
    spec: {
        containers: ContainerSpec[];
    };
}
export interface Selector {
    matchLabels: Record<string, any>;
}
/**
 * Kubernetes Job resource type
 *
 */
export interface Job extends KubeResource {
    apiVersion: 'batch/v1';
    kind: 'Job';
    spec: {
        backoffLimit: number;
        completions: number;
        parallelism: number;
        selector: Selector;
        template: ContainerTemplate<{
            'controller-uid': string;
            'job-name': string;
        }>;
    };
    status: {
        succeeded: number;
        startTime: string;
        completionTime: string;
        conditions: KubeStatusCondition[];
    };
}
/**
 * @return whether the given resource is an instance of a Deploymemt
 *
 */
export declare function isJob(resource: Pick<KubeResource, 'apiVersion' | 'kind'>): resource is Job;
/**
 * Kubernetes Deployment resource type
 *
 */
interface Deployment1 extends KubeResource {
    apiVersion: 'extensions/v1beta1';
    kind: 'Deployment';
}
/**
 * Kubernetes Deployment resource type
 *
 */
interface Deployment2 extends KubeResource {
    apiVersion: 'apps/v1';
    kind: 'Deployment';
}
export declare type Deployment = Deployment1 | Deployment2;
/**
 * @return whether the given resource is an instance of a Deployment
 *
 */
export declare function isDeployment(resource: KubeResource): resource is Deployment;
/**
 * Kubernetes ReplicaSet resource type
 *
 */
export interface ReplicaSet1 extends KubeResource {
    apiVersion: 'extensions/v1beta1';
    kind: 'ReplicaSet';
}
export interface ReplicaSet2 extends KubeResource {
    apiVersion: 'apps/v1';
    kind: 'ReplicaSet';
}
export declare type ReplicaSet = ReplicaSet1 | ReplicaSet2;
/**
 * @return whether the given resource is an instance of a ReplicaSet
 *
 */
export declare function isReplicaSet(resource: KubeResource): resource is ReplicaSet;
/**
 * Trait that defines an involvedObject, e.g. for Events
 *
 */
export interface InvolvedObject {
    involvedObject: {
        apiVersion: string;
        kind: string;
        name: string;
        namespace: string;
        uid?: string;
        fieldPath?: string;
        resourceVersion?: string;
    };
}
export declare type KubeResourceWithInvolvedObject = KubeResource & InvolvedObject;
export declare function hasInvolvedObject(resource: KubeResource | KubeResourceWithInvolvedObject): resource is KubeResourceWithInvolvedObject;
/**
 * Kubernetes Event resource type
 *
 */
export declare type Event = KubeResourceWithInvolvedObject & {
    apiVersion: 'v1';
    kind: 'Event';
    firstTimestamp: string;
    lastTimestamp: string;
    count: number;
    reason: string;
    message: string;
    type: 'Normal' | 'Warning' | 'Error';
    source: {
        component: string;
        host: string;
    };
    metadata: {
        name: string;
        namespace: string;
    };
    involvedObject: {
        apiVersion: string;
        kind: string;
        name: string;
        namespace: string;
    };
};
/**
 * @return whether the given resource is an instance of an Event
 *
 */
export declare function isEvent(resource: KubeResource): resource is Event;
/** is the command response a kube resource that can responds to "kubectl delete", etc.? */
export declare function isCrudableKubeResource(entity: ResourceWithMetadata): entity is KubeResource;
/**
 * e.g. `kubectl get pods -o json` will return a kind: items
 *
 */
export interface KubeItems<Item extends KubeResource = KubeResource> extends KubeResource {
    apiVersion: 'v1';
    kind: 'List';
    items: Item[];
}
export declare type KubePartial<R extends KubeResource = KubeResource> = Omit<R, 'apiVersion' | 'kind'>;
export interface PodList extends KubeResource {
    apiVersion: 'v1';
    kind: 'PodList';
    items: KubePartial<Pod>[];
}
export declare function isKubeItems(resource: KubeResource): resource is KubeItems;
export declare function isPodList(resource: KubeResource): resource is PodList;
export declare function isKubeItemsOfKind<Item extends KubeResource = KubeResource>(resource: KubeResource, isOfKind: (item: KubeResource) => item is Item): resource is KubeItems<Item>;
/** Scope */
declare type Scope = 'Namespaced' | 'Cluster';
/**
 * CustomResourceDefinition
 *
 */
export declare type CustomResourceDefinition = KubeResource & {
    apiVersion: 'apiextensions.k8s.io/v1' | 'apiextensions.k8s.io/v1beta1';
    kind: 'CustomResourceDefinition';
    spec: {
        scope: Scope;
        group: string;
        version: string;
        names: {
            categories: Record<string, string>;
            kind: string;
            listKind: string;
            plural: string;
            singular: string;
        };
    };
};
/**
 * @return whether the given resource is an instance of a CustomResourceDefinition
 *
 */
export declare function isCustomResourceDefinition(resource: KubeResource): resource is CustomResourceDefinition;
/**
 * ConfigMap
 *
 */
export declare type ConfigMap = KubeResource & {
    apiVersion: 'v1';
    kind: 'ConfigMap';
    data: Record<string, any>;
};
/**
 * @return whether the given resource is an instance of a CustomResourceDefinition
 *
 */
export declare function isConfigMap(resource: KubeResource): resource is ConfigMap;
/**
 * Kubernetes context
 *
 */
export interface KubeContext extends KubeResource {
    apiVersion: typeof kubeuiApiVersion;
    kind: 'Context';
    spec: {
        user: string;
        cluster: string;
        isCurrent: boolean;
    };
}
export interface Resource<T = KubeResource> {
    filepathForDrilldown?: string;
    kind?: string;
    name?: string;
    resource: T;
}
/**
 * Is the given resource kind cluster scoped (as opposed to namespace scoped)?
 * FIXME: apiVersion
 */
export declare function isClusterScoped(kind: string): boolean;
interface NodeCapacity {
    cpu: string;
    'ephemeral-storage': string;
    'hugepages-1Gi': string;
    'hugepages-2Mi': string;
    memory: string;
    pods: string;
}
export declare type AddressType = 'InternalIP' | 'ExternalIP' | 'Hostname';
interface NodeStatus {
    addresses: {
        address: string;
        type: AddressType;
    }[];
    allocatable: NodeCapacity;
    capacity: NodeCapacity;
    conditions: KubeStatusCondition[];
    images: {
        names: string[];
        sizeBytes: number;
    }[];
    nodeInfo: {
        architecture: string;
        bootId: string;
        containerRuntimeVersion: string;
        kernelVersion: string;
        kubeProxyVersion: string;
        kubeletVersion: string;
        machineID: string;
        operatingSystem: string;
        osImage: string;
        systemUUID: string;
    };
}
export interface Node extends KubeResource<NodeStatus> {
    apiVersion: 'v1';
    kind: 'Node';
}
export declare function isNode(resource: KubeResource): resource is Node;
/**
 * @return whether the given resource might possibly have events;
 * since Events never have Events, we can exclude those always
 *
 */
export declare function hasEvents(resource: KubeResource): boolean;
export default KubeResource;
interface PartialObjectMetadata extends KubeResource {
    apiVersion: 'meta.k8s.io/v1';
    kind: 'PartialObjectMetadata';
}
export interface MetaTable extends KubeResource {
    apiVersion: 'meta.k8s.io/v1';
    kind: 'Table';
    columnDefinitions: {
        name: string;
        priority: number;
        type: 'integer' | 'long' | 'float' | 'double' | 'string' | 'byte' | 'binary' | 'boolean' | 'date' | 'dateTime' | 'password';
    }[];
    rows: {
        cells: (string | number)[];
        object: PartialObjectMetadata;
    }[];
}
export declare function isMetaTable(response: KubeResource): response is MetaTable;
/** When calling the API directly, it may respond with a Status, e.g. for 404s */
export interface Status extends KubeResource {
    apiVersion: 'v1';
    kind: 'Status';
    code: number;
    reason: string;
    status: string;
    message: string;
}
export declare function isStatus(resource: string | Buffer | object | KubeResource): resource is Status;
/** Kubernetes Secret */
export interface Secret<T = Record<string, any>> extends KubeResource {
    apiVersion: 'v1';
    kind: 'Secret';
    type: string;
    data: T;
}
export declare function isSecret(resource: KubeResource): resource is Secret;
