/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { basename, dirname } from 'path';
import { findFile, expandHomeDir, i18n } from '@kui-shell/core';
import { getFileFromArgv, getFileForArgv, withKubeconfigFrom } from './options';
import { isUsage, doHelp } from '../../lib/util/help';
import { removeLastAppliedConfig } from '../../lib/util/util';
import { isKubeResource } from '../../lib/model/resource';
import { exec } from './exec';
const strings = i18n('plugin-kubectl');
const doDiff = (command) => function (args) {
    return __awaiter(this, void 0, void 0, function* () {
        if (isUsage(args)) {
            return doHelp(command, args);
        }
        else {
            try {
                const { REPL } = args;
                const filepath = getFileFromArgv(args);
                const fullpath = findFile(expandHomeDir(filepath));
                const name = basename(filepath);
                const enclosingDirectory = dirname(filepath);
                const packageName = enclosingDirectory === '.' ? undefined : enclosingDirectory;
                const [previous, current] = yield Promise.all([
                    REPL.qexec(withKubeconfigFrom(args, `${command} get ${getFileForArgv(args)} -o yaml`)),
                    REPL.qexec(withKubeconfigFrom(args, `${command} apply ${getFileForArgv(args)} --dry-run=server -o yaml`))
                ]);
                console.error('previous', previous);
                console.error('current', current);
                if (isKubeResource(previous) && isKubeResource(current)) {
                    const mode = {
                        mode: 'diff',
                        label: strings('Pending Changes'),
                        content: {
                            a: removeLastAppliedConfig(previous.kuiRawData),
                            b: removeLastAppliedConfig(current.kuiRawData)
                        },
                        contentType: 'yaml'
                    };
                    const response = {
                        apiVersion: 'kui-shell/v1',
                        kind: 'File',
                        metadata: {
                            name,
                            namespace: packageName
                        },
                        toolbarText: {
                            type: 'info',
                            text: strings('Showing changes between the working tree and previous commit.')
                        },
                        modes: [mode],
                        spec: {
                            filepath,
                            fullpath,
                            size: 0
                        }
                    };
                    return response;
                }
                else {
                    throw new Error('failed to get resources');
                }
            }
            catch (err) {
                console.error('error at kubectl diff', err);
                return exec(args, undefined, command);
            }
        }
    });
};
/** Register a command listener */
export function describer(registrar, command, cli = command) {
    registrar.listen(`/${command}/diff`, doDiff(cli));
}
export default (registrar) => {
    describer(registrar, 'kubectl');
    describer(registrar, 'k', 'kubectl');
};
//# sourceMappingURL=diff.js.map