"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _pluginKubectl = require("@kui-shell/plugin-kubectl");

/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = void 0 && (void 0).__awaiter || function (thisArg, _arguments, P, generator) {
  function adopt(value) {
    return value instanceof P ? value : new P(function (resolve) {
      resolve(value);
    });
  }

  return new (P || (P = Promise))(function (resolve, reject) {
    function fulfilled(value) {
      try {
        step(generator.next(value));
      } catch (e) {
        reject(e);
      }
    }

    function rejected(value) {
      try {
        step(generator["throw"](value));
      } catch (e) {
        reject(e);
      }
    }

    function step(result) {
      result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected);
    }

    step((generator = generator.apply(thisArg, _arguments || [])).next());
  });
};

function bodyToHeader(body) {
  if (body.length > 0) {
    return {
      key: body[0].key,
      name: body[0].key,
      attributes: body[0].attributes.map(({
        key
      }) => ({
        key,
        value: key
      }))
    };
  }
}

function formatIssueRow(group, message, level) {
  return {
    key: 'Group',
    name: group,
    attributes: [{
      key: 'STATUS',
      value: level === 3 ? 'Error' : level === 2 ? 'Warning' : 'OK',
      tag: 'badge',
      css: level === 3 ? 'red-background' : level === 2 ? 'yellow-background' : 'green-background'
    }, {
      key: 'Message',
      value: message
    }]
  };
}

var _default = command => args => __awaiter(void 0, void 0, void 0, function* () {
  if ((0, _pluginKubectl.isUsage)(args)) {
    return (0, _pluginKubectl.doHelp)(command, args);
  } else {
    const userAskForSection = args.parsedOptions['s'] || args.parsedOptions['sections'];
    let userAskResourceName;
    /**
     * override the popeye command to force json output
     * , and add support for fetching report by resource name
     *
     */

    const prepareArgsForPopeye = args => {
      userAskResourceName = userAskForSection && args.argvNoOptions[args.argvNoOptions.indexOf('popeye') + 1];
      /** popeye doesn't support showing report for a single resource name
       * e.g. popeye -s pod nginx
       * so we fetch the report of pod and filter the result
       *
       */

      return userAskResourceName ? `${args.command.replace(userAskResourceName, '')} -o json` : `${args.command} -o json`;
    };

    const stdout = yield (0, _pluginKubectl.doExecWithStdout)(args, prepareArgsForPopeye, command);
    const fullReport = JSON.parse(stdout).popeye; // console.error('fullReport', fullReport)

    const ns = yield (0, _pluginKubectl.getNamespace)(args);

    const formatTable = (body, title, footer) => {
      return {
        header: bodyToHeader(body),
        body,
        statusColumnIdx: 0,
        breadcrumbs: [{
          label: 'Popeye'
        }, {
          label: ns
        }],
        title,
        footer
      };
    };
    /**
     * full report issued by command kubectl popeye
     *
     */


    const fullReportTable = () => {
      const body = fullReport.sanitizers.map(({
        sanitizer,
        tally
      }) => {
        return {
          key: 'Name',
          name: sanitizer,
          onclick: `${command} popeye -s ${sanitizer} -n ${ns}`,
          attributes: [{
            key: 'Score',
            value: tally.score.toString(),
            tag: 'badge',
            css: tally.score === 100 ? 'green-background' : 'red-background'
          }, {
            key: 'Error',
            value: tally.error.toString()
          }, {
            key: 'Warning',
            value: tally.warning.toString()
          }, {
            key: 'Info',
            value: tally.info.toString()
          }, {
            key: 'Ok',
            value: tally.ok.toString()
          }]
        };
      });
      return formatTable(body, 'All resources', [`Overall score: ${fullReport.score} ${fullReport.grade}`]);
    };
    /**
     * section report issued by command e.g. kubectl popeye -s pod
     *
     */


    const sectionTable = () => __awaiter(void 0, void 0, void 0, function* () {
      const body = [];
      fullReport.sanitizers.forEach(({
        issues,
        sanitizer,
        tally
      }) => {
        if (issues) {
          Object.entries(issues).forEach(([resourceName, issueList]) => {
            if (resourceName) {
              let status = 0;
              issueList.forEach(({
                level
              }) => status = level > status ? level : status);
              const shortNames = resourceName.split(/\//);
              body.push({
                key: 'Name',
                name: shortNames.length > 1 ? shortNames[1] : shortNames[0],
                onclick: `${command} popeye -s ${sanitizer} ${resourceName} -n ${ns}`,
                attributes: [{
                  key: 'Status',
                  value: status === 3 ? 'Error' : status === 2 ? 'Warning' : 'OK',
                  tag: 'badge',
                  css: status === 3 ? 'red-background' : status === 2 ? 'yellow-background' : 'green-background'
                }]
              });
            } else {
              issueList.forEach(_ => body.push(formatIssueRow(_.group, _.message, _.level)));
            }
          });
        } else {
          // no issues, return the tally of this sanitizer
          body.push({
            key: 'Name',
            name: sanitizer,
            attributes: [{
              key: 'Score',
              value: tally.score.toString(),
              tag: 'badge',
              css: tally.score === 100 ? 'green-background' : 'red-background'
            }, {
              key: 'Error',
              value: tally.error.toString()
            }, {
              key: 'Warning',
              value: tally.warning.toString()
            }, {
              key: 'Info',
              value: tally.info.toString()
            }, {
              key: 'Ok',
              value: tally.ok.toString()
            }]
          });
        }
      });
      return formatTable(body, yield (0, _pluginKubectl.getKind)(command, args, userAskForSection), [`Overall score: ${fullReport.score} ${fullReport.grade}`]);
    });
    /**
     * all issues in e.g. pod nginx
     *
     */


    const resourceNameTable = () => {
      const body = fullReport.sanitizers[0].issues[userAskResourceName].map(({
        group,
        level,
        message
      }) => formatIssueRow(group, message, level));
      return formatTable(body, userAskResourceName.split(/\//)[1]);
    };

    if (userAskResourceName) {
      return resourceNameTable();
    } else if (userAskForSection) {
      return sectionTable();
    } else {
      return fullReportTable();
    }
  }
});

exports.default = _default;