"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useControllableState = useControllableState;

var _slicedToArray2 = _interopRequireDefault(require("@babel/runtime/helpers/slicedToArray"));

var _react = require("react");

var _warning = require("./warning");

/**
 * Copyright IBM Corp. 2016, 2018
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

/**
 * This custom hook simplifies the behavior of a component if it has state that
 * can be both controlled and uncontrolled. It functions identical to a
 * useState() hook and provides [state, setState] for you to use. You can use
 * the `onChange` argument to allow updates to the `state` to be communicated to
 * owners of controlled components.
 *
 * Note: this hook will warn if a component is switching from controlled to
 * uncontrolled, or vice-verse.
 *
 * @param {object} config
 * @param {string} config.name - the name of the custom component
 * @param {any} config.defaultValue - the default value used for the state. This will be
 * the fallback value used if `value` is not defined.
 * @param {Function} config.onChange - an optional function that is called when
 * the value of the state changes. This is useful for communicating to parents of
 * controlled components that the value is requesting to be changed.
 * @param {any} config.value - a controlled value. Omitting this means that the state is
 * uncontrolled
 * @returns {[any, Function]}
 */
function useControllableState(_ref) {
  var defaultValue = _ref.defaultValue,
      _ref$name = _ref.name,
      name = _ref$name === void 0 ? 'custom' : _ref$name,
      onChange = _ref.onChange,
      value = _ref.value;

  var _useState = (0, _react.useState)(value !== null && value !== void 0 ? value : defaultValue),
      _useState2 = (0, _slicedToArray2.default)(_useState, 2),
      state = _useState2[0],
      internalSetState = _useState2[1];

  var controlled = (0, _react.useRef)(null);

  if (controlled.current === null) {
    controlled.current = value !== undefined;
  }

  function setState(stateOrUpdater) {
    var value = typeof stateOrUpdater === 'function' ? stateOrUpdater(state) : stateOrUpdater;

    if (controlled.current === false) {
      internalSetState(value);
    }

    if (onChange) {
      onChange(value);
    }
  }

  (0, _react.useEffect)(function () {
    var controlledValue = value !== undefined; // Uncontrolled -> Controlled
    // If the component prop is uncontrolled, the prop value should be undefined

    if (controlled.current === false && controlledValue) {
      process.env.NODE_ENV !== "production" ? (0, _warning.warning)(false, 'A component is changing an uncontrolled %s component to be controlled. ' + 'This is likely caused by the value changing to a defined value ' + 'from undefined. Decide between using a controlled or uncontrolled ' + 'value for the lifetime of the component. ' + 'More info: https://reactjs.org/link/controlled-components', name) : void 0;
    } // Controlled -> Uncontrolled
    // If the component prop is controlled, the prop value should be defined


    if (controlled.current === true && !controlledValue) {
      process.env.NODE_ENV !== "production" ? (0, _warning.warning)(false, 'A component is changing a controlled %s component to be uncontrolled. ' + 'This is likely caused by the value changing to an undefined value ' + 'from a defined one. Decide between using a controlled or ' + 'uncontrolled value for the lifetime of the component. ' + 'More info: https://reactjs.org/link/controlled-components', name) : void 0;
    }
  }, [name, value]);

  if (controlled.current === true) {
    return [value, setState];
  }

  return [state, setState];
}