/**
 * Copyright IBM Corp. 2016, 2018
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
import React from 'react';
import { Close20 } from '@carbon/icons-react';
import Modal from './Modal';
import InlineLoading from '../../InlineLoading';
import { mount } from 'enzyme';
var prefix = 'bx'; // The modal is the 0th child inside the wrapper on account of focus-trap-react

var getModal = function getModal(wrapper) {
  return wrapper.find('.bx--modal');
};

var getModalBody = function getModalBody(wrapper) {
  return wrapper.find('.bx--modal-container');
};

describe('Modal', function () {
  describe('Renders as expected', function () {
    var wrapper = mount( /*#__PURE__*/React.createElement(Modal, {
      "aria-label": "test",
      className: "extra-class"
    }));
    var mounted = mount( /*#__PURE__*/React.createElement(Modal, {
      "aria-label": "test",
      className: "extra-class"
    }));
    it('has the expected classes', function () {
      expect(getModal(wrapper).hasClass("".concat(prefix, "--modal"))).toEqual(true);
    });
    it('should add extra classes that are passed via className', function () {
      expect(getModal(wrapper).hasClass('extra-class')).toEqual(true);
    });
    it('should not be a passive modal by default', function () {
      expect(getModal(wrapper).hasClass("".concat(prefix, "--modal-tall"))).toEqual(true);
    });
    it('should be a passive modal when passiveModal is passed', function () {
      wrapper.setProps({
        passiveModal: true
      });
      expect(getModal(wrapper).hasClass("".concat(prefix, "--modal-tall"))).toEqual(false);
    });
    it('should set id if one is passed via props', function () {
      var modal = mount( /*#__PURE__*/React.createElement(Modal, {
        "aria-label": "test",
        id: "modal-1"
      }));
      expect(getModal(modal).props().id).toEqual('modal-1');
    });
    it('should not place the svg icon in the accessibility tree', function () {
      var ariaHidden = mounted.find(Close20).props()['aria-hidden'];
      expect(ariaHidden).toEqual('true');
    });
    it("icon isn't a focusable tab stop", function () {
      var icon = mounted.find(Close20).props().tabIndex;
      expect(icon).toEqual('-1');
    });
    it('enables primary button by default', function () {
      var primaryButton = mounted.find(".".concat(prefix, "--btn.").concat(prefix, "--btn--primary")).at(0);
      expect(primaryButton.prop('disabled')).toEqual(false);
    });
    it('disables primary button when disablePrimaryButton prop is passed', function () {
      mounted.setProps({
        primaryButtonDisabled: true
      });
      var primaryButton = mounted.find(".".concat(prefix, "--btn.").concat(prefix, "--btn--primary")).at(0);
      expect(primaryButton.props().disabled).toEqual(true);
    });
    it('Should have node in primary', function () {
      mounted.setProps({
        primaryButtonText: /*#__PURE__*/React.createElement(InlineLoading, null)
      });
      var primaryButton = mounted.find(".".concat(prefix, "--btn.").concat(prefix, "--btn--primary")).at(0);
      expect(primaryButton.find('InlineLoading').exists()).toEqual(true);
    });
    it('Should have node in secondary', function () {
      mounted.setProps({
        secondaryButtonText: /*#__PURE__*/React.createElement(InlineLoading, null)
      });
      var secondaryButton = mounted.find(".".concat(prefix, "--btn.").concat(prefix, "--btn--secondary")).at(0);
      expect(secondaryButton.find('InlineLoading').exists()).toEqual(true);
    });
  });
  describe('Renders as expected with secondaryButtons prop', function () {
    var mounted = mount( /*#__PURE__*/React.createElement(Modal, {
      "aria-label": "test",
      className: "extra-class"
    }));
    it('Should support node in secondary', function () {
      mounted.setProps({
        secondaryButtons: [{
          buttonText: /*#__PURE__*/React.createElement(InlineLoading, null),
          onClick: jest.fn()
        }, {
          buttonText: 'Cancel',
          onClick: jest.fn()
        }]
      });
      var secondaryButtons = mounted.find(".".concat(prefix, "--btn.").concat(prefix, "--btn--secondary"));
      expect(secondaryButtons.length).toEqual(2);
      expect(secondaryButtons.at(0).find('InlineLoading').exists()).toEqual(true);
      expect(secondaryButtons.at(1).text()).toEqual('Cancel');
    });
  });
  describe('Adds props as expected to the right children', function () {
    it('should set label if one is passed via props', function () {
      var wrapper = mount( /*#__PURE__*/React.createElement(Modal, {
        "aria-label": "test",
        modalLabel: "modal-1"
      }));
      var label = wrapper.find(".".concat(prefix, "--modal-header__label"));
      expect(label.props().children).toEqual('modal-1');
    });
    it('should set modal heading if one is passed via props', function () {
      var wrapper = mount( /*#__PURE__*/React.createElement(Modal, {
        "aria-label": "test",
        modalHeading: "modal-1"
      }));
      var heading = wrapper.find(".".concat(prefix, "--modal-header__heading"));
      expect(heading.props().children).toEqual('modal-1');
    });
    it('should set button text if one is passed via props', function () {
      var wrapper = mount( /*#__PURE__*/React.createElement(Modal, {
        "aria-label": "test",
        primaryButtonText: "Submit",
        secondaryButtonText: "Cancel"
      }));
      var modalButtons = wrapper.find(".".concat(prefix, "--modal-footer Button"));
      expect(modalButtons.at(0).props().children).toEqual('Cancel');
      expect(modalButtons.at(1).props().children).toEqual('Submit');
    });
  });
  describe('events', function () {
    it('should handle close keyDown events', function () {
      var onRequestClose = jest.fn();
      var wrapper = mount( /*#__PURE__*/React.createElement(Modal, {
        "aria-label": "test",
        open: true,
        onRequestClose: onRequestClose
      }));
      wrapper.simulate('keyDown', {
        which: 26
      });
      expect(onRequestClose).not.toHaveBeenCalled();
      wrapper.simulate('keyDown', {
        which: 27
      });
      expect(onRequestClose).toHaveBeenCalled();
    });
    it('should handle submit keyDown events with shouldSubmitOnEnter enabled', function () {
      var onRequestSubmit = jest.fn();
      var wrapper = mount( /*#__PURE__*/React.createElement(Modal, {
        "aria-label": "test",
        open: true,
        onRequestSubmit: onRequestSubmit,
        shouldSubmitOnEnter: true
      }));
      wrapper.simulate('keyDown', {
        which: 14
      });
      expect(onRequestSubmit).not.toHaveBeenCalled();
      wrapper.simulate('keyDown', {
        which: 13
      });
      expect(onRequestSubmit).toHaveBeenCalled();
    });
    it('should not handle submit keyDown events with shouldSubmitOnEnter not enabled', function () {
      var onRequestSubmit = jest.fn();
      var wrapper = mount( /*#__PURE__*/React.createElement(Modal, {
        "aria-label": "test",
        open: true,
        onRequestSubmit: onRequestSubmit
      }));
      wrapper.simulate('keyDown', {
        which: 14
      });
      expect(onRequestSubmit).not.toHaveBeenCalled();
      wrapper.simulate('keyDown', {
        which: 13
      });
      expect(onRequestSubmit).not.toHaveBeenCalled();
    });
    it('should close by default on secondary button click', function () {
      var onRequestClose = jest.fn();
      var modal = mount( /*#__PURE__*/React.createElement(Modal, {
        "aria-label": "test",
        secondaryButtonText: "Cancel",
        onRequestClose: onRequestClose
      }));
      var secondaryBtn = modal.find(".".concat(prefix, "--btn--secondary"));
      secondaryBtn.simulate('click');
      expect(onRequestClose).toHaveBeenCalled();
    });
    it('should handle custom secondary button events', function () {
      var onSecondarySubmit = jest.fn();
      var modal = mount( /*#__PURE__*/React.createElement(Modal, {
        "aria-label": "test",
        secondaryButtonText: "Cancel",
        onSecondarySubmit: onSecondarySubmit
      }));
      var secondaryBtn = modal.find(".".concat(prefix, "--btn--secondary"));
      secondaryBtn.simulate('click');
      expect(onSecondarySubmit).toHaveBeenCalled();
    });
  });
});
describe('Danger Modal', function () {
  describe('Renders as expected', function () {
    var wrapper = mount( /*#__PURE__*/React.createElement(Modal, {
      "aria-label": "test",
      secondaryButtonText: "Cancel",
      danger: true
    }));
    it('has the expected classes', function () {
      expect(getModal(wrapper).hasClass("".concat(prefix, "--modal--danger"))).toEqual(true);
    });
    it('has correct button combination', function () {
      var modalButtons = wrapper.find(".".concat(prefix, "--modal-footer.").concat(prefix, "--btn-set Button"));
      expect(modalButtons.length).toEqual(2);
      expect(modalButtons.at(0).props().kind).toEqual('secondary');
      expect(modalButtons.at(1).props().kind).toEqual('danger');
    });
  });
});
describe('Alert Modal', function () {
  describe('Renders as expected', function () {
    var wrapper = mount( /*#__PURE__*/React.createElement(Modal, {
      "aria-label": "test",
      alert: true
    }));
    it('has the expected attributes', function () {
      expect(getModalBody(wrapper).props()).toEqual(expect.objectContaining({
        role: 'alertdialog',
        'aria-describedby': expect.any(String)
      }));
    });
    it('should be a passive modal when passiveModal is passed', function () {
      wrapper.setProps({
        passiveModal: true
      });
      expect(getModalBody(wrapper).props()).toEqual(expect.objectContaining({
        role: 'alert'
      }));
    });
  });
});