import _extends from "@babel/runtime/helpers/extends";
import _toConsumableArray from "@babel/runtime/helpers/toConsumableArray";
import _defineProperty from "@babel/runtime/helpers/defineProperty";
import _slicedToArray from "@babel/runtime/helpers/slicedToArray";
import _objectWithoutProperties from "@babel/runtime/helpers/objectWithoutProperties";
var _excluded = ["accept", "className", "id", "disabled", "labelText", "multiple", "name", "onAddFiles", "pattern", "role", "tabIndex"];

/**
 * Copyright IBM Corp. 2016, 2018
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
import React, { useRef, useState } from 'react';
import PropTypes from 'prop-types';
import classNames from 'classnames';
import { keys, matches } from '../../internal/keyboard';
import uniqueId from '../../tools/uniqueId';
import { usePrefix } from '../../internal/usePrefix';

function FileUploaderDropContainer(_ref) {
  var _classNames2;

  var accept = _ref.accept,
      className = _ref.className,
      id = _ref.id,
      disabled = _ref.disabled,
      labelText = _ref.labelText,
      multiple = _ref.multiple,
      name = _ref.name,
      onAddFiles = _ref.onAddFiles,
      pattern = _ref.pattern,
      role = _ref.role,
      tabIndex = _ref.tabIndex,
      rest = _objectWithoutProperties(_ref, _excluded);

  var prefix = usePrefix();
  var inputRef = useRef(null);

  var _useRef = useRef(id || uniqueId()),
      uid = _useRef.current;

  var _useState = useState(false),
      _useState2 = _slicedToArray(_useState, 2),
      isActive = _useState2[0],
      setActive = _useState2[1];

  var labelClasses = classNames("".concat(prefix, "--file-browse-btn"), _defineProperty({}, "".concat(prefix, "--file-browse-btn--disabled"), disabled));
  var dropareaClasses = classNames("".concat(prefix, "--file__drop-container"), (_classNames2 = {}, _defineProperty(_classNames2, "".concat(prefix, "--file__drop-container--drag-over"), isActive), _defineProperty(_classNames2, className, className), _classNames2));
  /**
   * Filters the array of added files based on file type restrictions
   * @param {Event} event - Event object, used to get the list of files added
   */

  function validateFiles(event) {
    var transferredFiles = event.type === 'drop' ? _toConsumableArray(event.dataTransfer.files) : _toConsumableArray(event.target.files);

    if (!accept.length) {
      return transferredFiles;
    }

    var acceptedTypes = new Set(accept);
    return transferredFiles.reduce(function (acc, curr) {
      var name = curr.name,
          _curr$type = curr.type,
          mimeType = _curr$type === void 0 ? '' : _curr$type;
      var fileExtensionRegExp = new RegExp(pattern, 'i');
      var hasFileExtension = fileExtensionRegExp.test(name);

      if (!hasFileExtension) {
        return acc;
      }

      var _name$match = name.match(fileExtensionRegExp),
          _name$match2 = _slicedToArray(_name$match, 1),
          fileExtension = _name$match2[0];

      if (acceptedTypes.has(mimeType) || acceptedTypes.has(fileExtension)) {
        return acc.concat([curr]);
      }

      curr.invalidFileType = true;
      return acc.concat([curr]);
    }, []);
  }

  function handleChange(event) {
    var addedFiles = validateFiles(event);
    return onAddFiles(event, {
      addedFiles: addedFiles
    });
  }

  return /*#__PURE__*/React.createElement("div", {
    className: "".concat(prefix, "--file"),
    onDragOver: function onDragOver(evt) {
      evt.stopPropagation();
      evt.preventDefault();

      if (disabled) {
        return;
      }

      setActive(true);
      evt.dataTransfer.dropEffect = 'copy';
    },
    onDragLeave: function onDragLeave(evt) {
      evt.stopPropagation();
      evt.preventDefault();

      if (disabled) {
        return;
      }

      setActive(false);
      evt.dataTransfer.dropEffect = 'move';
    },
    onDrop: function onDrop(evt) {
      evt.stopPropagation();
      evt.preventDefault();

      if (disabled) {
        return;
      }

      setActive(false);
      handleChange(evt);
    }
  }, /*#__PURE__*/React.createElement("label", _extends({
    className: labelClasses,
    htmlFor: uid,
    tabIndex: tabIndex || 0,
    onKeyDown: function onKeyDown(evt) {
      if (matches(evt, [keys.Enter, keys.Space])) {
        inputRef.current.click();
      }
    }
  }, rest), /*#__PURE__*/React.createElement("div", {
    className: dropareaClasses,
    role: role || 'button'
  }, labelText), /*#__PURE__*/React.createElement("input", {
    type: "file",
    id: uid,
    className: "".concat(prefix, "--file-input"),
    ref: inputRef,
    tabIndex: "-1",
    disabled: disabled,
    accept: accept,
    name: name,
    multiple: multiple,
    onChange: handleChange,
    onClick: function onClick(evt) {
      evt.target.value = null;
    }
  })));
}

FileUploaderDropContainer.propTypes = {
  /**
   * Specify the types of files that this input should be able to receive
   */
  accept: PropTypes.arrayOf(PropTypes.string),

  /**
   * Provide a custom className to be applied to the container node
   */
  className: PropTypes.string,

  /**
   * Specify whether file input is disabled
   */
  disabled: PropTypes.bool,

  /**
   * Provide a unique id for the underlying `<input>` node
   */
  id: PropTypes.string,

  /**
   * Provide the label text to be read by screen readers when interacting with
   * this control
   */
  labelText: PropTypes.string.isRequired,

  /**
   * Specify if the component should accept multiple files to upload
   */
  multiple: PropTypes.bool,

  /**
   * Provide a name for the underlying `<input>` node
   */
  name: PropTypes.string,

  /**
   * Event handler that is called after files are added to the uploader
   * The event handler signature looks like `onAddFiles(evt, { addedFiles })`
   */
  onAddFiles: PropTypes.func,

  /**
   * Provide a custom regex pattern for the acceptedTypes
   */
  pattern: PropTypes.string,

  /**
   * Provide an accessibility role for the <FileUploaderButton>
   */
  role: PropTypes.string,

  /**
   * Specify the size of the uploaded items, from a list of available
   * sizes. For `default` buttons, this prop can remain unspecified.
   * V11: `default`, `field`, and `small` will be removed
   */
  size: PropTypes.oneOf(['default', 'field', 'small', 'sm', 'md', 'lg']),

  /**
   * Provide a custom tabIndex value for the <FileUploaderButton>
   */
  tabIndex: PropTypes.number
};
FileUploaderDropContainer.defaultProps = {
  tabIndex: 0,
  labelText: 'Add file',
  multiple: false,
  onAddFiles: function onAddFiles() {},
  accept: [],
  pattern: '.[0-9a-z]+$'
};
export default FileUploaderDropContainer;