if (!window || !window.navigator || !window.navigator.userAgent) {
    Object.defineProperty(window, 'navigator', { value: { userAgent: 'node' }, writable: true });
    Object.defineProperty(document, 'getElementById', { value: (val) => document.querySelector('#' + val), writable: true });
}
import * as React from 'react';
import * as lodash from 'lodash';
import * as PropTypes from 'prop-types';
import tableDefinitions from '../definitions/search-definitions';
import Modal from './Modal';
import { Pagination, DataTable } from 'carbon-components-react';
import { getCurrentTab } from '@kui-shell/core';
import strings from '../util/i18n';
import { Delete16, Edit16, Share16 } from '@carbon/icons-react';
import { getStatusIcon } from '../util/status';
const { Table, TableHead, TableRow, TableBody, TableCell } = DataTable;
const PAGE_SIZES = { DEFAULT: 10, VALUES: [5, 10, 20, 50, 75, 100] };
export default class ResourceTable extends React.PureComponent {
    constructor(props) {
        super(props);
        this.tabHeaders = ['name', 'action', 'share', 'edit'];
        this.toggleCollapseTable = () => {
            this.setState(({ collapse }) => {
                return { collapse: !collapse };
            });
        };
        this.handleSort = (selectedKey) => () => {
            if (selectedKey) {
                this.setState((preState) => {
                    return { selectedKey, sortDirection: preState.sortDirection === 'asc' ? 'desc' : 'asc' };
                });
            }
        };
        this.handleEvent = (row, cell, e) => {
            if ((e && e.which === 13) || !e) {
                const item = this.props.items.filter((data) => data['name'] === lodash.get(row, '[0].value', ''));
                switch (cell.info['header']) {
                    case 'action':
                        this.setState({ itemForAction: lodash.get(item, '[0]', ''), modalOpen: true, action: 'remove' });
                        break;
                    case 'edit':
                        this.setState({ itemForAction: lodash.get(item, '[0]', ''), modalOpen: true, action: 'edit' });
                        break;
                    case 'share':
                        this.setState({ itemForAction: lodash.get(item, '[0]', ''), modalOpen: true, action: 'share' });
                        break;
                    default:
                        const headers = ['name', 'namespace', 'cluster'];
                        let filters = '';
                        const _ = row.filter((data) => (data && data.value && (headers.includes(data.info['header']))));
                        _.forEach((data) => {
                            filters += `${data.info['header']}:${data.value} `;
                        });
                        if (this.props.kind === 'savedSearches' && cell.info['header'] === 'name') {
                            return getCurrentTab().REPL.pexec(`search ${row[2].value}`);
                        }
                        else if (cell.info['header'] === 'name' && filters) {
                            return getCurrentTab().REPL.pexec(`search summary kind:${this.props.kind} ${filters}`);
                        }
                        else {
                            return null;
                        }
                }
            }
        };
        this.state = {
            action: '',
            itemForAction: {},
            page: 1,
            pageSize: PAGE_SIZES.DEFAULT,
            sortDirection: 'asc',
            selectedKey: '',
            modalOpen: false,
            collapse: false,
        };
        this.getHeaders = this.getHeaders.bind(this);
        this.getRows = this.getRows.bind(this);
        this.handleEvent = this.handleEvent.bind(this);
    }
    componentWillReceiveProps(nextProps) {
        this.setState({
            collapse: nextProps.collapseTable,
        });
    }
    getHeaders() {
        const { kind } = this.props;
        const resource = tableDefinitions[kind] || tableDefinitions['genericresource'];
        const headers = resource.columns.map((col) => ({
            key: col.key, header: strings(`table.header.${col.msgKey || col.key}`),
        }));
        headers.push({ key: 'action', header: '' });
        if (kind === 'savedSearches') {
            headers.push({ key: 'edit', header: '' });
            headers.push({ key: 'share', header: '' });
        }
        return headers;
    }
    getRows() {
        let { items } = this.props;
        const { kind } = this.props;
        let transforms = tableDefinitions[kind] || tableDefinitions['genericresource'];
        const { page, pageSize, selectedKey, sortDirection } = this.state;
        if (selectedKey) {
            items = lodash.orderBy(items, [selectedKey], [sortDirection]);
        }
        const startItem = (page - 1) * pageSize;
        const visibleItems = items.slice(startItem, startItem + pageSize);
        transforms = transforms.columns.filter((rowCol) => rowCol.transform);
        return visibleItems.map((item, i) => {
            const row = Object.assign({ id: i.toString(), action: null }, item);
            if (transforms.length > 0) {
                transforms.forEach((transform) => {
                    row[transform.key] = transform.transform(item, null, transform.key);
                });
            }
            if (this.props.kind !== 'cluster' && this.props.kind !== 'release') {
                row.action = (React.createElement(Delete16, { className: 'table-action', onClick: () => this.setState({ itemForAction: item, modalOpen: true, action: 'remove' }) }));
                if (kind === 'savedSearches') {
                    row['share'] = (React.createElement(Share16, { className: 'table-action', onClick: () => this.setState({ itemForAction: item, modalOpen: true, action: 'share' }) }));
                    row['edit'] = (React.createElement(Edit16, { className: 'table-action', onClick: () => this.setState({ itemForAction: item, modalOpen: true, action: 'edit' }) }));
                }
            }
            return row;
        });
    }
    render() {
        const { page, pageSize, sortDirection, selectedKey, modalOpen, collapse } = this.state;
        const totalItems = this.props.items.length;
        const sortColumn = selectedKey;
        return (React.createElement(React.Fragment, null,
            React.createElement("div", { className: 'search--resource-table-header' },
                React.createElement("div", null,
                    React.createElement("button", { onClick: this.toggleCollapseTable, className: 'search--resource-table-header-button' },
                        React.createElement("span", { className: 'linked-resources' }, `${this.props.kind}(${this.props.items.length})`),
                        !collapse ? React.createElement("span", { className: 'arrow-up' }, "\u25B2") : React.createElement("span", { className: 'arrow-down' }, "\u25BC")))),
            !collapse
                ? React.createElement(React.Fragment, null,
                    React.createElement(DataTable, { key: `${this.props.kind}-resource-table`, rows: this.getRows(), headers: this.getHeaders(), render: ({ rows, headers }) => {
                            return (React.createElement(Table, null,
                                React.createElement(TableHead, null,
                                    React.createElement(TableRow, null, headers.map((header) => (React.createElement("th", { scope: 'col', key: header.key }, header.key !== 'action'
                                        ? React.createElement("div", { onClick: this.handleSort(header.key), className: `bx--table-sort-v2${sortDirection === 'asc' ? ' bx--table-sort-v2--ascending' : ''}${sortColumn === header.key ? ' bx--table-sort-v2--active' : ''}`, "data-key": header.key },
                                            React.createElement("span", { className: 'bx--table-header-label' },
                                                header.header,
                                                React.createElement("span", { className: 'arrow-header-label' }, this.state.sortDirection === 'asc'
                                                    ? React.createElement("span", null, "\u25B2")
                                                    : React.createElement("span", null, "\u25BC"))))
                                        : null))))),
                                React.createElement(TableBody, null, rows.map((row) => (React.createElement(TableRow, { key: row.id, className: 'bx--data-table--compact' }, row.cells.map((cell) => React.createElement(TableCell, { onKeyPress: (e) => this.handleEvent(row.cells, cell, e), tabIndex: this.tabHeaders.includes(cell.info['header']) ? 0 : null, key: cell.id, onClick: () => this.handleEvent(row.cells, cell) }, cell.info['header'] !== 'status'
                                    ? cell.value
                                    : React.createElement("div", null,
                                        getStatusIcon(cell.value),
                                        React.createElement("span", { className: `status-name` }, `${cell.value}`))))))))));
                        } }),
                    this.props.items.length > PAGE_SIZES.DEFAULT
                        ? React.createElement(Pagination, { key: 'resource-table-pagination', id: 'resource-table-pagination', onChange: (pagination) => this.setState(pagination), pageSize: pageSize, pageSizes: PAGE_SIZES.VALUES, totalItems: totalItems, page: page, isLastPage: pageSize >= totalItems, itemsPerPageText: strings('pagination.itemsPerPage'), pageRangeText: (current, total) => strings('pagination.pageRange', [current, total]), itemRangeText: (min, max, total) => `${strings('pagination.itemRange', [min, max])} ${strings('pagination.itemRangeDescription', [total])}`, pageInputDisabled: pageSize >= totalItems })
                        : null)
                : null,
            React.createElement(Modal, { item: this.state.itemForAction, modalOpen: modalOpen, onClose: () => this.setState({ itemForAction: {}, modalOpen: false, action: '' }), action: this.state.action })));
    }
}
ResourceTable.propTypes = {
    collapseTable: PropTypes.bool,
    expandFullPage: PropTypes.bool,
    items: PropTypes.array,
    kind: PropTypes.string,
};
//# sourceMappingURL=ResourceTable.js.map