"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = onKeyDown;

var _debug = _interopRequireDefault(require("debug"));

var _core = require("@kui-shell/core");

var _TabCompletion = _interopRequireDefault(require("./TabCompletion"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright 2017-19 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = void 0 && (void 0).__awaiter || function (thisArg, _arguments, P, generator) {
  function adopt(value) {
    return value instanceof P ? value : new P(function (resolve) {
      resolve(value);
    });
  }

  return new (P || (P = Promise))(function (resolve, reject) {
    function fulfilled(value) {
      try {
        step(generator.next(value));
      } catch (e) {
        reject(e);
      }
    }

    function rejected(value) {
      try {
        step(generator["throw"](value));
      } catch (e) {
        reject(e);
      }
    }

    function step(result) {
      result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected);
    }

    step((generator = generator.apply(thisArg, _arguments || [])).next());
  });
};

const debug = (0, _debug.default)('Terminal/Block/OnKeyDown');

function isMSIEControl(ctrl) {
  return Object.prototype.hasOwnProperty.call(ctrl, 'createTextRange');
}
/**
 * Update the caret position in an html INPUT field
 *
 */


const setCaretPosition = (ctrl, pos) => {
  if (ctrl.setSelectionRange) {
    ctrl.focus();
    ctrl.setSelectionRange(pos, pos);
  } else if (isMSIEControl(ctrl)) {
    const range = ctrl.createTextRange();
    range.collapse(true);
    range.moveEnd('character', pos);
    range.moveStart('character', pos);
    range.select();
  }
};

const setCaretPositionToEnd = input => setCaretPosition(input, input.value.length);

const updateInputAndMoveCaretToEOL = (input, newValue) => {
  input.value = newValue;
  setTimeout(() => setCaretPositionToEnd(input), 0);
};

function onKeyDown(event) {
  return __awaiter(this, void 0, void 0, function* () {
    const tab = this.props.tab;
    const block = this.props._block;
    const prompt = this.state.prompt;
    const char = event.keyCode;

    if (this.state.tabCompletion) {
      this.state.tabCompletion.key(event);
      return;
    } else if (event.key === 'Tab') {
      (0, _TabCompletion.default)(this, event);
    }

    if (char === _core.KeyCodes.UP || char === _core.KeyCodes.P && event.ctrlKey) {
      // go to previous command in history
      const historyModel = yield (yield Promise.resolve().then(() => require('@kui-shell/core'))).History(tab);
      const newValue = (historyModel.previous() || {
        raw: ''
      }).raw;

      if (newValue) {
        updateInputAndMoveCaretToEOL(prompt, newValue);
      }
    } else if (char === _core.KeyCodes.D && event.ctrlKey) {
      if (prompt.value === '') {
        // <-- only if the line is blank
        debug('exit via ctrl+D');
        tab.REPL.pexec('exit', {
          tab
        });
      }
    } else if (char === _core.KeyCodes.PAGEUP) {
      if ((0, _core.inBrowser)()) {
        debug('pageup');
        const {
          height
        } = document.body.getBoundingClientRect();
        document.querySelector('.kui--tab-content.visible .repl-inner').scrollBy(0, -height);
      }
    } else if (char === _core.KeyCodes.PAGEDOWN) {
      if ((0, _core.inBrowser)()) {
        debug('pagedown');
        const {
          height
        } = document.body.getBoundingClientRect();
        document.querySelector('.kui--tab-content.visible .repl-inner').scrollBy(0, +height);
      }
    } else if (char === _core.KeyCodes.C && event.ctrlKey) {
      // Ctrl+C, cancel
      (0, _core.doCancel)(tab, block); // eslint-disable-line @typescript-eslint/no-use-before-define
    } else if (char === _core.KeyCodes.U && event.ctrlKey) {
      // clear line
      prompt.value = '';
    } else if (char === _core.KeyCodes.L && (event.ctrlKey || (0, _core.inElectron)() && event.metaKey) || process.platform === 'darwin' && char === _core.KeyCodes.K && event.metaKey) {
      // clear screen; capture and restore the current
      // prompt value, in keeping with unix terminal
      // behavior
      _core.eventChannelUnsafe.emit(`/terminal/clear/${this.props.uuid}`);

      _core.eventChannelUnsafe.emit(`/close/views/${this.props.uuid}`); // restore the prompt cursor position
      // debug('restoring cursor position', currentCursorPosition)
      // getCurrentPrompt().setSelectionRange(currentCursorPosition, currentCursorPosition)

    } else if (char === _core.KeyCodes.HOME) {
      // go to first command in history
      const historyModel = yield (yield Promise.resolve().then(() => require('@kui-shell/core'))).History(tab);
      const newValue = historyModel.first().raw;

      if (newValue) {
        updateInputAndMoveCaretToEOL(prompt, newValue);
      }
    } else if (char === _core.KeyCodes.END) {
      // go to last command in history
      const historyModel = yield (yield Promise.resolve().then(() => require('@kui-shell/core'))).History(tab);
      const newValue = (historyModel.last() || {
        raw: ''
      }).raw;
      updateInputAndMoveCaretToEOL(prompt, newValue);
    } else if (char === _core.KeyCodes.DOWN || char === _core.KeyCodes.N && event.ctrlKey) {
      // going DOWN past the last history item will result in '', i.e. a blank line
      const historyModel = yield (yield Promise.resolve().then(() => require('@kui-shell/core'))).History(tab);
      const newValue = (historyModel.next() || {
        raw: ''
      }).raw;
      updateInputAndMoveCaretToEOL(prompt, newValue);
    } else if (event.key === 'w' && event.ctrlKey) {
      const {
        prompt
      } = this.state;
      const idx = prompt.value.lastIndexOf(' ', prompt.value.charAt(prompt.value.length - 1) === ' ' ? prompt.value.length - 2 : prompt.value.length - 1);
      this.state.prompt.value = this.state.prompt.value.slice(0, idx + 1);
    }
  });
}