"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.computeExecutableSchemaId = exports.handleLegacyOptions = exports.signatureCacheKey = exports.EngineReportingAgent = exports.getEngineGraphVariant = exports.getEngineApiKey = void 0;
const os_1 = __importDefault(require("os"));
const zlib_1 = require("zlib");
const graphql_1 = require("graphql");
const apollo_engine_reporting_protobuf_1 = require("apollo-engine-reporting-protobuf");
const apollo_server_env_1 = require("apollo-server-env");
const async_retry_1 = __importDefault(require("async-retry"));
const plugin_1 = require("./plugin");
const apollo_server_caching_1 = require("apollo-server-caching");
const apollo_graphql_1 = require("apollo-graphql");
const schemaReporter_1 = require("./schemaReporter");
const uuid_1 = require("uuid");
const crypto_1 = require("crypto");
let warnedOnDeprecatedApiKey = false;
function getEngineApiKey({ engine, skipWarn = false, logger = console, }) {
    if (typeof engine === 'object') {
        if (engine.apiKey) {
            return engine.apiKey;
        }
    }
    const legacyApiKeyFromEnv = process.env.ENGINE_API_KEY;
    const apiKeyFromEnv = process.env.APOLLO_KEY;
    if (legacyApiKeyFromEnv && apiKeyFromEnv && !skipWarn) {
        logger.warn('Using `APOLLO_KEY` since `ENGINE_API_KEY` (deprecated) is also set in the environment.');
    }
    if (legacyApiKeyFromEnv && !warnedOnDeprecatedApiKey && !skipWarn) {
        logger.warn('[deprecated] The `ENGINE_API_KEY` environment variable has been renamed to `APOLLO_KEY`.');
        warnedOnDeprecatedApiKey = true;
    }
    return apiKeyFromEnv || legacyApiKeyFromEnv || '';
}
exports.getEngineApiKey = getEngineApiKey;
function getEngineGraphVariant(engine, logger = console) {
    if (engine === false) {
        return;
    }
    else if (typeof engine === 'object' &&
        (engine.graphVariant || engine.schemaTag)) {
        if (engine.graphVariant && engine.schemaTag) {
            throw new Error('Cannot set both engine.graphVariant and engine.schemaTag. Please use engine.graphVariant.');
        }
        if (engine.schemaTag) {
            logger.warn('[deprecated] The `schemaTag` property within `engine` configuration has been renamed to `graphVariant`.');
        }
        return engine.graphVariant || engine.schemaTag;
    }
    else {
        if (process.env.ENGINE_SCHEMA_TAG) {
            logger.warn('[deprecated] The `ENGINE_SCHEMA_TAG` environment variable has been renamed to `APOLLO_GRAPH_VARIANT`.');
        }
        if (process.env.ENGINE_SCHEMA_TAG && process.env.APOLLO_GRAPH_VARIANT) {
            throw new Error('`APOLLO_GRAPH_VARIANT` and `ENGINE_SCHEMA_TAG` (deprecated) environment variables must not both be set.');
        }
        return process.env.APOLLO_GRAPH_VARIANT || process.env.ENGINE_SCHEMA_TAG;
    }
}
exports.getEngineGraphVariant = getEngineGraphVariant;
const serviceHeaderDefaults = {
    hostname: os_1.default.hostname(),
    agentVersion: `apollo-engine-reporting@${require('../package.json').version}`,
    runtimeVersion: `node ${process.version}`,
    uname: `${os_1.default.platform()}, ${os_1.default.type()}, ${os_1.default.release()}, ${os_1.default.arch()})`,
};
class ReportData {
    constructor(executableSchemaId, graphVariant) {
        this.header = new apollo_engine_reporting_protobuf_1.ReportHeader(Object.assign(Object.assign({}, serviceHeaderDefaults), { executableSchemaId, schemaTag: graphVariant }));
        this.reset();
    }
    reset() {
        this.report = new apollo_engine_reporting_protobuf_1.Report({ header: this.header });
        this.size = 0;
    }
}
class EngineReportingAgent {
    constructor(options = {}) {
        this.logger = console;
        this.reportDataByExecutableSchemaId = Object.create(null);
        this.stopped = false;
        this.signalHandlers = new Map();
        this.options = options;
        this.apiKey = getEngineApiKey({
            engine: this.options,
            skipWarn: false,
            logger: this.logger,
        });
        if (options.logger)
            this.logger = options.logger;
        this.bootId = uuid_1.v4();
        this.graphVariant = getEngineGraphVariant(options, this.logger) || '';
        if (!this.apiKey) {
            throw new Error(`To use EngineReportingAgent, you must specify an API key via the apiKey option or the APOLLO_KEY environment variable.`);
        }
        if (options.experimental_schemaReporting !== undefined) {
            this.schemaReport = options.experimental_schemaReporting;
        }
        else {
            this.schemaReport = process.env.APOLLO_SCHEMA_REPORTING === "true";
        }
        this.signatureCache = createSignatureCache({ logger: this.logger });
        this.sendReportsImmediately = options.sendReportsImmediately;
        if (!this.sendReportsImmediately) {
            this.reportTimer = setInterval(() => this.sendAllReportsAndReportErrors(), this.options.reportIntervalMs || 10 * 1000);
        }
        if (this.options.handleSignals !== false) {
            const signals = ['SIGINT', 'SIGTERM'];
            signals.forEach((signal) => {
                const handler = () => __awaiter(this, void 0, void 0, function* () {
                    this.stop();
                    yield this.sendAllReportsAndReportErrors();
                    process.kill(process.pid, signal);
                });
                process.once(signal, handler);
                this.signalHandlers.set(signal, handler);
            });
        }
        if (this.options.endpointUrl) {
            this.logger.warn('[deprecated] The `endpointUrl` option within `engine` has been renamed to `tracesEndpointUrl`.');
        }
        this.tracesEndpointUrl =
            (this.options.endpointUrl ||
                this.options.tracesEndpointUrl ||
                'https://engine-report.apollodata.com') + '/api/ingress/traces';
        handleLegacyOptions(this.options);
    }
    executableSchemaIdGenerator(schema) {
        var _a;
        if (((_a = this.lastSeenExecutableSchemaToId) === null || _a === void 0 ? void 0 : _a.executableSchema) === schema) {
            return this.lastSeenExecutableSchemaToId.executableSchemaId;
        }
        const id = computeExecutableSchemaId(schema);
        this.lastSeenExecutableSchemaToId = {
            executableSchema: schema,
            executableSchemaId: id,
        };
        return id;
    }
    newPlugin() {
        return plugin_1.plugin(this.options, this.addTrace.bind(this), {
            startSchemaReporting: this.startSchemaReporting.bind(this),
            executableSchemaIdGenerator: this.executableSchemaIdGenerator.bind(this),
            schemaReport: this.schemaReport,
        });
    }
    getReportData(executableSchemaId) {
        const existing = this.reportDataByExecutableSchemaId[executableSchemaId];
        if (existing) {
            return existing;
        }
        const reportData = new ReportData(executableSchemaId, this.graphVariant);
        this.reportDataByExecutableSchemaId[executableSchemaId] = reportData;
        return reportData;
    }
    addTrace({ trace, queryHash, document, operationName, source, executableSchemaId, logger, }) {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.stopped) {
                return;
            }
            const reportData = this.getReportData(executableSchemaId);
            const { report } = reportData;
            const protobufError = apollo_engine_reporting_protobuf_1.Trace.verify(trace);
            if (protobufError) {
                throw new Error(`Error encoding trace: ${protobufError}`);
            }
            const encodedTrace = apollo_engine_reporting_protobuf_1.Trace.encode(trace).finish();
            const signature = yield this.getTraceSignature({
                queryHash,
                document,
                source,
                operationName,
                logger,
            });
            const statsReportKey = `# ${operationName || '-'}\n${signature}`;
            if (!report.tracesPerQuery.hasOwnProperty(statsReportKey)) {
                report.tracesPerQuery[statsReportKey] = new apollo_engine_reporting_protobuf_1.TracesAndStats();
                report.tracesPerQuery[statsReportKey].encodedTraces = [];
            }
            report.tracesPerQuery[statsReportKey].encodedTraces.push(encodedTrace);
            reportData.size += encodedTrace.length + Buffer.byteLength(statsReportKey);
            if (this.sendReportsImmediately ||
                reportData.size >=
                    (this.options.maxUncompressedReportSize || 4 * 1024 * 1024)) {
                yield this.sendReportAndReportErrors(executableSchemaId);
            }
        });
    }
    sendAllReports() {
        return __awaiter(this, void 0, void 0, function* () {
            yield Promise.all(Object.keys(this.reportDataByExecutableSchemaId).map((id) => this.sendReport(id)));
        });
    }
    sendReport(executableSchemaId) {
        return __awaiter(this, void 0, void 0, function* () {
            const reportData = this.getReportData(executableSchemaId);
            const { report } = reportData;
            reportData.reset();
            if (Object.keys(report.tracesPerQuery).length === 0) {
                return;
            }
            yield Promise.resolve();
            if (this.options.debugPrintReports) {
                this.logger.warn(`Engine sending report: ${JSON.stringify(report.toJSON())}`);
            }
            const protobufError = apollo_engine_reporting_protobuf_1.Report.verify(report);
            if (protobufError) {
                throw new Error(`Error encoding report: ${protobufError}`);
            }
            const message = apollo_engine_reporting_protobuf_1.Report.encode(report).finish();
            const compressed = yield new Promise((resolve, reject) => {
                const messageBuffer = Buffer.from(message.buffer, message.byteOffset, message.byteLength);
                zlib_1.gzip(messageBuffer, (err, gzipResult) => {
                    if (err) {
                        reject(err);
                    }
                    else {
                        resolve(gzipResult);
                    }
                });
            });
            const response = yield async_retry_1.default(() => __awaiter(this, void 0, void 0, function* () {
                const curResponse = yield apollo_server_env_1.fetch(this.tracesEndpointUrl, {
                    method: 'POST',
                    headers: {
                        'user-agent': 'apollo-engine-reporting',
                        'x-api-key': this.apiKey,
                        'content-encoding': 'gzip',
                    },
                    body: compressed,
                    agent: this.options.requestAgent,
                });
                if (curResponse.status >= 500 && curResponse.status < 600) {
                    throw new Error(`HTTP status ${curResponse.status}, ${(yield curResponse.text()) || '(no body)'}`);
                }
                else {
                    return curResponse;
                }
            }), {
                retries: (this.options.maxAttempts || 5) - 1,
                minTimeout: this.options.minimumRetryDelayMs || 100,
                factor: 2,
            }).catch((err) => {
                throw new Error(`Error sending report to Apollo Engine servers: ${err.message}`);
            });
            if (response.status < 200 || response.status >= 300) {
                throw new Error(`Error sending report to Apollo Engine servers: HTTP status ${response.status}, ${(yield response.text()) || '(no body)'}`);
            }
            if (this.options.debugPrintReports) {
                this.logger.warn(`Engine report: status ${response.status}`);
            }
        });
    }
    startSchemaReporting({ executableSchemaId, executableSchema, }) {
        this.logger.info('Starting schema reporter...');
        this.logger.info(`Schema reporter options: ${JSON.stringify({
            overrideReportedSchema: this.options.experimental_overrideReportedSchema,
            schemaReportingInitialDelayMaxMs: this.options.experimental_schemaReportingInitialDelayMaxMs,
            schemaReportingUrl: this.options.schemaReportingUrl,
        })}`);
        if (this.currentSchemaReporter) {
            this.currentSchemaReporter.stop();
        }
        const serverInfo = {
            bootId: this.bootId,
            graphVariant: this.graphVariant,
            platform: process.env.APOLLO_SERVER_PLATFORM || 'local',
            runtimeVersion: `node ${process.version}`,
            executableSchemaId: executableSchemaId,
            userVersion: process.env.APOLLO_SERVER_USER_VERSION,
            serverId: process.env.APOLLO_SERVER_ID || process.env.HOSTNAME || os_1.default.hostname(),
            libraryVersion: `apollo-engine-reporting@${require('../package.json').version}`,
        };
        this.logger.info(`Schema reporter EdgeServerInfo: ${JSON.stringify(serverInfo)}`);
        const delay = Math.floor(Math.random() *
            (this.options.experimental_schemaReportingInitialDelayMaxMs || 10000));
        const schemaReporter = new schemaReporter_1.SchemaReporter(serverInfo, executableSchema, this.apiKey, this.options.schemaReportingUrl);
        const fallbackReportingDelayInMs = 20000;
        this.currentSchemaReporter = schemaReporter;
        const logger = this.logger;
        setTimeout(function () {
            schemaReporter_1.reportingLoop(schemaReporter, logger, false, fallbackReportingDelayInMs);
        }, delay);
    }
    stop() {
        this.signalHandlers.forEach((handler, signal) => {
            process.removeListener(signal, handler);
        });
        if (this.reportTimer) {
            clearInterval(this.reportTimer);
            this.reportTimer = undefined;
        }
        if (this.currentSchemaReporter) {
            this.currentSchemaReporter.stop();
        }
        this.stopped = true;
    }
    getTraceSignature({ queryHash, operationName, document, source, logger, }) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!document && !source) {
                throw new Error('No document or source?');
            }
            const cacheKey = signatureCacheKey(queryHash, operationName);
            const cachedSignature = yield this.signatureCache.get(cacheKey);
            if (cachedSignature) {
                return cachedSignature;
            }
            if (!document) {
                return source;
            }
            const generatedSignature = (this.options.calculateSignature || apollo_graphql_1.defaultEngineReportingSignature)(document, operationName);
            this.signatureCache.set(cacheKey, generatedSignature)
                .catch(err => {
                logger.warn('Could not store signature cache. ' +
                    (err && err.message) || err);
            });
            return generatedSignature;
        });
    }
    sendAllReportsAndReportErrors() {
        return __awaiter(this, void 0, void 0, function* () {
            yield Promise.all(Object.keys(this.reportDataByExecutableSchemaId).map((executableSchemaId) => this.sendReportAndReportErrors(executableSchemaId)));
        });
    }
    sendReportAndReportErrors(executableSchemaId) {
        return this.sendReport(executableSchemaId).catch((err) => {
            if (this.options.reportErrorFunction) {
                this.options.reportErrorFunction(err);
            }
            else {
                this.logger.error(err.message);
            }
        });
    }
}
exports.EngineReportingAgent = EngineReportingAgent;
function createSignatureCache({ logger, }) {
    let lastSignatureCacheWarn;
    let lastSignatureCacheDisposals = 0;
    return new apollo_server_caching_1.InMemoryLRUCache({
        sizeCalculator(obj) {
            return Buffer.byteLength(JSON.stringify(obj), 'utf8');
        },
        maxSize: Math.pow(2, 20) * 3,
        onDispose() {
            lastSignatureCacheDisposals++;
            if (!lastSignatureCacheWarn ||
                new Date().getTime() - lastSignatureCacheWarn.getTime() > 60000) {
                lastSignatureCacheWarn = new Date();
                logger.warn([
                    'This server is processing a high number of unique operations.  ',
                    `A total of ${lastSignatureCacheDisposals} records have been `,
                    'ejected from the Engine Reporting signature cache in the past ',
                    'interval.  If you see this warning frequently, please open an ',
                    'issue on the Apollo Server repository.',
                ].join(''));
                lastSignatureCacheDisposals = 0;
            }
        },
    });
}
function signatureCacheKey(queryHash, operationName) {
    return `${queryHash}${operationName && ':' + operationName}`;
}
exports.signatureCacheKey = signatureCacheKey;
function handleLegacyOptions(options) {
    if (typeof options.privateVariables !== 'undefined' &&
        options.sendVariableValues) {
        throw new Error("You have set both the 'sendVariableValues' and the deprecated 'privateVariables' options. Please only set 'sendVariableValues'.");
    }
    else if (typeof options.privateVariables !== 'undefined') {
        if (options.privateVariables !== null) {
            options.sendVariableValues = makeSendValuesBaseOptionsFromLegacy(options.privateVariables);
        }
        delete options.privateVariables;
    }
    if (typeof options.privateHeaders !== 'undefined' && options.sendHeaders) {
        throw new Error("You have set both the 'sendHeaders' and the deprecated 'privateHeaders' options. Please only set 'sendHeaders'.");
    }
    else if (typeof options.privateHeaders !== 'undefined') {
        if (options.privateHeaders !== null) {
            options.sendHeaders = makeSendValuesBaseOptionsFromLegacy(options.privateHeaders);
        }
        delete options.privateHeaders;
    }
}
exports.handleLegacyOptions = handleLegacyOptions;
function makeSendValuesBaseOptionsFromLegacy(legacyPrivateOption) {
    return Array.isArray(legacyPrivateOption)
        ? {
            exceptNames: legacyPrivateOption,
        }
        : legacyPrivateOption
            ? { none: true }
            : { all: true };
}
function computeExecutableSchemaId(schema) {
    const sha256 = crypto_1.createHash('sha256');
    const schemaDocument = typeof schema === 'string' ? schema : graphql_1.printSchema(schema);
    return sha256.update(schemaDocument).digest('hex');
}
exports.computeExecutableSchemaId = computeExecutableSchemaId;
//# sourceMappingURL=agent.js.map