/*eslint no-magic-numbers: ["error", { "ignore": [-1, 0, 1, 2] }]*/
import React from "react";
import PropTypes from "prop-types";
import { assign } from "lodash";
import * as d3Shape from "d3-shape";
import { Helpers, CommonProps, Path } from "victory-core";

const defined = (d) => {
  const y = d._y1 !== undefined ? d._y1 : d._y;
  return y !== null && y !== undefined && d._y0 !== null;
};

const getXAccessor = (scale) => {
  return (d) => scale.x(d._x1 !== undefined ? d._x1 : d._x);
};

const getYAccessor = (scale) => {
  return (d) => scale.y(d._y1 !== undefined ? d._y1 : d._y);
};

const getAngleAccessor = (scale) => {
  return (d) => {
    const x = scale.x(d._x1 !== undefined ? d._x1 : d._x);
    return -1 * x + Math.PI / 2;
  };
};

const toNewName = (interpolation) => {
  // d3 shape changed the naming scheme for interpolators from "basis" -> "curveBasis" etc.
  const capitalize = (s) => s && s[0].toUpperCase() + s.slice(1);
  return `curve${capitalize(interpolation)}`;
};

const getLineFunction = (props) => {
  const { polar, scale, horizontal } = props;
  const defaultOpenCurve = polar ? false : true;
  const openCurve =
    props.openCurve === undefined ? defaultOpenCurve : props.openCurve;
  const interpolationFunction =
    typeof props.interpolation === "function" && props.interpolation;
  const interpolationName =
    typeof props.interpolation === "string" &&
    (!openCurve
      ? `${toNewName(props.interpolation)}Closed`
      : toNewName(props.interpolation));
  return polar
    ? d3Shape
        .lineRadial()
        .defined(defined)
        .curve(interpolationFunction || d3Shape[interpolationName])
        .angle(getAngleAccessor(scale))
        .radius(getYAccessor(scale))
    : d3Shape
        .line()
        .defined(defined)
        .curve(interpolationFunction || d3Shape[interpolationName])
        .x(horizontal ? getYAccessor(scale) : getXAccessor(scale))
        .y(horizontal ? getXAccessor(scale) : getYAccessor(scale));
};

const evaluateProps = (props) => {
  /**
   * Potential evaluated props are:
   * `ariaLabel`
   * `id`
   * `style`
   * `tabIndex`
   */
  const ariaLabel = Helpers.evaluateProp(props.ariaLabel, props);
  const id = Helpers.evaluateProp(props.id, props);
  const style = Helpers.evaluateStyle(
    assign({ fill: "none", stroke: "black" }, props.style),
    props
  );
  const tabIndex = Helpers.evaluateProp(props.tabIndex, props);

  return assign({}, props, { ariaLabel, id, style, tabIndex });
};

const Curve = (props) => {
  props = evaluateProps(props);
  const { polar, origin } = props;
  const lineFunction = getLineFunction(props);
  const defaultTransform =
    polar && origin ? `translate(${origin.x}, ${origin.y})` : undefined;

  return React.cloneElement(props.pathComponent, {
    ...props.events,
    "aria-label": props.ariaLabel,
    d: lineFunction(props.data),
    style: props.style,
    transform: props.transform || defaultTransform,
    className: props.className,
    role: props.role,
    shapeRendering: props.shapeRendering,
    clipPath: props.clipPath,
    tabIndex: props.tabIndex
  });
};

Curve.propTypes = {
  ...CommonProps.primitiveProps,
  interpolation: PropTypes.oneOfType([PropTypes.string, PropTypes.func]),
  openCurve: PropTypes.bool,
  origin: PropTypes.object,
  pathComponent: PropTypes.element,
  polar: PropTypes.bool
};

Curve.defaultProps = {
  pathComponent: <Path />,
  role: "presentation",
  shapeRendering: "auto"
};

export default Curve;
