/*
 *
 *  Wireless daemon for Linux
 *
 *  Copyright (C) 2015-2016  Intel Corporation. All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <stdlib.h>
#include <linux/if_ether.h>
#include <ell/ell.h>
#include <ell/key-private.h>

#include "src/wscutil.h"
#include "src/eapol.h"
#include "src/eap.h"
#include "src/eap-wsc.h"
#include "src/crypto.h"
#include "src/util.h"
#include "src/mpdu.h"
#include "src/ie.h"
#include "src/handshake.h"

struct test_handshake_state {
	struct handshake_state super;
};

static void test_handshake_state_free(struct handshake_state *hs)
{
	struct test_handshake_state *ths =
			container_of(hs, struct test_handshake_state, super);

	l_free(ths);
}

static struct handshake_state *test_handshake_state_new(uint32_t ifindex)
{
	struct test_handshake_state *ths;

	ths = l_new(struct test_handshake_state, 1);

	ths->super.ifindex = ifindex;
	ths->super.free = test_handshake_state_free;

	return &ths->super;
}

static const unsigned char wsc_attrs1[] = {
	0x10, 0x4a, 0x00, 0x01, 0x10, 0x10, 0x44, 0x00, 0x01, 0x02, 0x10, 0x41,
	0x00, 0x01, 0x01, 0x10, 0x12, 0x00, 0x02, 0x00, 0x04, 0x10, 0x53, 0x00,
	0x02, 0x26, 0x88, 0x10, 0x3b, 0x00, 0x01, 0x03, 0x10, 0x47, 0x00, 0x10,
	0xc4, 0x4a, 0xad, 0x8d, 0x25, 0x2f, 0x52, 0xc6, 0xf9, 0x6b, 0x38, 0x5d,
	0xcb, 0x23, 0x31, 0xae, 0x10, 0x21, 0x00, 0x15, 0x41, 0x53, 0x55, 0x53,
	0x54, 0x65, 0x4b, 0x20, 0x43, 0x6f, 0x6d, 0x70, 0x75, 0x74, 0x65, 0x72,
	0x20, 0x49, 0x6e, 0x63, 0x2e, 0x10, 0x23, 0x00, 0x1c, 0x57, 0x69, 0x2d,
	0x46, 0x69, 0x20, 0x50, 0x72, 0x6f, 0x74, 0x65, 0x63, 0x74, 0x65, 0x64,
	0x20, 0x53, 0x65, 0x74, 0x75, 0x70, 0x20, 0x52, 0x6f, 0x75, 0x74, 0x65,
	0x72, 0x10, 0x24, 0x00, 0x08, 0x52, 0x54, 0x2d, 0x41, 0x43, 0x36, 0x38,
	0x55, 0x10, 0x42, 0x00, 0x11, 0x31, 0x30, 0x3a, 0x63, 0x33, 0x3a, 0x37,
	0x62, 0x3a, 0x35, 0x34, 0x3a, 0x37, 0x34, 0x3a, 0x64, 0x30, 0x10, 0x54,
	0x00, 0x08, 0x00, 0x06, 0x00, 0x50, 0xf2, 0x04, 0x00, 0x01, 0x10, 0x11,
	0x00, 0x08, 0x52, 0x54, 0x2d, 0x41, 0x43, 0x36, 0x38, 0x55, 0x10, 0x08,
	0x00, 0x02, 0x20, 0x08, 0x10, 0x3c, 0x00, 0x01, 0x03, 0x10, 0x49, 0x00,
	0x0e, 0x00, 0x37, 0x2a, 0x00, 0x01, 0x20, 0x01, 0x06, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff,
};

static void wsc_test_iter_sanity_check(const void *data)
{
	struct wsc_attr_iter iter;
	struct wsc_wfa_ext_iter wfa_iter;

	wsc_attr_iter_init(&iter, wsc_attrs1, sizeof(wsc_attrs1));

	assert(wsc_attr_iter_next(&iter));
	assert(wsc_attr_iter_get_type(&iter) == WSC_ATTR_VERSION);
	assert(wsc_attr_iter_get_length(&iter) == 1);

	assert(wsc_attr_iter_next(&iter));
	assert(wsc_attr_iter_get_type(&iter) == WSC_ATTR_WSC_STATE);
	assert(wsc_attr_iter_get_length(&iter) == 1);

	assert(wsc_attr_iter_next(&iter));
	assert(wsc_attr_iter_get_type(&iter) == WSC_ATTR_SELECTED_REGISTRAR);
	assert(wsc_attr_iter_get_length(&iter) == 1);

	assert(wsc_attr_iter_next(&iter));
	assert(wsc_attr_iter_get_type(&iter) == WSC_ATTR_DEVICE_PASSWORD_ID);
	assert(wsc_attr_iter_get_length(&iter) == 2);

	assert(wsc_attr_iter_next(&iter));
	assert(wsc_attr_iter_get_type(&iter) ==
			WSC_ATTR_SELECTED_REGISTRAR_CONFIGURATION_METHODS);
	assert(wsc_attr_iter_get_length(&iter) == 2);

	assert(wsc_attr_iter_next(&iter));
	assert(wsc_attr_iter_get_type(&iter) == WSC_ATTR_RESPONSE_TYPE);
	assert(wsc_attr_iter_get_length(&iter) == 1);

	assert(wsc_attr_iter_next(&iter));
	assert(wsc_attr_iter_get_type(&iter) == WSC_ATTR_UUID_E);
	assert(wsc_attr_iter_get_length(&iter) == 16);

	assert(wsc_attr_iter_next(&iter));
	assert(wsc_attr_iter_get_type(&iter) == WSC_ATTR_MANUFACTURER);
	assert(wsc_attr_iter_get_length(&iter) == 21);

	assert(wsc_attr_iter_next(&iter));
	assert(wsc_attr_iter_get_type(&iter) == WSC_ATTR_MODEL_NAME);
	assert(wsc_attr_iter_get_length(&iter) == 28);

	assert(wsc_attr_iter_next(&iter));
	assert(wsc_attr_iter_get_type(&iter) == WSC_ATTR_MODEL_NUMBER);
	assert(wsc_attr_iter_get_length(&iter) == 8);

	assert(wsc_attr_iter_next(&iter));
	assert(wsc_attr_iter_get_type(&iter) == WSC_ATTR_SERIAL_NUMBER);
	assert(wsc_attr_iter_get_length(&iter) == 17);

	assert(wsc_attr_iter_next(&iter));
	assert(wsc_attr_iter_get_type(&iter) == WSC_ATTR_PRIMARY_DEVICE_TYPE);
	assert(wsc_attr_iter_get_length(&iter) == 8);

	assert(wsc_attr_iter_next(&iter));
	assert(wsc_attr_iter_get_type(&iter) == WSC_ATTR_DEVICE_NAME);
	assert(wsc_attr_iter_get_length(&iter) == 8);

	assert(wsc_attr_iter_next(&iter));
	assert(wsc_attr_iter_get_type(&iter) == WSC_ATTR_CONFIGURATION_METHODS);
	assert(wsc_attr_iter_get_length(&iter) == 2);

	assert(wsc_attr_iter_next(&iter));
	assert(wsc_attr_iter_get_type(&iter) == WSC_ATTR_RF_BANDS);
	assert(wsc_attr_iter_get_length(&iter) == 1);

	assert(wsc_attr_iter_next(&iter));
	assert(wsc_attr_iter_get_type(&iter) == WSC_ATTR_VENDOR_EXTENSION);
	assert(wsc_attr_iter_get_length(&iter) == 14);

	assert(wsc_attr_iter_recurse_wfa_ext(&iter, &wfa_iter));

	assert(wsc_wfa_ext_iter_next(&wfa_iter));
	assert(wsc_wfa_ext_iter_get_type(&wfa_iter) ==
			WSC_WFA_EXTENSION_VERSION2);
	assert(wsc_wfa_ext_iter_get_length(&wfa_iter) == 1);

	assert(wsc_wfa_ext_iter_next(&wfa_iter));
	assert(wsc_wfa_ext_iter_get_type(&wfa_iter) ==
			WSC_WFA_EXTENSION_AUTHORIZED_MACS);
	assert(wsc_wfa_ext_iter_get_length(&wfa_iter) == 6);

	assert(!wsc_attr_iter_next(&iter));
}


static const unsigned char beacon1[] = {
	0x10, 0x4a, 0x00, 0x01, 0x10, 0x10, 0x44, 0x00, 0x01, 0x02, 0x10, 0x49,
	0x00, 0x06, 0x00, 0x37, 0x2a, 0x00, 0x01, 0x20
};

struct beacon_data {
	struct wsc_beacon expected;
	const void *pdu;
	unsigned int len;
};

static const struct beacon_data beacon_data_1 = {
	.pdu = beacon1,
	.len = sizeof(beacon1),
	.expected = {
		.version2 = true,
		.state = WSC_STATE_CONFIGURED,
		.ap_setup_locked = false,
		.selected_registrar = false,
	},
};

static void wsc_test_parse_beacon(const void *data)
{
	const struct beacon_data *test = data;
	struct wsc_beacon beacon;
	const struct wsc_beacon *expected = &test->expected;
	int r;

	r = wsc_parse_beacon(test->pdu, test->len, &beacon);
	assert(r == 0);

	assert(expected->version2 == beacon.version2);
	assert(expected->state == beacon.state);
	assert(expected->ap_setup_locked == beacon.ap_setup_locked);
	assert(expected->selected_registrar == beacon.selected_registrar);
	assert(expected->device_password_id == beacon.device_password_id);
	assert(expected->selected_reg_config_methods ==
				beacon.selected_reg_config_methods);
	assert(!memcmp(expected->uuid_e, beacon.uuid_e, 16));
	assert(expected->rf_bands == beacon.rf_bands);

	assert(!memcmp(expected->authorized_macs,
				beacon.authorized_macs,
				sizeof(beacon.authorized_macs)));
	assert(expected->reg_config_methods ==
				beacon.reg_config_methods);
}

struct probe_response_data {
	struct wsc_probe_response expected;
	const void *pdu;
	unsigned int len;
};

static const struct probe_response_data probe_response_data_1 = {
	.pdu = wsc_attrs1,
	.len = sizeof(wsc_attrs1),
	.expected = {
		.version2 = true,
		.state = WSC_STATE_CONFIGURED,
		.ap_setup_locked = false,
		.selected_registrar = true,
		.device_password_id = WSC_DEVICE_PASSWORD_ID_PUSH_BUTTON,
		.selected_reg_config_methods =
				WSC_CONFIGURATION_METHOD_VIRTUAL_DISPLAY_PIN |
				WSC_CONFIGURATION_METHOD_VIRTUAL_PUSH_BUTTON |
				WSC_CONFIGURATION_METHOD_PHYSICAL_PUSH_BUTTON,
		.response_type = WSC_RESPONSE_TYPE_AP,
		.uuid_e = { 0xc4, 0x4a, 0xad, 0x8d, 0x25, 0x2f, 0x52, 0xc6,
			0xf9, 0x6b, 0x38, 0x5d, 0xcb, 0x23, 0x31, 0xae },
		.manufacturer = "ASUSTeK Computer Inc.",
		.model_name = "Wi-Fi Protected Setup Router",
		.model_number = "RT-AC68U",
		.serial_number = "10:c3:7b:54:74:d0",
		.primary_device_type = {
			.category = 6,
			.oui = { 0x00, 0x50, 0xf2 },
			.oui_type = 0x04,
			.subcategory = 1, },
		.device_name = "RT-AC68U",
		.config_methods = WSC_CONFIGURATION_METHOD_VIRTUAL_DISPLAY_PIN,
		.rf_bands = WSC_RF_BAND_2_4_GHZ | WSC_RF_BAND_5_0_GHZ,
		.authorized_macs = { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, },
	},
};

static void wsc_test_parse_probe_response(const void *data)
{
	const struct probe_response_data *test = data;
	struct wsc_probe_response probe_response;
	const struct wsc_probe_response *expected = &test->expected;
	int r;

	r = wsc_parse_probe_response(test->pdu, test->len, &probe_response);
	assert(r == 0);

	assert(expected->version2 == probe_response.version2);
	assert(expected->state == probe_response.state);
	assert(expected->ap_setup_locked == probe_response.ap_setup_locked);
	assert(expected->selected_registrar ==
					probe_response.selected_registrar);
	assert(expected->device_password_id ==
					probe_response.device_password_id);
	assert(expected->selected_reg_config_methods ==
				probe_response.selected_reg_config_methods);
	assert(expected->response_type == probe_response.response_type);
	assert(!memcmp(expected->uuid_e, probe_response.uuid_e, 16));
	assert(!strcmp(expected->manufacturer, probe_response.manufacturer));
	assert(!strcmp(expected->model_name, probe_response.model_name));
	assert(!strcmp(expected->model_number, probe_response.model_number));
	assert(!strcmp(expected->serial_number, probe_response.serial_number));

	assert(expected->primary_device_type.category ==
				probe_response.primary_device_type.category);
	assert(!memcmp(expected->primary_device_type.oui,
				probe_response.primary_device_type.oui, 3));
	assert(expected->primary_device_type.oui_type ==
				probe_response.primary_device_type.oui_type);
	assert(expected->primary_device_type.subcategory ==
				probe_response.primary_device_type.subcategory);

	assert(!strcmp(expected->device_name, probe_response.device_name));
	assert(expected->config_methods == probe_response.config_methods);
	assert(expected->rf_bands == probe_response.rf_bands);

	assert(!memcmp(expected->authorized_macs,
				probe_response.authorized_macs,
				sizeof(probe_response.authorized_macs)));
	assert(expected->reg_config_methods ==
				probe_response.reg_config_methods);
}

static const unsigned char probe_request1[] = {
	0x10, 0x4a, 0x00, 0x01, 0x10, 0x10, 0x3a, 0x00, 0x01, 0x01, 0x10, 0x08,
	0x00, 0x02, 0x21, 0x48, 0x10, 0x47, 0x00, 0x10, 0x79, 0x0c, 0x1f, 0x80,
	0x4f, 0x2b, 0x52, 0xb7, 0xbe, 0x30, 0xc0, 0xe9, 0x72, 0x92, 0x08, 0x8d,
	0x10, 0x54, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x10, 0x3c, 0x00, 0x01, 0x03, 0x10, 0x02, 0x00, 0x02, 0x00, 0x00, 0x10,
	0x09, 0x00, 0x02, 0x00, 0x00, 0x10, 0x12, 0x00, 0x02, 0x00, 0x04, 0x10,
	0x21, 0x00, 0x01, 0x20, 0x10, 0x23, 0x00, 0x01, 0x20, 0x10, 0x24, 0x00,
	0x01, 0x20, 0x10, 0x11, 0x00, 0x01, 0x20, 0x10, 0x49, 0x00, 0x09, 0x00,
	0x37, 0x2a, 0x00, 0x01, 0x20, 0x03, 0x01, 0x01,
};

struct probe_request_data {
	struct wsc_probe_request expected;
	const void *pdu;
	unsigned int len;
};

static const struct probe_request_data probe_request_data_1 = {
	.pdu = probe_request1,
	.len = sizeof(probe_request1),
	.expected = {
		.version2 = true,
		.request_type = WSC_REQUEST_TYPE_ENROLLEE_OPEN_8021X,
		.config_methods =
				WSC_CONFIGURATION_METHOD_VIRTUAL_DISPLAY_PIN |
				WSC_CONFIGURATION_METHOD_NFC_INTERFACE |
				WSC_CONFIGURATION_METHOD_KEYPAD,
		.uuid_e = { 0x79, 0x0c, 0x1f, 0x80, 0x4f, 0x2b, 0x52, 0xb7,
			0xbe, 0x30, 0xc0, 0xe9, 0x72, 0x92, 0x08, 0x8d },
		.primary_device_type = {
			.category = 0,
			.oui = { 0x00, 0x00, 0x00 },
			.oui_type = 0x00,
			.subcategory = 0, },
		.rf_bands = WSC_RF_BAND_2_4_GHZ | WSC_RF_BAND_5_0_GHZ,
		.association_state = WSC_ASSOCIATION_STATE_NOT_ASSOCIATED,
		.configuration_error = WSC_CONFIGURATION_ERROR_NO_ERROR,
		.device_password_id = WSC_DEVICE_PASSWORD_ID_PUSH_BUTTON,
		.manufacturer = " ",
		.model_name = " ",
		.model_number = " ",
		.device_name = " ",
		.request_to_enroll = true,
	},
};

static void wsc_test_parse_probe_request(const void *data)
{
	const struct probe_request_data *test = data;
	struct wsc_probe_request probe_request;
	const struct wsc_probe_request *expected = &test->expected;
	int r;

	r = wsc_parse_probe_request(test->pdu, test->len, &probe_request);
	assert(r == 0);

	assert(expected->version2 == probe_request.version2);
	assert(expected->request_type == probe_request.request_type);

	assert(expected->config_methods == probe_request.config_methods);
	assert(!memcmp(expected->uuid_e, probe_request.uuid_e, 16));

	assert(expected->primary_device_type.category ==
				probe_request.primary_device_type.category);
	assert(!memcmp(expected->primary_device_type.oui,
				probe_request.primary_device_type.oui, 3));
	assert(expected->primary_device_type.oui_type ==
				probe_request.primary_device_type.oui_type);
	assert(expected->primary_device_type.subcategory ==
				probe_request.primary_device_type.subcategory);

	assert(expected->rf_bands == probe_request.rf_bands);
	assert(expected->association_state == probe_request.association_state);
	assert(expected->configuration_error ==
				probe_request.configuration_error);
	assert(expected->device_password_id ==
					probe_request.device_password_id);

	assert(!strcmp(expected->manufacturer, probe_request.manufacturer));
	assert(!strcmp(expected->model_name, probe_request.model_name));
	assert(!strcmp(expected->model_number, probe_request.model_number));
	assert(!strcmp(expected->device_name, probe_request.device_name));

	assert(expected->request_to_enroll == probe_request.request_to_enroll);

	assert(expected->requested_device_type.category ==
				probe_request.requested_device_type.category);
	assert(!memcmp(expected->requested_device_type.oui,
				probe_request.requested_device_type.oui, 3));
	assert(expected->requested_device_type.oui_type ==
				probe_request.requested_device_type.oui_type);
	assert(expected->requested_device_type.subcategory ==
			probe_request.requested_device_type.subcategory);
}

static void wsc_test_build_probe_request(const void *data)
{
	const struct probe_request_data *test = data;
	uint8_t *pr;
	size_t prlen;

	pr = wsc_build_probe_request(&test->expected, &prlen);
	assert(pr);

	assert(!memcmp(test->pdu, pr, test->len));

	l_free(pr);
}

struct uuid_from_addr_data {
	uint8_t addr[6];
	uint8_t expected_uuid[16];
};

static const struct uuid_from_addr_data uuid_from_addr_data_1 = {
	.addr = { 0xa0, 0xa8, 0xcd, 0x1c, 0x7e, 0xc9 },
	.expected_uuid = { 0x79, 0x0c, 0x1f, 0x80, 0x4f, 0x2b, 0x52, 0xb7,
			0xbe, 0x30, 0xc0, 0xe9, 0x72, 0x92, 0x08, 0x8d },
};

static void wsc_test_uuid_from_addr(const void *data)
{
	const struct uuid_from_addr_data *test = data;
	uint8_t uuid[16];

	assert(wsc_uuid_from_addr(test->addr, uuid));

	assert(!memcmp(test->expected_uuid, uuid, 16));
}

static const unsigned char dh_private_key_1[] = {
	0x4b, 0x5e, 0x01, 0xfa, 0x2b, 0xd4, 0x91, 0xac, 0x34, 0x04, 0x28, 0xfb,
	0xe2, 0x5d, 0x06, 0x2d, 0x1a, 0xe7, 0x6c, 0xd0, 0xa3, 0x9d, 0x4a, 0x27,
	0x24, 0x51, 0x79, 0xdc, 0xa7, 0x47, 0x24, 0x1d, 0x19, 0x26, 0x9e, 0xb8,
	0xa0, 0x6e, 0xd9, 0x7e, 0x07, 0x4e, 0x02, 0xe1, 0xb2, 0x4e, 0x31, 0x12,
	0xa2, 0xc1, 0x41, 0x73, 0x6a, 0x43, 0xe2, 0x0e, 0xc8, 0xec, 0xb7, 0x41,
	0xdc, 0x25, 0x4c, 0x30, 0xd0, 0x3f, 0xa7, 0x97, 0x9f, 0x2e, 0xdc, 0x59,
	0x0d, 0x7a, 0x24, 0xcb, 0x1b, 0xc2, 0x8d, 0x30, 0xc5, 0x04, 0xc3, 0x50,
	0x61, 0xa7, 0xd2, 0xbd, 0x79, 0xb3, 0xec, 0xd2, 0xf5, 0x10, 0x20, 0x32,
	0x0c, 0x7a, 0xfd, 0x0c, 0xde, 0x07, 0x22, 0x94, 0x2d, 0x42, 0x16, 0xc8,
	0xa7, 0x17, 0x70, 0x16, 0x46, 0x72, 0xaa, 0xb5, 0xa7, 0xdc, 0xc0, 0xee,
	0x7d, 0xe8, 0x36, 0x0b, 0xba, 0x04, 0xd5, 0x5c, 0x5d, 0x40, 0x65, 0x58,
	0xd8, 0xe8, 0x9a, 0x27, 0xd2, 0xb3, 0x39, 0x2a, 0xe7, 0xfc, 0x16, 0x59,
	0xc6, 0x6c, 0x2f, 0xad, 0x6f, 0xd2, 0x0d, 0xf4, 0xb0, 0x8b, 0xdf, 0x9a,
	0x88, 0x84, 0xf9, 0x2a, 0xc4, 0x64, 0xb7, 0xb5, 0xe0, 0xf7, 0x3a, 0x8a,
	0xeb, 0xd0, 0xde, 0xad, 0x1b, 0x0a, 0x44, 0xf2, 0x6f, 0xf5, 0xce, 0x8c,
	0x70, 0xd6, 0x67, 0x1b, 0xb2, 0xa5, 0x9f, 0xeb, 0x63, 0x7f, 0x21, 0xdc,
};

static const unsigned char dh_public_key_1[] = {
	0xb2, 0xfc, 0xd6, 0x4f, 0xf6, 0x71, 0x5a, 0x33, 0x84, 0x60, 0x4a, 0xe8,
	0x2c, 0x1e, 0x55, 0x4a, 0xdb, 0xd5, 0x18, 0x17, 0x91, 0xa6, 0xf5, 0x70,
	0xcd, 0x23, 0xd7, 0x12, 0x6e, 0x4c, 0xaf, 0x27, 0x9a, 0x4e, 0xf5, 0x37,
	0xea, 0x8f, 0x03, 0xc9, 0x0e, 0x79, 0xc5, 0x8d, 0x37, 0xf8, 0xfb, 0x11,
	0xa1, 0x39, 0x19, 0x9b, 0x5a, 0x3a, 0x66, 0x36, 0x6d, 0xfb, 0xae, 0xed,
	0xfc, 0xa5, 0x90, 0xcb, 0xb3, 0xe1, 0xd5, 0x92, 0x2e, 0xe9, 0x99, 0xbd,
	0x0b, 0x93, 0x82, 0x57, 0xe1, 0xbd, 0x70, 0x17, 0xa7, 0x78, 0x7a, 0x0a,
	0xff, 0x42, 0x06, 0x95, 0x2c, 0x0b, 0x6c, 0x1a, 0x6b, 0x2f, 0x6b, 0xed,
	0x42, 0xa5, 0x60, 0x8a, 0xb0, 0xb5, 0x79, 0x1b, 0xa9, 0xe6, 0x15, 0x17,
	0xa3, 0x6c, 0xe9, 0x84, 0xb3, 0x77, 0x48, 0x9b, 0x7a, 0x4d, 0x04, 0xf6,
	0xb8, 0x27, 0xe5, 0x0c, 0xcb, 0x76, 0xfc, 0x3c, 0x65, 0x49, 0xd7, 0x28,
	0x06, 0x8d, 0x99, 0x18, 0x0f, 0xa7, 0x35, 0xb2, 0x9d, 0x15, 0x35, 0x51,
	0xea, 0x83, 0xb6, 0x4d, 0x14, 0xb0, 0x21, 0xa4, 0x82, 0x1f, 0xb8, 0x73,
	0x2b, 0x15, 0x1d, 0x48, 0x99, 0x9f, 0x32, 0x2c, 0xe1, 0xe1, 0xab, 0x66,
	0x3f, 0xb4, 0x40, 0x79, 0xe8, 0x96, 0xe1, 0x9d, 0x54, 0x8b, 0xb6, 0x7f,
	0x1a, 0x5b, 0x5f, 0x09, 0x9f, 0x40, 0xa7, 0x8b, 0xc8, 0xf6, 0x27, 0x80,
};

static const unsigned char dh_private_key_2[] = {
	0x45, 0x2d, 0x9c, 0x56, 0x48, 0xe0, 0xe2, 0x35, 0x6b, 0x47, 0x04, 0x08,
	0x47, 0xf6, 0x3a, 0x8a, 0xa8, 0x65, 0xff, 0xf8, 0x7c, 0x55, 0x16, 0xf4,
	0xf6, 0x11, 0xcd, 0xb8, 0xc6, 0x2c, 0x5f, 0x94, 0x04, 0xdc, 0x69, 0x47,
	0xd0, 0x9c, 0x8d, 0x78, 0x18, 0xd9, 0xab, 0x51, 0xf6, 0xb4, 0x81, 0xec,
	0xd8, 0x4d, 0x61, 0xfe, 0xda, 0x78, 0x9c, 0x95, 0xa1, 0x91, 0xe3, 0x52,
	0x1d, 0x03, 0xdd, 0x70, 0x2f, 0x83, 0x3e, 0x9c, 0xc4, 0x7f, 0x1a, 0x6f,
	0x91, 0xeb, 0xc9, 0x2c, 0xbf, 0xfa, 0x4d, 0x1e, 0xb7, 0x5f, 0x70, 0x72,
	0x92, 0x73, 0xed, 0x28, 0xd5, 0xdc, 0x4b, 0x0d, 0xc2, 0xa6, 0x2b, 0xdf,
	0x2f, 0xdf, 0xf7, 0x16, 0xff, 0xf6, 0x68, 0x63, 0x68, 0xde, 0x72, 0x8a,
	0x9c, 0x67, 0x8e, 0xb7, 0xd9, 0xb1, 0xe9, 0x03, 0x6d, 0xb7, 0xcb, 0xcd,
	0x93, 0x0c, 0x7b, 0x59, 0x26, 0x62, 0xa1, 0x04, 0x41, 0x8d, 0xc7, 0xc3,
	0x02, 0xbf, 0x8c, 0xcf, 0x93, 0x7d, 0xd0, 0xd8, 0x26, 0x19, 0x41, 0x04,
	0x92, 0x4e, 0x4f, 0x11, 0x88, 0x72, 0x78, 0x8b, 0x6d, 0x22, 0x62, 0xdf,
	0x80, 0xd3, 0x79, 0xf5, 0xa5, 0xfa, 0xee, 0xc6, 0x9e, 0xee, 0x4b, 0x94,
	0xd2, 0x77, 0x1e, 0x71, 0xf6, 0xc2, 0x5f, 0xbb, 0xb7, 0xff, 0xcd, 0x49,
	0x1a, 0x80, 0xc3, 0xf8, 0x3f, 0xb4, 0x02, 0x38, 0x84, 0x06, 0xd1, 0xc0,
};

static const unsigned char dh_public_key_2[] = {
	0x42, 0x76, 0x1f, 0xac, 0x65, 0x7b, 0x25, 0xd3, 0x59, 0xe8, 0x5d, 0xde,
	0x22, 0xb1, 0x74, 0x39, 0x7d, 0x0d, 0x6c, 0x56, 0xc9, 0x4f, 0xfc, 0xf2,
	0x1f, 0xd0, 0x4f, 0x70, 0xe9, 0xd5, 0x5c, 0xb6, 0x79, 0x07, 0x54, 0xb6,
	0x8e, 0x67, 0xe1, 0xb5, 0x35, 0x18, 0xc0, 0xe3, 0x36, 0xf9, 0x58, 0x15,
	0x3f, 0x4e, 0xed, 0x74, 0xb5, 0x46, 0xd4, 0x2f, 0x30, 0x22, 0xb2, 0xe1,
	0xa9, 0x4d, 0xf6, 0x3a, 0x00, 0x22, 0x34, 0xf0, 0x14, 0xc5, 0xa3, 0x53,
	0x28, 0x78, 0x91, 0x46, 0x1e, 0x80, 0x3d, 0x7e, 0x41, 0x4a, 0x44, 0x3a,
	0xd0, 0x83, 0x7e, 0xcf, 0xa4, 0x31, 0x0e, 0x0c, 0xfc, 0xcd, 0x63, 0x37,
	0x05, 0x8b, 0x7f, 0x0d, 0x65, 0xef, 0x52, 0x58, 0xb9, 0x9e, 0xb2, 0xfc,
	0x77, 0x7b, 0xdb, 0x79, 0x17, 0x3d, 0xb2, 0x84, 0x4a, 0xaf, 0x09, 0x93,
	0x1c, 0x45, 0x9d, 0x0a, 0x9f, 0xb9, 0x64, 0xec, 0x6f, 0x66, 0x01, 0x49,
	0xb6, 0x00, 0xf7, 0xff, 0x85, 0x4d, 0xa8, 0x60, 0x07, 0xf6, 0x33, 0x1f,
	0xe6, 0x48, 0x9e, 0x71, 0x88, 0x60, 0x88, 0x02, 0xbb, 0x1d, 0x07, 0x68,
	0x7a, 0xa0, 0xfa, 0x4f, 0xfb, 0x42, 0xde, 0x6f, 0x8e, 0x4e, 0x4a, 0xa8,
	0xa7, 0xd3, 0x5d, 0xcb, 0x38, 0x47, 0x16, 0xb6, 0xea, 0x99, 0x9f, 0x7d,
	0xf6, 0xcd, 0xd4, 0xe4, 0x5f, 0x9f, 0xb5, 0xb6, 0x61, 0x9d, 0x42, 0x78,
};

struct dh_generate_pubkey_test {
	const uint8_t *private_key;
	uint32_t private_key_size;
	const uint8_t *public_key;
	uint32_t public_key_size;
};

static const struct dh_generate_pubkey_test dh_generate_pubkey_test_data_1 = {
	.private_key = dh_private_key_1,
	.private_key_size = sizeof(dh_private_key_1),
	.public_key = dh_public_key_1,
	.public_key_size = sizeof(dh_public_key_1),
};

static const struct dh_generate_pubkey_test dh_generate_pubkey_test_data_2 = {
	.private_key = dh_private_key_2,
	.private_key_size = sizeof(dh_private_key_2),
	.public_key = dh_public_key_2,
	.public_key_size = sizeof(dh_public_key_2),
};

static void wsc_test_dh_generate_pubkey(const void *data)
{
	const struct dh_generate_pubkey_test *test = data;
	struct l_key *private;
	struct l_key *generator;
	struct l_key *prime;
	uint8_t public_key[test->private_key_size];
	size_t len;

	memset(public_key, 0, sizeof(public_key));

	generator = l_key_new(L_KEY_RAW, crypto_dh5_generator,
						crypto_dh5_generator_size);
	assert(generator);

	prime = l_key_new(L_KEY_RAW, crypto_dh5_prime, crypto_dh5_prime_size);
	assert(prime);

	private = l_key_new(L_KEY_RAW, test->private_key,
						test->private_key_size);
	assert(private);

	len = test->private_key_size;
	assert(l_key_compute_dh_public(generator, private, prime,
							public_key, &len));
	assert(len == test->public_key_size);
	assert(!memcmp(public_key, test->public_key, test->public_key_size));

	l_key_free(prime);
	l_key_free(generator);
	l_key_free(private);
}

static const unsigned char eap_wsc_m1_1[] = {
	0x01, 0x00, 0x01, 0x78, 0x02, 0x01, 0x01, 0x78, 0xfe, 0x00, 0x37, 0x2a,
	0x00, 0x00, 0x00, 0x01, 0x04, 0x00, 0x10, 0x4a, 0x00, 0x01, 0x10, 0x10,
	0x22, 0x00, 0x01, 0x04, 0x10, 0x47, 0x00, 0x10, 0x79, 0x0c, 0x1f, 0x80,
	0x4f, 0x2b, 0x52, 0xb7, 0xbe, 0x30, 0xc0, 0xe9, 0x72, 0x92, 0x08, 0x8d,
	0x10, 0x20, 0x00, 0x06, 0xa0, 0xa8, 0xcd, 0x1c, 0x7e, 0xc9, 0x10, 0x1a,
	0x00, 0x10, 0xab, 0x84, 0x41, 0x2f, 0xe7, 0xc3, 0xc9, 0xc9, 0xd7, 0xf4,
	0xe8, 0xc1, 0x4f, 0x49, 0x2b, 0x79, 0x10, 0x32, 0x00, 0xc0, 0xb2, 0xfc,
	0xd6, 0x4f, 0xf6, 0x71, 0x5a, 0x33, 0x84, 0x60, 0x4a, 0xe8, 0x2c, 0x1e,
	0x55, 0x4a, 0xdb, 0xd5, 0x18, 0x17, 0x91, 0xa6, 0xf5, 0x70, 0xcd, 0x23,
	0xd7, 0x12, 0x6e, 0x4c, 0xaf, 0x27, 0x9a, 0x4e, 0xf5, 0x37, 0xea, 0x8f,
	0x03, 0xc9, 0x0e, 0x79, 0xc5, 0x8d, 0x37, 0xf8, 0xfb, 0x11, 0xa1, 0x39,
	0x19, 0x9b, 0x5a, 0x3a, 0x66, 0x36, 0x6d, 0xfb, 0xae, 0xed, 0xfc, 0xa5,
	0x90, 0xcb, 0xb3, 0xe1, 0xd5, 0x92, 0x2e, 0xe9, 0x99, 0xbd, 0x0b, 0x93,
	0x82, 0x57, 0xe1, 0xbd, 0x70, 0x17, 0xa7, 0x78, 0x7a, 0x0a, 0xff, 0x42,
	0x06, 0x95, 0x2c, 0x0b, 0x6c, 0x1a, 0x6b, 0x2f, 0x6b, 0xed, 0x42, 0xa5,
	0x60, 0x8a, 0xb0, 0xb5, 0x79, 0x1b, 0xa9, 0xe6, 0x15, 0x17, 0xa3, 0x6c,
	0xe9, 0x84, 0xb3, 0x77, 0x48, 0x9b, 0x7a, 0x4d, 0x04, 0xf6, 0xb8, 0x27,
	0xe5, 0x0c, 0xcb, 0x76, 0xfc, 0x3c, 0x65, 0x49, 0xd7, 0x28, 0x06, 0x8d,
	0x99, 0x18, 0x0f, 0xa7, 0x35, 0xb2, 0x9d, 0x15, 0x35, 0x51, 0xea, 0x83,
	0xb6, 0x4d, 0x14, 0xb0, 0x21, 0xa4, 0x82, 0x1f, 0xb8, 0x73, 0x2b, 0x15,
	0x1d, 0x48, 0x99, 0x9f, 0x32, 0x2c, 0xe1, 0xe1, 0xab, 0x66, 0x3f, 0xb4,
	0x40, 0x79, 0xe8, 0x96, 0xe1, 0x9d, 0x54, 0x8b, 0xb6, 0x7f, 0x1a, 0x5b,
	0x5f, 0x09, 0x9f, 0x40, 0xa7, 0x8b, 0xc8, 0xf6, 0x27, 0x80, 0x10, 0x04,
	0x00, 0x02, 0x00, 0x23, 0x10, 0x10, 0x00, 0x02, 0x00, 0x0d, 0x10, 0x0d,
	0x00, 0x01, 0x01, 0x10, 0x08, 0x00, 0x02, 0x21, 0x48, 0x10, 0x44, 0x00,
	0x01, 0x01, 0x10, 0x21, 0x00, 0x01, 0x20, 0x10, 0x23, 0x00, 0x01, 0x20,
	0x10, 0x24, 0x00, 0x01, 0x20, 0x10, 0x42, 0x00, 0x01, 0x20, 0x10, 0x54,
	0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x11,
	0x00, 0x01, 0x20, 0x10, 0x3c, 0x00, 0x01, 0x02, 0x10, 0x02, 0x00, 0x02,
	0x00, 0x00, 0x10, 0x12, 0x00, 0x02, 0x00, 0x04, 0x10, 0x09, 0x00, 0x02,
	0x00, 0x00, 0x10, 0x2d, 0x00, 0x04, 0x80, 0x00, 0x00, 0x00, 0x10, 0x49,
	0x00, 0x06, 0x00, 0x37, 0x2a, 0x00, 0x01, 0x20,
};

static const unsigned char eap_wsc_m1_2[] = {
	0x01, 0x00, 0x01, 0x78, 0x02, 0x01, 0x01, 0x78, 0xfe, 0x00, 0x37, 0x2a,
	0x00, 0x00, 0x00, 0x01, 0x04, 0x00, 0x10, 0x4a, 0x00, 0x01, 0x10, 0x10,
	0x22, 0x00, 0x01, 0x04, 0x10, 0x47, 0x00, 0x10, 0x79, 0x0c, 0x1f, 0x80,
	0x4f, 0x2b, 0x52, 0xb7, 0xbe, 0x30, 0xc0, 0xe9, 0x72, 0x92, 0x08, 0x8d,
	0x10, 0x20, 0x00, 0x06, 0xa0, 0xa8, 0xcd, 0x1c, 0x7e, 0xc9, 0x10, 0x1a,
	0x00, 0x10, 0x19, 0x8d, 0x0d, 0x25, 0x91, 0x2c, 0x37, 0x1c, 0xeb, 0x07,
	0x89, 0x33, 0xe1, 0x25, 0xd7, 0x43, 0x10, 0x32, 0x00, 0xc0, 0x42, 0x76,
	0x1f, 0xac, 0x65, 0x7b, 0x25, 0xd3, 0x59, 0xe8, 0x5d, 0xde, 0x22, 0xb1,
	0x74, 0x39, 0x7d, 0x0d, 0x6c, 0x56, 0xc9, 0x4f, 0xfc, 0xf2, 0x1f, 0xd0,
	0x4f, 0x70, 0xe9, 0xd5, 0x5c, 0xb6, 0x79, 0x07, 0x54, 0xb6, 0x8e, 0x67,
	0xe1, 0xb5, 0x35, 0x18, 0xc0, 0xe3, 0x36, 0xf9, 0x58, 0x15, 0x3f, 0x4e,
	0xed, 0x74, 0xb5, 0x46, 0xd4, 0x2f, 0x30, 0x22, 0xb2, 0xe1, 0xa9, 0x4d,
	0xf6, 0x3a, 0x00, 0x22, 0x34, 0xf0, 0x14, 0xc5, 0xa3, 0x53, 0x28, 0x78,
	0x91, 0x46, 0x1e, 0x80, 0x3d, 0x7e, 0x41, 0x4a, 0x44, 0x3a, 0xd0, 0x83,
	0x7e, 0xcf, 0xa4, 0x31, 0x0e, 0x0c, 0xfc, 0xcd, 0x63, 0x37, 0x05, 0x8b,
	0x7f, 0x0d, 0x65, 0xef, 0x52, 0x58, 0xb9, 0x9e, 0xb2, 0xfc, 0x77, 0x7b,
	0xdb, 0x79, 0x17, 0x3d, 0xb2, 0x84, 0x4a, 0xaf, 0x09, 0x93, 0x1c, 0x45,
	0x9d, 0x0a, 0x9f, 0xb9, 0x64, 0xec, 0x6f, 0x66, 0x01, 0x49, 0xb6, 0x00,
	0xf7, 0xff, 0x85, 0x4d, 0xa8, 0x60, 0x07, 0xf6, 0x33, 0x1f, 0xe6, 0x48,
	0x9e, 0x71, 0x88, 0x60, 0x88, 0x02, 0xbb, 0x1d, 0x07, 0x68, 0x7a, 0xa0,
	0xfa, 0x4f, 0xfb, 0x42, 0xde, 0x6f, 0x8e, 0x4e, 0x4a, 0xa8, 0xa7, 0xd3,
	0x5d, 0xcb, 0x38, 0x47, 0x16, 0xb6, 0xea, 0x99, 0x9f, 0x7d, 0xf6, 0xcd,
	0xd4, 0xe4, 0x5f, 0x9f, 0xb5, 0xb6, 0x61, 0x9d, 0x42, 0x78, 0x10, 0x04,
	0x00, 0x02, 0x00, 0x23, 0x10, 0x10, 0x00, 0x02, 0x00, 0x0d, 0x10, 0x0d,
	0x00, 0x01, 0x01, 0x10, 0x08, 0x00, 0x02, 0x21, 0x48, 0x10, 0x44, 0x00,
	0x01, 0x01, 0x10, 0x21, 0x00, 0x01, 0x20, 0x10, 0x23, 0x00, 0x01, 0x20,
	0x10, 0x24, 0x00, 0x01, 0x20, 0x10, 0x42, 0x00, 0x01, 0x20, 0x10, 0x54,
	0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x11,
	0x00, 0x01, 0x20, 0x10, 0x3c, 0x00, 0x01, 0x01, 0x10, 0x02, 0x00, 0x02,
	0x00, 0x00, 0x10, 0x12, 0x00, 0x02, 0x00, 0x04, 0x10, 0x09, 0x00, 0x02,
	0x00, 0x00, 0x10, 0x2d, 0x00, 0x04, 0x80, 0x00, 0x00, 0x00, 0x10, 0x49,
	0x00, 0x06, 0x00, 0x37, 0x2a, 0x00, 0x01, 0x20,
};

struct m1_data {
	struct wsc_m1 expected;
	const void *pdu;
	unsigned int len;
	const uint8_t *private_key;
	uint32_t private_key_size;
	const uint8_t *public_key;
	uint32_t public_key_size;
};

static const struct m1_data m1_data_1 = {
	.pdu = eap_wsc_m1_1 + 18,
	.len = sizeof(eap_wsc_m1_1) - 18,
	.expected = {
		.version2 = true,
		.uuid_e = { 0x79, 0x0c, 0x1f, 0x80, 0x4f, 0x2b, 0x52, 0xb7,
			0xbe, 0x30, 0xc0, 0xe9, 0x72, 0x92, 0x08, 0x8d },
		.addr = { 0xa0, 0xa8, 0xcd, 0x1c, 0x7e, 0xc9 },
		.enrollee_nonce = { 0xab, 0x84, 0x41, 0x2f, 0xe7, 0xc3, 0xc9,
					0xc9, 0xd7, 0xf4, 0xe8, 0xc1, 0x4f,
					0x49, 0x2b, 0x79 },
		.public_key = { }, /* Tested elsewhere */
		.auth_type_flags =
			WSC_AUTHENTICATION_TYPE_WPA2_PERSONAL |
			WSC_AUTHENTICATION_TYPE_WPA_PERSONAL |
			WSC_AUTHENTICATION_TYPE_OPEN,
		.encryption_type_flags = WSC_ENCRYPTION_TYPE_AES_TKIP |
						WSC_ENCRYPTION_TYPE_NONE,
		.connection_type_flags = WSC_CONNECTION_TYPE_ESS,
		.config_methods = WSC_CONFIGURATION_METHOD_VIRTUAL_DISPLAY_PIN |
				WSC_CONFIGURATION_METHOD_KEYPAD |
				WSC_CONFIGURATION_METHOD_DISPLAY |
				WSC_CONFIGURATION_METHOD_NFC_INTERFACE,
		.state = WSC_STATE_NOT_CONFIGURED,
		.manufacturer = " ",
		.model_name = " ",
		.model_number = " ",
		.serial_number = " ",
		.primary_device_type = {
			.category = 0,
			.oui = { 0x00, 0x00, 0x00 },
			.oui_type = 0x00,
			.subcategory = 0, },
		.device_name = " ",
		.rf_bands = WSC_RF_BAND_5_0_GHZ,
		.association_state = WSC_ASSOCIATION_STATE_NOT_ASSOCIATED,
		.device_password_id = WSC_DEVICE_PASSWORD_ID_PUSH_BUTTON,
		.configuration_error = WSC_CONFIGURATION_ERROR_NO_ERROR,
		.os_version = 0,
		.request_to_enroll = false,
	},
	.private_key = dh_private_key_1,
	.private_key_size = sizeof(dh_private_key_1),
	.public_key = dh_public_key_1,
	.public_key_size = sizeof(dh_public_key_1),
};

static const struct m1_data m1_data_2 = {
	.pdu = eap_wsc_m1_2 + 18,
	.len = sizeof(eap_wsc_m1_2) - 18,
	.expected = {
		.version2 = true,
		.uuid_e = { 0x79, 0x0c, 0x1f, 0x80, 0x4f, 0x2b, 0x52, 0xb7,
			0xbe, 0x30, 0xc0, 0xe9, 0x72, 0x92, 0x08, 0x8d },
		.addr = { 0xa0, 0xa8, 0xcd, 0x1c, 0x7e, 0xc9 },
		.enrollee_nonce = { 0x19, 0x8d, 0x0d, 0x25, 0x91, 0x2c, 0x37,
					0x1c, 0xeb, 0x07, 0x89, 0x33, 0xe1,
					0x25, 0xd7, 0x43 },
		.public_key = { }, /* Tested elsewhere */
		.auth_type_flags =
			WSC_AUTHENTICATION_TYPE_WPA2_PERSONAL |
			WSC_AUTHENTICATION_TYPE_WPA_PERSONAL |
			WSC_AUTHENTICATION_TYPE_OPEN,
		.encryption_type_flags = WSC_ENCRYPTION_TYPE_AES_TKIP |
						WSC_ENCRYPTION_TYPE_NONE,
		.connection_type_flags = WSC_CONNECTION_TYPE_ESS,
		.config_methods = WSC_CONFIGURATION_METHOD_VIRTUAL_DISPLAY_PIN |
				WSC_CONFIGURATION_METHOD_KEYPAD |
				WSC_CONFIGURATION_METHOD_DISPLAY |
				WSC_CONFIGURATION_METHOD_NFC_INTERFACE,
		.state = WSC_STATE_NOT_CONFIGURED,
		.manufacturer = " ",
		.model_name = " ",
		.model_number = " ",
		.serial_number = " ",
		.primary_device_type = {
			.category = 0,
			.oui = { 0x00, 0x00, 0x00 },
			.oui_type = 0x00,
			.subcategory = 0, },
		.device_name = " ",
		.rf_bands = WSC_RF_BAND_2_4_GHZ,
		.association_state = WSC_ASSOCIATION_STATE_NOT_ASSOCIATED,
		.device_password_id = WSC_DEVICE_PASSWORD_ID_PUSH_BUTTON,
		.configuration_error = WSC_CONFIGURATION_ERROR_NO_ERROR,
		.os_version = 0,
		.request_to_enroll = false,
	},
	.private_key = dh_private_key_2,
	.private_key_size = sizeof(dh_private_key_2),
	.public_key = dh_public_key_2,
	.public_key_size = sizeof(dh_public_key_2),
};

static void wsc_test_parse_m1(const void *data)
{
	const struct m1_data *test = data;
	struct wsc_m1 m1;
	const struct wsc_m1 *expected = &test->expected;
	int r;

	r = wsc_parse_m1(test->pdu, test->len, &m1);
	assert(r == 0);

	assert(expected->version2 == m1.version2);
	assert(!memcmp(expected->uuid_e, m1.uuid_e, 16));
	assert(!memcmp(expected->addr, m1.addr, 6));
	assert(!memcmp(expected->enrollee_nonce, m1.enrollee_nonce, 16));

	/* Skip public_key testing */

	assert(expected->auth_type_flags == m1.auth_type_flags);
	assert(expected->encryption_type_flags == m1.encryption_type_flags);
	assert(expected->connection_type_flags == m1.connection_type_flags);
	assert(expected->config_methods == m1.config_methods);
	assert(expected->state == m1.state);

	assert(!strcmp(expected->manufacturer, m1.manufacturer));
	assert(!strcmp(expected->model_name, m1.model_name));
	assert(!strcmp(expected->model_number, m1.model_number));
	assert(!strcmp(expected->serial_number, m1.serial_number));

	assert(expected->primary_device_type.category ==
				m1.primary_device_type.category);
	assert(!memcmp(expected->primary_device_type.oui,
				m1.primary_device_type.oui, 3));
	assert(expected->primary_device_type.oui_type ==
				m1.primary_device_type.oui_type);
	assert(expected->primary_device_type.subcategory ==
				m1.primary_device_type.subcategory);

	assert(!strcmp(expected->device_name, m1.device_name));
	assert(expected->rf_bands == m1.rf_bands);
	assert(expected->association_state == m1.association_state);
	assert(expected->device_password_id == m1.device_password_id);
	assert(expected->configuration_error == m1.configuration_error);
	assert(expected->os_version == m1.os_version);
	assert(expected->request_to_enroll == m1.request_to_enroll);

	assert(!memcmp(test->public_key, m1.public_key, 192));
}

static void wsc_test_build_m1(const void *data)
{
	const struct m1_data *test = data;
	struct wsc_m1 m1;
	uint8_t *out;
	size_t out_len;

	memcpy(&m1, &test->expected, sizeof(m1));
	memcpy(m1.public_key, test->public_key, 192);

	out = wsc_build_m1(&m1, &out_len);
	assert(out);

	assert(out_len == test->len);
	assert(!memcmp(test->pdu, out, test->len));

	l_free(out);
}

static bool wsc_compute_authenticator(struct l_key *peer_public_key,
					struct l_key *private_key,
					struct l_key *prime,
					const uint8_t *enrollee_nonce,
					const uint8_t *enrollee_mac,
					const uint8_t *registrar_nonce,
					const uint8_t *prev_msg,
					size_t prev_msg_len,
					const uint8_t *cur_msg,
					size_t cur_msg_len,
					uint8_t *authenticator)
{
	uint8_t shared_secret[192];
	size_t shared_secret_len;
	struct l_checksum *sha256;
	uint8_t dhkey[32];
	struct l_checksum *hmac_sha256;
	uint8_t kdk[32];
	ssize_t digest_len;
	bool r;
	struct iovec iov[3];
	struct wsc_session_key session_key;

	/* Compute shared secret */
	shared_secret_len = sizeof(shared_secret);
	r = l_key_compute_dh_secret(peer_public_key, private_key, prime,
					shared_secret, &shared_secret_len);
	assert(r);

	sha256 = l_checksum_new(L_CHECKSUM_SHA256);
	assert(sha256);

	/* Compute DHKey */
	assert(l_checksum_update(sha256, shared_secret, shared_secret_len));
	digest_len = l_checksum_get_digest(sha256, dhkey, sizeof(dhkey));
	assert(digest_len == 32);
	l_checksum_free(sha256);

	hmac_sha256 = l_checksum_new_hmac(L_CHECKSUM_SHA256,
							dhkey, sizeof(dhkey));
	assert(hmac_sha256);

	iov[0].iov_base = (void *) enrollee_nonce;
	iov[0].iov_len = 16;
	iov[1].iov_base = (void *) enrollee_mac;
	iov[1].iov_len = 6;
	iov[2].iov_base = (void *) registrar_nonce;
	iov[2].iov_len = 16;

	assert(l_checksum_updatev(hmac_sha256, iov, 3));
	digest_len = l_checksum_get_digest(hmac_sha256, kdk, sizeof(kdk));
	assert(digest_len == 32);
	l_checksum_free(hmac_sha256);

	assert(wsc_kdf(kdk, &session_key, sizeof(session_key)));

	hmac_sha256 = l_checksum_new_hmac(L_CHECKSUM_SHA256,
						session_key.auth_key,
						sizeof(session_key.auth_key));
	iov[0].iov_base = (void *) prev_msg;
	iov[0].iov_len = prev_msg_len;
	iov[1].iov_base = (void *) cur_msg;
	iov[1].iov_len = cur_msg_len - 12;

	assert(l_checksum_updatev(hmac_sha256, iov, 2));
	digest_len = l_checksum_get_digest(hmac_sha256, authenticator, 8);
	assert(digest_len == 8);
	l_checksum_free(hmac_sha256);

	return true;
}

static const unsigned char eap_wsc_m2_1[] = {
	0x01, 0x00, 0x01, 0xd6, 0x01, 0x02, 0x01, 0xd6, 0xfe, 0x00, 0x37, 0x2a,
	0x00, 0x00, 0x00, 0x01, 0x04, 0x00, 0x10, 0x4a, 0x00, 0x01, 0x10, 0x10,
	0x22, 0x00, 0x01, 0x05, 0x10, 0x1a, 0x00, 0x10, 0xab, 0x84, 0x41, 0x2f,
	0xe7, 0xc3, 0xc9, 0xc9, 0xd7, 0xf4, 0xe8, 0xc1, 0x4f, 0x49, 0x2b, 0x79,
	0x10, 0x39, 0x00, 0x10, 0x1a, 0xe1, 0xa2, 0xc1, 0x5a, 0xff, 0xa8, 0xb4,
	0x3a, 0x4a, 0xe1, 0xea, 0x71, 0x3f, 0xfb, 0x49, 0x10, 0x48, 0x00, 0x10,
	0xc4, 0x4a, 0xad, 0x8d, 0x25, 0x2f, 0x52, 0xc6, 0xf9, 0x6b, 0x38, 0x5d,
	0xcb, 0x23, 0x31, 0xae, 0x10, 0x32, 0x00, 0xc0, 0xe8, 0x23, 0x5b, 0x62,
	0x65, 0x41, 0x83, 0xfa, 0xc3, 0x23, 0x6f, 0x30, 0xc9, 0x4e, 0x32, 0x34,
	0x8c, 0xfa, 0xf0, 0xfb, 0xff, 0x09, 0xc1, 0x93, 0x13, 0x7a, 0x80, 0xe2,
	0xce, 0x5a, 0x5e, 0xed, 0x55, 0xcd, 0xdb, 0x87, 0x05, 0xbf, 0x06, 0x47,
	0xf9, 0x96, 0xdd, 0x96, 0x41, 0xdf, 0x56, 0x6e, 0xb1, 0x20, 0xee, 0xf7,
	0x97, 0xf7, 0x59, 0x9f, 0x94, 0x78, 0xe5, 0xff, 0xd7, 0xa0, 0x99, 0xcf,
	0x34, 0x00, 0xb4, 0xbd, 0xaf, 0xaf, 0x8f, 0xd9, 0x7d, 0x10, 0x2e, 0x4d,
	0x8c, 0xac, 0x0e, 0x8f, 0x19, 0xcf, 0x07, 0xc7, 0x24, 0x6c, 0xa8, 0xf4,
	0x18, 0x56, 0x04, 0xbf, 0x80, 0x34, 0x72, 0x18, 0x65, 0x09, 0x7a, 0x85,
	0x7b, 0xe2, 0xe0, 0xeb, 0x19, 0x15, 0x69, 0x7f, 0xef, 0x60, 0x78, 0x01,
	0x73, 0xb8, 0xd0, 0x33, 0x76, 0x18, 0xf2, 0xf8, 0x29, 0x39, 0x11, 0x1a,
	0xcd, 0x6c, 0x5c, 0x10, 0xf5, 0xbf, 0x4e, 0xf3, 0xda, 0x9a, 0x48, 0x46,
	0x5c, 0x93, 0x8e, 0x18, 0x03, 0xd0, 0x42, 0x8f, 0xef, 0x5a, 0xc7, 0x9c,
	0x1a, 0x10, 0x3a, 0xd3, 0x74, 0x6a, 0x3b, 0x23, 0x37, 0x7b, 0xcd, 0x4a,
	0x52, 0xfd, 0x11, 0x4d, 0x62, 0xf8, 0x2e, 0xac, 0xae, 0x2e, 0xcf, 0x25,
	0x5c, 0xce, 0xd4, 0xd9, 0x8b, 0x30, 0x06, 0x74, 0x87, 0xd1, 0xb8, 0x64,
	0x1a, 0xf5, 0xc6, 0x28, 0x89, 0x1f, 0x68, 0xd7, 0x10, 0x04, 0x00, 0x02,
	0x00, 0x23, 0x10, 0x10, 0x00, 0x02, 0x00, 0x0d, 0x10, 0x0d, 0x00, 0x01,
	0x01, 0x10, 0x08, 0x00, 0x02, 0x26, 0x88, 0x10, 0x21, 0x00, 0x15, 0x41,
	0x53, 0x55, 0x53, 0x54, 0x65, 0x4b, 0x20, 0x43, 0x6f, 0x6d, 0x70, 0x75,
	0x74, 0x65, 0x72, 0x20, 0x49, 0x6e, 0x63, 0x2e, 0x10, 0x23, 0x00, 0x1c,
	0x57, 0x69, 0x2d, 0x46, 0x69, 0x20, 0x50, 0x72, 0x6f, 0x74, 0x65, 0x63,
	0x74, 0x65, 0x64, 0x20, 0x53, 0x65, 0x74, 0x75, 0x70, 0x20, 0x52, 0x6f,
	0x75, 0x74, 0x65, 0x72, 0x10, 0x24, 0x00, 0x08, 0x52, 0x54, 0x2d, 0x41,
	0x43, 0x36, 0x38, 0x55, 0x10, 0x42, 0x00, 0x11, 0x31, 0x30, 0x3a, 0x63,
	0x33, 0x3a, 0x37, 0x62, 0x3a, 0x35, 0x34, 0x3a, 0x37, 0x34, 0x3a, 0x64,
	0x30, 0x10, 0x54, 0x00, 0x08, 0x00, 0x06, 0x00, 0x50, 0xf2, 0x04, 0x00,
	0x01, 0x10, 0x11, 0x00, 0x08, 0x52, 0x54, 0x2d, 0x41, 0x43, 0x36, 0x38,
	0x55, 0x10, 0x3c, 0x00, 0x01, 0x02, 0x10, 0x02, 0x00, 0x02, 0x00, 0x00,
	0x10, 0x09, 0x00, 0x02, 0x00, 0x00, 0x10, 0x12, 0x00, 0x02, 0x00, 0x04,
	0x10, 0x2d, 0x00, 0x04, 0x80, 0x00, 0x00, 0x00, 0x10, 0x49, 0x00, 0x06,
	0x00, 0x37, 0x2a, 0x00, 0x01, 0x20, 0x10, 0x05, 0x00, 0x08, 0xec, 0xb1,
	0x17, 0xac, 0xd9, 0x33, 0x84, 0x5f,
};

static const unsigned char eap_wsc_m2_2[] = {
	0x01, 0x00, 0x01, 0xd3, 0x01, 0x02, 0x01, 0xd3, 0xfe, 0x00, 0x37, 0x2a,
	0x00, 0x00, 0x00, 0x01, 0x04, 0x00, 0x10, 0x4a, 0x00, 0x01, 0x10, 0x10,
	0x22, 0x00, 0x01, 0x05, 0x10, 0x1a, 0x00, 0x10, 0x19, 0x8d, 0x0d, 0x25,
	0x91, 0x2c, 0x37, 0x1c, 0xeb, 0x07, 0x89, 0x33, 0xe1, 0x25, 0xd7, 0x43,
	0x10, 0x39, 0x00, 0x10, 0x67, 0xa5, 0x53, 0x61, 0xee, 0x72, 0xc2, 0x21,
	0x48, 0x78, 0xc5, 0x70, 0x89, 0x9f, 0xf2, 0xa0, 0x10, 0x48, 0x00, 0x10,
	0x8e, 0xd8, 0x32, 0x02, 0xa3, 0x24, 0x52, 0xc0, 0xb0, 0x3e, 0x0c, 0x95,
	0x18, 0x27, 0xd6, 0xe7, 0x10, 0x32, 0x00, 0xc0, 0x29, 0xdd, 0x2e, 0x36,
	0x97, 0xa4, 0x25, 0x1c, 0x93, 0x56, 0xef, 0xdd, 0x8c, 0xf4, 0x9e, 0x1c,
	0xab, 0xbe, 0x42, 0xd1, 0xba, 0xc4, 0xb0, 0xa4, 0x2b, 0xaf, 0x9e, 0xcc,
	0xe0, 0xfb, 0x94, 0x0c, 0xf1, 0xdf, 0x5d, 0x71, 0x2b, 0xcb, 0x24, 0xd7,
	0x83, 0x9f, 0xf4, 0xa2, 0xa9, 0x88, 0x86, 0xa2, 0x7f, 0x4f, 0x42, 0xe2,
	0x76, 0x4a, 0xac, 0x4e, 0x5d, 0x45, 0x15, 0x01, 0x4f, 0x29, 0xbe, 0xa2,
	0x44, 0xdf, 0x36, 0xb9, 0xff, 0xe3, 0xdf, 0x14, 0x30, 0x21, 0xf2, 0x74,
	0x9d, 0x57, 0x5b, 0x08, 0xe9, 0x78, 0xad, 0x5c, 0x28, 0x2c, 0x15, 0xcb,
	0xa2, 0x0d, 0xdb, 0x8c, 0xc5, 0x0f, 0xda, 0x6f, 0x98, 0xd0, 0x84, 0x0b,
	0x4c, 0xab, 0x0b, 0x1c, 0xd9, 0xee, 0x78, 0xa4, 0xfe, 0x8e, 0x51, 0x8d,
	0xa1, 0xc5, 0xf1, 0xe1, 0x5e, 0xf1, 0x9b, 0xc4, 0x17, 0xa5, 0x41, 0xb1,
	0x46, 0x75, 0x9b, 0x03, 0x4e, 0x6c, 0x0b, 0x3a, 0xb9, 0x25, 0xb4, 0x76,
	0x79, 0x18, 0xdd, 0xba, 0x73, 0xca, 0x32, 0x33, 0x60, 0xd5, 0xb3, 0xef,
	0xfe, 0x13, 0x3f, 0xfc, 0x91, 0xf3, 0x31, 0xd6, 0x4a, 0xbf, 0xfa, 0x3e,
	0x89, 0x4d, 0xe6, 0xfa, 0x09, 0x71, 0x0f, 0x8a, 0xc8, 0x67, 0x1f, 0x7e,
	0xc0, 0xae, 0xb8, 0x15, 0x4d, 0xb9, 0x20, 0xec, 0x40, 0xd2, 0xdf, 0x73,
	0x09, 0x13, 0x5b, 0x06, 0xe0, 0xc1, 0xfa, 0xe8, 0x10, 0x04, 0x00, 0x02,
	0x00, 0x3f, 0x10, 0x10, 0x00, 0x02, 0x00, 0x0c, 0x10, 0x0d, 0x00, 0x01,
	0x01, 0x10, 0x08, 0x00, 0x02, 0x00, 0x00, 0x10, 0x21, 0x00, 0x0a, 0x41,
	0x70, 0x70, 0x6c, 0x65, 0x20, 0x49, 0x6e, 0x63, 0x2e, 0x10, 0x23, 0x00,
	0x04, 0x4a, 0x32, 0x38, 0x45, 0x10, 0x24, 0x00, 0x19, 0x41, 0x70, 0x70,
	0x6c, 0x65, 0x20, 0x42, 0x61, 0x73, 0x65, 0x20, 0x53, 0x74, 0x61, 0x74,
	0x69, 0x6f, 0x6e, 0x20, 0x56, 0x37, 0x2e, 0x37, 0x2e, 0x33, 0x10, 0x42,
	0x00, 0x0c, 0x43, 0x38, 0x36, 0x4d, 0x44, 0x38, 0x57, 0x48, 0x46, 0x4a,
	0x31, 0x52, 0x10, 0x54, 0x00, 0x08, 0x00, 0x06, 0x00, 0x50, 0xf2, 0x04,
	0x00, 0x01, 0x10, 0x11, 0x00, 0x17, 0x44, 0x65, 0x6e, 0x69, 0x73, 0x27,
	0x73, 0x20, 0x41, 0x69, 0x72, 0x50, 0x6f, 0x72, 0x74, 0x20, 0x45, 0x78,
	0x74, 0x72, 0x65, 0x6d, 0x65, 0x10, 0x3c, 0x00, 0x01, 0x01, 0x10, 0x02,
	0x00, 0x02, 0x00, 0x00, 0x10, 0x09, 0x00, 0x02, 0x00, 0x00, 0x10, 0x12,
	0x00, 0x02, 0x00, 0x04, 0x10, 0x2d, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00,
	0x10, 0x49, 0x00, 0x0b, 0x00, 0x00, 0x3f, 0x00, 0x04, 0x00, 0x04, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x05, 0x00, 0x08, 0x45, 0x33, 0x07, 0x57, 0x77,
	0x40, 0x50, 0x37,
};

struct m2_data {
	struct wsc_m2 expected;
	const void *pdu;
	unsigned int len;
	const uint8_t *public_key;
	uint32_t public_key_size;
	const struct m1_data *m1;
};

static const struct m2_data m2_data_1 = {
	.pdu = eap_wsc_m2_1 + 18,
	.len = sizeof(eap_wsc_m2_1) - 18,
	.expected = {
		.version2 = true,
		.enrollee_nonce = { 0xab, 0x84, 0x41, 0x2f, 0xe7, 0xc3, 0xc9,
					0xc9, 0xd7, 0xf4, 0xe8, 0xc1, 0x4f,
					0x49, 0x2b, 0x79 },
		.registrar_nonce = { 0x1a, 0xe1, 0xa2, 0xc1, 0x5a, 0xff, 0xa8,
					0xb4, 0x3a, 0x4a, 0xe1, 0xea, 0x71,
					0x3f, 0xfb, 0x49 },
		.uuid_r = { 0xc4, 0x4a, 0xad, 0x8d, 0x25, 0x2f, 0x52, 0xc6,
			0xf9, 0x6b, 0x38, 0x5d, 0xcb, 0x23, 0x31, 0xae },
		.public_key = { }, /* Tested elsewhere */
		.auth_type_flags =
			WSC_AUTHENTICATION_TYPE_WPA2_PERSONAL |
			WSC_AUTHENTICATION_TYPE_WPA_PERSONAL |
			WSC_AUTHENTICATION_TYPE_OPEN,
		.encryption_type_flags = WSC_ENCRYPTION_TYPE_AES_TKIP |
						WSC_ENCRYPTION_TYPE_NONE,
		.connection_type_flags = WSC_CONNECTION_TYPE_ESS,
		.config_methods = WSC_CONFIGURATION_METHOD_VIRTUAL_PUSH_BUTTON |
				WSC_CONFIGURATION_METHOD_PHYSICAL_PUSH_BUTTON |
				WSC_CONFIGURATION_METHOD_VIRTUAL_DISPLAY_PIN,
		.manufacturer = "ASUSTeK Computer Inc.",
		.model_name = "Wi-Fi Protected Setup Router",
		.model_number = "RT-AC68U",
		.serial_number = "10:c3:7b:54:74:d0",
		.primary_device_type = {
			.category = 6,
			.oui = { 0x00, 0x50, 0xf2 },
			.oui_type = 0x04,
			.subcategory = 1, },
		.device_name = "RT-AC68U",
		.rf_bands = WSC_RF_BAND_5_0_GHZ,
		.association_state = WSC_ASSOCIATION_STATE_NOT_ASSOCIATED,
		.device_password_id = WSC_DEVICE_PASSWORD_ID_PUSH_BUTTON,
		.configuration_error = WSC_CONFIGURATION_ERROR_NO_ERROR,
		.os_version = 0,
		.authenticator = { 0xec, 0xb1, 0x17, 0xac, 0xd9, 0x33, 0x84,
					0x5f },
	},
	.public_key = eap_wsc_m2_1 + 92,
	.public_key_size = 192,
	.m1 = &m1_data_1,
};

static const struct m2_data m2_data_2 = {
	.pdu = eap_wsc_m2_2 + 18,
	.len = sizeof(eap_wsc_m2_2) - 18,
	.expected = {
		.version2 = false,
		.enrollee_nonce = { 0x19, 0x8d, 0x0d, 0x25, 0x91, 0x2c, 0x37,
					0x1c, 0xeb, 0x07, 0x89, 0x33, 0xe1,
					0x25, 0xd7, 0x43 },
		.registrar_nonce = { 0x67, 0xa5, 0x53, 0x61, 0xee, 0x72, 0xc2,
					0x21, 0x48, 0x78, 0xc5, 0x70, 0x89,
					0x9f, 0xf2, 0xa0 },
		.uuid_r = { 0x8e, 0xd8, 0x32, 0x02, 0xa3, 0x24, 0x52, 0xc0,
			0xb0, 0x3e, 0x0c, 0x95, 0x18, 0x27, 0xd6, 0xe7, },
		.public_key = { }, /* Tested elsewhere */
		.auth_type_flags =
			WSC_AUTHENTICATION_TYPE_OPEN |
			WSC_AUTHENTICATION_TYPE_WPA_PERSONAL |
			WSC_AUTHENTICATION_TYPE_SHARED |
			WSC_AUTHENTICATION_TYPE_WPA_ENTERPRISE |
			WSC_AUTHENTICATION_TYPE_WPA2_PERSONAL |
			WSC_AUTHENTICATION_TYPE_WPA2_ENTERPRISE,
		.encryption_type_flags = WSC_ENCRYPTION_TYPE_AES_TKIP,
		.connection_type_flags = WSC_CONNECTION_TYPE_ESS,
		.config_methods = 0,
		.manufacturer = "Apple Inc.",
		.model_name = "J28E",
		.model_number = "Apple Base Station V7.7.3",
		.serial_number = "C86MD8WHFJ1R",
		.primary_device_type = {
			.category = 6,
			.oui = { 0x00, 0x50, 0xf2 },
			.oui_type = 0x04,
			.subcategory = 1, },
		.device_name = "Denis's AirPort Extreme",
		.rf_bands = WSC_RF_BAND_2_4_GHZ,
		.association_state = WSC_ASSOCIATION_STATE_NOT_ASSOCIATED,
		.device_password_id = WSC_DEVICE_PASSWORD_ID_PUSH_BUTTON,
		.configuration_error = WSC_CONFIGURATION_ERROR_NO_ERROR,
		.os_version = 0,
		.authenticator = { 0x45, 0x33, 0x07, 0x57, 0x77, 0x40, 0x50,
						0x37 },
	},
	.public_key = eap_wsc_m2_2 + 92,
	.public_key_size = 192,
	.m1 = &m1_data_2,
};

static void wsc_test_parse_m2(const void *data)
{
	const struct m2_data *test = data;
	struct wsc_m2 m2;
	const struct wsc_m2 *expected = &test->expected;
	struct l_key *private;
	struct l_key *prime;
	struct l_key *peer_public;
	uint8_t authenticator[8];
	int r;

	r = wsc_parse_m2(test->pdu, test->len, &m2);
	assert(r == 0);

	assert(expected->version2 == m2.version2);
	assert(!memcmp(expected->uuid_r, m2.uuid_r, 16));
	assert(!memcmp(expected->enrollee_nonce, m2.enrollee_nonce, 16));
	assert(!memcmp(expected->registrar_nonce, m2.registrar_nonce, 16));

	/* Skip public_key testing */

	assert(expected->auth_type_flags == m2.auth_type_flags);
	assert(expected->encryption_type_flags == m2.encryption_type_flags);
	assert(expected->connection_type_flags == m2.connection_type_flags);
	assert(expected->config_methods == m2.config_methods);

	assert(!strcmp(expected->manufacturer, m2.manufacturer));
	assert(!strcmp(expected->model_name, m2.model_name));
	assert(!strcmp(expected->model_number, m2.model_number));
	assert(!strcmp(expected->serial_number, m2.serial_number));

	assert(expected->primary_device_type.category ==
				m2.primary_device_type.category);
	assert(!memcmp(expected->primary_device_type.oui,
				m2.primary_device_type.oui, 3));
	assert(expected->primary_device_type.oui_type ==
				m2.primary_device_type.oui_type);
	assert(expected->primary_device_type.subcategory ==
				m2.primary_device_type.subcategory);

	assert(!strcmp(expected->device_name, m2.device_name));
	assert(expected->rf_bands == m2.rf_bands);
	assert(expected->association_state == m2.association_state);
	assert(expected->device_password_id == m2.device_password_id);
	assert(expected->configuration_error == m2.configuration_error);
	assert(expected->os_version == m2.os_version);
	assert(!memcmp(expected->authenticator, m2.authenticator, 8));

	assert(!memcmp(test->public_key, m2.public_key, 192));

	prime = l_key_new(L_KEY_RAW, crypto_dh5_prime, crypto_dh5_prime_size);
	assert(prime);

	private = l_key_new(L_KEY_RAW, test->m1->private_key,
						test->m1->private_key_size);
	assert(private);

	peer_public = l_key_new(L_KEY_RAW, test->public_key,
						test->public_key_size);
	assert(peer_public);

	assert(wsc_compute_authenticator(peer_public, private, prime,
						m2.enrollee_nonce,
						test->m1->expected.addr,
						m2.registrar_nonce,
						test->m1->pdu,
						test->m1->len,
						test->pdu,
						test->len,
						authenticator));

	assert(!memcmp(m2.authenticator, authenticator, 8));

	l_key_free(prime);
	l_key_free(private);
	l_key_free(peer_public);
}

static void wsc_test_build_m2(const void *data)
{
	const struct m2_data *test = data;
	struct wsc_m2 m2;
	uint8_t *out;
	size_t out_len;

	memcpy(&m2, &test->expected, sizeof(m2));
	memcpy(m2.public_key, test->public_key, 192);

	out = wsc_build_m2(&m2, &out_len);
	assert(out);

	assert(out_len == test->len);
	assert(!memcmp(test->pdu, out, test->len));

	l_free(out);
}

static const unsigned char eap_wsc_m3[] = {
	0x01, 0x00, 0x00, 0x8a, 0x02, 0x02, 0x00, 0x8a, 0xfe, 0x00, 0x37, 0x2a,
	0x00, 0x00, 0x00, 0x01, 0x04, 0x00, 0x10, 0x4a, 0x00, 0x01, 0x10, 0x10,
	0x22, 0x00, 0x01, 0x07, 0x10, 0x39, 0x00, 0x10, 0x67, 0xa5, 0x53, 0x61,
	0xee, 0x72, 0xc2, 0x21, 0x48, 0x78, 0xc5, 0x70, 0x89, 0x9f, 0xf2, 0xa0,
	0x10, 0x14, 0x00, 0x20, 0x70, 0x9b, 0x9a, 0x2d, 0x26, 0x98, 0x0d, 0x46,
	0x8e, 0x1c, 0xb8, 0x8a, 0xbf, 0xcb, 0x83, 0xb9, 0x65, 0xba, 0x0b, 0xd1,
	0x20, 0xd9, 0x36, 0x9e, 0xc3, 0xc7, 0xb8, 0x9b, 0xe9, 0x0a, 0xa5, 0x50,
	0x10, 0x15, 0x00, 0x20, 0xfd, 0x5f, 0xfa, 0xb0, 0xa1, 0x4b, 0x78, 0xf9,
	0xa4, 0x9c, 0x84, 0x72, 0x2b, 0x08, 0x2e, 0x73, 0x73, 0x4e, 0xfb, 0x4d,
	0xbf, 0x93, 0x08, 0xa9, 0x56, 0xa5, 0x52, 0x89, 0x4c, 0x74, 0x44, 0x8f,
	0x10, 0x49, 0x00, 0x06, 0x00, 0x37, 0x2a, 0x00, 0x01, 0x20, 0x10, 0x05,
	0x00, 0x08, 0xaf, 0x03, 0x0e, 0x95, 0xd6, 0xe6, 0x72, 0xaf,
};

struct m3_data {
	struct wsc_m3 expected;
	const void *pdu;
	unsigned int len;
};

static const struct m3_data m3_data_1 = {
	.pdu = eap_wsc_m3 + 18,
	.len = sizeof(eap_wsc_m3) - 18,
	.expected = {
		.version2 = true,
		.registrar_nonce = { 0x67, 0xa5, 0x53, 0x61, 0xee, 0x72, 0xc2,
					0x21, 0x48, 0x78, 0xc5, 0x70, 0x89,
					0x9f, 0xf2, 0xa0 },
		.e_hash1 = { 0x70, 0x9b, 0x9a, 0x2d, 0x26, 0x98, 0x0d, 0x46,
			0x8e, 0x1c, 0xb8, 0x8a, 0xbf, 0xcb, 0x83, 0xb9, 0x65,
			0xba, 0x0b, 0xd1, 0x20, 0xd9, 0x36, 0x9e, 0xc3, 0xc7,
			0xb8, 0x9b, 0xe9, 0x0a, 0xa5, 0x50 },
		.e_hash2 = { 0xfd, 0x5f, 0xfa, 0xb0, 0xa1, 0x4b, 0x78, 0xf9,
			0xa4, 0x9c, 0x84, 0x72, 0x2b, 0x08, 0x2e, 0x73, 0x73,
			0x4e, 0xfb, 0x4d, 0xbf, 0x93, 0x08, 0xa9, 0x56, 0xa5,
			0x52, 0x89, 0x4c, 0x74, 0x44, 0x8f },
		.authenticator = { 0xaf, 0x03, 0x0e, 0x95, 0xd6, 0xe6,
					0x72, 0xaf },
	},
};

static void wsc_test_parse_m3(const void *data)
{
	const struct m3_data *test = data;
	struct wsc_m3 m3;
	const struct wsc_m3 *expected = &test->expected;
	int r;

	r = wsc_parse_m3(test->pdu, test->len, &m3);
	assert(r == 0);

	assert(expected->version2 == m3.version2);
	assert(!memcmp(expected->registrar_nonce, m3.registrar_nonce, 16));
	assert(!memcmp(expected->e_hash1, m3.e_hash1, 32));
	assert(!memcmp(expected->e_hash2, m3.e_hash2, 32));
	assert(!memcmp(expected->authenticator, m3.authenticator, 8));
}

static void wsc_test_build_m3(const void *data)
{
	const struct m3_data *test = data;
	uint8_t *out;
	size_t out_len;

	out = wsc_build_m3(&test->expected, &out_len);
	assert(out);

	assert(out_len == test->len);
	assert(!memcmp(test->pdu, out, test->len));

	l_free(out);
}

static const unsigned char eap_wsc_m4[] = {
	0x01, 0x00, 0x00, 0xc4, 0x01, 0x03, 0x00, 0xc4, 0xfe, 0x00, 0x37, 0x2a,
	0x00, 0x00, 0x00, 0x01, 0x04, 0x00, 0x10, 0x4a, 0x00, 0x01, 0x10, 0x10,
	0x22, 0x00, 0x01, 0x08, 0x10, 0x1a, 0x00, 0x10, 0x19, 0x8d, 0x0d, 0x25,
	0x91, 0x2c, 0x37, 0x1c, 0xeb, 0x07, 0x89, 0x33, 0xe1, 0x25, 0xd7, 0x43,
	0x10, 0x3d, 0x00, 0x20, 0x33, 0x81, 0x27, 0xf0, 0xb6, 0x75, 0x4c, 0x11,
	0x83, 0x05, 0x3f, 0xa6, 0x10, 0x4b, 0xb5, 0x55, 0x2d, 0x7e, 0x65, 0x29,
	0x48, 0x8c, 0xbd, 0xc4, 0x47, 0x49, 0xbd, 0xc9, 0x80, 0x67, 0x0e, 0x78,
	0x10, 0x3e, 0x00, 0x20, 0x3f, 0x7c, 0x94, 0x89, 0xf8, 0x10, 0x4e, 0x9b,
	0x37, 0xf6, 0x64, 0x28, 0xe1, 0x63, 0xdb, 0xca, 0x90, 0x42, 0xf4, 0x84,
	0x3f, 0x1f, 0x2a, 0x7a, 0x3e, 0x4b, 0x29, 0xc3, 0xd2, 0x93, 0x79, 0x37,
	0x10, 0x18, 0x00, 0x40, 0x8e, 0x02, 0xd6, 0x5f, 0xff, 0x43, 0xdf, 0x55,
	0x19, 0xe2, 0x48, 0x30, 0xce, 0x02, 0x7c, 0xd6, 0x73, 0x6e, 0x19, 0x61,
	0x38, 0x99, 0x5d, 0xb3, 0x63, 0x05, 0xba, 0xf2, 0xcc, 0xa4, 0xcf, 0x24,
	0x33, 0x26, 0xb3, 0x93, 0x92, 0xbc, 0xb6, 0x65, 0x9d, 0x94, 0xe3, 0xe4,
	0x39, 0xf3, 0xd0, 0xc5, 0x51, 0x79, 0x00, 0x67, 0xa2, 0x4e, 0xd6, 0x56,
	0xfa, 0xb3, 0x0e, 0x0d, 0xca, 0x45, 0x76, 0xdf, 0x10, 0x05, 0x00, 0x08,
	0x30, 0xf1, 0xf6, 0x4d, 0x5c, 0x55, 0x36, 0x3c,
};

struct m4_data {
	struct wsc_m4 expected;
	const uint8_t *expected_encrypted;
	size_t expected_encrypted_size;
	const void *pdu;
	unsigned int len;
};

static const struct m4_data m4_data_1 = {
	.pdu = eap_wsc_m4 + 18,
	.len = sizeof(eap_wsc_m4) - 18,
	.expected = {
		.version2 = false,
		.enrollee_nonce = { 0x19, 0x8d, 0x0d, 0x25, 0x91, 0x2c, 0x37,
					0x1c, 0xeb, 0x07, 0x89, 0x33, 0xe1,
					0x25, 0xd7, 0x43 },
		.r_hash1 = { 0x33, 0x81, 0x27, 0xf0, 0xb6, 0x75, 0x4c, 0x11,
			0x83, 0x05, 0x3f, 0xa6, 0x10, 0x4b, 0xb5, 0x55, 0x2d,
			0x7e, 0x65, 0x29, 0x48, 0x8c, 0xbd, 0xc4, 0x47, 0x49,
			0xbd, 0xc9, 0x80, 0x67, 0x0e, 0x78 },
		.r_hash2 = { 0x3f, 0x7c, 0x94, 0x89, 0xf8, 0x10, 0x4e, 0x9b,
			0x37, 0xf6, 0x64, 0x28, 0xe1, 0x63, 0xdb, 0xca, 0x90,
			0x42, 0xf4, 0x84, 0x3f, 0x1f, 0x2a, 0x7a, 0x3e, 0x4b,
			0x29, 0xc3, 0xd2, 0x93, 0x79, 0x37 },
		.authenticator = { 0x30, 0xf1, 0xf6, 0x4d, 0x5c, 0x55,
					0x36, 0x3c },
	},
	.expected_encrypted = eap_wsc_m4 + 124,
	.expected_encrypted_size = 64,
};

static void wsc_test_parse_m4(const void *data)
{
	const struct m4_data *test = data;
	struct wsc_m4 m4;
	const struct wsc_m4 *expected = &test->expected;
	struct iovec encrypted;
	int r;

	r = wsc_parse_m4(test->pdu, test->len, &m4, &encrypted);
	assert(r == 0);

	assert(expected->version2 == m4.version2);
	assert(!memcmp(expected->enrollee_nonce, m4.enrollee_nonce, 16));
	assert(!memcmp(expected->r_hash1, m4.r_hash1, 32));
	assert(!memcmp(expected->r_hash2, m4.r_hash2, 32));
	assert(!memcmp(expected->authenticator, m4.authenticator, 8));

	assert(test->expected_encrypted_size == encrypted.iov_len);
	assert(!memcmp(test->expected_encrypted,
					encrypted.iov_base, encrypted.iov_len));
}

static void wsc_test_build_m4(const void *data)
{
	const struct m4_data *test = data;
	uint8_t *out;
	size_t out_len;

	out = wsc_build_m4(&test->expected, test->expected_encrypted,
				test->expected_encrypted_size, &out_len);
	assert(out);

	assert(out_len == test->len);
	assert(!memcmp(test->pdu, out, test->len));

	l_free(out);
}

static const unsigned char m4_encrypted_settings[] = {
	0x10, 0x3f, 0x00, 0x10, 0xaf, 0xbe, 0xbc, 0x19, 0x44, 0xdf, 0xf3, 0x40,
	0x3c, 0xdc, 0x70, 0xc0, 0x63, 0x63, 0xc7, 0x53, 0x10, 0x1e, 0x00, 0x08,
	0x47, 0x82, 0x81, 0xb3, 0x1e, 0xe0, 0xa4, 0xcb,
};

struct m4_encrypted_settings_data {
	struct wsc_m4_encrypted_settings expected;
	const void *pdu;
	unsigned int len;
};

static const struct m4_encrypted_settings_data m4_encrypted_settings_data_1 = {
	.pdu = m4_encrypted_settings,
	.len = sizeof(m4_encrypted_settings),
	.expected = {
		.r_snonce1 = { 0xaf, 0xbe, 0xbc, 0x19, 0x44, 0xdf, 0xf3, 0x40,
			0x3c, 0xdc, 0x70, 0xc0, 0x63, 0x63, 0xc7, 0x53 },
		.authenticator = { 0x47, 0x82, 0x81, 0xb3, 0x1e, 0xe0,
					0xa4, 0xcb },
	},
};

static void wsc_test_parse_m4_encrypted_settings(const void *data)
{
	const struct m4_encrypted_settings_data *test = data;
	struct wsc_m4_encrypted_settings m4es;
	const struct wsc_m4_encrypted_settings *expected = &test->expected;
	int r;

	r = wsc_parse_m4_encrypted_settings(test->pdu, test->len, &m4es);
	assert(r == 0);

	assert(!memcmp(expected->r_snonce1, m4es.r_snonce1, 16));
	assert(!memcmp(expected->authenticator, m4es.authenticator, 8));
}

static void wsc_test_build_m4_encrypted_settings(const void *data)
{
	const struct m4_encrypted_settings_data *test = data;
	uint8_t *out;
	size_t out_len;

	out = wsc_build_m4_encrypted_settings(&test->expected, &out_len);
	assert(out);

	assert(out_len == test->len);
	assert(!memcmp(test->pdu, out, test->len));

	l_free(out);
}

static const unsigned char eap_wsc_m5[] = {
	0x01, 0x00, 0x00, 0x86, 0x02, 0x03, 0x00, 0x86, 0xfe, 0x00, 0x37, 0x2a,
	0x00, 0x00, 0x00, 0x01, 0x04, 0x00, 0x10, 0x4a, 0x00, 0x01, 0x10, 0x10,
	0x22, 0x00, 0x01, 0x09, 0x10, 0x39, 0x00, 0x10, 0x67, 0xa5, 0x53, 0x61,
	0xee, 0x72, 0xc2, 0x21, 0x48, 0x78, 0xc5, 0x70, 0x89, 0x9f, 0xf2, 0xa0,
	0x10, 0x18, 0x00, 0x40, 0x9a, 0x31, 0xf8, 0x4b, 0x46, 0x72, 0xf2, 0xcc,
	0xf6, 0x3c, 0x84, 0x5e, 0xed, 0x34, 0x64, 0xec, 0xe6, 0x04, 0xe5, 0xb1,
	0xac, 0xac, 0xa5, 0x5d, 0x50, 0x5f, 0xcc, 0x5c, 0x99, 0x13, 0x3f, 0xe7,
	0x24, 0xb8, 0x10, 0x41, 0x04, 0x94, 0x2a, 0x3c, 0x56, 0xe5, 0x14, 0xee,
	0x57, 0xda, 0xdf, 0x25, 0x57, 0xbe, 0x3e, 0x71, 0x8a, 0x10, 0x3a, 0x88,
	0xd9, 0x74, 0x42, 0x2d, 0x53, 0x6a, 0xa6, 0xe5, 0x10, 0x49, 0x00, 0x06,
	0x00, 0x37, 0x2a, 0x00, 0x01, 0x20, 0x10, 0x05, 0x00, 0x08, 0x6c, 0x80,
	0x5a, 0x2f, 0xac, 0x19, 0x7c, 0x35,
};

struct m5_data {
	struct wsc_m5 expected;
	const uint8_t *expected_encrypted;
	size_t expected_encrypted_size;
	const void *pdu;
	unsigned int len;
};

static const struct m5_data m5_data_1 = {
	.pdu = eap_wsc_m5 + 18,
	.len = sizeof(eap_wsc_m5) - 18,
	.expected = {
		.version2 = true,
		.registrar_nonce = { 0x67, 0xa5, 0x53, 0x61, 0xee, 0x72, 0xc2,
					0x21, 0x48, 0x78, 0xc5, 0x70, 0x89,
					0x9f, 0xf2, 0xa0 },
		.authenticator = { 0x6c, 0x80, 0x5a, 0x2f, 0xac, 0x19,
						0x7c, 0x35 },
	},
	.expected_encrypted = eap_wsc_m5 + 52,
	.expected_encrypted_size = 64,
};

static void wsc_test_parse_m5(const void *data)
{
	const struct m5_data *test = data;
	struct wsc_m5 m5;
	const struct wsc_m5 *expected = &test->expected;
	struct iovec encrypted;
	int r;

	r = wsc_parse_m5(test->pdu, test->len, &m5, &encrypted);
	assert(r == 0);

	assert(expected->version2 == m5.version2);
	assert(!memcmp(expected->registrar_nonce, m5.registrar_nonce, 16));
	assert(!memcmp(expected->authenticator, m5.authenticator, 8));

	assert(test->expected_encrypted_size == encrypted.iov_len);
	assert(!memcmp(test->expected_encrypted,
					encrypted.iov_base, encrypted.iov_len));
}

static void wsc_test_build_m5(const void *data)
{
	const struct m5_data *test = data;
	uint8_t *out;
	size_t out_len;

	out = wsc_build_m5(&test->expected, test->expected_encrypted,
				test->expected_encrypted_size, &out_len);
	assert(out);

	assert(out_len == test->len);
	assert(!memcmp(test->pdu, out, test->len));

	l_free(out);
}

static const unsigned char eap_wsc_m6[] = {
	0x01, 0x00, 0x00, 0x7c, 0x01, 0xaa, 0x00, 0x7c, 0xfe, 0x00, 0x37, 0x2a,
	0x00, 0x00, 0x00, 0x01, 0x04, 0x00, 0x10, 0x4a, 0x00, 0x01, 0x10, 0x10,
	0x22, 0x00, 0x01, 0x0a, 0x10, 0x1a, 0x00, 0x10, 0x19, 0x8d, 0x0d, 0x25,
	0x91, 0x2c, 0x37, 0x1c, 0xeb, 0x07, 0x89, 0x33, 0xe1, 0x25, 0xd7, 0x43,
	0x10, 0x18, 0x00, 0x40, 0x62, 0xfe, 0xc7, 0xf5, 0x1d, 0xa1, 0x0e, 0x5b,
	0xe9, 0x2f, 0xcb, 0x4f, 0x30, 0x7b, 0x15, 0x17, 0x9c, 0xa1, 0x92, 0x1d,
	0xd4, 0x7f, 0x06, 0x2f, 0x14, 0xcd, 0x4b, 0x8f, 0xe6, 0x76, 0xe7, 0x6c,
	0x9a, 0xc1, 0xd0, 0x1f, 0xab, 0x5e, 0x0f, 0xf5, 0xa7, 0xa9, 0x9a, 0xca,
	0xf3, 0xcc, 0xbd, 0x5f, 0xfc, 0xfb, 0x6f, 0x5b, 0x01, 0xb7, 0x05, 0x39,
	0x11, 0xc3, 0xaa, 0x92, 0x5c, 0x06, 0xdb, 0x21, 0x10, 0x05, 0x00, 0x08,
	0xc4, 0x40, 0xd2, 0x39, 0xf1, 0x07, 0x72, 0xcb,
};

struct m6_data {
	struct wsc_m6 expected;
	const uint8_t *expected_encrypted;
	size_t expected_encrypted_size;
	const void *pdu;
	unsigned int len;
};

static const struct m6_data m6_data_1 = {
	.pdu = eap_wsc_m6 + 18,
	.len = sizeof(eap_wsc_m6) - 18,
	.expected = {
		.version2 = false,
		.enrollee_nonce = { 0x19, 0x8d, 0x0d, 0x25, 0x91, 0x2c, 0x37,
					0x1c, 0xeb, 0x07, 0x89, 0x33, 0xe1,
					0x25, 0xd7, 0x43 },
		.authenticator = { 0xc4, 0x40, 0xd2, 0x39, 0xf1, 0x07,
								0x72, 0xcb },
	},
	.expected_encrypted = eap_wsc_m6 + 52,
	.expected_encrypted_size = 64,
};

static void wsc_test_parse_m6(const void *data)
{
	const struct m6_data *test = data;
	struct wsc_m6 m6;
	const struct wsc_m6 *expected = &test->expected;
	struct iovec encrypted;
	int r;

	r = wsc_parse_m6(test->pdu, test->len, &m6, &encrypted);
	assert(r == 0);

	assert(expected->version2 == m6.version2);
	assert(!memcmp(expected->enrollee_nonce, m6.enrollee_nonce, 16));
	assert(!memcmp(expected->authenticator, m6.authenticator, 8));

	assert(test->expected_encrypted_size == encrypted.iov_len);
	assert(!memcmp(test->expected_encrypted,
					encrypted.iov_base, encrypted.iov_len));
}

static void wsc_test_build_m6(const void *data)
{
	const struct m6_data *test = data;
	uint8_t *out;
	size_t out_len;

	out = wsc_build_m6(&test->expected, test->expected_encrypted,
				test->expected_encrypted_size, &out_len);
	assert(out);

	assert(out_len == test->len);
	assert(!memcmp(test->pdu, out, test->len));

	l_free(out);
}

static const unsigned char m6_encrypted_settings[] = {
	0x10, 0x40, 0x00, 0x10, 0x57, 0x29, 0x13, 0xb5, 0x89, 0xf1, 0x50, 0x7e,
	0x4d, 0x5c, 0x88, 0xda, 0x25, 0x9b, 0xce, 0xf0, 0x10, 0x1e, 0x00, 0x08,
	0x72, 0x8c, 0x3c, 0xf8, 0xd4, 0xd2, 0xb1, 0xa1,
};

struct m6_encrypted_settings_data {
	struct wsc_m6_encrypted_settings expected;
	const void *pdu;
	unsigned int len;
};

static const struct m6_encrypted_settings_data m6_encrypted_settings_data_1 = {
	.pdu = m6_encrypted_settings,
	.len = sizeof(m6_encrypted_settings),
	.expected = {
		.r_snonce2 = { 0x57, 0x29, 0x13, 0xb5, 0x89, 0xf1, 0x50, 0x7e,
			0x4d, 0x5c, 0x88, 0xda, 0x25, 0x9b, 0xce, 0xf0 },
		.authenticator = { 0x72, 0x8c, 0x3c, 0xf8, 0xd4, 0xd2,
					0xb1, 0xa1 },
	},
};

static void wsc_test_parse_m6_encrypted_settings(const void *data)
{
	const struct m6_encrypted_settings_data *test = data;
	struct wsc_m6_encrypted_settings m6es;
	const struct wsc_m6_encrypted_settings *expected = &test->expected;
	int r;

	r = wsc_parse_m6_encrypted_settings(test->pdu, test->len, &m6es);
	assert(r == 0);

	assert(!memcmp(expected->r_snonce2, m6es.r_snonce2, 16));
	assert(!memcmp(expected->authenticator, m6es.authenticator, 8));
}

static void wsc_test_build_m6_encrypted_settings(const void *data)
{
	const struct m6_encrypted_settings_data *test = data;
	uint8_t *out;
	size_t out_len;

	out = wsc_build_m6_encrypted_settings(&test->expected, &out_len);
	assert(out);

	assert(out_len == test->len);
	assert(!memcmp(test->pdu, out, test->len));

	l_free(out);
}

static const unsigned char eap_wsc_m7[] = {
	0x01, 0x00, 0x00, 0x86, 0x02, 0xaa, 0x00, 0x86, 0xfe, 0x00, 0x37, 0x2a,
	0x00, 0x00, 0x00, 0x01, 0x04, 0x00, 0x10, 0x4a, 0x00, 0x01, 0x10, 0x10,
	0x22, 0x00, 0x01, 0x0b, 0x10, 0x39, 0x00, 0x10, 0x67, 0xa5, 0x53, 0x61,
	0xee, 0x72, 0xc2, 0x21, 0x48, 0x78, 0xc5, 0x70, 0x89, 0x9f, 0xf2, 0xa0,
	0x10, 0x18, 0x00, 0x40, 0x4e, 0x3a, 0x4c, 0xf0, 0x88, 0x17, 0x69, 0x89,
	0xe1, 0x48, 0xd4, 0xc1, 0x0b, 0x96, 0xe8, 0xfd, 0x1e, 0x9b, 0xf4, 0xc0,
	0x74, 0xcc, 0xa2, 0xeb, 0x87, 0x58, 0xc9, 0x42, 0xd1, 0xf3, 0x89, 0xe2,
	0x27, 0x13, 0xfe, 0x40, 0x10, 0x45, 0x51, 0xfd, 0xc2, 0x1f, 0xe6, 0x60,
	0x84, 0xc2, 0x8f, 0x9d, 0x43, 0xbc, 0x03, 0xf3, 0x0a, 0x64, 0x8f, 0x18,
	0xe9, 0x96, 0x0f, 0x9e, 0xeb, 0x52, 0x73, 0x70, 0x10, 0x49, 0x00, 0x06,
	0x00, 0x37, 0x2a, 0x00, 0x01, 0x20, 0x10, 0x05, 0x00, 0x08, 0x05, 0x05,
	0x2f, 0x5f, 0x3c, 0x42, 0xf8, 0xa6,
};

struct m7_data {
	struct wsc_m7 expected;
	const uint8_t *expected_encrypted;
	size_t expected_encrypted_size;
	const void *pdu;
	unsigned int len;
};

static const struct m7_data m7_data_1 = {
	.pdu = eap_wsc_m7 + 18,
	.len = sizeof(eap_wsc_m7) - 18,
	.expected = {
		.version2 = true,
		.registrar_nonce = { 0x67, 0xa5, 0x53, 0x61, 0xee, 0x72, 0xc2,
					0x21, 0x48, 0x78, 0xc5, 0x70, 0x89,
					0x9f, 0xf2, 0xa0 },
		.authenticator = { 0x05, 0x05, 0x2f, 0x5f, 0x3c, 0x42,
					0xf8, 0xa6 },
	},
	.expected_encrypted = eap_wsc_m7 + 52,
	.expected_encrypted_size = 64,
};

static void wsc_test_parse_m7(const void *data)
{
	const struct m7_data *test = data;
	struct wsc_m7 m7;
	const struct wsc_m7 *expected = &test->expected;
	struct iovec encrypted;
	int r;

	r = wsc_parse_m7(test->pdu, test->len, &m7, &encrypted);
	assert(r == 0);

	assert(expected->version2 == m7.version2);
	assert(!memcmp(expected->registrar_nonce, m7.registrar_nonce, 16));
	assert(!memcmp(expected->authenticator, m7.authenticator, 8));

	assert(test->expected_encrypted_size == encrypted.iov_len);
	assert(!memcmp(test->expected_encrypted,
					encrypted.iov_base, encrypted.iov_len));
}

static void wsc_test_build_m7(const void *data)
{
	const struct m7_data *test = data;
	uint8_t *out;
	size_t out_len;

	out = wsc_build_m7(&test->expected, test->expected_encrypted,
				test->expected_encrypted_size, &out_len);
	assert(out);

	assert(out_len == test->len);
	assert(!memcmp(test->pdu, out, test->len));

	l_free(out);
}

static const unsigned char eap_wsc_m8[] = {
	0x01, 0x00, 0x00, 0xcc, 0x01, 0xab, 0x00, 0xcc, 0xfe, 0x00, 0x37, 0x2a,
	0x00, 0x00, 0x00, 0x01, 0x04, 0x00, 0x10, 0x4a, 0x00, 0x01, 0x10, 0x10,
	0x22, 0x00, 0x01, 0x0c, 0x10, 0x1a, 0x00, 0x10, 0x19, 0x8d, 0x0d, 0x25,
	0x91, 0x2c, 0x37, 0x1c, 0xeb, 0x07, 0x89, 0x33, 0xe1, 0x25, 0xd7, 0x43,
	0x10, 0x18, 0x00, 0x90, 0x7e, 0x2f, 0x4a, 0xae, 0x60, 0x1e, 0x54, 0x75,
	0x3b, 0x6e, 0xfc, 0x0a, 0x84, 0xdb, 0xae, 0x62, 0x74, 0xf4, 0xe5, 0xe0,
	0xf2, 0xc5, 0xf1, 0x78, 0xc0, 0x76, 0x4d, 0xc4, 0x08, 0x6d, 0xd5, 0xb7,
	0xa2, 0x74, 0x18, 0x26, 0xee, 0x01, 0xe4, 0x3d, 0x5f, 0x44, 0xca, 0x93,
	0xad, 0x07, 0xc7, 0xa5, 0xcf, 0xf1, 0xb9, 0xb9, 0x68, 0xf1, 0x15, 0xa9,
	0xbc, 0x66, 0x7f, 0x03, 0xa1, 0x84, 0x80, 0x78, 0x71, 0x13, 0x09, 0xaa,
	0x74, 0x60, 0xa2, 0x0f, 0x49, 0x4a, 0x8b, 0x8f, 0xd9, 0xab, 0x07, 0x5f,
	0x79, 0xc1, 0xb2, 0x22, 0x16, 0xc7, 0x05, 0xf8, 0xe8, 0x3e, 0x46, 0x26,
	0x9a, 0xc5, 0x01, 0xfe, 0xa2, 0x15, 0x1c, 0xbe, 0x97, 0x87, 0xc8, 0x83,
	0xb9, 0xf2, 0x89, 0x82, 0xb8, 0xe0, 0x46, 0x08, 0xb1, 0xbd, 0x1d, 0xe8,
	0x9f, 0x1d, 0x50, 0xbd, 0x79, 0x2c, 0xa7, 0xf7, 0xe7, 0x34, 0xf4, 0x7c,
	0x28, 0xa4, 0x9a, 0x42, 0x6e, 0x94, 0x88, 0x78, 0x28, 0xfd, 0xea, 0xdf,
	0x13, 0x5c, 0x66, 0x79, 0x10, 0x05, 0x00, 0x08, 0xba, 0x2b, 0xfc, 0x68,
	0x49, 0x8e, 0x88, 0xfa,
};

struct m8_data {
	struct wsc_m8 expected;
	const uint8_t *expected_encrypted;
	size_t expected_encrypted_size;
	const void *pdu;
	unsigned int len;
};

static const struct m8_data m8_data_1 = {
	.pdu = eap_wsc_m8 + 18,
	.len = sizeof(eap_wsc_m8) - 18,
	.expected = {
		.version2 = false,
		.enrollee_nonce = { 0x19, 0x8d, 0x0d, 0x25, 0x91, 0x2c, 0x37,
					0x1c, 0xeb, 0x07, 0x89, 0x33, 0xe1,
					0x25, 0xd7, 0x43 },
		.authenticator = { 0xba, 0x2b, 0xfc, 0x68, 0x49, 0x8e,
					0x88, 0xfa },

	},
	.expected_encrypted = eap_wsc_m8 + 52,
	.expected_encrypted_size = 144,
};

static void wsc_test_parse_m8(const void *data)
{
	const struct m8_data *test = data;
	struct wsc_m8 m8;
	const struct wsc_m8 *expected = &test->expected;
	struct iovec encrypted;
	int r;

	r = wsc_parse_m8(test->pdu, test->len, &m8, &encrypted);
	assert(r == 0);

	assert(expected->version2 == m8.version2);
	assert(!memcmp(expected->enrollee_nonce, m8.enrollee_nonce, 16));
	assert(!memcmp(expected->authenticator, m8.authenticator, 8));

	assert(test->expected_encrypted_size == encrypted.iov_len);
	assert(!memcmp(test->expected_encrypted,
					encrypted.iov_base, encrypted.iov_len));
}

static void wsc_test_build_m8(const void *data)
{
	const struct m8_data *test = data;
	uint8_t *out;
	size_t out_len;

	out = wsc_build_m8(&test->expected, test->expected_encrypted,
				test->expected_encrypted_size, &out_len);
	assert(out);

	assert(out_len == test->len);
	assert(!memcmp(test->pdu, out, test->len));

	l_free(out);
}

static const unsigned char m8_encrypted_settings[] = {
	0x10, 0x0e, 0x00, 0x6f, 0x10, 0x26, 0x00, 0x01, 0x00, 0x10, 0x45, 0x00,
	0x07, 0x54, 0x65, 0x73, 0x74, 0x57, 0x50, 0x41, 0x10, 0x03, 0x00, 0x02,
	0x00, 0x20, 0x10, 0x0f, 0x00, 0x02, 0x00, 0x08, 0x10, 0x28, 0x00, 0x01,
	0x00, 0x10, 0x27, 0x00, 0x40, 0x34, 0x36, 0x30, 0x34, 0x44, 0x30, 0x31,
	0x46, 0x46, 0x44, 0x42, 0x30, 0x42, 0x32, 0x39, 0x32, 0x45, 0x33, 0x37,
	0x37, 0x33, 0x32, 0x44, 0x44, 0x34, 0x45, 0x31, 0x31, 0x43, 0x32, 0x34,
	0x30, 0x31, 0x31, 0x35, 0x34, 0x32, 0x38, 0x39, 0x41, 0x30, 0x39, 0x41,
	0x33, 0x33, 0x41, 0x44, 0x37, 0x30, 0x34, 0x31, 0x37, 0x37, 0x41, 0x42,
	0x30, 0x44, 0x31, 0x42, 0x37, 0x35, 0x38, 0x44, 0x30, 0x10, 0x20, 0x00,
	0x06, 0xa0, 0xa8, 0xcd, 0x1c, 0x7e, 0xc9, 0x10, 0x1e, 0x00, 0x08, 0xe8,
	0x3b, 0x3b, 0xe7, 0x9e, 0x72, 0x06, 0x46,
};

struct wsc_credential creds_1[1] = {
	{
		.ssid = "TestWPA",
		.ssid_len = 7,
		.auth_type = WSC_AUTHENTICATION_TYPE_WPA2_PERSONAL,
		.encryption_type = WSC_ENCRYPTION_TYPE_AES,
		.network_key = { 0x34, 0x36, 0x30, 0x34, 0x44, 0x30, 0x31, 0x46,
			0x46, 0x44, 0x42, 0x30, 0x42, 0x32, 0x39, 0x32, 0x45,
			0x33, 0x37, 0x37, 0x33, 0x32, 0x44, 0x44, 0x34, 0x45,
			0x31, 0x31, 0x43, 0x32, 0x34, 0x30, 0x31, 0x31, 0x35,
			0x34, 0x32, 0x38, 0x39, 0x41, 0x30, 0x39, 0x41, 0x33,
			0x33, 0x41, 0x44, 0x37, 0x30, 0x34, 0x31, 0x37, 0x37,
			0x41, 0x42, 0x30, 0x44, 0x31, 0x42, 0x37, 0x35, 0x38,
			0x44, 0x30 },
		.network_key_len = 64,
		.addr = { 0xa0, 0xa8, 0xcd, 0x1c, 0x7e, 0xc9 },
	},
};

struct m8_encrypted_settings_data {
	struct wsc_m8_encrypted_settings expected;
	struct wsc_credential *creds;
	unsigned int n_creds;
	const void *pdu;
	unsigned int len;
};

static const struct m8_encrypted_settings_data m8_encrypted_settings_data_1 = {
	.pdu = m8_encrypted_settings,
	.len = sizeof(m8_encrypted_settings),
	.creds = creds_1,
	.n_creds = 1,
	.expected = {
		.authenticator = { 0xe8, 0x3b, 0x3b, 0xe7, 0x9e, 0x72,
					0x06, 0x46 },
	},
};

static void wsc_test_parse_m8_encrypted_settings(const void *data)
{
	const struct m8_encrypted_settings_data *test = data;
	struct wsc_m8_encrypted_settings m8es;
	const struct wsc_m8_encrypted_settings *expected = &test->expected;
	struct iovec creds[10];
	size_t n_creds = 10;
	size_t i;
	int r;

	r = wsc_parse_m8_encrypted_settings(test->pdu, test->len, &m8es,
							creds, &n_creds);
	assert(r == 0);

	assert(expected->new_password_len == m8es.new_password_len);

	if (m8es.new_password_len > 0) {
		assert(!memcmp(expected->new_password, m8es.new_password,
					m8es.new_password_len));
		assert(expected->device_password_id == m8es.device_password_id);
	}

	assert(n_creds == test->n_creds);

	for (i = 0; i < n_creds; i++) {
		struct wsc_credential cred;

		assert(!wsc_parse_credential(creds[i].iov_base,
						creds[i].iov_len,
						&cred));

		assert(test->creds[i].ssid_len == cred.ssid_len);
		assert(!memcmp(test->creds[i].ssid, cred.ssid, cred.ssid_len));
		assert(test->creds[i].auth_type == cred.auth_type);
		assert(test->creds[i].encryption_type == cred.encryption_type);
		assert(test->creds[i].network_key_len == cred.network_key_len);
		assert(!memcmp(test->creds[i].network_key, cred.network_key,
							cred.network_key_len));
		assert(!memcmp(test->creds[i].addr, cred.addr, 6));
	}

	assert(!memcmp(expected->authenticator, m8es.authenticator, 8));
}

static const unsigned char eap_wsc_done[] = {
	0x01, 0x00, 0x00, 0x4a, 0x02, 0xab, 0x00, 0x4a, 0xfe, 0x00, 0x37, 0x2a,
	0x00, 0x00, 0x00, 0x01, 0x05, 0x00, 0x10, 0x4a, 0x00, 0x01, 0x10, 0x10,
	0x22, 0x00, 0x01, 0x0f, 0x10, 0x1a, 0x00, 0x10, 0x19, 0x8d, 0x0d, 0x25,
	0x91, 0x2c, 0x37, 0x1c, 0xeb, 0x07, 0x89, 0x33, 0xe1, 0x25, 0xd7, 0x43,
	0x10, 0x39, 0x00, 0x10, 0x67, 0xa5, 0x53, 0x61, 0xee, 0x72, 0xc2, 0x21,
	0x48, 0x78, 0xc5, 0x70, 0x89, 0x9f, 0xf2, 0xa0, 0x10, 0x49, 0x00, 0x06,
	0x00, 0x37, 0x2a, 0x00, 0x01, 0x20,
};

struct wsc_done_data {
	struct wsc_done expected;
	const void *pdu;
	unsigned int len;
};

static const struct wsc_done_data wsc_done_data_1 = {
	.pdu = eap_wsc_done + 18,
	.len = sizeof(eap_wsc_done) - 18,
	.expected = {
		.version2 = true,
		.enrollee_nonce = { 0x19, 0x8d, 0x0d, 0x25, 0x91, 0x2c, 0x37,
					0x1c, 0xeb, 0x07, 0x89, 0x33, 0xe1,
					0x25, 0xd7, 0x43 },
		.registrar_nonce = { 0x67, 0xa5, 0x53, 0x61, 0xee, 0x72, 0xc2,
					0x21, 0x48, 0x78, 0xc5, 0x70, 0x89,
					0x9f, 0xf2, 0xa0 },
	},
};

static void wsc_test_parse_wsc_done(const void *data)
{
	const struct wsc_done_data *test = data;
	struct wsc_done done;
	const struct wsc_done *expected = &test->expected;
	int r;

	r = wsc_parse_wsc_done(test->pdu, test->len, &done);
	assert(r == 0);

	assert(expected->version2 == done.version2);
	assert(!memcmp(expected->enrollee_nonce, done.enrollee_nonce, 16));
	assert(!memcmp(expected->registrar_nonce, done.registrar_nonce, 16));
}

static void wsc_test_build_wsc_done(const void *data)
{
	const struct wsc_done_data *test = data;
	struct wsc_done done;
	uint8_t *out;
	size_t out_len;

	memcpy(&done, &test->expected, sizeof(done));

	out = wsc_build_wsc_done(&done, &out_len);
	assert(out);

	assert(out_len == test->len);
	assert(!memcmp(test->pdu, out, test->len));

	l_free(out);
}

static const uint8_t eap_identity_req[] = {
	0x01, 0x00, 0x00, 0x05, 0x01, 0x00, 0x00, 0x05, 0x01
};

static const uint8_t eap_identity_resp[] = {
	0x01, 0x00, 0x00, 0x22, 0x02, 0x00, 0x00, 0x22,
	0x01, 0x57, 0x46, 0x41, 0x2d, 0x53, 0x69, 0x6d,
	0x70, 0x6c, 0x65, 0x43, 0x6f, 0x6e, 0x66, 0x69,
	0x67, 0x2d, 0x45, 0x6e, 0x72, 0x6f, 0x6c, 0x6c,
	0x65, 0x65, 0x2d, 0x31, 0x2d, 0x30,
};

static const uint8_t eap_wsc_start[] = {
	0x01, 0x00, 0x00, 0x0e, 0x01, 0x01, 0x00, 0x0e, 0xfe, 0x00, 0x37, 0x2a,
	0x00, 0x00, 0x00, 0x01, 0x01, 0x00,
};

static const uint8_t eap_fail[] = {
	0x01, 0x00, 0x00, 0x04, 0x04, 0xab, 0x00, 0x04,
};

struct verify_data {
	bool response_sent;
	bool eapol_failed;
	const void *expected;
	size_t expected_len;
	struct wsc_credential *expected_creds;
	unsigned int n_creds;
	unsigned int cur_cred;
};

#define VERIFY_RESET(verify, e)				\
	verify.response_sent = false;			\
	verify.expected = e;				\
	verify.expected_len = sizeof(e)

static void verify_handshake_event(struct handshake_state *hs,
		enum handshake_event event, void *event_data, void *user_data)
{
	struct verify_data *data = user_data;

	switch (event) {
	case HANDSHAKE_EVENT_FAILED:
		assert(l_get_u16(event_data) ==
				MMPDU_REASON_CODE_IEEE8021X_FAILED);
		data->eapol_failed = true;
		break;
	default:
		break;
	}
}

static int verify_8021x(uint32_t ifindex,
			const uint8_t *aa_addr, uint16_t proto,
			const struct eapol_frame *ef, bool noencrypt,
			void *user_data)
{
	struct verify_data *data = user_data;
	size_t len = sizeof(struct eapol_header) +
		L_BE16_TO_CPU(ef->header.packet_len);

	assert(len == data->expected_len);
	assert(!memcmp(ef, data->expected, data->expected_len));

	data->response_sent = true;

	return 0;
}

static void verify_credential(unsigned int event, const void *event_data,
					void *user_data)
{
	struct verify_data *data = user_data;
	const struct wsc_credential *cred;

	if (event != EAP_WSC_EVENT_CREDENTIAL_OBTAINED)
		assert(false);

	cred = event_data;
	assert(!memcmp(cred, data->expected_creds + data->cur_cred,
						sizeof(struct wsc_credential)));

	data->cur_cred += 1;
}

static void wsc_test_pbc_handshake(const void *data)
{
	static uint8_t ap_address[] = { 0x24, 0xa2, 0xe1, 0xec, 0x17, 0x04 };
	static uint8_t sta_address[] = { 0xa0, 0xa8, 0xcd, 0x1c, 0x7e, 0xc9 };
	struct verify_data verify;
	struct handshake_state *hs;
	struct eapol_sm *sm;
	char *hex;
	struct l_settings *settings;

	eap_init(1400);
	eapol_init();

	hs = test_handshake_state_new(1);
	sm = eapol_sm_new(hs);
	eapol_register(sm);

	handshake_state_set_authenticator_address(hs, ap_address);
	handshake_state_set_supplicant_address(hs, sta_address);
	handshake_state_set_event_func(hs, verify_handshake_event, &verify);

	__eapol_set_tx_packet_func(verify_8021x);
	__eapol_set_tx_user_data(&verify);
	eapol_sm_set_user_data(sm, &verify);
	eapol_sm_set_event_func(sm, verify_credential);

	settings = l_settings_new();
	l_settings_set_string(settings, "Security", "EAP-Identity",
					"WFA-SimpleConfig-Enrollee-1-0");
	l_settings_set_string(settings, "Security", "EAP-Method", "WSC");

	l_settings_set_uint(settings, "WSC", "RFBand", WSC_RF_BAND_2_4_GHZ);
	l_settings_set_uint(settings, "WSC", "ConfigurationMethods",
				WSC_CONFIGURATION_METHOD_VIRTUAL_DISPLAY_PIN |
				WSC_CONFIGURATION_METHOD_KEYPAD |
				WSC_CONFIGURATION_METHOD_DISPLAY |
				WSC_CONFIGURATION_METHOD_NFC_INTERFACE);
	l_settings_set_string(settings, "WSC", "PrimaryDeviceType",
					"0-00000000-0");
	l_settings_set_string(settings, "WSC", "EnrolleeMAC",
					util_address_to_string(sta_address));

	hex = l_util_hexstring(m1_data_2.expected.enrollee_nonce, 16);
	l_settings_set_string(settings, "WSC", "EnrolleeNonce", hex);
	l_free(hex);

	hex = l_util_hexstring(m1_data_2.private_key,
						m1_data_2.private_key_size);
	l_settings_set_string(settings, "WSC", "PrivateKey", hex);
	l_free(hex);

	l_settings_set_string(settings, "WSC", "E-SNonce1",
					"fdbb480ee6f572f3591cc3b364f2185b");
	l_settings_set_string(settings, "WSC", "E-SNonce2",
					"c12698739faf385920ba659d524c71c9");
	l_settings_set_string(settings, "WSC", "IV1",
					"9a31f84b4672f2ccf63c845eed3464ec");
	l_settings_set_string(settings, "WSC", "IV2",
					"4e3a4cf088176989e148d4c10b96e8fd");

	handshake_state_set_8021x_config(hs, settings);
	eapol_start(sm);

	l_settings_free(settings);

	VERIFY_RESET(verify, eap_identity_resp);
	__eapol_rx_packet(1, ap_address, ETH_P_PAE, eap_identity_req,
					sizeof(eap_identity_req), false);
	assert(verify.response_sent);

	VERIFY_RESET(verify, eap_wsc_m1_2);
	__eapol_rx_packet(1, ap_address, ETH_P_PAE, eap_wsc_start,
						sizeof(eap_wsc_start), false);
	assert(verify.response_sent);

	VERIFY_RESET(verify, eap_wsc_m3);
	__eapol_rx_packet(1, ap_address, ETH_P_PAE, eap_wsc_m2_2,
						sizeof(eap_wsc_m2_2), false);
	assert(verify.response_sent);

	VERIFY_RESET(verify, eap_wsc_m5);
	__eapol_rx_packet(1, ap_address, ETH_P_PAE, eap_wsc_m4,
						sizeof(eap_wsc_m4), false);
	assert(verify.response_sent);

	VERIFY_RESET(verify, eap_wsc_m7);
	__eapol_rx_packet(1, ap_address, ETH_P_PAE, eap_wsc_m6,
						sizeof(eap_wsc_m6), false);
	assert(verify.response_sent);

	VERIFY_RESET(verify, eap_wsc_done);
	verify.expected_creds = creds_1;
	verify.n_creds = 1;
	verify.cur_cred = 0;
	__eapol_rx_packet(1, ap_address, ETH_P_PAE, eap_wsc_m8,
						sizeof(eap_wsc_m8), false);
	assert(verify.response_sent);
	assert(verify.cur_cred == 1);

	__eapol_rx_packet(1, ap_address, ETH_P_PAE,
					eap_fail, sizeof(eap_fail), false);
	assert(verify.eapol_failed);

	handshake_state_free(hs);
	eapol_exit();
	eap_exit();
}

static void wsc_test_retransmission_no_fragmentation(const void *data)
{
	static uint8_t ap_address[] = { 0x24, 0xa2, 0xe1, 0xec, 0x17, 0x04 };
	static uint8_t sta_address[] = { 0xa0, 0xa8, 0xcd, 0x1c, 0x7e, 0xc9 };
	struct verify_data verify;
	struct handshake_state *hs;
	struct eapol_sm *sm;
	char *hex;
	struct l_settings *settings;

	eap_init(1400);
	eapol_init();

	hs = test_handshake_state_new(1);
	sm = eapol_sm_new(hs);
	eapol_register(sm);

	handshake_state_set_authenticator_address(hs, ap_address);
	handshake_state_set_supplicant_address(hs, sta_address);
	handshake_state_set_event_func(hs, verify_handshake_event, &verify);

	__eapol_set_tx_packet_func(verify_8021x);
	__eapol_set_tx_user_data(&verify);
	eapol_sm_set_user_data(sm, &verify);
	eapol_sm_set_event_func(sm, verify_credential);

	settings = l_settings_new();
	l_settings_set_string(settings, "Security", "EAP-Identity",
					"WFA-SimpleConfig-Enrollee-1-0");
	l_settings_set_string(settings, "Security", "EAP-Method", "WSC");

	l_settings_set_uint(settings, "WSC", "RFBand", WSC_RF_BAND_2_4_GHZ);
	l_settings_set_uint(settings, "WSC", "ConfigurationMethods",
				WSC_CONFIGURATION_METHOD_VIRTUAL_DISPLAY_PIN |
				WSC_CONFIGURATION_METHOD_KEYPAD |
				WSC_CONFIGURATION_METHOD_DISPLAY |
				WSC_CONFIGURATION_METHOD_NFC_INTERFACE);
	l_settings_set_string(settings, "WSC", "PrimaryDeviceType",
					"0-00000000-0");
	l_settings_set_string(settings, "WSC", "EnrolleeMAC",
					util_address_to_string(sta_address));

	hex = l_util_hexstring(m1_data_2.expected.enrollee_nonce, 16);
	l_settings_set_string(settings, "WSC", "EnrolleeNonce", hex);
	l_free(hex);

	hex = l_util_hexstring(m1_data_2.private_key,
						m1_data_2.private_key_size);
	l_settings_set_string(settings, "WSC", "PrivateKey", hex);
	l_free(hex);

	l_settings_set_string(settings, "WSC", "E-SNonce1",
					"fdbb480ee6f572f3591cc3b364f2185b");
	l_settings_set_string(settings, "WSC", "E-SNonce2",
					"c12698739faf385920ba659d524c71c9");
	l_settings_set_string(settings, "WSC", "IV1",
					"9a31f84b4672f2ccf63c845eed3464ec");
	l_settings_set_string(settings, "WSC", "IV2",
					"4e3a4cf088176989e148d4c10b96e8fd");

	handshake_state_set_8021x_config(hs, settings);
	eapol_start(sm);

	l_settings_free(settings);

	VERIFY_RESET(verify, eap_identity_resp);
	__eapol_rx_packet(1, ap_address, ETH_P_PAE, eap_identity_req,
					sizeof(eap_identity_req), false);
	assert(verify.response_sent);
	__eapol_rx_packet(1, ap_address, ETH_P_PAE, eap_identity_req,
					sizeof(eap_identity_req), false);
	assert(verify.response_sent);

	VERIFY_RESET(verify, eap_wsc_m1_2);
	__eapol_rx_packet(1, ap_address, ETH_P_PAE, eap_wsc_start,
						sizeof(eap_wsc_start), false);
	assert(verify.response_sent);
	__eapol_rx_packet(1, ap_address, ETH_P_PAE, eap_wsc_start,
						sizeof(eap_wsc_start), false);
	assert(verify.response_sent);

	VERIFY_RESET(verify, eap_wsc_m3);
	__eapol_rx_packet(1, ap_address, ETH_P_PAE, eap_wsc_m2_2,
						sizeof(eap_wsc_m2_2), false);
	assert(verify.response_sent);
	__eapol_rx_packet(1, ap_address, ETH_P_PAE, eap_wsc_m2_2,
						sizeof(eap_wsc_m2_2), false);
	assert(verify.response_sent);

	VERIFY_RESET(verify, eap_wsc_m3);
	__eapol_rx_packet(1, ap_address, ETH_P_PAE, eap_wsc_m2_2,
						sizeof(eap_wsc_m2_2), false);
	assert(verify.response_sent);

	VERIFY_RESET(verify, eap_wsc_m5);
	__eapol_rx_packet(1, ap_address, ETH_P_PAE, eap_wsc_m4,
						sizeof(eap_wsc_m4), false);
	assert(verify.response_sent);
	__eapol_rx_packet(1, ap_address, ETH_P_PAE, eap_wsc_m4,
						sizeof(eap_wsc_m4), false);
	assert(verify.response_sent);

	VERIFY_RESET(verify, eap_wsc_m7);
	__eapol_rx_packet(1, ap_address, ETH_P_PAE, eap_wsc_m6,
						sizeof(eap_wsc_m6), false);
	assert(verify.response_sent);
	__eapol_rx_packet(1, ap_address, ETH_P_PAE, eap_wsc_m6,
						sizeof(eap_wsc_m6), false);
	assert(verify.response_sent);

	VERIFY_RESET(verify, eap_wsc_done);
	verify.expected_creds = creds_1;
	verify.n_creds = 1;
	verify.cur_cred = 0;
	__eapol_rx_packet(1, ap_address, ETH_P_PAE, eap_wsc_m8,
						sizeof(eap_wsc_m8), false);
	assert(verify.response_sent);
	assert(verify.cur_cred == 1);

	__eapol_rx_packet(1, ap_address, ETH_P_PAE, eap_fail,
						sizeof(eap_fail), false);
	assert(verify.eapol_failed);
	__eapol_rx_packet(1, ap_address, ETH_P_PAE, eap_fail,
						sizeof(eap_fail), false);
	assert(verify.eapol_failed);

	handshake_state_free(hs);
	eapol_exit();
	eap_exit();
}

static void wsc_test_pin_valid(const void *data)
{
	assert(wsc_pin_is_valid("1234"));
	assert(wsc_pin_is_valid("12345678"));
	assert(!wsc_pin_is_valid(""));
	assert(!wsc_pin_is_valid("123456789"));
	assert(!wsc_pin_is_valid("abcdefgh"));
}

static void wsc_test_pin_checksum(const void *data)
{
	assert(!wsc_pin_is_checksum_valid("12345671"));
	assert(wsc_pin_is_checksum_valid("12345670"));
	assert(wsc_pin_is_checksum_valid("24681353"));
}

static void wsc_test_pin_generate(const void *data)
{
	int i;
	char pin[9];

	for (i = 0; i < 1000; i++) {
		assert(wsc_pin_generate(pin));
		assert(wsc_pin_is_valid(pin));
		assert(wsc_pin_is_checksum_valid(pin));
	}
}

int main(int argc, char *argv[])
{
	l_test_init(&argc, &argv);

	l_test_add("/wsc/iter/sanity-check", wsc_test_iter_sanity_check, NULL);

	l_test_add("/wsc/parse/beacon 1", wsc_test_parse_beacon,
					&beacon_data_1);

	l_test_add("/wsc/parse/probe response 1", wsc_test_parse_probe_response,
					&probe_response_data_1);

	l_test_add("/wsc/parse/probe request 1", wsc_test_parse_probe_request,
					&probe_request_data_1);

	l_test_add("/wsc/build/probe request 1", wsc_test_build_probe_request,
					&probe_request_data_1);

	l_test_add("/wsc/pin/valid pin", wsc_test_pin_valid, NULL);
	l_test_add("/wsc/pin/valid checksum", wsc_test_pin_checksum, NULL);

	if (l_getrandom_is_supported())
		l_test_add("/wsc/pin/generate", wsc_test_pin_generate, NULL);

	l_test_add("/wsc/gen_uuid/1", wsc_test_uuid_from_addr,
					&uuid_from_addr_data_1);

	l_test_add("/wsc/parse/m1 1", wsc_test_parse_m1, &m1_data_1);
	l_test_add("/wsc/parse/m1 2", wsc_test_parse_m1, &m1_data_2);

	l_test_add("/wsc/build/m1 1", wsc_test_build_m1, &m1_data_1);
	l_test_add("/wsc/build/m1 2", wsc_test_build_m1, &m1_data_2);

	if (!l_checksum_is_supported(L_CHECKSUM_SHA256, true)) {
		printf("SHA256 support missing, skipping other tests...\n");
		goto done;
	}

	if (!l_key_is_supported(L_KEY_FEATURE_CRYPTO)) {
		printf("Key crypto not supported, skipping other tests...\n");
		goto done;
	}

	if (l_key_is_supported(L_KEY_FEATURE_DH)) {
		l_test_add("/wsc/parse/m2 1", wsc_test_parse_m2, &m2_data_1);
		l_test_add("/wsc/parse/m2 2", wsc_test_parse_m2, &m2_data_2);
	}

	l_test_add("/wsc/build/m2 1", wsc_test_build_m2, &m2_data_1);

	l_test_add("/wsc/parse/m3 1", wsc_test_parse_m3, &m3_data_1);
	l_test_add("/wsc/build/m3 1", wsc_test_build_m3, &m3_data_1);

	l_test_add("/wsc/parse/m4 1", wsc_test_parse_m4, &m4_data_1);
	l_test_add("/wsc/build/m4 1", wsc_test_build_m4, &m4_data_1);

	l_test_add("/wsc/parse/m4 encrypted settings 1",
			wsc_test_parse_m4_encrypted_settings,
			&m4_encrypted_settings_data_1);
	l_test_add("/wsc/build/m4 encrypted settings 1",
			wsc_test_build_m4_encrypted_settings,
			&m4_encrypted_settings_data_1);

	l_test_add("/wsc/parse/m5 1", wsc_test_parse_m5, &m5_data_1);
	l_test_add("/wsc/build/m5 1", wsc_test_build_m5, &m5_data_1);

	l_test_add("/wsc/parse/m6 1", wsc_test_parse_m6, &m6_data_1);
	l_test_add("/wsc/build/m6 1", wsc_test_build_m6, &m6_data_1);

	l_test_add("/wsc/parse/m6 encrypted settings 1",
			wsc_test_parse_m6_encrypted_settings,
			&m6_encrypted_settings_data_1);
	l_test_add("/wsc/build/m6 encrypted settings 1",
			wsc_test_build_m6_encrypted_settings,
			&m6_encrypted_settings_data_1);

	l_test_add("/wsc/parse/m7 1", wsc_test_parse_m7, &m7_data_1);
	l_test_add("/wsc/build/m7 1", wsc_test_build_m7, &m7_data_1);

	l_test_add("/wsc/parse/m8 1", wsc_test_parse_m8, &m8_data_1);
	l_test_add("/wsc/build/m8 1", wsc_test_build_m8, &m8_data_1);

	l_test_add("/wsc/parse/m8 encrypted settings 1",
			wsc_test_parse_m8_encrypted_settings,
			&m8_encrypted_settings_data_1);

	l_test_add("/wsc/parse/wsc_done 1", wsc_test_parse_wsc_done,
							&wsc_done_data_1);
	l_test_add("/wsc/build/wsc_done 1", wsc_test_build_wsc_done,
							&wsc_done_data_1);

	if (!l_key_is_supported(L_KEY_FEATURE_DH))
		goto done;

	l_test_add("/wsc/diffie-hellman/generate pubkey 1",
					wsc_test_dh_generate_pubkey,
					&dh_generate_pubkey_test_data_1);
	l_test_add("/wsc/diffie-hellman/generate pubkey 2",
					wsc_test_dh_generate_pubkey,
					&dh_generate_pubkey_test_data_2);

	if (!l_cipher_is_supported(L_CIPHER_AES_CBC))
		goto done;

	l_test_add("/wsc/handshake/PBC Handshake Test",
						wsc_test_pbc_handshake, NULL);

	l_test_add("/wsc/retransmission/no fragmentation",
				wsc_test_retransmission_no_fragmentation, NULL);

done:
	return l_test_run();
}
