/*
 *
 *  oFono - Open Source Telephony
 *
 *  Copyright (C) 2008-2009  Intel Corporation. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifndef __OFONO_SIM_H
#define __OFONO_SIM_H

#ifdef __cplusplus
extern "C" {
#endif

#include <ofono/types.h>

struct ofono_sim;

/* 51.011 Section 9.3 */
enum ofono_sim_file_structure {
	OFONO_SIM_FILE_STRUCTURE_TRANSPARENT = 0,
	OFONO_SIM_FILE_STRUCTURE_FIXED = 1,
	OFONO_SIM_FILE_STRUCTURE_CYCLIC = 3
};

enum ofono_sim_password_type {
	OFONO_SIM_PASSWORD_NONE = 0,
	OFONO_SIM_PASSWORD_SIM_PIN,
	OFONO_SIM_PASSWORD_PHSIM_PIN,
	OFONO_SIM_PASSWORD_PHFSIM_PIN,
	OFONO_SIM_PASSWORD_SIM_PIN2,
	OFONO_SIM_PASSWORD_PHNET_PIN,
	OFONO_SIM_PASSWORD_PHNETSUB_PIN,
	OFONO_SIM_PASSWORD_PHSP_PIN,
	OFONO_SIM_PASSWORD_PHCORP_PIN,
	OFONO_SIM_PASSWORD_SIM_PUK,
	OFONO_SIM_PASSWORD_PHFSIM_PUK,
	OFONO_SIM_PASSWORD_SIM_PUK2,
	OFONO_SIM_PASSWORD_PHNET_PUK,
	OFONO_SIM_PASSWORD_PHNETSUB_PUK,
	OFONO_SIM_PASSWORD_PHSP_PUK,
	OFONO_SIM_PASSWORD_PHCORP_PUK,
	OFONO_SIM_PASSWORD_INVALID,
};

typedef void (*ofono_sim_file_info_cb_t)(const struct ofono_error *error,
					int filelength,
					enum ofono_sim_file_structure structure,
					int recordlength,
					const unsigned char access[3],
					void *data);

typedef void (*ofono_sim_read_cb_t)(const struct ofono_error *error,
					const unsigned char *sdata, int length,
					void *data);

typedef void (*ofono_sim_write_cb_t)(const struct ofono_error *error,
					void *data);

typedef void (*ofono_sim_imsi_cb_t)(const struct ofono_error *error,
					const char *imsi, void *data);

typedef void (*ofono_sim_ready_notify_cb_t)(void *data);

typedef void (*ofono_sim_file_read_cb_t)(int ok, int total_length, int record,
					const unsigned char *data,
					int record_length, void *userdata);

typedef void (*ofono_sim_file_write_cb_t)(int ok, void *userdata);

typedef void (*ofono_sim_passwd_cb_t)(const struct ofono_error *error,
					enum ofono_sim_password_type type,
					void *data);

typedef void (*ofono_sim_lock_unlock_cb_t)(const struct ofono_error *error,
					void *data);

typedef void (*ofono_sim_locked_cb_t)(const struct ofono_error *error,
					int locked, void *data);

struct ofono_sim_driver {
	const char *name;
	int (*probe)(struct ofono_sim *sim, unsigned int vendor, void *data);
	void (*remove)(struct ofono_sim *sim);
	void (*read_file_info)(struct ofono_sim *sim, int fileid,
			ofono_sim_file_info_cb_t cb, void *data);
	void (*read_file_transparent)(struct ofono_sim *sim, int fileid,
			int start, int length,
			ofono_sim_read_cb_t cb, void *data);
	void (*read_file_linear)(struct ofono_sim *sim, int fileid,
			int record, int length,
			ofono_sim_read_cb_t cb, void *data);
	void (*read_file_cyclic)(struct ofono_sim *sim, int fileid,
			int record, int length,
			ofono_sim_read_cb_t cb, void *data);
	void (*write_file_transparent)(struct ofono_sim *sim, int fileid,
			int start, int length, const unsigned char *value,
			ofono_sim_write_cb_t cb, void *data);
	void (*write_file_linear)(struct ofono_sim *sim, int fileid,
			int record, int length, const unsigned char *value,
			ofono_sim_write_cb_t cb, void *data);
	void (*write_file_cyclic)(struct ofono_sim *sim, int fileid,
			int length, const unsigned char *value,
			ofono_sim_write_cb_t cb, void *data);
	void (*read_imsi)(struct ofono_sim *sim,
			ofono_sim_imsi_cb_t cb, void *data);
	void (*query_passwd_state)(struct ofono_sim *sim,
			ofono_sim_passwd_cb_t cb, void *data);
	void (*send_passwd)(struct ofono_sim *sim, const char *passwd,
			ofono_sim_lock_unlock_cb_t cb, void *data);
	void (*reset_passwd)(struct ofono_sim *sim, const char *puk,
			const char *passwd,
			ofono_sim_lock_unlock_cb_t cb, void *data);
	void (*change_passwd)(struct ofono_sim *sim,
			enum ofono_sim_password_type type,
			const char *old, const char *new,
			ofono_sim_lock_unlock_cb_t cb, void *data);
	void (*lock)(struct ofono_sim *sim, enum ofono_sim_password_type type,
			int enable, const char *passwd,
			ofono_sim_lock_unlock_cb_t cb, void *data);
	void (*query_locked)(struct ofono_sim *sim,
			enum ofono_sim_password_type type,
			ofono_sim_locked_cb_t cb, void *data);
};

int ofono_sim_driver_register(const struct ofono_sim_driver *d);
void ofono_sim_driver_unregister(const struct ofono_sim_driver *d);

struct ofono_sim *ofono_sim_create(struct ofono_modem *modem,
					unsigned int vendor,
					const char *driver, void *data);

void ofono_sim_register(struct ofono_sim *sim);
void ofono_sim_remove(struct ofono_sim *sim);

void ofono_sim_set_data(struct ofono_sim *sim, void *data);
void *ofono_sim_get_data(struct ofono_sim *sim);

const char *ofono_sim_get_imsi(struct ofono_sim *sim);

unsigned int ofono_sim_add_ready_watch(struct ofono_sim *sim,
				ofono_sim_ready_notify_cb_t cb,
				void *data, ofono_destroy_func destroy);

void ofono_sim_remove_ready_watch(struct ofono_sim *sim, unsigned int id);

int ofono_sim_get_ready(struct ofono_sim *sim);
void ofono_sim_set_ready(struct ofono_sim *sim);

/* This will queue an operation to read all available records with id from the
 * SIM.  Callback cb will be called every time a record has been read, or once
 * if an error has occurred.  For transparent files, the callback will only
 * be called once.
 *
 * Returns 0 if the request could be queued, -1 otherwise.
 */
int ofono_sim_read(struct ofono_sim *sim, int id,
			enum ofono_sim_file_structure expected,
			ofono_sim_file_read_cb_t cb, void *data);

int ofono_sim_write(struct ofono_sim *sim, int id,
			ofono_sim_file_write_cb_t cb,
			enum ofono_sim_file_structure structure, int record,
			const unsigned char *data, int length, void *userdata);
#ifdef __cplusplus
}
#endif

#endif /* __OFONO_SIM_H */
