/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.server.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import org.apache.ratis.metrics.Timekeeper;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.protocol.RaftGroupMemberId;
import org.apache.ratis.protocol.RaftPeer;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.server.RaftConfiguration;
import org.apache.ratis.server.RaftServerConfigKeys;
import org.apache.ratis.server.impl.RaftConfigurationImpl;
import org.apache.ratis.server.impl.RaftServerImpl;
import org.apache.ratis.server.impl.ServerProtoUtils;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.server.util.ServerStringUtils;
import org.apache.ratis.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.ratis.util.Daemon;
import org.apache.ratis.util.LifeCycle;
import org.apache.ratis.util.LogUtils;
import org.apache.ratis.util.Preconditions;
import org.apache.ratis.util.TimeDuration;
import org.apache.ratis.util.Timestamp;
import org.apache.ratis.util.UncheckedAutoCloseable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class LeaderElection
implements Runnable {
    public static final Logger LOG = LoggerFactory.getLogger(LeaderElection.class);
    private static final AtomicInteger COUNT = new AtomicInteger();
    private final String name;
    private final LifeCycle lifeCycle;
    private final Daemon daemon;
    private final CompletableFuture<Void> stopped = new CompletableFuture();
    private final ServerInterface server;
    private final boolean skipPreVote;
    private final ConfAndTerm round0;

    private ResultAndTerm logAndReturn(Phase phase, Result result, Map<RaftPeerId, RaftProtos.RequestVoteReplyProto> responses, List<Exception> exceptions) {
        return this.logAndReturn(phase, result, responses, exceptions, null);
    }

    private ResultAndTerm logAndReturn(Phase phase, Result result, Map<RaftPeerId, RaftProtos.RequestVoteReplyProto> responses, List<Exception> exceptions, Long newTerm) {
        ResultAndTerm resultAndTerm = new ResultAndTerm(result, newTerm);
        LOG.info("{}: {} {} received {} response(s) and {} exception(s):", new Object[]{this, phase, resultAndTerm, responses.size(), exceptions.size()});
        int i = 0;
        for (RaftProtos.RequestVoteReplyProto reply : responses.values()) {
            LOG.info("  Response {}: {}", (Object)i++, (Object)ServerStringUtils.toRequestVoteReplyString(reply));
        }
        for (Exception e : exceptions) {
            int j = i++;
            LogUtils.infoOrTrace(LOG, () -> "  Exception " + j, (Throwable)e);
        }
        return resultAndTerm;
    }

    LeaderElection(RaftServerImpl server, boolean force) {
        this(ServerInterface.get(server), force);
    }

    LeaderElection(ServerInterface server, boolean force) {
        this.name = ServerStringUtils.generateUnifiedName(server.getMemberId(), this.getClass()) + COUNT.incrementAndGet();
        this.lifeCycle = new LifeCycle(this);
        this.daemon = Daemon.newBuilder().setName(this.name).setRunnable(this).setThreadGroup(server.getThreadGroup()).build();
        this.server = server;
        this.skipPreVote = force || !server.isPreVoteEnabled();
        try {
            this.round0 = force ? server.initElection(Phase.ELECTION) : null;
        }
        catch (IOException e) {
            throw new IllegalStateException(this.name + ": Failed to initialize election", e);
        }
    }

    void start() {
        this.startIfNew(this.daemon::start);
    }

    @VisibleForTesting
    void startInForeground() {
        this.startIfNew(this);
    }

    private void startIfNew(Runnable starter) {
        if (this.lifeCycle.compareAndTransition(LifeCycle.State.NEW, LifeCycle.State.STARTING)) {
            starter.run();
        } else {
            LifeCycle.State state = this.lifeCycle.getCurrentState();
            LOG.info("{}: skip starting since this is already {}", (Object)this, (Object)state);
        }
    }

    CompletableFuture<Void> shutdown() {
        this.lifeCycle.checkStateAndClose();
        return this.stopped;
    }

    @VisibleForTesting
    LifeCycle.State getCurrentState() {
        return this.lifeCycle.getCurrentState();
    }

    @Override
    public void run() {
        try {
            this.runImpl();
        }
        finally {
            this.stopped.complete(null);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void runImpl() {
        if (!this.lifeCycle.compareAndTransition(LifeCycle.State.STARTING, LifeCycle.State.RUNNING)) {
            LifeCycle.State state = this.lifeCycle.getCurrentState();
            LOG.info("{}: skip running since this is already {}", (Object)this, (Object)state);
            return;
        }
        try (UncheckedAutoCloseable ignored = Timekeeper.start(this.server.getLeaderElectionTimer());){
            int round = 0;
            while (this.shouldRun()) {
                if ((this.skipPreVote || this.askForVotes(Phase.PRE_VOTE, round)) && this.askForVotes(Phase.ELECTION, round)) {
                    this.server.changeToLeader();
                }
                ++round;
            }
        }
        catch (Exception e) {
            LifeCycle.State state;
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            if ((state = this.lifeCycle.getCurrentState()).isClosingOrClosed()) {
                LOG.info("{}: since this is already {}, safely ignore {}", new Object[]{this, state, e.toString()});
            } else {
                if (!this.server.isAlive()) {
                    LOG.info("{}: since the server is not alive, safely ignore {}", (Object)this, (Object)e.toString());
                } else {
                    LOG.error("{}: Failed, state={}", new Object[]{this, state, e});
                }
                this.shutdown();
            }
        }
        finally {
            this.server.onNewLeaderElectionCompletion();
            this.lifeCycle.checkStateAndClose(() -> {});
        }
    }

    private boolean shouldRun() {
        return this.lifeCycle.getCurrentState().isRunning() && this.server.isCandidate() && this.server.isAlive();
    }

    private boolean shouldRun(long electionTerm) {
        return this.shouldRun() && this.server.getCurrentTerm() == electionTerm;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private ResultAndTerm submitRequestAndWaitResult(Phase phase, RaftConfigurationImpl conf, long electionTerm) throws InterruptedException {
        ResultAndTerm r;
        if (!conf.containsInConf(this.server.getId(), new RaftProtos.RaftPeerRole[0])) {
            return new ResultAndTerm(Result.NOT_IN_CONF, electionTerm);
        }
        Collection<RaftPeer> others = conf.getOtherPeers(this.server.getId());
        if (others.isEmpty()) {
            r = new ResultAndTerm(Result.PASSED, electionTerm);
        } else {
            TermIndex lastEntry = this.server.getLastEntry();
            Executor voteExecutor = new Executor(this, others.size());
            try {
                int submitted = this.submitRequests(phase, electionTerm, lastEntry, others, voteExecutor);
                r = this.waitForResults(phase, electionTerm, submitted, conf, voteExecutor);
            }
            finally {
                voteExecutor.shutdown();
            }
        }
        return r;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean askForVotes(Phase phase, int round) throws InterruptedException, IOException {
        RaftConfigurationImpl conf;
        long electionTerm;
        ServerInterface serverInterface = this.server;
        synchronized (serverInterface) {
            if (!this.shouldRun()) {
                return false;
            }
            ConfAndTerm confAndTerm = round == 0 && this.round0 != null ? this.round0 : this.server.initElection(phase);
            electionTerm = confAndTerm.getTerm();
            conf = confAndTerm.getConf();
        }
        LOG.info("{} {} round {}: submit vote requests at term {} for {}", new Object[]{this, phase, round, electionTerm, conf});
        ResultAndTerm r = this.submitRequestAndWaitResult(phase, conf, electionTerm);
        LOG.info("{} {} round {}: result {}", new Object[]{this, phase, round, r});
        ServerInterface serverInterface2 = this.server;
        synchronized (serverInterface2) {
            if (!this.shouldRun(electionTerm)) {
                return false;
            }
            switch (r.getResult()) {
                case PASSED: 
                case SINGLE_MODE_PASSED: {
                    return true;
                }
                case NOT_IN_CONF: 
                case SHUTDOWN: {
                    this.server.shutdown();
                    return false;
                }
                case TIMEOUT: {
                    return false;
                }
                case REJECTED: 
                case DISCOVERED_A_NEW_TERM: {
                    long term = r.maxTerm(this.server.getCurrentTerm());
                    this.server.rejected(term, r);
                    return false;
                }
            }
            throw new IllegalArgumentException("Unable to process result " + (Object)((Object)r.result));
        }
    }

    private int submitRequests(Phase phase, long electionTerm, TermIndex lastEntry, Collection<RaftPeer> others, Executor voteExecutor) {
        int submitted = 0;
        for (RaftPeer peer : others) {
            RaftProtos.RequestVoteRequestProto r = ServerProtoUtils.toRequestVoteRequestProto(this.server.getMemberId(), peer.getId(), electionTerm, lastEntry, phase == Phase.PRE_VOTE);
            voteExecutor.submit(() -> this.server.requestVote(r));
            ++submitted;
        }
        return submitted;
    }

    private Set<RaftPeerId> getHigherPriorityPeers(RaftConfiguration conf) {
        Optional<Integer> priority = Optional.ofNullable(conf.getPeer(this.server.getId(), new RaftProtos.RaftPeerRole[0])).map(RaftPeer::getPriority);
        return conf.getAllPeers().stream().filter(peer -> priority.filter(p -> peer.getPriority() > p).isPresent()).map(RaftPeer::getId).collect(Collectors.toSet());
    }

    private ResultAndTerm waitForResults(Phase phase, long electionTerm, int submitted, RaftConfigurationImpl conf, Executor voteExecutor) throws InterruptedException {
        boolean emptyCommit;
        Timestamp timeout = Timestamp.currentTime().addTime(this.server.getRandomElectionTimeout());
        HashMap<RaftPeerId, RaftProtos.RequestVoteReplyProto> responses = new HashMap<RaftPeerId, RaftProtos.RequestVoteReplyProto>();
        ArrayList<Exception> exceptions = new ArrayList<Exception>();
        int waitForNum = submitted;
        ArrayList<RaftPeerId> votedPeers = new ArrayList<RaftPeerId>();
        ArrayList<RaftPeerId> rejectedPeers = new ArrayList<RaftPeerId>();
        Set<RaftPeerId> higherPriorityPeers = this.getHigherPriorityPeers(conf);
        boolean singleMode = conf.isSingleMode(this.server.getId());
        boolean bl = emptyCommit = this.server.getLastCommittedIndex() < 0L;
        while (waitForNum > 0 && this.shouldRun(electionTerm)) {
            TimeDuration waitTime = timeout.elapsedTime().apply(n -> -n);
            if (waitTime.isNonPositive()) {
                if (conf.hasMajority(votedPeers, this.server.getId())) {
                    return this.logAndReturn(phase, Result.PASSED, responses, exceptions);
                }
                if (singleMode) {
                    return this.logAndReturn(phase, Result.SINGLE_MODE_PASSED, responses, exceptions);
                }
                return this.logAndReturn(phase, Result.TIMEOUT, responses, exceptions);
            }
            try {
                boolean acceptVote;
                Future<RaftProtos.RequestVoteReplyProto> future = voteExecutor.poll(waitTime);
                if (future == null) continue;
                RaftProtos.RequestVoteReplyProto r = future.get();
                RaftPeerId replierId = RaftPeerId.valueOf(r.getServerReply().getReplyId());
                RaftProtos.RequestVoteReplyProto previous = responses.putIfAbsent(replierId, r);
                if (previous != null) {
                    if (!LOG.isWarnEnabled()) continue;
                    LOG.warn("{} received duplicated replies from {}, the 2nd reply is ignored: 1st={}, 2nd={}", this, replierId, ServerStringUtils.toRequestVoteReplyString(previous), ServerStringUtils.toRequestVoteReplyString(r));
                    continue;
                }
                if (r.getShouldShutdown()) {
                    return this.logAndReturn(phase, Result.SHUTDOWN, responses, exceptions);
                }
                if (r.getTerm() > electionTerm) {
                    return this.logAndReturn(phase, Result.DISCOVERED_A_NEW_TERM, responses, exceptions, r.getTerm());
                }
                if (!r.getServerReply().getSuccess() && higherPriorityPeers.contains(replierId) && !singleMode) {
                    return this.logAndReturn(phase, Result.REJECTED, responses, exceptions);
                }
                higherPriorityPeers.remove(replierId);
                boolean bl2 = acceptVote = r.getServerReply().getSuccess() && (emptyCommit || LeaderElection.nonEmptyLog(r));
                if (acceptVote) {
                    votedPeers.add(replierId);
                    if (higherPriorityPeers.isEmpty() && conf.hasMajority(votedPeers, this.server.getId())) {
                        return this.logAndReturn(phase, Result.PASSED, responses, exceptions);
                    }
                } else {
                    rejectedPeers.add(replierId);
                    if (conf.majorityRejectVotes(rejectedPeers)) {
                        LOG.info("rejectedPeers: {}, emptyCommit? {}", (Object)rejectedPeers, (Object)emptyCommit);
                        return this.logAndReturn(phase, Result.REJECTED, responses, exceptions);
                    }
                }
            }
            catch (ExecutionException e) {
                LogUtils.infoOrTrace(LOG, () -> this + " got exception when requesting votes", (Throwable)e);
                exceptions.add(e);
            }
            --waitForNum;
        }
        if (conf.hasMajority(votedPeers, this.server.getId())) {
            return this.logAndReturn(phase, Result.PASSED, responses, exceptions);
        }
        if (singleMode) {
            return this.logAndReturn(phase, Result.SINGLE_MODE_PASSED, responses, exceptions);
        }
        return this.logAndReturn(phase, Result.REJECTED, responses, exceptions);
    }

    static boolean nonEmptyLog(RaftProtos.RequestVoteReplyProto reply) {
        RaftProtos.TermIndexProto lastEntry = reply.getLastEntry();
        if (lastEntry.equals(RaftProtos.TermIndexProto.getDefaultInstance())) {
            LOG.info("Reply missing lastEntry: {} ", (Object)ServerStringUtils.toRequestVoteReplyString(reply));
            return true;
        }
        if (lastEntry.getTerm() > 0L) {
            return true;
        }
        LOG.info("Replier log is empty: {} ", (Object)ServerStringUtils.toRequestVoteReplyString(reply));
        return false;
    }

    public String toString() {
        return this.name;
    }

    static class ConfAndTerm {
        private final RaftConfigurationImpl conf;
        private final long term;

        ConfAndTerm(RaftConfigurationImpl conf, long term) {
            this.conf = conf;
            this.term = term;
        }

        long getTerm() {
            return this.term;
        }

        RaftConfigurationImpl getConf() {
            return this.conf;
        }

        public String toString() {
            return "term=" + this.term + ", " + this.conf;
        }
    }

    static class Executor {
        private final ExecutorCompletionService<RaftProtos.RequestVoteReplyProto> service;
        private final ExecutorService executor;
        private final AtomicInteger count = new AtomicInteger();

        Executor(Object name, int size) {
            Preconditions.assertTrue(size > 0);
            this.executor = Executors.newFixedThreadPool(size, r -> Daemon.newBuilder().setName(name + "-" + this.count.incrementAndGet()).setRunnable(r).build());
            this.service = new ExecutorCompletionService(this.executor);
        }

        void shutdown() {
            this.executor.shutdownNow();
        }

        void submit(Callable<RaftProtos.RequestVoteReplyProto> task) {
            this.service.submit(task);
        }

        Future<RaftProtos.RequestVoteReplyProto> poll(TimeDuration waitTime) throws InterruptedException {
            return this.service.poll(waitTime.getDuration(), waitTime.getUnit());
        }
    }

    static class ResultAndTerm {
        private final Result result;
        private final Long term;

        ResultAndTerm(Result result, Long term) {
            this.result = result;
            this.term = term;
        }

        long maxTerm(long thatTerm) {
            return this.term != null && this.term > thatTerm ? this.term : thatTerm;
        }

        Result getResult() {
            return this.result;
        }

        public String toString() {
            return (Object)((Object)this.result) + (this.term != null ? " (term=" + this.term + ")" : "");
        }
    }

    static enum Result {
        PASSED,
        SINGLE_MODE_PASSED,
        REJECTED,
        TIMEOUT,
        DISCOVERED_A_NEW_TERM,
        SHUTDOWN,
        NOT_IN_CONF;

    }

    static enum Phase {
        PRE_VOTE,
        ELECTION;

    }

    static interface ServerInterface {
        default public RaftPeerId getId() {
            return this.getMemberId().getPeerId();
        }

        public RaftGroupMemberId getMemberId();

        public boolean isAlive();

        public boolean isCandidate();

        public long getCurrentTerm();

        public long getLastCommittedIndex();

        public TermIndex getLastEntry();

        public boolean isPreVoteEnabled();

        public ConfAndTerm initElection(Phase var1) throws IOException;

        public RaftProtos.RequestVoteReplyProto requestVote(RaftProtos.RequestVoteRequestProto var1) throws IOException;

        public void changeToLeader();

        public void rejected(long var1, ResultAndTerm var3) throws IOException;

        public void shutdown();

        public Timekeeper getLeaderElectionTimer();

        public void onNewLeaderElectionCompletion();

        public TimeDuration getRandomElectionTimeout();

        public ThreadGroup getThreadGroup();

        public static ServerInterface get(final RaftServerImpl server) {
            final boolean preVote = RaftServerConfigKeys.LeaderElection.preVote(server.getRaftServer().getProperties());
            return new ServerInterface(){

                @Override
                public RaftGroupMemberId getMemberId() {
                    return server.getMemberId();
                }

                @Override
                public boolean isAlive() {
                    return server.getInfo().isAlive();
                }

                @Override
                public boolean isCandidate() {
                    return server.getInfo().isCandidate();
                }

                @Override
                public long getCurrentTerm() {
                    return server.getState().getCurrentTerm();
                }

                @Override
                public long getLastCommittedIndex() {
                    return server.getRaftLog().getLastCommittedIndex();
                }

                @Override
                public TermIndex getLastEntry() {
                    return server.getState().getLastEntry();
                }

                @Override
                public boolean isPreVoteEnabled() {
                    return preVote;
                }

                @Override
                public ConfAndTerm initElection(Phase phase) throws IOException {
                    return server.getState().initElection(phase);
                }

                @Override
                public RaftProtos.RequestVoteReplyProto requestVote(RaftProtos.RequestVoteRequestProto r) throws IOException {
                    return server.getServerRpc().requestVote(r);
                }

                @Override
                public void changeToLeader() {
                    server.changeToLeader();
                }

                @Override
                public void rejected(long term, ResultAndTerm result) throws IOException {
                    server.changeToFollowerAndPersistMetadata(term, false, result);
                }

                @Override
                public void shutdown() {
                    server.close();
                    server.getStateMachine().event().notifyServerShutdown(server.getRoleInfoProto(), false);
                }

                @Override
                public Timekeeper getLeaderElectionTimer() {
                    return server.getLeaderElectionMetrics().getLeaderElectionTimer();
                }

                @Override
                public void onNewLeaderElectionCompletion() {
                    server.getLeaderElectionMetrics().onNewLeaderElectionCompletion();
                }

                @Override
                public TimeDuration getRandomElectionTimeout() {
                    return server.getRandomElectionTimeout();
                }

                @Override
                public ThreadGroup getThreadGroup() {
                    return server.getThreadGroup();
                }
            };
        }
    }
}

