/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.sidecar.restore;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.SidecarRateLimiter;
import io.vertx.core.Future;
import java.io.File;
import java.io.IOException;
import java.nio.channels.Channel;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.EnumSet;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.BiConsumer;
import org.apache.cassandra.sidecar.common.data.StorageCredentials;
import org.apache.cassandra.sidecar.common.server.utils.ThrowableUtils;
import org.apache.cassandra.sidecar.common.utils.HttpRange;
import org.apache.cassandra.sidecar.common.utils.Preconditions;
import org.apache.cassandra.sidecar.concurrent.TaskExecutorPool;
import org.apache.cassandra.sidecar.db.RestoreJob;
import org.apache.cassandra.sidecar.db.RestoreRange;
import org.apache.cassandra.sidecar.exceptions.RestoreJobFatalException;
import org.apache.cassandra.sidecar.restore.HttpRangesIterator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.auth.credentials.AwsCredentials;
import software.amazon.awssdk.auth.credentials.AwsCredentialsProvider;
import software.amazon.awssdk.auth.credentials.AwsSessionCredentials;
import software.amazon.awssdk.auth.credentials.StaticCredentialsProvider;
import software.amazon.awssdk.core.ResponseBytes;
import software.amazon.awssdk.core.async.AsyncResponseTransformer;
import software.amazon.awssdk.services.s3.S3AsyncClient;
import software.amazon.awssdk.services.s3.model.GetObjectRequest;
import software.amazon.awssdk.services.s3.model.HeadObjectRequest;
import software.amazon.awssdk.services.s3.model.HeadObjectResponse;

public class StorageClient {
    private static final Logger LOGGER = LoggerFactory.getLogger(StorageClient.class);
    private final S3AsyncClient client;
    private final int rangeHeaderSize;
    private final SidecarRateLimiter downloadRateLimiter;
    private final Map<UUID, Credentials> credentialsProviders = new ConcurrentHashMap<UUID, Credentials>();

    @VisibleForTesting
    StorageClient(S3AsyncClient client) {
        this(client, 0x500000, SidecarRateLimiter.create(-1.0));
    }

    StorageClient(S3AsyncClient client, int rangeHeaderSize, SidecarRateLimiter downloadRateLimiter) {
        this.client = client;
        this.rangeHeaderSize = rangeHeaderSize;
        this.downloadRateLimiter = downloadRateLimiter;
    }

    public StorageClient authenticate(RestoreJob restoreJob) throws RestoreJobFatalException {
        Credentials newCredentials = new Credentials(restoreJob);
        this.credentialsProviders.compute(restoreJob.jobId, (jobId, credentials) -> {
            if (credentials == null || !this.matches((Credentials)credentials, newCredentials)) {
                LOGGER.info("Credentials are updated in the storage client. jobId={} credentials={}", (Object)restoreJob.jobId, (Object)newCredentials.readCredentials);
                newCredentials.init();
                return newCredentials;
            }
            return credentials;
        });
        return this;
    }

    public void revokeCredentials(UUID jobId) {
        LOGGER.info("Revoke credentials for job. jobId={}", (Object)jobId);
        this.credentialsProviders.remove(jobId);
    }

    public CompletableFuture<HeadObjectResponse> objectExists(RestoreRange range) {
        Credentials credentials = this.credentialsProviders.get(range.jobId());
        if (credentials == null) {
            LOGGER.warn("Credentials not found. jobId={}", (Object)range.jobId());
            return StorageClient.failedFuture(this.credentialsNotFound(range));
        }
        HeadObjectRequest request = (HeadObjectRequest)HeadObjectRequest.builder().overrideConfiguration(b -> b.credentialsProvider(credentials.awsCredentialsProvider())).bucket(range.sliceBucket()).key(range.sliceKey()).ifMatch(this.quoteIfNeeded(range.sliceChecksum())).build();
        return this.client.headObject(request).whenComplete(this.logCredentialOnRequestFailure(range, credentials));
    }

    public Future<File> downloadObjectIfAbsent(RestoreRange range, TaskExecutorPool taskExecutorPool) {
        Credentials credentials = this.credentialsProviders.get(range.jobId());
        if (credentials == null) {
            LOGGER.warn("Credentials to download object not found. jobId={}", (Object)range.jobId());
            return Future.failedFuture((Throwable)this.credentialsNotFound(range));
        }
        Path objectPath = range.stagedObjectPath();
        File object = objectPath.toFile();
        if (object.exists()) {
            LOGGER.info("Skipping download, file already exists. jobId={} sliceKey={}", (Object)range.jobId(), (Object)range.sliceKey());
            return Future.succeededFuture((Object)object);
        }
        try {
            Files.createDirectories(objectPath.getParent(), new FileAttribute[0]);
        }
        catch (Exception ex) {
            LOGGER.error("Error occurred while creating directory. jobId={} sliceKey={}", new Object[]{range.jobId(), range.sliceKey(), ex});
            return Future.failedFuture((Throwable)ex);
        }
        LOGGER.info("Downloading object. jobId={} sliceKey={}", (Object)range.jobId(), (Object)range.sliceKey());
        return this.rangeGetObject(range, credentials, objectPath, taskExecutorPool);
    }

    public void close() {
        try {
            this.client.close();
        }
        catch (Exception ex) {
            LOGGER.warn("Error when closing", (Throwable)ex);
        }
    }

    private Future<File> rangeGetObject(RestoreRange range, Credentials credentials, Path destinationPath, TaskExecutorPool taskExecutorPool) {
        SeekableByteChannel seekableByteChannel;
        HttpRangesIterator iterator = new HttpRangesIterator(range.sliceObjectLength(), this.rangeHeaderSize);
        Preconditions.checkState((boolean)iterator.hasNext(), (String)("SliceObject is empty. sliceKey=" + range.sliceKey()));
        try {
            seekableByteChannel = Files.newByteChannel(destinationPath, EnumSet.of(StandardOpenOption.CREATE_NEW, StandardOpenOption.WRITE), new FileAttribute[0]);
        }
        catch (IOException e) {
            LOGGER.error("Failed to create file channel for downloading. jobId={} sliceKey={}", new Object[]{range.jobId(), range.sliceKey(), e});
            return Future.failedFuture((Throwable)e);
        }
        Future channelFuture = Future.succeededFuture((Object)seekableByteChannel);
        while (iterator.hasNext()) {
            HttpRange httpRange = iterator.next();
            channelFuture = channelFuture.compose(channel -> taskExecutorPool.executeBlocking(() -> {
                int actualRangeSize = (int)httpRange.length();
                this.downloadRateLimiter.acquire(actualRangeSize);
                GetObjectRequest request = (GetObjectRequest)GetObjectRequest.builder().overrideConfiguration(b -> b.credentialsProvider(credentials.awsCredentialsProvider())).bucket(range.sliceBucket()).key(range.sliceKey()).range(httpRange.toString()).build();
                ResponseBytes bytes = (ResponseBytes)this.client.getObject(request, AsyncResponseTransformer.toBytes()).get();
                channel.write(bytes.asByteBuffer());
                return channel;
            }, true));
        }
        return channelFuture.eventually(() -> taskExecutorPool.runBlocking(() -> ThrowableUtils.propagate(() -> this.closeChannel(seekableByteChannel)), true)).compose(channel -> Future.succeededFuture((Object)destinationPath.toFile()), failure -> {
            LOGGER.error("Request is not successful. jobId={} credentials={}", new Object[]{range.jobId(), credentials.readCredentials, failure});
            try {
                Files.deleteIfExists(destinationPath);
            }
            catch (IOException e) {
                LOGGER.warn("Failed to clean up the failed download. jobId={} sliceKey={}", new Object[]{range.jobId(), range.sliceKey(), e});
                failure.addSuppressed(e);
            }
            return Future.failedFuture((Throwable)failure);
        });
    }

    private boolean matches(Credentials c1, Credentials c2) {
        if (c1 == c2) {
            return true;
        }
        return Objects.equals(c1.readCredentials, c2.readCredentials);
    }

    private String quoteIfNeeded(String input) {
        if (input.startsWith("\"") && input.endsWith("\"")) {
            return input;
        }
        return "\"" + input + "\"";
    }

    private IllegalStateException credentialsNotFound(RestoreRange range) {
        return new IllegalStateException("No credential available. The job might already have failed.jobId: " + range.jobId());
    }

    private BiConsumer<Object, ? super Throwable> logCredentialOnRequestFailure(RestoreRange range, Credentials credentials) {
        return (ignored, cause) -> {
            if (cause != null) {
                LOGGER.error("Request is not successful. jobId={} credentials={}", new Object[]{range.jobId(), credentials.readCredentials, cause});
            }
        };
    }

    private void closeChannel(Channel channel) throws IOException {
        if (channel != null && channel.isOpen()) {
            channel.close();
        }
    }

    private static <T> CompletableFuture<T> failedFuture(Throwable cause) {
        CompletableFuture failure = new CompletableFuture();
        failure.completeExceptionally(cause);
        return failure;
    }

    private static class Credentials {
        final StorageCredentials readCredentials;
        private AwsCredentialsProvider awsCredential;

        Credentials(RestoreJob restoreJob) throws RestoreJobFatalException {
            if (restoreJob.secrets == null) {
                throw new RestoreJobFatalException("Restore job is missing credentials. JobId: " + restoreJob.jobId);
            }
            this.readCredentials = restoreJob.secrets.readCredentials();
        }

        void init() {
            AwsSessionCredentials credentials = AwsSessionCredentials.create((String)this.readCredentials.accessKeyId(), (String)this.readCredentials.secretAccessKey(), (String)this.readCredentials.sessionToken());
            this.awsCredential = StaticCredentialsProvider.create((AwsCredentials)credentials);
        }

        AwsCredentialsProvider awsCredentialsProvider() {
            return this.awsCredential;
        }
    }
}

