/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.core.spi.crypto;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.crypto.Cipher;
import javax.crypto.CipherInputStream;
import javax.crypto.CipherOutputStream;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.spec.GCMParameterSpec;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import org.apache.accumulo.core.crypto.streams.BlockedInputStream;
import org.apache.accumulo.core.crypto.streams.BlockedOutputStream;
import org.apache.accumulo.core.crypto.streams.DiscardCloseOutputStream;
import org.apache.accumulo.core.crypto.streams.RFileCipherOutputStream;
import org.apache.accumulo.core.spi.crypto.CryptoEnvironment;
import org.apache.accumulo.core.spi.crypto.CryptoService;
import org.apache.accumulo.core.spi.crypto.FileDecrypter;
import org.apache.accumulo.core.spi.crypto.FileEncrypter;
import org.apache.accumulo.core.spi.crypto.NoFileDecrypter;
import org.apache.accumulo.core.spi.crypto.NoFileEncrypter;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AESCryptoService
implements CryptoService {
    private static final Logger log = LoggerFactory.getLogger(AESCryptoService.class);
    private volatile boolean initialized = false;
    public static final String KEY_URI_PROPERTY = "general.custom.crypto.key.uri";
    public static final String ENCRYPT_ENABLED_PROPERTY = "general.custom.crypto.enabled";
    private static final String NO_CRYPTO_VERSION = "U+1F47B";
    private static final String URI = "uri";
    private static final String KEY_WRAP_TRANSFORM = "AESWrap";
    private static final SecureRandom random = new SecureRandom();
    private Key encryptingKek = null;
    private String keyLocation = null;
    private String keyManager = null;
    private HashMap<String, Key> decryptingKeys = null;
    private boolean encryptEnabled = true;
    private static final FileEncrypter DISABLED = new NoFileEncrypter();
    private static final ThreadLocal<Cipher> KEY_WRAP_CIPHER = new ThreadLocal<Cipher>(){

        @Override
        @SuppressFBWarnings(value={"CIPHER_INTEGRITY"}, justification="integrity not needed for key wrap")
        protected Cipher initialValue() {
            try {
                return Cipher.getInstance(AESCryptoService.KEY_WRAP_TRANSFORM);
            }
            catch (NoSuchAlgorithmException | NoSuchPaddingException e) {
                throw new CryptoService.CryptoException("Error creating Cipher for AESWrap", e);
            }
        }
    };
    private static final ThreadLocal<Cipher> KEY_UNWRAP_CIPHER = new ThreadLocal<Cipher>(){

        @Override
        @SuppressFBWarnings(value={"CIPHER_INTEGRITY"}, justification="integrity not needed for key wrap")
        protected Cipher initialValue() {
            try {
                return Cipher.getInstance(AESCryptoService.KEY_WRAP_TRANSFORM);
            }
            catch (NoSuchAlgorithmException | NoSuchPaddingException e) {
                throw new CryptoService.CryptoException("Error creating Cipher for AESWrap", e);
            }
        }
    };

    @Override
    public void init(Map<String, String> conf) throws CryptoService.CryptoException {
        this.ensureNotInit();
        String keyLocation = Objects.requireNonNull(conf.get(KEY_URI_PROPERTY), "Config property general.custom.crypto.key.uri is required.");
        String enabledProp = conf.get(ENCRYPT_ENABLED_PROPERTY);
        if (enabledProp != null) {
            this.encryptEnabled = Boolean.parseBoolean(enabledProp);
        }
        String keyMgr = URI;
        this.decryptingKeys = new HashMap();
        switch (keyMgr) {
            case "uri": {
                this.keyManager = keyMgr;
                this.keyLocation = keyLocation;
                this.encryptingKek = AESCryptoService.loadKekFromUri(keyLocation);
                break;
            }
            default: {
                throw new CryptoService.CryptoException("Unrecognized key manager");
            }
        }
        Objects.requireNonNull(this.encryptingKek, "Encrypting Key Encryption Key was null, init failed");
        log.debug("Successfully initialized crypto Key from {}", (Object)keyLocation);
        this.initialized = true;
    }

    @Override
    public FileEncrypter getFileEncrypter(CryptoEnvironment environment) {
        this.ensureInit();
        if (!this.encryptEnabled) {
            return DISABLED;
        }
        switch (environment.getScope()) {
            case WAL: {
                AESCBCCryptoModule cm = new AESCBCCryptoModule(this.encryptingKek, this.keyLocation, this.keyManager);
                return cm.getEncrypter();
            }
            case TABLE: {
                AESGCMCryptoModule cm = new AESGCMCryptoModule(this.encryptingKek, this.keyLocation, this.keyManager);
                return cm.getEncrypter();
            }
        }
        throw new CryptoService.CryptoException("Unknown scope: " + String.valueOf((Object)environment.getScope()));
    }

    @Override
    public FileDecrypter getFileDecrypter(CryptoEnvironment environment) {
        this.ensureInit();
        Optional<byte[]> decryptionParams = environment.getDecryptionParams();
        if (decryptionParams.isEmpty() || AESCryptoService.checkNoCrypto(decryptionParams.orElseThrow())) {
            return new NoFileDecrypter();
        }
        ParsedCryptoParameters parsed = AESCryptoService.parseCryptoParameters(decryptionParams.orElseThrow());
        Key kek = this.loadDecryptionKek(parsed);
        Key fek = AESCryptoService.unwrapKey(parsed.getEncFek(), kek);
        switch (parsed.getCryptoServiceVersion()) {
            case "U+1f600": {
                AESCBCCryptoModule cm = new AESCBCCryptoModule(this.encryptingKek, this.keyLocation, this.keyManager);
                return cm.getDecrypter(fek);
            }
            case "U+1F43B": {
                AESGCMCryptoModule cm = new AESGCMCryptoModule(this.encryptingKek, this.keyLocation, this.keyManager);
                return cm.getDecrypter(fek);
            }
        }
        throw new CryptoService.CryptoException("Unknown crypto module version: " + parsed.getCryptoServiceVersion());
    }

    private static boolean checkNoCrypto(byte[] params) {
        byte[] noCryptoBytes = NO_CRYPTO_VERSION.getBytes(StandardCharsets.UTF_8);
        return Arrays.equals(params, noCryptoBytes);
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private static byte[] createCryptoParameters(String version, Key encryptingKek, String encryptingKekId, String encryptingKeyManager, Key fek) {
        try (ByteArrayOutputStream baos = new ByteArrayOutputStream();){
            byte[] byArray;
            try (DataOutputStream params = new DataOutputStream(baos);){
                params.writeUTF(AESCryptoService.class.getName());
                params.writeUTF(version);
                params.writeUTF(encryptingKeyManager);
                params.writeUTF(encryptingKekId);
                byte[] wrappedFek = AESCryptoService.wrapKey(fek, encryptingKek);
                params.writeInt(wrappedFek.length);
                params.write(wrappedFek);
                params.flush();
                byArray = baos.toByteArray();
            }
            return byArray;
        }
        catch (IOException e) {
            throw new CryptoService.CryptoException("Error creating crypto params", e);
        }
    }

    private static ParsedCryptoParameters parseCryptoParameters(byte[] parameters) {
        ParsedCryptoParameters parsed = new ParsedCryptoParameters();
        try (ByteArrayInputStream bais = new ByteArrayInputStream(parameters);
             DataInputStream params = new DataInputStream(bais);){
            parsed.setCryptoServiceName(params.readUTF());
            parsed.setCryptoServiceVersion(params.readUTF());
            parsed.setKeyManagerVersion(params.readUTF());
            parsed.setKekId(params.readUTF());
            int encFekLen = params.readInt();
            byte[] encFek = new byte[encFekLen];
            int bytesRead = params.read(encFek);
            if (bytesRead != encFekLen) {
                throw new CryptoService.CryptoException("Incorrect number of bytes read for encrypted FEK");
            }
            parsed.setEncFek(encFek);
        }
        catch (IOException e) {
            throw new CryptoService.CryptoException("Error creating crypto params", e);
        }
        return parsed;
    }

    private Key loadDecryptionKek(ParsedCryptoParameters params) {
        Key ret = null;
        String keyTag = params.getKeyManagerVersion() + "!" + params.getKekId();
        if (this.decryptingKeys.get(keyTag) != null) {
            return this.decryptingKeys.get(keyTag);
        }
        switch (params.keyManagerVersion) {
            case "uri": {
                ret = AESCryptoService.loadKekFromUri(params.kekId);
                break;
            }
            default: {
                throw new CryptoService.CryptoException("Unable to load kek: " + params.kekId);
            }
        }
        this.decryptingKeys.put(keyTag, ret);
        if (ret == null) {
            throw new CryptoService.CryptoException("Unable to load decryption KEK");
        }
        return ret;
    }

    public static Key generateKey(SecureRandom random, int size) {
        byte[] bytes = new byte[size];
        random.nextBytes(bytes);
        return new SecretKeySpec(bytes, "AES");
    }

    public static Key unwrapKey(byte[] fek, Key kek) {
        try {
            Cipher c = KEY_UNWRAP_CIPHER.get();
            c.init(4, kek);
            return c.unwrap(fek, "AES", 3);
        }
        catch (InvalidKeyException | NoSuchAlgorithmException e) {
            throw new CryptoService.CryptoException("Unable to unwrap file encryption key", e);
        }
    }

    public static byte[] wrapKey(Key fek, Key kek) {
        try {
            Cipher c = KEY_WRAP_CIPHER.get();
            c.init(3, kek);
            return c.wrap(fek);
        }
        catch (InvalidKeyException | IllegalBlockSizeException e) {
            throw new CryptoService.CryptoException("Unable to wrap file encryption key", e);
        }
    }

    @SuppressFBWarnings(value={"PATH_TRAVERSAL_IN"}, justification="keyId specified by admin")
    public static Key loadKekFromUri(String keyId) {
        try {
            URI uri = new URI(keyId);
            return new SecretKeySpec(Files.readAllBytes(Paths.get(uri.getPath(), new String[0])), "AES");
        }
        catch (IOException | IllegalArgumentException | URISyntaxException e) {
            throw new CryptoService.CryptoException("Unable to load key encryption key.", e);
        }
    }

    private void ensureInit() {
        if (!this.initialized) {
            throw new IllegalStateException("This Crypto Service has not been initialized.");
        }
    }

    private void ensureNotInit() {
        if (this.initialized) {
            throw new IllegalStateException("This Crypto Service has already been initialized.");
        }
    }

    public class AESCBCCryptoModule
    implements CryptoModule {
        public static final String VERSION = "U+1f600";
        private final Integer IV_LENGTH_IN_BYTES = 16;
        private final Integer KEY_LENGTH_IN_BYTES = 16;
        private final String transformation = "AES/CBC/NoPadding";
        private final Key encryptingKek;
        private final String keyLocation;
        private final String keyManager;

        public AESCBCCryptoModule(Key encryptingKek, String keyLocation, String keyManager) {
            this.encryptingKek = encryptingKek;
            this.keyLocation = keyLocation;
            this.keyManager = keyManager;
        }

        @Override
        public FileEncrypter getEncrypter() {
            return new AESCBCFileEncrypter();
        }

        @Override
        public FileDecrypter getDecrypter(Key fek) {
            return new AESCBCFileDecrypter(fek);
        }

        @SuppressFBWarnings(value={"CIPHER_INTEGRITY"}, justification="CBC is provided for WALs")
        public class AESCBCFileEncrypter
        implements FileEncrypter {
            private final Cipher cipher;
            private final Key fek;
            private final byte[] initVector;
            private final byte[] decryptionParameters;
            private final AtomicBoolean openTracker;

            AESCBCFileEncrypter() {
                this.initVector = new byte[AESCBCCryptoModule.this.IV_LENGTH_IN_BYTES.intValue()];
                this.openTracker = new AtomicBoolean();
                try {
                    this.cipher = Cipher.getInstance("AES/CBC/NoPadding");
                }
                catch (NoSuchAlgorithmException | NoSuchPaddingException e) {
                    throw new CryptoService.CryptoException("Error obtaining cipher for transform AES/CBC/NoPadding", e);
                }
                this.fek = AESCryptoService.generateKey(random, AESCBCCryptoModule.this.KEY_LENGTH_IN_BYTES);
                this.decryptionParameters = AESCryptoService.createCryptoParameters(AESCBCCryptoModule.VERSION, AESCBCCryptoModule.this.encryptingKek, AESCBCCryptoModule.this.keyLocation, AESCBCCryptoModule.this.keyManager, this.fek);
            }

            @Override
            public OutputStream encryptStream(OutputStream outputStream) throws CryptoService.CryptoException {
                if (!this.openTracker.compareAndSet(false, true)) {
                    throw new CryptoService.CryptoException("Attempted to obtain new stream without closing previous one.");
                }
                random.nextBytes(this.initVector);
                try {
                    outputStream.write(this.initVector);
                }
                catch (IOException e) {
                    throw new CryptoService.CryptoException("Unable to write IV to stream", e);
                }
                try {
                    this.cipher.init(1, this.fek, new IvParameterSpec(this.initVector));
                }
                catch (InvalidAlgorithmParameterException | InvalidKeyException e) {
                    throw new CryptoService.CryptoException("Unable to initialize cipher", e);
                }
                CipherOutputStream cos = new CipherOutputStream(outputStream, this.cipher);
                return new BlockedOutputStream(cos, this.cipher.getBlockSize(), 1024, this.openTracker);
            }

            @Override
            public byte[] getDecryptionParameters() {
                return this.decryptionParameters;
            }
        }

        @SuppressFBWarnings(value={"CIPHER_INTEGRITY"}, justification="CBC is provided for WALs")
        public class AESCBCFileDecrypter
        implements FileDecrypter {
            private final Key fek;

            AESCBCFileDecrypter(Key fek) {
                this.fek = fek;
            }

            @Override
            public InputStream decryptStream(InputStream inputStream) throws CryptoService.CryptoException {
                Cipher cipher;
                try {
                    cipher = Cipher.getInstance("AES/CBC/NoPadding");
                }
                catch (NoSuchAlgorithmException | NoSuchPaddingException e) {
                    throw new CryptoService.CryptoException("Error obtaining cipher for transform AES/CBC/NoPadding", e);
                }
                byte[] initVector = new byte[AESCBCCryptoModule.this.IV_LENGTH_IN_BYTES.intValue()];
                try {
                    IOUtils.readFully((InputStream)inputStream, (byte[])initVector);
                }
                catch (IOException e) {
                    throw new CryptoService.CryptoException("Unable to read IV from stream", e);
                }
                try {
                    cipher.init(2, this.fek, new IvParameterSpec(initVector));
                }
                catch (InvalidAlgorithmParameterException | InvalidKeyException e) {
                    throw new CryptoService.CryptoException("Unable to initialize cipher", e);
                }
                CipherInputStream cis = new CipherInputStream(inputStream, cipher);
                return new BlockedInputStream(cis, cipher.getBlockSize(), 1024);
            }
        }
    }

    private static interface CryptoModule {
        public FileEncrypter getEncrypter();

        public FileDecrypter getDecrypter(Key var1);
    }

    public class AESGCMCryptoModule
    implements CryptoModule {
        private static final String VERSION = "U+1F43B";
        private final Integer GCM_IV_LENGTH_IN_BYTES = 12;
        private final Integer KEY_LENGTH_IN_BYTES = 16;
        private final Integer GCM_TAG_LENGTH_IN_BITS = 128;
        private final String transformation = "AES/GCM/NoPadding";
        private boolean ivReused = false;
        private final Key encryptingKek;
        private final String keyLocation;
        private final String keyManager;

        public AESGCMCryptoModule(Key encryptingKek, String keyLocation, String keyManager) {
            this.encryptingKek = encryptingKek;
            this.keyLocation = keyLocation;
            this.keyManager = keyManager;
        }

        @Override
        public FileEncrypter getEncrypter() {
            return new AESGCMFileEncrypter();
        }

        @Override
        public FileDecrypter getDecrypter(Key fek) {
            return new AESGCMFileDecrypter(fek);
        }

        public class AESGCMFileEncrypter
        implements FileEncrypter {
            private final byte[] firstInitVector;
            private final Key fek;
            private final byte[] initVector;
            private final Cipher cipher;
            private final byte[] decryptionParameters;
            private final AtomicBoolean openTracker;

            AESGCMFileEncrypter() {
                this.initVector = new byte[AESGCMCryptoModule.this.GCM_IV_LENGTH_IN_BYTES.intValue()];
                this.openTracker = new AtomicBoolean();
                try {
                    this.cipher = Cipher.getInstance("AES/GCM/NoPadding");
                }
                catch (NoSuchAlgorithmException | NoSuchPaddingException e) {
                    throw new CryptoService.CryptoException("Error obtaining cipher for transform AES/GCM/NoPadding", e);
                }
                this.fek = AESCryptoService.generateKey(random, AESGCMCryptoModule.this.KEY_LENGTH_IN_BYTES);
                random.nextBytes(this.initVector);
                this.firstInitVector = Arrays.copyOf(this.initVector, this.initVector.length);
                this.decryptionParameters = AESCryptoService.createCryptoParameters(AESGCMCryptoModule.VERSION, AESGCMCryptoModule.this.encryptingKek, AESGCMCryptoModule.this.keyLocation, AESGCMCryptoModule.this.keyManager, this.fek);
            }

            @Override
            public OutputStream encryptStream(OutputStream outputStream) throws CryptoService.CryptoException {
                if (AESGCMCryptoModule.this.ivReused) {
                    throw new CryptoService.CryptoException("Key/IV reuse is forbidden in AESGCMCryptoModule. Too many RBlocks.");
                }
                if (!this.openTracker.compareAndSet(false, true)) {
                    throw new CryptoService.CryptoException("Attempted to obtain new stream without closing previous one.");
                }
                this.incrementIV(this.initVector, this.initVector.length - 1);
                if (Arrays.equals(this.initVector, this.firstInitVector)) {
                    AESGCMCryptoModule.this.ivReused = true;
                }
                try {
                    outputStream.write(this.initVector);
                }
                catch (IOException e) {
                    throw new CryptoService.CryptoException("Unable to write IV to stream", e);
                }
                try {
                    this.cipher.init(1, this.fek, new GCMParameterSpec(AESGCMCryptoModule.this.GCM_TAG_LENGTH_IN_BITS, this.initVector));
                }
                catch (InvalidAlgorithmParameterException | InvalidKeyException e) {
                    throw new CryptoService.CryptoException("Unable to initialize cipher", e);
                }
                RFileCipherOutputStream cos = new RFileCipherOutputStream(new DiscardCloseOutputStream(outputStream), this.cipher);
                return new BlockedOutputStream(cos, this.cipher.getBlockSize(), 1024, this.openTracker);
            }

            void incrementIV(byte[] iv, int i) {
                int n = i;
                iv[n] = (byte)(iv[n] + 1);
                if (iv[i] == 0) {
                    if (i == 0) {
                        return;
                    }
                    this.incrementIV(iv, i - 1);
                }
            }

            @Override
            public byte[] getDecryptionParameters() {
                return this.decryptionParameters;
            }
        }

        public class AESGCMFileDecrypter
        implements FileDecrypter {
            private final Key fek;

            AESGCMFileDecrypter(Key fek) {
                this.fek = fek;
            }

            @Override
            public InputStream decryptStream(InputStream inputStream) throws CryptoService.CryptoException {
                Cipher cipher;
                try {
                    cipher = Cipher.getInstance("AES/GCM/NoPadding");
                }
                catch (NoSuchAlgorithmException | NoSuchPaddingException e) {
                    throw new CryptoService.CryptoException("Error obtaining cipher for transform AES/GCM/NoPadding", e);
                }
                byte[] initVector = new byte[AESGCMCryptoModule.this.GCM_IV_LENGTH_IN_BYTES.intValue()];
                try {
                    IOUtils.readFully((InputStream)inputStream, (byte[])initVector);
                }
                catch (IOException e) {
                    throw new CryptoService.CryptoException("Unable to read IV from stream", e);
                }
                try {
                    cipher.init(2, this.fek, new GCMParameterSpec(AESGCMCryptoModule.this.GCM_TAG_LENGTH_IN_BITS, initVector));
                }
                catch (InvalidAlgorithmParameterException | InvalidKeyException e) {
                    throw new CryptoService.CryptoException("Unable to initialize cipher", e);
                }
                CipherInputStream cis = new CipherInputStream(inputStream, cipher);
                return new BlockedInputStream(cis, cipher.getBlockSize(), 1024);
            }
        }
    }

    static class ParsedCryptoParameters {
        String cryptoServiceName;
        String cryptoServiceVersion;
        String keyManagerVersion;
        String kekId;
        byte[] encFek;

        ParsedCryptoParameters() {
        }

        public void setCryptoServiceName(String cryptoServiceName) {
            this.cryptoServiceName = cryptoServiceName;
        }

        public String getCryptoServiceVersion() {
            return this.cryptoServiceVersion;
        }

        public void setCryptoServiceVersion(String cryptoServiceVersion) {
            this.cryptoServiceVersion = cryptoServiceVersion;
        }

        public String getKeyManagerVersion() {
            return this.keyManagerVersion;
        }

        public void setKeyManagerVersion(String keyManagerVersion) {
            this.keyManagerVersion = keyManagerVersion;
        }

        public String getKekId() {
            return this.kekId;
        }

        public void setKekId(String kekId) {
            this.kekId = kekId;
        }

        public byte[] getEncFek() {
            return this.encFek;
        }

        public void setEncFek(byte[] encFek) {
            this.encFek = encFek;
        }
    }
}

