/*******************************************************************************
 * Copyright (c) 2015 itemis AG and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Matthias Wienand (itemis AG) - initial API and implementation
 *
 *******************************************************************************/
package org.eclipse.gef4.mvc.fx.operations;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.commands.operations.AbstractOperation;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.gef4.fx.nodes.InfiniteCanvas;
import org.eclipse.gef4.geometry.convert.fx.Geometry2FX;
import org.eclipse.gef4.geometry.convert.fx.FX2Geometry;
import org.eclipse.gef4.geometry.planar.AffineTransform;
import org.eclipse.gef4.mvc.operations.ITransactionalOperation;

import javafx.scene.transform.Affine;

/**
 * The {@link FXChangeViewportOperation} can be used to alter the scroll offset
 * and the content transformation of an {@link InfiniteCanvas}. It is used by
 * scroll/pan and zoom policies.
 *
 * @author mwienand
 *
 */
public class FXChangeViewportOperation extends AbstractOperation
		implements ITransactionalOperation {

	/**
	 * The {@link InfiniteCanvas} that is manipulated by this operation.
	 */
	private InfiniteCanvas canvas;

	/**
	 * The viewport width that is applied when undoing this operation.
	 */
	private double initialWidth;

	/**
	 * The viewport width that is applied when executing this operation.
	 */
	private double newWidth;

	/**
	 * The viewport height that is applied when undoing this operation.
	 */
	private double initialHeight;

	/**
	 * The viewport height that is applied when executing this operation.
	 */
	private double newHeight;

	/**
	 * The contents transformation that is applied when undoing this operation.
	 */
	private AffineTransform initialContentTransform;

	/**
	 * The contents transformation that is applied when executing this
	 * operation.
	 */
	private AffineTransform newContentTransform;

	/**
	 * The horizontal translation that is applied when undoing this operation.
	 */
	private double initialHorizontalScrollOffset;

	/**
	 * The horizontal translation that is applied when executing this operation.
	 */
	private double horizontalScrollOffset;

	/**
	 * The vertical translation that is applied when undoing this operation.
	 */
	private double initialVerticalScrollOffset;

	/**
	 * The vertical translation that is applied when executing this operation.
	 */
	private double verticalScrollOffset;

	/**
	 * Creates a new {@link FXChangeViewportOperation} to manipulate the given
	 * {@link InfiniteCanvas}. The current viewport values are read and used
	 * when undoing this operation.
	 *
	 * @param canvas
	 *            The {@link InfiniteCanvas} which is manipulated by this
	 *            operation.
	 */
	public FXChangeViewportOperation(InfiniteCanvas canvas) {
		super("Change Viewport");
		readViewport(canvas);
	}

	/**
	 * Creates a new {@link FXChangeViewportOperation} to manipulate the given
	 * {@link InfiniteCanvas}. The current viewport values are read and used
	 * when undoing this operation. The given
	 * {@link java.awt.geom.AffineTransform} will be applied when executing this
	 * operation.
	 *
	 * @param canvas
	 *            The {@link InfiniteCanvas} that is manipulated.
	 * @param newContentTransform
	 *            The contents transformation which is applied when executing
	 *            this operation.
	 */
	public FXChangeViewportOperation(InfiniteCanvas canvas,
			AffineTransform newContentTransform) {
		this(canvas);
		this.newContentTransform = newContentTransform;
	}

	/**
	 * Creates a new {@link FXChangeViewportOperation} to manipulate the given
	 * {@link InfiniteCanvas}. The current viewport values are read and used
	 * when undoing this operation. The given translation values will be applied
	 * when executing this operation.
	 *
	 * @param canvas
	 *            The {@link InfiniteCanvas} that is manipulated.
	 * @param newHorizontalScrollOffset
	 *            The horizontal translation that is applied when executing this
	 *            operation.
	 * @param newVerticalScrollOffset
	 *            The vertical translation that is applied when executing this
	 *            operation.
	 */
	public FXChangeViewportOperation(InfiniteCanvas canvas,
			double newHorizontalScrollOffset, double newVerticalScrollOffset) {
		this(canvas);
		this.horizontalScrollOffset = newHorizontalScrollOffset;
		this.verticalScrollOffset = newVerticalScrollOffset;
	}

	/**
	 * Creates a new {@link FXChangeViewportOperation} to manipulate the given
	 * {@link InfiniteCanvas}. The current viewport values are read and used
	 * when undoing this operation. The given translation values and contents
	 * transformation will be applied when executing this operation.
	 *
	 * @param canvas
	 *            The {@link InfiniteCanvas} that is manipulated.
	 * @param newHorizontalScrollOffset
	 *            The horizontal translation that is applied when executing this
	 *            operation.
	 * @param newVerticalScrollOffset
	 *            The vertical translation that is applied when executing this
	 *            operation.
	 * @param newContentTransform
	 *            The contents transformation which is applied when executing
	 *            this operation.
	 */
	public FXChangeViewportOperation(InfiniteCanvas canvas,
			double newHorizontalScrollOffset, double newVerticalScrollOffset,
			AffineTransform newContentTransform) {
		this(canvas);
		this.newContentTransform = newContentTransform;
		this.horizontalScrollOffset = newHorizontalScrollOffset;
		this.verticalScrollOffset = newVerticalScrollOffset;
	}

	/**
	 * Creates a new {@link FXChangeViewportOperation} to manipulate the given
	 * {@link InfiniteCanvas}. The current viewport values are read and used
	 * when undoing this operation. The given translation values, dimensions,
	 * and contents transformation will be applied when executing this
	 * operation.
	 *
	 * @param canvas
	 *            The {@link InfiniteCanvas} that is manipulated.
	 * @param newHorizontalScrollOffset
	 *            The horizontal translation that is applied when executing this
	 *            operation.
	 * @param newVerticalScrollOffset
	 *            The vertical translation that is applied when executing this
	 *            operation.
	 * @param newWidth
	 *            The viewport width that is applied when executing this
	 *            operation.
	 * @param newHeight
	 *            The viewport height that is applied when executing this
	 *            operation.
	 * @param newContentTransform
	 *            The contents transformation which is applied when executing
	 *            this operation.
	 */
	public FXChangeViewportOperation(InfiniteCanvas canvas,
			double newHorizontalScrollOffset, double newVerticalScrollOffset,
			double newWidth, double newHeight,
			AffineTransform newContentTransform) {
		this(canvas);
		this.newWidth = newWidth;
		this.newHeight = newHeight;
		this.newContentTransform = newContentTransform;
		this.horizontalScrollOffset = newHorizontalScrollOffset;
		this.verticalScrollOffset = newVerticalScrollOffset;
	}

	/**
	 * Concatenates the given {@link java.awt.geom.AffineTransform} to the
	 * contents transformation that will be applied when executing this
	 * operation.
	 *
	 * @param t
	 *            The {@link java.awt.geom.AffineTransform} which is
	 *            concatenated to the transformation that will be applied when
	 *            executing this operation.
	 */
	public void concatenateToNewContentTransform(AffineTransform t) {
		newContentTransform.concatenate(t);
	}

	@Override
	public IStatus execute(IProgressMonitor monitor, IAdaptable info)
			throws ExecutionException {
		if (canvas.getPrefWidth() != newWidth) {
			canvas.setPrefWidth(newWidth);
		}
		if (canvas.getPrefHeight() != newHeight) {
			canvas.setPrefHeight(newHeight);
		}
		Affine newContentAffine = Geometry2FX
				.toFXAffine(newContentTransform);
		if (!canvas.getContentTransform().equals(newContentAffine)) {
			canvas.setContentTransform(newContentAffine);
		}
		if (canvas.getHorizontalScrollOffset() != horizontalScrollOffset) {
			canvas.setHorizontalScrollOffset(horizontalScrollOffset);
		}
		if (canvas.getVerticalScrollOffset() != verticalScrollOffset) {
			canvas.setVerticalScrollOffset(verticalScrollOffset);
		}
		return Status.OK_STATUS;
	}

	/**
	 * Returns the {@link InfiniteCanvas} that is manipulated by this operation.
	 *
	 * @return The {@link InfiniteCanvas} that is manipulated by this operation.
	 */
	public InfiniteCanvas getInfiniteCanvas() {
		return canvas;
	}

	/**
	 * Returns the contents transformation that will be applied when undoing
	 * this operation.
	 *
	 * @return The contents transformation that will be applied when undoing
	 *         this operation.
	 */
	public AffineTransform getInitialContentTransform() {
		return initialContentTransform;
	}

	/**
	 * Returns the viewport height that will be applied when undoing this
	 * operation.
	 *
	 * @return The viewport height that will be applied when undoing this
	 *         operation.
	 */
	public double getInitialHeight() {
		return initialHeight;
	}

	/**
	 * Returns the horizontal translation that will be applied when undoing this
	 * operation.
	 *
	 * @return The horizontal translation that will be applied when undoing this
	 *         operation.
	 */
	public double getInitialHorizontalScrollOffset() {
		return initialHorizontalScrollOffset;
	}

	/**
	 * Returns the vertical translation that will be applied when undoing this
	 * operation.
	 *
	 * @return The vertical translation that will be applied when undoing this
	 *         operation.
	 */
	public double getInitialVerticalScrollOffset() {
		return initialVerticalScrollOffset;
	}

	/**
	 * Returns the viewport width that will be applied when undoing this
	 * operation.
	 *
	 * @return The viewport width that will be applied when undoing this
	 *         operation.
	 */
	public double getInitialWidth() {
		return initialWidth;
	}

	/**
	 * Returns the viewport height that will be applied when executing this
	 * operation.
	 *
	 * @return The viewport height that will be applied when executing this
	 *         operation.
	 */
	public double getNewHeight() {
		return newHeight;
	}

	/**
	 * Returns the horizontal translation that will be applied when executing
	 * this operation.
	 *
	 * @return The horizontal translation that will be applied when executing
	 *         this operation.
	 */
	public double getNewHorizontalScrollOffset() {
		return horizontalScrollOffset;
	}

	/**
	 * Returns the contents transformation that will be applied when executing
	 * this operation.
	 *
	 * @return The contents transformation that will be applied when executing
	 *         this operation.
	 */
	public AffineTransform getNewTransform() {
		return newContentTransform;
	}

	/**
	 * Returns the vertical translation that will be applied when executing this
	 * operation.
	 *
	 * @return The vertical translation that will be applied when executing this
	 *         operation.
	 */
	public double getNewVerticalScrollOffset() {
		return verticalScrollOffset;
	}

	/**
	 * Returns the viewport width that will be applied when executing this
	 * operation.
	 *
	 * @return The viewport width that will be applied when executing this
	 *         operation.
	 */
	public double getNewWidth() {
		return newWidth;
	}

	@Override
	public boolean isContentRelevant() {
		return false;
	}

	@Override
	public boolean isNoOp() {
		return getNewWidth() == getInitialWidth()
				&& getNewHeight() == getInitialHeight()
				&& (getNewTransform() == null
						? getInitialContentTransform() == null
						: getNewTransform()
								.equals(getInitialContentTransform()))
				&& getNewHorizontalScrollOffset() == getInitialHorizontalScrollOffset()
				&& getNewVerticalScrollOffset() == getInitialVerticalScrollOffset();
	}

	/**
	 * Stores all relevant viewport values in fields, so that they can be
	 * restored later.
	 *
	 * @param canvas
	 *            The {@link InfiniteCanvas} from which the values are read.
	 */
	protected void readViewport(InfiniteCanvas canvas) {
		this.canvas = canvas;
		initialWidth = canvas.getWidth();
		initialHeight = canvas.getHeight();
		initialContentTransform = FX2Geometry
				.toAffineTransform(canvas.getContentTransform());
		initialHorizontalScrollOffset = canvas.getHorizontalScrollOffset();
		initialVerticalScrollOffset = canvas.getVerticalScrollOffset();
		// use old values for new values per default
		newWidth = initialWidth;
		newHeight = initialHeight;
		newContentTransform = initialContentTransform.getCopy();
		horizontalScrollOffset = initialHorizontalScrollOffset;
		verticalScrollOffset = initialVerticalScrollOffset;
	}

	@Override
	public IStatus redo(IProgressMonitor monitor, IAdaptable info)
			throws ExecutionException {
		return execute(monitor, info);
	}

	/**
	 * Sets the contents transformation that will be applied when executing this
	 * operation to the given value.
	 *
	 * @param newContentTransform
	 *            The contents transformation to apply when executing this
	 *            operation.
	 */
	public void setNewContentTransform(AffineTransform newContentTransform) {
		this.newContentTransform = newContentTransform;
	}

	/**
	 * Sets the viewport height that will be applied when executing this
	 * operation to the given value.
	 *
	 * @param newHeight
	 *            The viewport height to apply when executing this operation.
	 */
	public void setNewHeight(double newHeight) {
		this.newHeight = newHeight;
	}

	/**
	 * Sets the horizontal translation that will be applied when executing this
	 * operation to the given value.
	 *
	 * @param horizontalScrollOffset
	 *            The horizontal translation to apply when executing this
	 *            operation.
	 */
	public void setNewHorizontalScrollOffset(double horizontalScrollOffset) {
		this.horizontalScrollOffset = horizontalScrollOffset;
	}

	/**
	 * Sets the vertical translation that will be applied when executing this
	 * operation to the given value.
	 *
	 * @param verticalScrollOffset
	 *            The vertical translation to apply when executing this
	 *            operation.
	 */
	public void setNewVerticalScrollOffset(double verticalScrollOffset) {
		this.verticalScrollOffset = verticalScrollOffset;
	}

	/**
	 * Sets the viewport width that will be applied when executing this
	 * operation to the given value.
	 *
	 * @param newWidth
	 *            The viewport width to apply when executing this operation.
	 */
	public void setNewWidth(double newWidth) {
		this.newWidth = newWidth;
	}

	@Override
	public IStatus undo(IProgressMonitor monitor, IAdaptable info)
			throws ExecutionException {
		if (canvas.getPrefWidth() != initialWidth) {
			canvas.setPrefWidth(initialWidth);
		}
		if (canvas.getPrefHeight() != initialHeight) {
			canvas.setPrefHeight(initialHeight);
		}
		Affine initialContentAffine = Geometry2FX
				.toFXAffine(initialContentTransform);
		if (!canvas.getContentTransform().equals(initialContentAffine)) {
			canvas.setContentTransform(initialContentAffine);
		}
		if (canvas
				.getHorizontalScrollOffset() != initialHorizontalScrollOffset) {
			canvas.setHorizontalScrollOffset(initialHorizontalScrollOffset);
		}
		if (canvas.getVerticalScrollOffset() != initialVerticalScrollOffset) {
			canvas.setVerticalScrollOffset(initialVerticalScrollOffset);
		}
		return Status.OK_STATUS;
	}

}
