/**
 * Copyright (c) 2016, 2019 itemis AG and others.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Alexander Nyßen    (itemis AG) - initial API and implementation
 *     Tamas Miklossy     (itemis AG) - Merge DotInterpreter into DotImport (bug #491261)
 *                                    - Add support for all dot attributes (bug #461506)
 *     Zoey Gerrit Prigge (itemis AG) - Add support for all dot attributes (bug #461506)
 */
package org.eclipse.gef.dot.internal;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.io.File;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gef.dot.internal.DotAttributes;
import org.eclipse.gef.dot.internal.DotFileUtils;
import org.eclipse.gef.dot.internal.language.DotAstHelper;
import org.eclipse.gef.dot.internal.language.DotStandaloneSetup;
import org.eclipse.gef.dot.internal.language.dot.AttrList;
import org.eclipse.gef.dot.internal.language.dot.AttrStmt;
import org.eclipse.gef.dot.internal.language.dot.Attribute;
import org.eclipse.gef.dot.internal.language.dot.AttributeType;
import org.eclipse.gef.dot.internal.language.dot.DotAst;
import org.eclipse.gef.dot.internal.language.dot.DotFactory;
import org.eclipse.gef.dot.internal.language.dot.DotGraph;
import org.eclipse.gef.dot.internal.language.dot.EdgeRhs;
import org.eclipse.gef.dot.internal.language.dot.EdgeRhsNode;
import org.eclipse.gef.dot.internal.language.dot.EdgeStmtNode;
import org.eclipse.gef.dot.internal.language.dot.NodeId;
import org.eclipse.gef.dot.internal.language.dot.NodeStmt;
import org.eclipse.gef.dot.internal.language.dot.Stmt;
import org.eclipse.gef.dot.internal.language.dot.Subgraph;
import org.eclipse.gef.dot.internal.language.parser.antlr.DotParser;
import org.eclipse.gef.dot.internal.language.terminals.ID;
import org.eclipse.gef.graph.Edge;
import org.eclipse.gef.graph.Graph;
import org.eclipse.gef.graph.Node;
import org.eclipse.xtext.nodemodel.INode;
import org.eclipse.xtext.parser.IParseResult;
import org.eclipse.xtext.parser.IParser;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure2;

/**
 * A parser that creates a {@link Graph} with {@link DotAttributes} from a Graphviz DOT string or file.
 * The created {@link Graph} follows the structure of the DOT input very closely.
 * Subgraphs (including clusters) are represented by a {@link Node} with a nested {@link Graph},
 * where the graph holds all attributes (like the name). If a node is used in multiple (sub-)graphs,
 * it will be contained in the graph where it is defined (first occurrence).
 * 
 * @author anyssen
 */
@SuppressWarnings("all")
public class DotImport {
  @Inject
  private static IParser dotParser;
  
  private static IParser getDotParser() {
    if ((DotImport.dotParser == null)) {
      DotImport.dotParser = new DotStandaloneSetup().createInjectorAndDoEMFRegistration().<DotParser>getInstance(DotParser.class);
    }
    return DotImport.dotParser;
  }
  
  public List<Graph> importDot(final File dotFile) {
    return this.importDot(DotFileUtils.read(dotFile));
  }
  
  public List<Graph> importDot(final String dotString) {
    List<Graph> _xblockexpression = null;
    {
      IParser _dotParser = DotImport.getDotParser();
      StringReader _stringReader = new StringReader(dotString);
      IParseResult parseResult = _dotParser.parse(_stringReader);
      boolean _hasSyntaxErrors = parseResult.hasSyntaxErrors();
      if (_hasSyntaxErrors) {
        final Function1<INode, String> _function = (INode it) -> {
          return it.getSyntaxErrorMessage().getMessage();
        };
        String _join = IterableExtensions.join(IterableExtensions.<INode, String>map(parseResult.getSyntaxErrors(), _function), ",");
        String _plus = ("Given DOT string is not valid: " + _join);
        throw new IllegalArgumentException(_plus);
      }
      EObject _rootASTElement = parseResult.getRootASTElement();
      _xblockexpression = this.importDot(((DotAst) _rootASTElement));
    }
    return _xblockexpression;
  }
  
  public List<Graph> importDot(final DotAst dotAst) {
    final Function1<DotGraph, Graph> _function = (DotGraph it) -> {
      return this.transformDotGraph(it);
    };
    return IterableExtensions.<Graph>toList(IterableExtensions.<Graph>filterNull(ListExtensions.<DotGraph, Graph>map(dotAst.getGraphs(), _function)));
  }
  
  private Map<String, ID> globalGraphAttributes(final Graph.Builder context) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(context);
    final HashMap<String, ID> _result;
    synchronized (_createCache_globalGraphAttributes) {
      if (_createCache_globalGraphAttributes.containsKey(_cacheKey)) {
        return _createCache_globalGraphAttributes.get(_cacheKey);
      }
      HashMap<String, ID> _newHashMap = CollectionLiterals.<String, ID>newHashMap();
      _result = _newHashMap;
      _createCache_globalGraphAttributes.put(_cacheKey, _result);
    }
    _init_globalGraphAttributes(_result, context);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, Map<String, ID>> _createCache_globalGraphAttributes = CollectionLiterals.newHashMap();
  
  private void _init_globalGraphAttributes(final HashMap<String, ID> it, final Graph.Builder context) {
  }
  
  private Map<String, ID> globalNodeAttributes(final Graph.Builder context) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(context);
    final HashMap<String, ID> _result;
    synchronized (_createCache_globalNodeAttributes) {
      if (_createCache_globalNodeAttributes.containsKey(_cacheKey)) {
        return _createCache_globalNodeAttributes.get(_cacheKey);
      }
      HashMap<String, ID> _newHashMap = CollectionLiterals.<String, ID>newHashMap();
      _result = _newHashMap;
      _createCache_globalNodeAttributes.put(_cacheKey, _result);
    }
    _init_globalNodeAttributes(_result, context);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, Map<String, ID>> _createCache_globalNodeAttributes = CollectionLiterals.newHashMap();
  
  private void _init_globalNodeAttributes(final HashMap<String, ID> it, final Graph.Builder context) {
  }
  
  private Map<String, ID> globalEdgeAttributes(final Graph.Builder context) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(context);
    final HashMap<String, ID> _result;
    synchronized (_createCache_globalEdgeAttributes) {
      if (_createCache_globalEdgeAttributes.containsKey(_cacheKey)) {
        return _createCache_globalEdgeAttributes.get(_cacheKey);
      }
      HashMap<String, ID> _newHashMap = CollectionLiterals.<String, ID>newHashMap();
      _result = _newHashMap;
      _createCache_globalEdgeAttributes.put(_cacheKey, _result);
    }
    _init_globalEdgeAttributes(_result, context);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, Map<String, ID>> _createCache_globalEdgeAttributes = CollectionLiterals.newHashMap();
  
  private void _init_globalEdgeAttributes(final HashMap<String, ID> it, final Graph.Builder context) {
  }
  
  private Graph transformDotGraph(final DotGraph it) {
    Graph _xblockexpression = null;
    {
      this._createCache_globalGraphAttributes.clear();
      this._createCache_globalNodeAttributes.clear();
      this._createCache_globalEdgeAttributes.clear();
      this._createCache_createNode.clear();
      this._createCache_createSubgraph.clear();
      final Graph.Builder graphBuilder = new Graph.Builder();
      ID _name = it.getName();
      boolean _tripleNotEquals = (_name != null);
      if (_tripleNotEquals) {
        graphBuilder.attr(DotAttributes._NAME__GNE, it.getName());
      }
      graphBuilder.attr(DotAttributes._TYPE__G, it.getType());
      final Function1<Stmt, Boolean> _function = (Stmt it_1) -> {
        return Boolean.valueOf((!(it_1 instanceof Attribute)));
      };
      final Consumer<Stmt> _function_1 = (Stmt it_1) -> {
        this.transformStmt(it_1, graphBuilder);
      };
      IterableExtensions.<Stmt>filter(it.getStmts(), _function).forEach(_function_1);
      final Graph graph = graphBuilder.build();
      final Procedure2<String, Procedure2<? super Graph, ? super ID>> _function_2 = (String attributeName, Procedure2<? super Graph, ? super ID> f) -> {
        final ID attributeValue = DotAstHelper.getAttributeValue(it, attributeName);
        if ((attributeValue != null)) {
          f.apply(graph, attributeValue);
        } else {
          boolean _containsKey = this.globalGraphAttributes(graphBuilder).containsKey(attributeName);
          if (_containsKey) {
            f.apply(graph, this.globalGraphAttributes(graphBuilder).get(attributeName));
          }
        }
      };
      final Procedure2<String, Procedure2<? super Graph, ? super ID>> setter = _function_2;
      final Procedure2<Graph, ID> _function_3 = (Graph g, ID value) -> {
        DotAttributes.setBbRaw(g, value);
      };
      setter.apply(DotAttributes.BB__GC, _function_3);
      final Procedure2<Graph, ID> _function_4 = (Graph g, ID value) -> {
        DotAttributes.setBgcolorRaw(g, value);
      };
      setter.apply(DotAttributes.BGCOLOR__GC, _function_4);
      final Procedure2<Graph, ID> _function_5 = (Graph g, ID value) -> {
        DotAttributes.setClusterrankRaw(g, value);
      };
      setter.apply(DotAttributes.CLUSTERRANK__G, _function_5);
      final Procedure2<Graph, ID> _function_6 = (Graph g, ID value) -> {
        DotAttributes.setColorschemeRaw(g, value);
      };
      setter.apply(DotAttributes.COLORSCHEME__GCNE, _function_6);
      final Procedure2<Graph, ID> _function_7 = (Graph g, ID value) -> {
        DotAttributes.setFontcolorRaw(g, value);
      };
      setter.apply(DotAttributes.FONTCOLOR__GCNE, _function_7);
      final Procedure2<Graph, ID> _function_8 = (Graph g, ID value) -> {
        DotAttributes.setFontnameRaw(g, value);
      };
      setter.apply(DotAttributes.FONTNAME__GCNE, _function_8);
      final Procedure2<Graph, ID> _function_9 = (Graph g, ID value) -> {
        DotAttributes.setFontsizeRaw(g, value);
      };
      setter.apply(DotAttributes.FONTSIZE__GCNE, _function_9);
      final Procedure2<Graph, ID> _function_10 = (Graph g, ID value) -> {
        DotAttributes.setLabelRaw(g, value);
      };
      setter.apply(DotAttributes.LABEL__GCNE, _function_10);
      final Procedure2<Graph, ID> _function_11 = (Graph g, ID value) -> {
        DotAttributes.setLayoutRaw(g, value);
      };
      setter.apply(DotAttributes.LAYOUT__G, _function_11);
      final Procedure2<Graph, ID> _function_12 = (Graph g, ID value) -> {
        DotAttributes.setOutputorderRaw(g, value);
      };
      setter.apply(DotAttributes.OUTPUTORDER__G, _function_12);
      final Procedure2<Graph, ID> _function_13 = (Graph g, ID value) -> {
        DotAttributes.setPagedirRaw(g, value);
      };
      setter.apply(DotAttributes.PAGEDIR__G, _function_13);
      final Procedure2<Graph, ID> _function_14 = (Graph g, ID value) -> {
        DotAttributes.setRankdirRaw(g, value);
      };
      setter.apply(DotAttributes.RANKDIR__G, _function_14);
      final Procedure2<Graph, ID> _function_15 = (Graph g, ID value) -> {
        DotAttributes.setSplinesRaw(g, value);
      };
      setter.apply(DotAttributes.SPLINES__G, _function_15);
      _xblockexpression = graph;
    }
    return _xblockexpression;
  }
  
  private Node transformNodeId(final NodeId it, final Graph.Builder graphBuilder) {
    AttrList _createAttrList = DotFactory.eINSTANCE.createAttrList();
    return this.transformNodeId(it, Collections.<AttrList>unmodifiableList(CollectionLiterals.<AttrList>newArrayList(_createAttrList)), graphBuilder);
  }
  
  private Node transformNodeId(final NodeId it, final List<AttrList> attrLists, final Graph.Builder graphBuilder) {
    Node _xblockexpression = null;
    {
      final boolean isExistingNode = this._createCache_createNode.containsKey(CollectionLiterals.<String>newArrayList(it.getName().toValue()));
      final Node node = this.createNode(it.getName().toValue());
      if ((!isExistingNode)) {
        DotAttributes._setNameRaw(node, it.getName());
        graphBuilder.nodes(node);
      }
      final Procedure2<String, Procedure2<? super Node, ? super ID>> _function = (String attributeName, Procedure2<? super Node, ? super ID> f) -> {
        final ID attributeValue = DotAstHelper.getAttributeValue(attrLists, attributeName);
        if ((attributeValue != null)) {
          f.apply(node, attributeValue);
        } else {
          if (((!isExistingNode) && this.globalNodeAttributes(graphBuilder).containsKey(attributeName))) {
            f.apply(node, this.globalNodeAttributes(graphBuilder).get(attributeName));
          }
        }
      };
      final Procedure2<String, Procedure2<? super Node, ? super ID>> setter = _function;
      final Procedure2<Node, ID> _function_1 = (Node n, ID value) -> {
        DotAttributes.setColorRaw(n, value);
      };
      setter.apply(DotAttributes.COLOR__CNE, _function_1);
      final Procedure2<Node, ID> _function_2 = (Node n, ID value) -> {
        DotAttributes.setColorschemeRaw(n, value);
      };
      setter.apply(DotAttributes.COLORSCHEME__GCNE, _function_2);
      final Procedure2<Node, ID> _function_3 = (Node n, ID value) -> {
        DotAttributes.setDistortionRaw(n, value);
      };
      setter.apply(DotAttributes.DISTORTION__N, _function_3);
      final Procedure2<Node, ID> _function_4 = (Node n, ID value) -> {
        DotAttributes.setFillcolorRaw(n, value);
      };
      setter.apply(DotAttributes.FILLCOLOR__CNE, _function_4);
      final Procedure2<Node, ID> _function_5 = (Node n, ID value) -> {
        DotAttributes.setFixedsizeRaw(n, value);
      };
      setter.apply(DotAttributes.FIXEDSIZE__N, _function_5);
      final Procedure2<Node, ID> _function_6 = (Node n, ID value) -> {
        DotAttributes.setFontcolorRaw(n, value);
      };
      setter.apply(DotAttributes.FONTCOLOR__GCNE, _function_6);
      final Procedure2<Node, ID> _function_7 = (Node n, ID value) -> {
        DotAttributes.setFontnameRaw(n, value);
      };
      setter.apply(DotAttributes.FONTNAME__GCNE, _function_7);
      final Procedure2<Node, ID> _function_8 = (Node n, ID value) -> {
        DotAttributes.setFontsizeRaw(n, value);
      };
      setter.apply(DotAttributes.FONTSIZE__GCNE, _function_8);
      final Procedure2<Node, ID> _function_9 = (Node n, ID value) -> {
        DotAttributes.setHeightRaw(n, value);
      };
      setter.apply(DotAttributes.HEIGHT__N, _function_9);
      final Procedure2<Node, ID> _function_10 = (Node n, ID value) -> {
        DotAttributes.setIdRaw(n, value);
      };
      setter.apply(DotAttributes.ID__GCNE, _function_10);
      final Procedure2<Node, ID> _function_11 = (Node n, ID value) -> {
        DotAttributes.setLabelRaw(n, value);
      };
      setter.apply(DotAttributes.LABEL__GCNE, _function_11);
      final Procedure2<Node, ID> _function_12 = (Node g, ID value) -> {
        DotAttributes.setPenwidthRaw(g, value);
      };
      setter.apply(DotAttributes.PENWIDTH__CNE, _function_12);
      final Procedure2<Node, ID> _function_13 = (Node n, ID value) -> {
        DotAttributes.setPosRaw(n, value);
      };
      setter.apply(DotAttributes.POS__NE, _function_13);
      final Procedure2<Node, ID> _function_14 = (Node n, ID value) -> {
        DotAttributes.setShapeRaw(n, value);
      };
      setter.apply(DotAttributes.SHAPE__N, _function_14);
      final Procedure2<Node, ID> _function_15 = (Node n, ID value) -> {
        DotAttributes.setSidesRaw(n, value);
      };
      setter.apply(DotAttributes.SIDES__N, _function_15);
      final Procedure2<Node, ID> _function_16 = (Node n, ID value) -> {
        DotAttributes.setSkewRaw(n, value);
      };
      setter.apply(DotAttributes.SKEW__N, _function_16);
      final Procedure2<Node, ID> _function_17 = (Node n, ID value) -> {
        DotAttributes.setStyleRaw(n, value);
      };
      setter.apply(DotAttributes.STYLE__GCNE, _function_17);
      final Procedure2<Node, ID> _function_18 = (Node n, ID value) -> {
        DotAttributes.setTooltipRaw(n, value);
      };
      setter.apply(DotAttributes.TOOLTIP__CNE, _function_18);
      final Procedure2<Node, ID> _function_19 = (Node n, ID value) -> {
        DotAttributes.setWidthRaw(n, value);
      };
      setter.apply(DotAttributes.WIDTH__N, _function_19);
      final Procedure2<Node, ID> _function_20 = (Node n, ID value) -> {
        DotAttributes.setXlabelRaw(n, value);
      };
      setter.apply(DotAttributes.XLABEL__NE, _function_20);
      final Procedure2<Node, ID> _function_21 = (Node n, ID value) -> {
        DotAttributes.setXlpRaw(n, value);
      };
      setter.apply(DotAttributes.XLP__NE, _function_21);
      _xblockexpression = node;
    }
    return _xblockexpression;
  }
  
  /**
   * dynamic dispatch methods
   */
  private void _transformStmt(final Stmt it, final Graph.Builder graphBuilder) {
    System.err.println(("DotImport cannot transform Stmt: " + it));
  }
  
  private void _transformStmt(final AttrStmt it, final Graph.Builder graphBuilder) {
    AttributeType _type = it.getType();
    if (_type != null) {
      switch (_type) {
        case GRAPH:
          final Consumer<AttrList> _function = (AttrList it_1) -> {
            final Consumer<Attribute> _function_1 = (Attribute it_2) -> {
              this.globalGraphAttributes(graphBuilder).put(it_2.getName().toValue(), it_2.getValue());
            };
            it_1.getAttributes().forEach(_function_1);
          };
          it.getAttrLists().forEach(_function);
          break;
        case NODE:
          final Consumer<AttrList> _function_1 = (AttrList it_1) -> {
            final Consumer<Attribute> _function_2 = (Attribute it_2) -> {
              this.globalNodeAttributes(graphBuilder).put(it_2.getName().toValue(), it_2.getValue());
            };
            it_1.getAttributes().forEach(_function_2);
          };
          it.getAttrLists().forEach(_function_1);
          break;
        case EDGE:
          final Consumer<AttrList> _function_2 = (AttrList it_1) -> {
            final Consumer<Attribute> _function_3 = (Attribute it_2) -> {
              this.globalEdgeAttributes(graphBuilder).put(it_2.getName().toValue(), it_2.getValue());
            };
            it_1.getAttributes().forEach(_function_3);
          };
          it.getAttrLists().forEach(_function_2);
          break;
        default:
          break;
      }
    }
  }
  
  private void _transformStmt(final NodeStmt it, final Graph.Builder graphBuilder) {
    this.transformNodeId(it.getNode(), it.getAttrLists(), graphBuilder);
  }
  
  private void _transformStmt(final EdgeStmtNode it, final Graph.Builder graphBuilder) {
    Node sourceNode = this.transformNodeId(it.getNode(), graphBuilder);
    EList<EdgeRhs> _edgeRHS = it.getEdgeRHS();
    for (final EdgeRhs edgeRhs : _edgeRHS) {
      boolean _matched = false;
      if (edgeRhs instanceof EdgeRhsNode) {
        _matched=true;
        final Node targetNode = this.transformNodeId(((EdgeRhsNode)edgeRhs).getNode(), graphBuilder);
        graphBuilder.edges(this.createEdge(sourceNode, ((EdgeRhsNode)edgeRhs).getOp().getLiteral(), targetNode, it.getAttrLists(), graphBuilder));
        sourceNode = targetNode;
      }
      if (!_matched) {
        System.err.println(("DotImport cannot transform EdgeStmtNode: " + it));
      }
    }
  }
  
  private void _transformStmt(final Subgraph it, final Graph.Builder graphBuilder) {
    final boolean isExistingSubgraph = ((it.getName() != null) && this._createCache_createSubgraph.containsKey(CollectionLiterals.<String>newArrayList(it.getName().toValue())));
    final Graph.Builder subgraphBuilder = new Graph.Builder();
    Node _xifexpression = null;
    ID _name = it.getName();
    boolean _tripleEquals = (_name == null);
    if (_tripleEquals) {
      _xifexpression = this.createSubgraph(Integer.valueOf(System.identityHashCode(subgraphBuilder)).toString());
    } else {
      _xifexpression = this.createSubgraph(it.getName().toValue());
    }
    final Node subgraphNode = _xifexpression;
    ID _name_1 = it.getName();
    boolean _tripleNotEquals = (_name_1 != null);
    if (_tripleNotEquals) {
      subgraphBuilder.attr(DotAttributes._NAME__GNE, it.getName());
    }
    this.globalGraphAttributes(subgraphBuilder).putAll(this.globalGraphAttributes(graphBuilder));
    this.globalNodeAttributes(subgraphBuilder).putAll(this.globalNodeAttributes(graphBuilder));
    this.globalEdgeAttributes(subgraphBuilder).putAll(this.globalEdgeAttributes(graphBuilder));
    final Function1<Stmt, Boolean> _function = (Stmt it_1) -> {
      return Boolean.valueOf((!(it_1 instanceof Attribute)));
    };
    final Consumer<Stmt> _function_1 = (Stmt it_1) -> {
      this.transformStmt(it_1, subgraphBuilder);
    };
    IterableExtensions.<Stmt>filter(it.getStmts(), _function).forEach(_function_1);
    final Graph subgraph = subgraphBuilder.build();
    if ((!isExistingSubgraph)) {
      subgraphNode.setNestedGraph(subgraph);
      subgraph.setNestingNode(subgraphNode);
      graphBuilder.nodes(subgraphNode);
    } else {
      subgraphNode.getNestedGraph().getAttributes().putAll(subgraph.getAttributes());
      final Function1<Node, Boolean> _function_2 = (Node it_1) -> {
        boolean _contains = subgraphNode.getNestedGraph().getNodes().contains(it_1);
        return Boolean.valueOf((!_contains));
      };
      Iterables.<Node>addAll(subgraphNode.getNestedGraph().getNodes(), IterableExtensions.<Node>filter(subgraph.getNodes(), _function_2));
      final Function1<Edge, Boolean> _function_3 = (Edge it_1) -> {
        boolean _contains = subgraphNode.getNestedGraph().getNodes().contains(it_1);
        return Boolean.valueOf((!_contains));
      };
      Iterables.<Edge>addAll(subgraphNode.getNestedGraph().getEdges(), IterableExtensions.<Edge>filter(subgraph.getEdges(), _function_3));
    }
    final Procedure2<String, Procedure2<? super Graph, ? super ID>> _function_4 = (String attributeName, Procedure2<? super Graph, ? super ID> f) -> {
      final ID attributeValue = DotAstHelper.getAttributeValue(it, attributeName);
      if ((attributeValue != null)) {
        f.apply(subgraph, attributeValue);
      } else {
        boolean _containsKey = this.globalGraphAttributes(subgraphBuilder).containsKey(attributeName);
        if (_containsKey) {
          f.apply(subgraph, this.globalGraphAttributes(subgraphBuilder).get(attributeName));
        }
      }
    };
    final Procedure2<String, Procedure2<? super Graph, ? super ID>> setter = _function_4;
    final Procedure2<Graph, ID> _function_5 = (Graph g, ID value) -> {
      DotAttributes.setBbRaw(g, value);
    };
    setter.apply(DotAttributes.BB__GC, _function_5);
    final Procedure2<Graph, ID> _function_6 = (Graph g, ID value) -> {
      DotAttributes.setBgcolorRaw(g, value);
    };
    setter.apply(DotAttributes.BGCOLOR__GC, _function_6);
    final Procedure2<Graph, ID> _function_7 = (Graph g, ID value) -> {
      DotAttributes.setColorRaw(g, value);
    };
    setter.apply(DotAttributes.COLOR__CNE, _function_7);
    final Procedure2<Graph, ID> _function_8 = (Graph g, ID value) -> {
      DotAttributes.setColorschemeRaw(g, value);
    };
    setter.apply(DotAttributes.COLORSCHEME__GCNE, _function_8);
    final Procedure2<Graph, ID> _function_9 = (Graph g, ID value) -> {
      DotAttributes.setFillcolorRaw(g, value);
    };
    setter.apply(DotAttributes.FILLCOLOR__CNE, _function_9);
    final Procedure2<Graph, ID> _function_10 = (Graph g, ID value) -> {
      DotAttributes.setFontcolorRaw(g, value);
    };
    setter.apply(DotAttributes.FONTCOLOR__GCNE, _function_10);
    final Procedure2<Graph, ID> _function_11 = (Graph g, ID value) -> {
      DotAttributes.setFontnameRaw(g, value);
    };
    setter.apply(DotAttributes.FONTNAME__GCNE, _function_11);
    final Procedure2<Graph, ID> _function_12 = (Graph g, ID value) -> {
      DotAttributes.setFontsizeRaw(g, value);
    };
    setter.apply(DotAttributes.FONTSIZE__GCNE, _function_12);
    final Procedure2<Graph, ID> _function_13 = (Graph g, ID value) -> {
      DotAttributes.setIdRaw(g, value);
    };
    setter.apply(DotAttributes.ID__GCNE, _function_13);
    final Procedure2<Graph, ID> _function_14 = (Graph g, ID value) -> {
      DotAttributes.setLabelRaw(g, value);
    };
    setter.apply(DotAttributes.LABEL__GCNE, _function_14);
    final Procedure2<Graph, ID> _function_15 = (Graph g, ID value) -> {
      DotAttributes.setLpRaw(g, value);
    };
    setter.apply(DotAttributes.LP__GCE, _function_15);
    final Procedure2<Graph, ID> _function_16 = (Graph g, ID value) -> {
      DotAttributes.setRankRaw(g, value);
    };
    setter.apply(DotAttributes.RANK__S, _function_16);
    final Procedure2<Graph, ID> _function_17 = (Graph g, ID value) -> {
      DotAttributes.setPenwidthRaw(g, value);
    };
    setter.apply(DotAttributes.PENWIDTH__CNE, _function_17);
    final Procedure2<Graph, ID> _function_18 = (Graph g, ID value) -> {
      DotAttributes.setStyleRaw(g, value);
    };
    setter.apply(DotAttributes.STYLE__GCNE, _function_18);
    final Procedure2<Graph, ID> _function_19 = (Graph g, ID value) -> {
      DotAttributes.setTooltipRaw(g, value);
    };
    setter.apply(DotAttributes.TOOLTIP__CNE, _function_19);
  }
  
  private Node createSubgraph(final String subgraphName) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(subgraphName);
    final Node _result;
    synchronized (_createCache_createSubgraph) {
      if (_createCache_createSubgraph.containsKey(_cacheKey)) {
        return _createCache_createSubgraph.get(_cacheKey);
      }
      Node _buildNode = new Node.Builder().buildNode();
      _result = _buildNode;
      _createCache_createSubgraph.put(_cacheKey, _result);
    }
    _init_createSubgraph(_result, subgraphName);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, Node> _createCache_createSubgraph = CollectionLiterals.newHashMap();
  
  private void _init_createSubgraph(final Node it, final String subgraphName) {
  }
  
  private Node createNode(final String nodeName) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(nodeName);
    final Node _result;
    synchronized (_createCache_createNode) {
      if (_createCache_createNode.containsKey(_cacheKey)) {
        return _createCache_createNode.get(_cacheKey);
      }
      Node _buildNode = new Node.Builder().buildNode();
      _result = _buildNode;
      _createCache_createNode.put(_cacheKey, _result);
    }
    _init_createNode(_result, nodeName);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, Node> _createCache_createNode = CollectionLiterals.newHashMap();
  
  private void _init_createNode(final Node it, final String nodeName) {
  }
  
  private Edge createEdge(final Node sourceNode, final String edgeOp, final Node targetNode, final List<AttrList> attrLists, final Graph.Builder graphBuilder) {
    Edge _xblockexpression = null;
    {
      final Edge edge = new Edge.Builder(sourceNode, targetNode).buildEdge();
      final Procedure2<String, Procedure2<? super Edge, ? super ID>> _function = (String attributeName, Procedure2<? super Edge, ? super ID> f) -> {
        final ID attributeValue = DotAstHelper.getAttributeValue(attrLists, attributeName);
        if ((attributeValue != null)) {
          f.apply(edge, attributeValue);
        } else {
          boolean _containsKey = this.globalEdgeAttributes(graphBuilder).containsKey(attributeName);
          if (_containsKey) {
            f.apply(edge, this.globalEdgeAttributes(graphBuilder).get(attributeName));
          }
        }
      };
      final Procedure2<String, Procedure2<? super Edge, ? super ID>> setter = _function;
      final Procedure2<Edge, ID> _function_1 = (Edge e, ID value) -> {
        DotAttributes.setArrowheadRaw(e, value);
      };
      setter.apply(DotAttributes.ARROWHEAD__E, _function_1);
      final Procedure2<Edge, ID> _function_2 = (Edge e, ID value) -> {
        DotAttributes.setArrowsizeRaw(e, value);
      };
      setter.apply(DotAttributes.ARROWSIZE__E, _function_2);
      final Procedure2<Edge, ID> _function_3 = (Edge e, ID value) -> {
        DotAttributes.setArrowtailRaw(e, value);
      };
      setter.apply(DotAttributes.ARROWTAIL__E, _function_3);
      final Procedure2<Edge, ID> _function_4 = (Edge e, ID value) -> {
        DotAttributes.setColorRaw(e, value);
      };
      setter.apply(DotAttributes.COLOR__CNE, _function_4);
      final Procedure2<Edge, ID> _function_5 = (Edge e, ID value) -> {
        DotAttributes.setColorschemeRaw(e, value);
      };
      setter.apply(DotAttributes.COLORSCHEME__GCNE, _function_5);
      final Procedure2<Edge, ID> _function_6 = (Edge e, ID value) -> {
        DotAttributes.setDirRaw(e, value);
      };
      setter.apply(DotAttributes.DIR__E, _function_6);
      final Procedure2<Edge, ID> _function_7 = (Edge e, ID value) -> {
        DotAttributes.setEdgetooltipRaw(e, value);
      };
      setter.apply(DotAttributes.EDGETOOLTIP__E, _function_7);
      final Procedure2<Edge, ID> _function_8 = (Edge e, ID value) -> {
        DotAttributes.setFillcolorRaw(e, value);
      };
      setter.apply(DotAttributes.FILLCOLOR__CNE, _function_8);
      final Procedure2<Edge, ID> _function_9 = (Edge e, ID value) -> {
        DotAttributes.setFontcolorRaw(e, value);
      };
      setter.apply(DotAttributes.FONTCOLOR__GCNE, _function_9);
      final Procedure2<Edge, ID> _function_10 = (Edge e, ID value) -> {
        DotAttributes.setFontnameRaw(e, value);
      };
      setter.apply(DotAttributes.FONTNAME__GCNE, _function_10);
      final Procedure2<Edge, ID> _function_11 = (Edge e, ID value) -> {
        DotAttributes.setFontsizeRaw(e, value);
      };
      setter.apply(DotAttributes.FONTSIZE__GCNE, _function_11);
      final Procedure2<Edge, ID> _function_12 = (Edge e, ID value) -> {
        DotAttributes.setHeadLpRaw(e, value);
      };
      setter.apply(DotAttributes.HEAD_LP__E, _function_12);
      final Procedure2<Edge, ID> _function_13 = (Edge e, ID value) -> {
        DotAttributes.setHeadlabelRaw(e, value);
      };
      setter.apply(DotAttributes.HEADLABEL__E, _function_13);
      final Procedure2<Edge, ID> _function_14 = (Edge e, ID value) -> {
        DotAttributes.setHeadportRaw(e, value);
      };
      setter.apply(DotAttributes.HEADPORT__E, _function_14);
      final Procedure2<Edge, ID> _function_15 = (Edge e, ID value) -> {
        DotAttributes.setHeadtooltipRaw(e, value);
      };
      setter.apply(DotAttributes.HEADTOOLTIP__E, _function_15);
      final Procedure2<Edge, ID> _function_16 = (Edge e, ID value) -> {
        DotAttributes.setIdRaw(e, value);
      };
      setter.apply(DotAttributes.ID__GCNE, _function_16);
      final Procedure2<Edge, ID> _function_17 = (Edge e, ID value) -> {
        DotAttributes.setLabelRaw(e, value);
      };
      setter.apply(DotAttributes.LABEL__GCNE, _function_17);
      final Procedure2<Edge, ID> _function_18 = (Edge e, ID value) -> {
        DotAttributes.setLabelfontcolorRaw(e, value);
      };
      setter.apply(DotAttributes.LABELFONTCOLOR__E, _function_18);
      final Procedure2<Edge, ID> _function_19 = (Edge e, ID value) -> {
        DotAttributes.setLabelfontnameRaw(e, value);
      };
      setter.apply(DotAttributes.LABELFONTNAME__E, _function_19);
      final Procedure2<Edge, ID> _function_20 = (Edge e, ID value) -> {
        DotAttributes.setLabelfontsizeRaw(e, value);
      };
      setter.apply(DotAttributes.LABELFONTSIZE__E, _function_20);
      final Procedure2<Edge, ID> _function_21 = (Edge e, ID value) -> {
        DotAttributes.setLabeltooltipRaw(e, value);
      };
      setter.apply(DotAttributes.LABELTOOLTIP__E, _function_21);
      final Procedure2<Edge, ID> _function_22 = (Edge e, ID value) -> {
        DotAttributes.setLpRaw(e, value);
      };
      setter.apply(DotAttributes.LP__GCE, _function_22);
      final Procedure2<Edge, ID> _function_23 = (Edge g, ID value) -> {
        DotAttributes.setPenwidthRaw(g, value);
      };
      setter.apply(DotAttributes.PENWIDTH__CNE, _function_23);
      final Procedure2<Edge, ID> _function_24 = (Edge e, ID value) -> {
        DotAttributes.setPosRaw(e, value);
      };
      setter.apply(DotAttributes.POS__NE, _function_24);
      final Procedure2<Edge, ID> _function_25 = (Edge e, ID value) -> {
        DotAttributes.setStyleRaw(e, value);
      };
      setter.apply(DotAttributes.STYLE__GCNE, _function_25);
      final Procedure2<Edge, ID> _function_26 = (Edge e, ID value) -> {
        DotAttributes.setTaillabelRaw(e, value);
      };
      setter.apply(DotAttributes.TAILLABEL__E, _function_26);
      final Procedure2<Edge, ID> _function_27 = (Edge e, ID value) -> {
        DotAttributes.setTailportRaw(e, value);
      };
      setter.apply(DotAttributes.TAILPORT__E, _function_27);
      final Procedure2<Edge, ID> _function_28 = (Edge e, ID value) -> {
        DotAttributes.setTailtooltipRaw(e, value);
      };
      setter.apply(DotAttributes.TAILTOOLTIP__E, _function_28);
      final Procedure2<Edge, ID> _function_29 = (Edge e, ID value) -> {
        DotAttributes.setTailLpRaw(e, value);
      };
      setter.apply(DotAttributes.TAIL_LP__E, _function_29);
      final Procedure2<Edge, ID> _function_30 = (Edge e, ID value) -> {
        DotAttributes.setTooltipRaw(e, value);
      };
      setter.apply(DotAttributes.TOOLTIP__CNE, _function_30);
      final Procedure2<Edge, ID> _function_31 = (Edge e, ID value) -> {
        DotAttributes.setXlabelRaw(e, value);
      };
      setter.apply(DotAttributes.XLABEL__NE, _function_31);
      final Procedure2<Edge, ID> _function_32 = (Edge e, ID value) -> {
        DotAttributes.setXlpRaw(e, value);
      };
      setter.apply(DotAttributes.XLP__NE, _function_32);
      _xblockexpression = edge;
    }
    return _xblockexpression;
  }
  
  private void transformStmt(final Stmt it, final Graph.Builder graphBuilder) {
    if (it instanceof Subgraph) {
      _transformStmt((Subgraph)it, graphBuilder);
      return;
    } else if (it instanceof AttrStmt) {
      _transformStmt((AttrStmt)it, graphBuilder);
      return;
    } else if (it instanceof EdgeStmtNode) {
      _transformStmt((EdgeStmtNode)it, graphBuilder);
      return;
    } else if (it instanceof NodeStmt) {
      _transformStmt((NodeStmt)it, graphBuilder);
      return;
    } else if (it != null) {
      _transformStmt(it, graphBuilder);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(it, graphBuilder).toString());
    }
  }
}
