"use strict";
/********************************************************************************
 * Copyright (c) 2021 STMicroelectronics and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 *******************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.getClangdContextsConfig = exports.COMPILE_COMMANDS_FILE = void 0;
const fs_1 = require("fs");
const path = require("path");
const _1 = require(".");
const file_util_1 = require("./util/file-util");
exports.COMPILE_COMMANDS_FILE = 'compile_commands.json';
const CONFIG_FILE = '.clangd-contexts';
class ClangdContextsConfigImpl {
    constructor(clangdContextsFile) {
        try {
            const data = JSON.parse(fs_1.readFileSync(clangdContextsFile, 'utf8'));
            Object.assign(this, data);
        }
        catch (e) {
            console.log(`Failed to load ${clangdContextsFile}:`, e);
            throw e;
        }
        this.path = path.dirname(clangdContextsFile);
        this.projects.forEach(this.initProjectConfig.bind(this));
    }
    /**
     * Initialize the defaults in a project configuration.
     *
     * @param projectConfig the context configuration to initialize
     * @returns the initialized context configuration
     */
    initProjectConfig(projectConfig) {
        if (!projectConfig.contextDirs) {
            projectConfig.contextDirs = 'flat';
        }
        return projectConfig;
    }
    getClangdProjects() {
        return this.projects.map(proj => path.resolve(this.path, proj.path));
    }
    getClangdContexts() {
        const result = new Set();
        this.projects.forEach(proj => {
            const projPath = path.resolve(this.path, proj.path);
            if (proj.contextDirs === 'flat') {
                this.collectFlatClangdContexts(projPath, result);
            }
            else {
                this.collectNestedClangdContexts(projPath, result);
            }
        });
        return Array.from(result);
    }
    getClangdConfig(projectConfig) {
        return _1.loadOrCreate(path.join(this.path, projectConfig.path));
    }
    toContextDir(projectConfig, contextName) {
        return path.join(this.path, projectConfig.path, contextName);
    }
    collectFlatClangdContexts(projectDir, contexts) {
        const listing = file_util_1.isDirectory(projectDir) ? fs_1.readdirSync(projectDir) : [];
        listing.forEach(entry => {
            const contextDir = path.join(projectDir, entry);
            if (file_util_1.isDirectory(contextDir)) {
                // Does it have a compilation database?
                const ccFile = path.join(contextDir, exports.COMPILE_COMMANDS_FILE);
                if (fs_1.existsSync(ccFile)) {
                    contexts.add(entry);
                }
            }
        });
    }
    collectNestedClangdContexts(projectDir, contexts) {
        this.collectNestedClangdContextsRecursive(projectDir, [], contexts);
    }
    collectNestedClangdContextsRecursive(projectRoot, contextPath, contexts) {
        const possibleContextDir = path.join(...contextPath);
        const qualifiedContextDir = path.join(projectRoot, ...contextPath);
        // Use a uniform, POSIX-like segmentation of the path on all platforms, including Windows
        const possibleContextName = file_util_1.toPortablePath(possibleContextDir);
        // Check for compilation database in a context below the root. At the root, the context name would be empty
        const ccFile = path.join(qualifiedContextDir, exports.COMPILE_COMMANDS_FILE);
        if (contextPath.length > 0 && fs_1.existsSync(ccFile)) {
            // This is a context directory
            contexts.add(possibleContextName);
        }
        if (file_util_1.isDirectory(qualifiedContextDir)) {
            // Look deeper for context directories
            const listing = fs_1.readdirSync(qualifiedContextDir);
            listing.forEach(entry => {
                const nestedDir = path.join(qualifiedContextDir, entry);
                if (file_util_1.isDirectory(nestedDir)) {
                    this.collectNestedClangdContextsRecursive(projectRoot, [...contextPath, entry], contexts);
                }
            });
        }
    }
}
/**
 * Find and load the `.clangd-contexts` file that configures clangd contexts for projects in the given
 * `directory`. Searches that `directory` and up its parent chain until a `.clangd-contexts`
 * file is found or else the search is exhausted.
 *
 * @param directory the directory in which to search for the `.clangd-contexts` file
 * @returns the loaded `.clangd-contexts` configuration if it is found
 */
function getClangdContextsConfig(directory) {
    for (;;) {
        const maybeConfig = path.join(directory, CONFIG_FILE);
        if (fs_1.existsSync(maybeConfig)) {
            const result = new ClangdContextsConfigImpl(maybeConfig);
            return result;
        }
        // Continue searching up the directory hierarchy
        const parent = path.dirname(directory);
        if (parent === directory) {
            // This happens at the root
            break;
        }
        else {
            directory = parent;
        }
    }
    return undefined;
}
exports.getClangdContextsConfig = getClangdContextsConfig;
//# sourceMappingURL=clangd-contexts-config.js.map