/**
 * Copyright (c) 2015, 2021, 2023 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 * Christian W. Damus - Initial API and implementation
 * Vincent Lorenzo (CEA LIST) - vincent.lorenzo@cea.fr - bug 569174, 570944
 * Ansgar Radermacher - bug 582492, move to com.google.inject
 */
package org.eclipse.papyrus.dev.assistants.codegen.generator;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.function.Consumer;
import org.eclipse.papyrus.gmf.codegen.gmfgen.ElementType;
import org.eclipse.papyrus.gmf.codegen.gmfgen.GenEditorGenerator;
import org.eclipse.papyrus.gmf.codegen.gmfgen.GenLink;
import org.eclipse.papyrus.gmf.codegen.gmfgen.GenNode;
import org.eclipse.papyrus.infra.gmfdiag.assistant.AssistantFactory;
import org.eclipse.papyrus.infra.gmfdiag.assistant.ConnectionAssistant;
import org.eclipse.papyrus.infra.gmfdiag.assistant.ModelingAssistantProvider;
import org.eclipse.papyrus.infra.gmfdiag.assistant.PopupAssistant;
import org.eclipse.papyrus.infra.types.ElementTypeConfiguration;
import org.eclipse.papyrus.infra.types.ElementTypeSetConfiguration;
import org.eclipse.papyrus.uml.profile.types.generator.UMLElementTypes;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Transformation rule for generating a {@link ModelingAssistantProvider} from a UML {@link Profile}.
 */
@Singleton
@SuppressWarnings("all")
public class ModelingAssistantProviderRule extends org.eclipse.papyrus.uml.profile.assistants.generator.ModelingAssistantProviderRule {
  @Extension
  private static AssistantFactory assistantFactory = AssistantFactory.eINSTANCE;

  @Inject
  @Extension
  private ElementTypeToAssistantRule _elementTypeToAssistantRule;

  @Inject
  @Extension
  private GMFGenToAssistantRule _gMFGenToAssistantRule;

  @Inject
  @Extension
  private UMLElementTypes _uMLElementTypes;

  public ModelingAssistantProvider toModelingAssistantProvider(final ElementTypeSetConfiguration elementTypeSet) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(elementTypeSet);
    final ModelingAssistantProvider _result;
    synchronized (_createCache_toModelingAssistantProvider) {
      if (_createCache_toModelingAssistantProvider.containsKey(_cacheKey)) {
        return _createCache_toModelingAssistantProvider.get(_cacheKey);
      }
      ModelingAssistantProvider _createModelingAssistantProvider = ModelingAssistantProviderRule.assistantFactory.createModelingAssistantProvider();
      _result = _createModelingAssistantProvider;
      _createCache_toModelingAssistantProvider.put(_cacheKey, _result);
    }
    _init_toModelingAssistantProvider(_result, elementTypeSet);
    return _result;
  }

  private final HashMap<ArrayList<?>, ModelingAssistantProvider> _createCache_toModelingAssistantProvider = CollectionLiterals.newHashMap();

  private void _init_toModelingAssistantProvider(final ModelingAssistantProvider it, final ElementTypeSetConfiguration elementTypeSet) {
    it.setName(elementTypeSet.getName());
    final Consumer<ElementTypeConfiguration> _function = (ElementTypeConfiguration type) -> {
      boolean _isRelationship = this._uMLElementTypes.isRelationship(type);
      boolean _not = (!_isRelationship);
      if (_not) {
        it.getElementTypeIDs().add(type.getIdentifier());
        it.getPopupAssistants().add(this._elementTypeToAssistantRule.toPopupAssistant(type));
      } else {
        it.getRelationshipTypeIDs().add(type.getIdentifier());
        it.getConnectionAssistants().add(this._elementTypeToAssistantRule.toConnectionAssistant(type));
      }
    };
    elementTypeSet.getElementTypeConfigurations().forEach(_function);
  }

  public ModelingAssistantProvider toModelingAssistantProvider(final GenEditorGenerator editor) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(editor);
    final ModelingAssistantProvider _result;
    synchronized (_createCache_toModelingAssistantProvider_1) {
      if (_createCache_toModelingAssistantProvider_1.containsKey(_cacheKey)) {
        return _createCache_toModelingAssistantProvider_1.get(_cacheKey);
      }
      ModelingAssistantProvider _createModelingAssistantProvider = ModelingAssistantProviderRule.assistantFactory.createModelingAssistantProvider();
      _result = _createModelingAssistantProvider;
      _createCache_toModelingAssistantProvider_1.put(_cacheKey, _result);
    }
    _init_toModelingAssistantProvider_1(_result, editor);
    return _result;
  }

  private final HashMap<ArrayList<?>, ModelingAssistantProvider> _createCache_toModelingAssistantProvider_1 = CollectionLiterals.newHashMap();

  private void _init_toModelingAssistantProvider_1(final ModelingAssistantProvider it, final GenEditorGenerator editor) {
    it.setName(editor.getModelID());
    final Function1<GenNode, String> _function = (GenNode it_1) -> {
      ElementType _elementType = it_1.getElementType();
      String _uniqueIdentifier = null;
      if (_elementType!=null) {
        _uniqueIdentifier=_elementType.getUniqueIdentifier();
      }
      return _uniqueIdentifier;
    };
    Iterables.<String>addAll(it.getElementTypeIDs(), IterableExtensions.<String>filterNull(IterableExtensions.<GenNode, String>map(this._gMFGenToAssistantRule.validNodes(editor.getDiagram()), _function)));
    final Function1<GenLink, String> _function_1 = (GenLink it_1) -> {
      ElementType _elementType = it_1.getElementType();
      String _uniqueIdentifier = null;
      if (_elementType!=null) {
        _uniqueIdentifier=_elementType.getUniqueIdentifier();
      }
      return _uniqueIdentifier;
    };
    Iterables.<String>addAll(it.getRelationshipTypeIDs(), IterableExtensions.<String>filterNull(IterableExtensions.<GenLink, String>map(this._gMFGenToAssistantRule.validLinks(editor.getDiagram()), _function_1)));
    final Function1<GenNode, PopupAssistant> _function_2 = (GenNode it_1) -> {
      return this._gMFGenToAssistantRule.toPopupAssistant(it_1);
    };
    Iterables.<PopupAssistant>addAll(it.getPopupAssistants(), IterableExtensions.<GenNode, PopupAssistant>map(this._gMFGenToAssistantRule.validNodes(editor.getDiagram()), _function_2));
    final Function1<GenLink, ConnectionAssistant> _function_3 = (GenLink it_1) -> {
      return this._gMFGenToAssistantRule.toConnectionAssistant(it_1);
    };
    Iterables.<ConnectionAssistant>addAll(it.getConnectionAssistants(), IterableExtensions.<GenLink, ConnectionAssistant>map(this._gMFGenToAssistantRule.validLinks(editor.getDiagram()), _function_3));
  }
}
