/**
 * Copyright (c) 2014, 2015, 2023 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 * Christian W. Damus - Initial API and implementation
 * Ansgar Radermacher - bug 582492, move to com.google.inject
 */
package org.eclipse.papyrus.uml.profile.assistants.generator;

import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.util.ArrayList;
import java.util.HashMap;
import org.eclipse.papyrus.infra.filters.Filter;
import org.eclipse.papyrus.infra.gmfdiag.assistant.AssistantFactory;
import org.eclipse.papyrus.infra.gmfdiag.assistant.ConnectionAssistant;
import org.eclipse.papyrus.infra.gmfdiag.assistant.ElementTypeFilter;
import org.eclipse.papyrus.infra.types.ElementTypeConfiguration;
import org.eclipse.papyrus.uml.profile.types.generator.Identifiers;
import org.eclipse.papyrus.uml.profile.types.generator.ImpliedExtension;
import org.eclipse.papyrus.uml.profile.types.generator.UML;
import org.eclipse.papyrus.uml.profile.types.generator.UMLElementTypes;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.Functions.Function2;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Transformation rule for generating a {@link ConnectionAssistant} from a UML {@link Extension}.
 */
@Singleton
@SuppressWarnings("all")
public class ConnectionAssistantRule {
  @Extension
  private static AssistantFactory assistantFactory = AssistantFactory.eINSTANCE;

  @Inject
  @Extension
  private Identifiers _identifiers;

  @Inject
  @Extension
  private UMLElementTypes _uMLElementTypes;

  @Inject
  @Extension
  private FiltersUtil _filtersUtil;

  @Inject
  @Extension
  private UML _uML;

  public ConnectionAssistant toConnectionAssistant(final ImpliedExtension umlExtension, final ElementTypeConfiguration basetype) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(umlExtension, basetype);
    final ConnectionAssistant _result;
    synchronized (_createCache_toConnectionAssistant) {
      if (_createCache_toConnectionAssistant.containsKey(_cacheKey)) {
        return _createCache_toConnectionAssistant.get(_cacheKey);
      }
      ConnectionAssistant _createConnectionAssistant = ConnectionAssistantRule.assistantFactory.createConnectionAssistant();
      _result = _createConnectionAssistant;
      _createCache_toConnectionAssistant.put(_cacheKey, _result);
    }
    _init_toConnectionAssistant(_result, umlExtension, basetype);
    return _result;
  }

  private final HashMap<ArrayList<?>, ConnectionAssistant> _createCache_toConnectionAssistant = CollectionLiterals.newHashMap();

  private void _init_toConnectionAssistant(final ConnectionAssistant it, final ImpliedExtension umlExtension, final ElementTypeConfiguration basetype) {
    it.setElementTypeID(this._identifiers.toElementTypeID(umlExtension, basetype));
    it.setOwnedSourceFilter(this._filtersUtil.andProfileApplied(this._filtersUtil.reduce(this.createPossibleSourcesFilter(basetype, umlExtension)), this._uML.getProfile(umlExtension)));
    it.setOwnedTargetFilter(this._filtersUtil.andProfileApplied(this._filtersUtil.reduce(this.createPossibleTargetsFilter(basetype, umlExtension)), this._uML.getProfile(umlExtension)));
  }

  private Filter createPossibleSourcesFilter(final ElementTypeConfiguration basetype, final ImpliedExtension umlExtension) {
    final Function1<ElementTypeConfiguration, Boolean> _function = (ElementTypeConfiguration it) -> {
      boolean _isRelationship = this._uMLElementTypes.isRelationship(it);
      return Boolean.valueOf((!_isRelationship));
    };
    final Function2<Filter, ElementTypeConfiguration, Filter> _function_1 = (Filter filter, ElementTypeConfiguration elementType) -> {
      Filter _xifexpression = null;
      boolean _canSourceToType = this._uMLElementTypes.canSourceToType(elementType, basetype);
      if (_canSourceToType) {
        ElementTypeFilter _filter = this._filtersUtil.toFilter(elementType, this._uML.getProfile(umlExtension));
        _xifexpression = this._filtersUtil.operator_or(filter, _filter);
      } else {
        _xifexpression = filter;
      }
      return _xifexpression;
    };
    return IterableExtensions.<ElementTypeConfiguration, Filter>fold(IterableExtensions.<ElementTypeConfiguration>filter(this._uMLElementTypes.getBaseElementTypes(), _function), null, _function_1);
  }

  private Filter createPossibleTargetsFilter(final ElementTypeConfiguration basetype, final ImpliedExtension umlExtension) {
    final Function1<ElementTypeConfiguration, Boolean> _function = (ElementTypeConfiguration it) -> {
      boolean _isRelationship = this._uMLElementTypes.isRelationship(it);
      return Boolean.valueOf((!_isRelationship));
    };
    final Function2<Filter, ElementTypeConfiguration, Filter> _function_1 = (Filter filter, ElementTypeConfiguration elementType) -> {
      Filter _xifexpression = null;
      boolean _canTargetFromType = this._uMLElementTypes.canTargetFromType(elementType, basetype);
      if (_canTargetFromType) {
        ElementTypeFilter _filter = this._filtersUtil.toFilter(elementType, this._uML.getProfile(umlExtension));
        _xifexpression = this._filtersUtil.operator_or(filter, _filter);
      } else {
        _xifexpression = filter;
      }
      return _xifexpression;
    };
    return IterableExtensions.<ElementTypeConfiguration, Filter>fold(IterableExtensions.<ElementTypeConfiguration>filter(this._uMLElementTypes.getBaseElementTypes(), _function), null, _function_1);
  }
}
