/**
 * Copyright (c) 2014, 2015, 2023 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 * Christian W. Damus - Initial API and implementation
 * Ansgar Radermacher - bug 582492, move to com.google.inject
 */
package org.eclipse.papyrus.uml.profile.types.generator;

import com.google.common.collect.Iterables;
import com.google.inject.Singleton;
import java.util.Arrays;
import java.util.Collections;
import java.util.Set;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Namespace;
import org.eclipse.uml2.uml.Profile;
import org.eclipse.uml2.uml.Stereotype;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * Utility extensions for working with UML models and elements.
 */
@Singleton
@SuppressWarnings("all")
public class UML {
  private final Set<EClass> relationshipLikeUMLMetaclasses = Collections.<EClass>unmodifiableSet(CollectionLiterals.<EClass>newHashSet(UMLPackage.Literals.RELATIONSHIP, UMLPackage.Literals.GENERALIZATION_SET, UMLPackage.Literals.CONNECTOR, UMLPackage.Literals.ACTIVITY_EDGE, UMLPackage.Literals.TRANSITION, UMLPackage.Literals.MESSAGE));

  protected Profile _getProfile(final Profile element) {
    return element;
  }

  protected Profile _getProfile(final NamedElement element) {
    return IterableExtensions.<Profile>head(Iterables.<Profile>filter(element.allOwningPackages(), Profile.class));
  }

  protected Profile _getProfile(final ImpliedExtension umlExtension) {
    return umlExtension.getStereotype().getProfile();
  }

  protected Profile _getRootProfile(final Profile element) {
    Profile _elvis = null;
    Namespace _namespace = element.getNamespace();
    Profile _rootProfile = null;
    if (_namespace!=null) {
      _rootProfile=this.getRootProfile(_namespace);
    }
    if (_rootProfile != null) {
      _elvis = _rootProfile;
    } else {
      _elvis = element;
    }
    return _elvis;
  }

  protected Profile _getRootProfile(final NamedElement element) {
    return IterableExtensions.<Profile>last(Iterables.<Profile>filter(element.allOwningPackages(), Profile.class));
  }

  public Iterable<ImpliedExtension> getAllExtensions(final org.eclipse.uml2.uml.Package package_) {
    final Function1<Stereotype, Iterable<ImpliedExtension>> _function = (Stereotype it) -> {
      return this.impliedExtensions(it);
    };
    Iterable<ImpliedExtension> _flatten = Iterables.<ImpliedExtension>concat(IterableExtensions.<Stereotype, Iterable<ImpliedExtension>>map(Iterables.<Stereotype>filter(package_.getOwnedTypes(), Stereotype.class), _function));
    final Function1<org.eclipse.uml2.uml.Package, Iterable<ImpliedExtension>> _function_1 = (org.eclipse.uml2.uml.Package it) -> {
      return this.getAllExtensions(it);
    };
    Iterable<ImpliedExtension> _flatten_1 = Iterables.<ImpliedExtension>concat(ListExtensions.<org.eclipse.uml2.uml.Package, Iterable<ImpliedExtension>>map(package_.getNestedPackages(), _function_1));
    return Iterables.<ImpliedExtension>concat(_flatten, _flatten_1);
  }

  public Iterable<Stereotype> getAllStereotypes(final org.eclipse.uml2.uml.Package package_) {
    Iterable<Stereotype> _filter = Iterables.<Stereotype>filter(package_.getOwnedTypes(), Stereotype.class);
    final Function1<org.eclipse.uml2.uml.Package, Iterable<Stereotype>> _function = (org.eclipse.uml2.uml.Package it) -> {
      return this.getAllStereotypes(it);
    };
    Iterable<Stereotype> _flatten = Iterables.<Stereotype>concat(ListExtensions.<org.eclipse.uml2.uml.Package, Iterable<Stereotype>>map(package_.getNestedPackages(), _function));
    return Iterables.<Stereotype>concat(_filter, _flatten);
  }

  public Iterable<ImpliedExtension> impliedExtensions(final Stereotype stereotype) {
    final Function1<org.eclipse.uml2.uml.Class, ImpliedExtension> _function = (org.eclipse.uml2.uml.Class it) -> {
      return new ImpliedExtension(stereotype, it);
    };
    return ListExtensions.<org.eclipse.uml2.uml.Class, ImpliedExtension>map(stereotype.getAllExtendedMetaclasses(), _function);
  }

  public boolean isRelationship(final EClass eClass) {
    final Function1<EClass, Boolean> _function = (EClass it) -> {
      return Boolean.valueOf(it.isSuperTypeOf(eClass));
    };
    return IterableExtensions.<EClass>exists(this.relationshipLikeUMLMetaclasses, _function);
  }

  public Iterable<EReference> getSourceReferences(final EClass relationshipClass) {
    Iterable<EReference> _switchResult = null;
    boolean _matched = false;
    boolean _isSuperTypeOf = UMLPackage.Literals.DIRECTED_RELATIONSHIP.isSuperTypeOf(relationshipClass);
    if (_isSuperTypeOf) {
      _matched=true;
      final Function1<EReference, Boolean> _function = (EReference it) -> {
        return Boolean.valueOf((((!it.isDerived()) && it.isChangeable()) && this.subsets(it, UMLPackage.Literals.DIRECTED_RELATIONSHIP__SOURCE)));
      };
      _switchResult = IterableExtensions.<EReference>filter(relationshipClass.getEAllReferences(), _function);
    }
    if (!_matched) {
      boolean _isSuperTypeOf_1 = UMLPackage.Literals.ASSOCIATION.isSuperTypeOf(relationshipClass);
      if (_isSuperTypeOf_1) {
        _matched=true;
        _switchResult = Collections.<EReference>unmodifiableList(CollectionLiterals.<EReference>newArrayList(UMLPackage.Literals.ASSOCIATION__END_TYPE));
      }
    }
    if (!_matched) {
      boolean _isSuperTypeOf_2 = UMLPackage.Literals.CONNECTOR.isSuperTypeOf(relationshipClass);
      if (_isSuperTypeOf_2) {
        _matched=true;
        _switchResult = Collections.<EReference>unmodifiableList(CollectionLiterals.<EReference>newArrayList(UMLPackage.Literals.CONNECTOR__END));
      }
    }
    if (!_matched) {
      boolean _isSuperTypeOf_3 = UMLPackage.Literals.ACTIVITY_EDGE.isSuperTypeOf(relationshipClass);
      if (_isSuperTypeOf_3) {
        _matched=true;
        _switchResult = Collections.<EReference>unmodifiableList(CollectionLiterals.<EReference>newArrayList(UMLPackage.Literals.ACTIVITY_EDGE__SOURCE));
      }
    }
    if (!_matched) {
      boolean _isSuperTypeOf_4 = UMLPackage.Literals.TRANSITION.isSuperTypeOf(relationshipClass);
      if (_isSuperTypeOf_4) {
        _matched=true;
        _switchResult = Collections.<EReference>unmodifiableList(CollectionLiterals.<EReference>newArrayList(UMLPackage.Literals.TRANSITION__SOURCE));
      }
    }
    if (!_matched) {
      boolean _isSuperTypeOf_5 = UMLPackage.Literals.MESSAGE.isSuperTypeOf(relationshipClass);
      if (_isSuperTypeOf_5) {
        _matched=true;
        _switchResult = Collections.<EReference>unmodifiableList(CollectionLiterals.<EReference>newArrayList(UMLPackage.Literals.MESSAGE__SEND_EVENT));
      }
    }
    if (!_matched) {
      boolean _isSuperTypeOf_6 = UMLPackage.Literals.GENERALIZATION_SET.isSuperTypeOf(relationshipClass);
      if (_isSuperTypeOf_6) {
        _matched=true;
        _switchResult = Collections.<EReference>unmodifiableList(CollectionLiterals.<EReference>newArrayList(UMLPackage.Literals.GENERALIZATION_SET__GENERALIZATION));
      }
    }
    if (!_matched) {
      _switchResult = Collections.<EReference>unmodifiableList(CollectionLiterals.<EReference>newArrayList());
    }
    return _switchResult;
  }

  public Iterable<EReference> getTargetReferences(final EClass relationshipClass) {
    Iterable<EReference> _switchResult = null;
    boolean _matched = false;
    boolean _isSuperTypeOf = UMLPackage.Literals.DIRECTED_RELATIONSHIP.isSuperTypeOf(relationshipClass);
    if (_isSuperTypeOf) {
      _matched=true;
      final Function1<EReference, Boolean> _function = (EReference it) -> {
        return Boolean.valueOf((((!it.isDerived()) && it.isChangeable()) && this.subsets(it, UMLPackage.Literals.DIRECTED_RELATIONSHIP__TARGET)));
      };
      _switchResult = IterableExtensions.<EReference>filter(relationshipClass.getEAllReferences(), _function);
    }
    if (!_matched) {
      boolean _isSuperTypeOf_1 = UMLPackage.Literals.ASSOCIATION.isSuperTypeOf(relationshipClass);
      if (_isSuperTypeOf_1) {
        _matched=true;
        _switchResult = Collections.<EReference>unmodifiableList(CollectionLiterals.<EReference>newArrayList(UMLPackage.Literals.ASSOCIATION__END_TYPE));
      }
    }
    if (!_matched) {
      boolean _isSuperTypeOf_2 = UMLPackage.Literals.CONNECTOR.isSuperTypeOf(relationshipClass);
      if (_isSuperTypeOf_2) {
        _matched=true;
        _switchResult = Collections.<EReference>unmodifiableList(CollectionLiterals.<EReference>newArrayList(UMLPackage.Literals.CONNECTOR__END));
      }
    }
    if (!_matched) {
      boolean _isSuperTypeOf_3 = UMLPackage.Literals.ACTIVITY_EDGE.isSuperTypeOf(relationshipClass);
      if (_isSuperTypeOf_3) {
        _matched=true;
        _switchResult = Collections.<EReference>unmodifiableList(CollectionLiterals.<EReference>newArrayList(UMLPackage.Literals.ACTIVITY_EDGE__TARGET));
      }
    }
    if (!_matched) {
      boolean _isSuperTypeOf_4 = UMLPackage.Literals.TRANSITION.isSuperTypeOf(relationshipClass);
      if (_isSuperTypeOf_4) {
        _matched=true;
        _switchResult = Collections.<EReference>unmodifiableList(CollectionLiterals.<EReference>newArrayList(UMLPackage.Literals.TRANSITION__TARGET));
      }
    }
    if (!_matched) {
      boolean _isSuperTypeOf_5 = UMLPackage.Literals.MESSAGE.isSuperTypeOf(relationshipClass);
      if (_isSuperTypeOf_5) {
        _matched=true;
        _switchResult = Collections.<EReference>unmodifiableList(CollectionLiterals.<EReference>newArrayList(UMLPackage.Literals.MESSAGE__RECEIVE_EVENT));
      }
    }
    if (!_matched) {
      boolean _isSuperTypeOf_6 = UMLPackage.Literals.GENERALIZATION_SET.isSuperTypeOf(relationshipClass);
      if (_isSuperTypeOf_6) {
        _matched=true;
        _switchResult = Collections.<EReference>unmodifiableList(CollectionLiterals.<EReference>newArrayList(UMLPackage.Literals.GENERALIZATION_SET__GENERALIZATION));
      }
    }
    if (!_matched) {
      _switchResult = Collections.<EReference>unmodifiableList(CollectionLiterals.<EReference>newArrayList());
    }
    return _switchResult;
  }

  public boolean subsets(final EReference subset, final EReference superset) {
    EAnnotation _eAnnotation = subset.getEAnnotation("subsets");
    EList<EObject> _references = null;
    if (_eAnnotation!=null) {
      _references=_eAnnotation.getReferences();
    }
    boolean _contains = false;
    if (_references!=null) {
      _contains=_references.contains(superset);
    }
    return _contains;
  }

  public Profile getProfile(final Object element) {
    if (element instanceof Profile) {
      return _getProfile((Profile)element);
    } else if (element instanceof NamedElement) {
      return _getProfile((NamedElement)element);
    } else if (element instanceof ImpliedExtension) {
      return _getProfile((ImpliedExtension)element);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(element).toString());
    }
  }

  public Profile getRootProfile(final NamedElement element) {
    if (element instanceof Profile) {
      return _getRootProfile((Profile)element);
    } else if (element != null) {
      return _getRootProfile(element);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(element).toString());
    }
  }
}
