/**
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.dispatching.teditor.validation;

import activity.Activity;
import activity.ActivityPackage;
import activity.Claim;
import activity.Event;
import activity.JitBounds;
import activity.LocationPrerequisite;
import activity.Move;
import activity.TimeBounds;
import activity.util.ActivityParametersUtil;
import activity.util.EventSequenceValidator;
import com.google.common.collect.Iterables;
import common.CommonPackage;
import common.HasName;
import common.Import;
import common.Parameter;
import common.ParameterReference;
import common.TypeDefinition;
import dispatching.ActivityDispatching;
import dispatching.Attribute;
import dispatching.Constraint;
import dispatching.Dispatch;
import dispatching.DispatchGroup;
import dispatching.DispatchingPackage;
import dispatching.impl.AttributesMapEntryImpl;
import dispatching.util.DispatchingUtil;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Consumer;
import machine.ActionType;
import machine.IResource;
import machine.Peripheral;
import machine.Resource;
import machine.SymbolicPosition;
import machine.util.ResourcePeripheralKey;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.lsat.common.graph.directed.editable.EdgQueries;
import org.eclipse.lsat.common.graph.directed.editable.Node;
import org.eclipse.lsat.common.queries.QueryableIterable;
import org.eclipse.lsat.common.xtend.Queries;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.Functions.Function2;
import org.eclipse.xtext.xbase.lib.IntegerRange;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure2;

/**
 * Custom validation rules.
 * 
 * see http://www.eclipse.org/Xtext/documentation.html#validation
 */
@SuppressWarnings("all")
public class DispatchingValidator extends AbstractDispatchingValidator {
  private static final class PositionInfo {
    private final SymbolicPosition position;

    private final Activity activity;

    PositionInfo(final SymbolicPosition position, final Activity activity) {
      this.position = position;
      this.activity = activity;
    }
  }

  public static final String GREATER_OFFSET = "greaterOffset";

  public static final String INVALID_IMPORT = "invalidImport";

  public static final String REMOVE_ITEM = "removeItem";

  public static final String SELECT_ITEM_FOR_RESOURCE = "selectItemForResource";

  public static final String REPLACE_ITEM_FOR_RESOURCE = "replaceItemForResource";

  public static final String CONVERT_YIELD = "convertYield";

  public static final String CONSTRAINT_DISPATCH_NAME = "constraintDispatchName";

  public static final String CONSTRAINT_LIMITS = "constraintLimits";

  @Check
  public void checkObsoleteThroughput(final ActivityDispatching ad) {
    boolean _eIsSet = ad.eIsSet(DispatchingPackage.Literals.ACTIVITY_DISPATCHING__NUMBER_OF_ITERATIONS);
    if (_eIsSet) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Througput is obsolete. Specify yield below in activities");
      this.warning(_builder.toString(), ad, 
        DispatchingPackage.Literals.ACTIVITY_DISPATCHING__NUMBER_OF_ITERATIONS, DispatchingValidator.CONVERT_YIELD);
    }
    for (int i = 0; (i < ad.getResourceIterations().size()); i++) {
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("Througput is obsolete. Specify yield below in activities");
      this.warning(_builder_1.toString(), ad, 
        DispatchingPackage.Literals.ACTIVITY_DISPATCHING__RESOURCE_ITERATIONS, i, DispatchingValidator.CONVERT_YIELD);
    }
  }

  @Check
  public void checkDuplicateDispatchGroup(final ActivityDispatching ad) {
    final Function1<DispatchGroup, Boolean> _function = (DispatchGroup it) -> {
      return Boolean.valueOf(it.eIsSet(DispatchingPackage.Literals.DISPATCH_GROUP__NAME));
    };
    final Function1<DispatchGroup, String> _function_1 = (DispatchGroup it) -> {
      return it.getName();
    };
    final Function1<List<DispatchGroup>, Boolean> _function_2 = (List<DispatchGroup> it) -> {
      int _size = it.size();
      return Boolean.valueOf((_size > 1));
    };
    final Consumer<DispatchGroup> _function_3 = (DispatchGroup it) -> {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Duplicate dispatch group name  \'");
      String _name = it.getName();
      _builder.append(_name);
      _builder.append("\' Rename one");
      this.error(_builder.toString(), ad, 
        DispatchingPackage.Literals.ACTIVITY_DISPATCHING__DISPATCH_GROUPS, ad.getDispatchGroups().indexOf(it));
    };
    Iterables.<DispatchGroup>concat(IterableExtensions.<List<DispatchGroup>>filter(IterableExtensions.<String, DispatchGroup>groupBy(IterableExtensions.<DispatchGroup>filter(ad.getDispatchGroups(), _function), _function_1).values(), _function_2)).forEach(_function_3);
  }

  @Check
  public void checkImportIsValid(final Import imp) {
    try {
      final boolean isImportUriValid = EcoreUtil2.isValidUri(imp, URI.createURI(imp.getImportURI()));
      if ((!isImportUriValid)) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("The import ");
        String _importURI = imp.getImportURI();
        _builder.append(_importURI);
        _builder.append(" cannot be resolved. Make sure that the name is spelled correctly.");
        this.error(_builder.toString(), imp, CommonPackage.Literals.IMPORT__IMPORT_URI, DispatchingValidator.INVALID_IMPORT);
      }
      final boolean isUnderstood = imp.getImportURI().matches(".*\\.(activity)");
      if ((!isUnderstood)) {
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append("Importing ");
        String _importURI_1 = imp.getImportURI();
        _builder_1.append(_importURI_1);
        _builder_1.append(" is not allowed. Only \'activity\' files are allowed");
        this.error(_builder_1.toString(), imp, CommonPackage.Literals.IMPORT__IMPORT_URI, DispatchingValidator.INVALID_IMPORT);
      }
    } catch (final Throwable _t) {
      if (_t instanceof IllegalArgumentException) {
        StringConcatenation _builder_2 = new StringConcatenation();
        _builder_2.append("The import ");
        String _importURI_2 = imp.getImportURI();
        _builder_2.append(_importURI_2);
        _builder_2.append(" is not a valid URI.");
        this.error(_builder_2.toString(), imp, 
          CommonPackage.Literals.IMPORT__IMPORT_URI, DispatchingValidator.INVALID_IMPORT);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }

  @Check
  public void checkOffsetShouldBeGreaterThanPrevious(final ActivityDispatching activityDispatching) {
    EList<DispatchGroup> dispatchGroups = activityDispatching.getDispatchGroups();
    for (int i = 1; (i < dispatchGroups.size()); i++) {
      BigDecimal _offset = dispatchGroups.get(i).getOffset();
      BigDecimal _offset_1 = dispatchGroups.get((i - 1)).getOffset();
      boolean _lessThan = (_offset.compareTo(_offset_1) < 0);
      if (_lessThan) {
        this.error("Offset should be greater then previously defined offset", dispatchGroups.get(i), 
          DispatchingPackage.Literals.DISPATCH_GROUP__OFFSET, DispatchingValidator.GREATER_OFFSET);
      }
    }
  }

  @Check
  public void validateAttributesDuplicate(final Dispatch dis) {
    final Function1<Map.Entry<Attribute, String>, String> _function = (Map.Entry<Attribute, String> it) -> {
      return it.getKey().getName();
    };
    final Function1<List<Map.Entry<Attribute, String>>, Boolean> _function_1 = (List<Map.Entry<Attribute, String>> it) -> {
      int _size = it.size();
      return Boolean.valueOf((_size > 1));
    };
    final Consumer<Map.Entry<Attribute, String>> _function_2 = (Map.Entry<Attribute, String> it) -> {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Duplicate item specified for Attribute \'");
      String _name = it.getKey().getName();
      _builder.append(_name);
      _builder.append("\' Remove one");
      this.error(_builder.toString(), dis, 
        DispatchingPackage.Literals.HAS_USER_ATTRIBUTES__USER_ATTRIBUTES, dis.getUserAttributes().indexOf(it), 
        DispatchingValidator.REMOVE_ITEM);
    };
    Iterables.<Map.Entry<Attribute, String>>concat(IterableExtensions.<List<Map.Entry<Attribute, String>>>filter(IterableExtensions.<String, Map.Entry<Attribute, String>>groupBy(dis.getUserAttributes(), _function).values(), _function_1)).forEach(_function_2);
  }

  @Check
  public void validateAttributeValue(final AttributesMapEntryImpl entry) {
    String _value = entry.getValue();
    boolean _tripleEquals = (_value == null);
    if (_tripleEquals) {
      return;
    }
    final String illegalChars = entry.getValue().replaceAll("[\\w\\.\\-\\s]", "");
    boolean _isEmpty = illegalChars.isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      final Dispatch parent = EcoreUtil2.<Dispatch>getContainerOfType(entry, Dispatch.class);
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Attribute value may not contain  \'");
      _builder.append(illegalChars);
      _builder.append("\'. only \'a-zA-Z0-9-_.<space><tab>\' are allowed ");
      this.error(_builder.toString(), parent, DispatchingPackage.Literals.HAS_USER_ATTRIBUTES__USER_ATTRIBUTES, 
        parent.getUserAttributes().indexOf(entry), DispatchingValidator.REMOVE_ITEM);
    }
  }

  @Check
  public void validatePosition(final ActivityDispatching activityDispatching) {
    final LinkedHashMap<ResourcePeripheralKey, DispatchingValidator.PositionInfo> positionState = CollectionLiterals.<ResourcePeripheralKey, DispatchingValidator.PositionInfo>newLinkedHashMap();
    final Function1<DispatchGroup, EList<Dispatch>> _function = (DispatchGroup it) -> {
      return it.getDispatches();
    };
    final Function1<Dispatch, Boolean> _function_1 = (Dispatch it) -> {
      boolean _isEmpty = it.getParameters().isEmpty();
      return Boolean.valueOf((!_isEmpty));
    };
    Dispatch _findFirst = IterableExtensions.<Dispatch>findFirst(IterableExtensions.<DispatchGroup, Dispatch>flatMap(activityDispatching.getDispatchGroups(), _function), _function_1);
    final boolean hasParameters = (_findFirst != null);
    if (hasParameters) {
      return;
    }
    final Consumer<DispatchGroup> _function_2 = (DispatchGroup dispatchGroup) -> {
      final Procedure2<Dispatch, Integer> _function_3 = (Dispatch disp, Integer index) -> {
        Activity activity = disp.getActivity();
        LinkedHashMap<ResourcePeripheralKey, SymbolicPosition> locationPrerequisites = this.indexPrerequisites(disp, (index).intValue());
        Set<Map.Entry<ResourcePeripheralKey, SymbolicPosition>> _entrySet = locationPrerequisites.entrySet();
        for (final Map.Entry<ResourcePeripheralKey, SymbolicPosition> prerequisite : _entrySet) {
          {
            ResourcePeripheralKey _key = prerequisite.getKey();
            SymbolicPosition _value = prerequisite.getValue();
            DispatchingValidator.PositionInfo _positionInfo = new DispatchingValidator.PositionInfo(_value, activity);
            DispatchingValidator.PositionInfo positionInfo = positionState.put(_key, _positionInfo);
            if (((null != positionInfo) && (!positionInfo.position.equals(prerequisite.getValue())))) {
              StringConcatenation _builder = new StringConcatenation();
              _builder.append("Activity ");
              String _name = activity.getName();
              _builder.append(_name);
              _builder.append(" requires peripheral ");
              String _fqn = prerequisite.getKey().fqn();
              _builder.append(_fqn);
              _builder.append(" at position ");
              String _name_1 = prerequisite.getValue().getName();
              _builder.append(_name_1);
              _builder.append(", but activity ");
              String _name_2 = positionInfo.activity.getName();
              _builder.append(_name_2);
              _builder.append(" leaves it at position ");
              String _name_3 = positionInfo.position.getName();
              _builder.append(_name_3);
              this.warning(_builder.toString(), disp, 
                DispatchingPackage.Literals.DISPATCH__ACTIVITY);
            }
          }
        }
        EList<Node> topologicalOrder = EdgQueries.<Node>topologicalOrdering(activity.getNodes());
        if ((null == topologicalOrder)) {
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("Location prerequisites cannot be validated for the next activity as activity ");
          String _name = activity.getName();
          _builder.append(_name);
          _builder.append(" contains a cycle");
          this.error(_builder.toString(), disp, 
            DispatchingPackage.Literals.DISPATCH__ACTIVITY);
          positionState.clear();
        } else {
          final Function1<Move, Boolean> _function_4 = (Move it) -> {
            return Boolean.valueOf(it.isPositionMove());
          };
          Iterable<Move> _filter = IterableExtensions.<Move>filter(QueryableIterable.<Node>from(topologicalOrder).<Move>objectsOfKind(Move.class), _function_4);
          for (final Move move : _filter) {
            IResource _actualResource = DispatchingValidator.getActualResource(disp, move.getResource());
            Peripheral _peripheral = move.getPeripheral();
            ResourcePeripheralKey _resourcePeripheralKey = new ResourcePeripheralKey(_actualResource, _peripheral);
            SymbolicPosition _targetPosition = move.getTargetPosition();
            DispatchingValidator.PositionInfo _positionInfo = new DispatchingValidator.PositionInfo(_targetPosition, activity);
            positionState.put(_resourcePeripheralKey, _positionInfo);
          }
        }
      };
      IterableExtensions.<Dispatch>forEach(dispatchGroup.getDispatches(), _function_3);
    };
    activityDispatching.getDispatchGroups().forEach(_function_2);
  }

  /**
   * Checks the number if events raised and required match test rules:
   * <ul>
   *  <li>#receives <= #sends at any point in the sequence</li>
   *  <li>0 <= #sends - #receives <= 1, such that at most one event can be outstanding</li>
   *  <li>the number of sends and receives matches over the full sequence for the event</li>
   * </ul>
   */
  @Check
  public void checkUnfolded(final ActivityDispatching activityDispatching) {
    final List<Dispatch> unfoldedDispatches = DispatchingValidator.unfoldedDispatches(activityDispatching);
    final EventSequenceValidator.ErrorRaiser errorRaiser = new EventSequenceValidator.ErrorRaiser() {
      @Override
      public void raise(final int listIndex, final String msg) {
        if ((listIndex < 0)) {
          DispatchingValidator.this.error(msg, activityDispatching, DispatchingPackage.Literals.ACTIVITY_DISPATCHING__DISPATCH_GROUPS);
        } else {
          final Dispatch d = unfoldedDispatches.get(listIndex);
          EObject _eContainer = d.eContainer();
          DispatchGroup dg = ((DispatchGroup) _eContainer);
          DispatchingValidator.this.error(msg, dg, DispatchingPackage.Literals.DISPATCH_GROUP__DISPATCHES, dg.getDispatches().indexOf(d));
        }
      }
    };
    final Function1<Dispatch, Activity> _function = (Dispatch it) -> {
      return it.getActivity();
    };
    EventSequenceValidator.validate(IterableExtensions.<Activity>toList(ListExtensions.<Dispatch, Activity>map(unfoldedDispatches, _function)), errorRaiser);
    final Function1<Dispatch, Iterable<Pair<Claim, Dispatch>>> _function_1 = (Dispatch d) -> {
      final Function1<Claim, Pair<Claim, Dispatch>> _function_2 = (Claim c) -> {
        return Pair.<Claim, Dispatch>of(c, d);
      };
      return IterableExtensions.<Claim, Pair<Claim, Dispatch>>map(Iterables.<Claim>filter(d.getActivity().getNodes(), Claim.class), _function_2);
    };
    final Iterable<Pair<Claim, Dispatch>> claimsWithDispatches = IterableExtensions.<Dispatch, Pair<Claim, Dispatch>>flatMap(unfoldedDispatches, _function_1);
    final Function1<Pair<Claim, Dispatch>, IResource> _function_2 = (Pair<Claim, Dispatch> it) -> {
      return it.getKey().getResource();
    };
    final Function1<List<Pair<Claim, Dispatch>>, Iterable<Dispatch>> _function_3 = (List<Pair<Claim, Dispatch>> claims) -> {
      final Function2<Pair<Claim, Dispatch>, Pair<Claim, Dispatch>, Boolean> _function_4 = (Pair<Claim, Dispatch> p1, Pair<Claim, Dispatch> p2) -> {
        boolean _isPassive = p1.getKey().isPassive();
        boolean _isPassive_1 = p2.getKey().isPassive();
        return Boolean.valueOf((_isPassive == _isPassive_1));
      };
      final Function1<List<Pair<Claim, Dispatch>>, Boolean> _function_5 = (List<Pair<Claim, Dispatch>> it) -> {
        return Boolean.valueOf(IterableExtensions.<Pair<Claim, Dispatch>>head(it).getKey().isPassive());
      };
      final Iterable<List<Pair<Claim, Dispatch>>> passiveClaimsGroups = IterableExtensions.<List<Pair<Claim, Dispatch>>>filter(Queries.<Pair<Claim, Dispatch>>segment(claims, _function_4), _function_5);
      final Function1<List<Pair<Claim, Dispatch>>, Iterable<Dispatch>> _function_6 = (List<Pair<Claim, Dispatch>> passiveClaims) -> {
        final Function1<Pair<Claim, Dispatch>, Activity> _function_7 = (Pair<Claim, Dispatch> it) -> {
          return it.getValue().getActivity();
        };
        final Function1<List<Pair<Claim, Dispatch>>, Boolean> _function_8 = (List<Pair<Claim, Dispatch>> it) -> {
          int _size = it.size();
          return Boolean.valueOf((_size > 1));
        };
        final Iterable<Pair<Claim, Dispatch>> duplicatePassiveClaims = Iterables.<Pair<Claim, Dispatch>>concat(IterableExtensions.<List<Pair<Claim, Dispatch>>>filter(IterableExtensions.<Activity, Pair<Claim, Dispatch>>groupBy(passiveClaims, _function_7).values(), _function_8));
        final Function1<Pair<Claim, Dispatch>, Dispatch> _function_9 = (Pair<Claim, Dispatch> it) -> {
          return it.getValue();
        };
        return IterableExtensions.<Pair<Claim, Dispatch>, Dispatch>map(duplicatePassiveClaims, _function_9);
      };
      return IterableExtensions.<List<Pair<Claim, Dispatch>>, Dispatch>flatMap(passiveClaimsGroups, _function_6);
    };
    final Iterable<Dispatch> duplicatePassiveClaimDispatches = IterableExtensions.<List<Pair<Claim, Dispatch>>, Dispatch>flatMap(IterableExtensions.<IResource, Pair<Claim, Dispatch>>groupBy(claimsWithDispatches, _function_2).values(), _function_3);
    final Consumer<Dispatch> _function_4 = (Dispatch dispatch) -> {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Activity ");
      String _name = dispatch.getActivity().getName();
      _builder.append(_name);
      _builder.append(" cannot be repeated within a passive claim sequence.");
      this.error(_builder.toString(), dispatch, 
        DispatchingPackage.Literals.DISPATCH__ACTIVITY);
    };
    Queries.<Dispatch>unique(duplicatePassiveClaimDispatches).forEach(_function_4);
  }

  private static List<Dispatch> unfoldedDispatches(final ActivityDispatching activityDispatching) {
    final ArrayList<Dispatch> unfoldedDispatches = CollectionLiterals.<Dispatch>newArrayList();
    final Consumer<DispatchGroup> _function = (DispatchGroup dg) -> {
      int _numberRepeats = DispatchingUtil.getNumberRepeats(dg);
      IntegerRange _upTo = new IntegerRange(1, _numberRepeats);
      for (final Integer i : _upTo) {
        unfoldedDispatches.addAll(dg.getDispatches());
      }
    };
    activityDispatching.getDispatchGroups().forEach(_function);
    return unfoldedDispatches;
  }

  @Check
  public void validateRepeatAndOffset(final DispatchGroup group) {
    if (((BigDecimal.ZERO.compareTo(group.getOffset()) != 0) && (DispatchingUtil.getNumberRepeats(group) > 1))) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("offset must be ommited or zero when repeats are specified");
      final String msg = _builder.toString();
      this.error(msg, group, DispatchingPackage.Literals.DISPATCH_GROUP__OFFSET);
      this.error(msg, group, DispatchingPackage.Literals.DISPATCH_GROUP__REPEATS, 0);
    }
  }

  @Check
  public void validateDispatchNameDuplicate(final DispatchGroup group) {
    final Function1<Dispatch, Boolean> _function = (Dispatch it) -> {
      String _name = it.getName();
      return Boolean.valueOf((_name != null));
    };
    final Function1<Dispatch, String> _function_1 = (Dispatch it) -> {
      return it.getName();
    };
    final Function1<List<Dispatch>, Boolean> _function_2 = (List<Dispatch> it) -> {
      int _size = it.size();
      return Boolean.valueOf((_size > 1));
    };
    final Consumer<Dispatch> _function_3 = (Dispatch it) -> {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Duplicate name specified for Dispatch \'");
      String _name = it.getName();
      _builder.append(_name);
      _builder.append("\' Remove one");
      this.error(_builder.toString(), it, 
        DispatchingPackage.Literals.DISPATCH__NAME, group.getUserAttributes().indexOf(it), 
        DispatchingValidator.REMOVE_ITEM);
    };
    Iterables.<Dispatch>concat(IterableExtensions.<List<Dispatch>>filter(IterableExtensions.<String, Dispatch>groupBy(IterableExtensions.<Dispatch>filter(group.getDispatches(), _function), _function_1).values(), _function_2)).forEach(_function_3);
  }

  @Check
  public void validatePhase(final AttributesMapEntryImpl entry) {
    final DispatchGroup dispatchGroup = EcoreUtil2.<DispatchGroup>getContainerOfType(entry, DispatchGroup.class);
    if ((dispatchGroup == null)) {
      return;
    }
    boolean _and = false;
    Attribute _key = entry.getKey();
    String _name = null;
    if (_key!=null) {
      _name=_key.getName();
    }
    boolean _equalsIgnoreCase = "phase".equalsIgnoreCase(_name);
    if (!_equalsIgnoreCase) {
      _and = false;
    } else {
      boolean _equalsIgnoreCase_1 = dispatchGroup.getName().equalsIgnoreCase(entry.getValue());
      boolean _not = (!_equalsIgnoreCase_1);
      _and = _not;
    }
    if (_and) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("phase is a reserved attribute and may not be used");
      final String msg = _builder.toString();
      this.error(msg, entry, DispatchingPackage.Literals.ATTRIBUTES_MAP_ENTRY__KEY);
    }
  }

  @Check
  public void validateIteratorName(final AttributesMapEntryImpl entry) {
    final DispatchGroup dispatchGroup = EcoreUtil2.<DispatchGroup>getContainerOfType(entry, DispatchGroup.class);
    if ((dispatchGroup == null)) {
      return;
    }
    boolean _and = false;
    String _iteratorName = dispatchGroup.getIteratorName();
    Attribute _key = entry.getKey();
    String _name = null;
    if (_key!=null) {
      _name=_key.getName();
    }
    boolean _equalsIgnoreCase = _iteratorName.equalsIgnoreCase(_name);
    if (!_equalsIgnoreCase) {
      _and = false;
    } else {
      int _numberRepeats = DispatchingUtil.getNumberRepeats(dispatchGroup);
      boolean _greaterThan = (_numberRepeats > 1);
      _and = _greaterThan;
    }
    if (_and) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("attribute key may not be equal to activites iterator name. Change key or iterator name ");
      final String msg = _builder.toString();
      this.error(msg, entry, DispatchingPackage.Literals.ATTRIBUTES_MAP_ENTRY__KEY);
      boolean _eIsSet = dispatchGroup.eIsSet(DispatchingPackage.Literals.DISPATCH_GROUP__ITERATOR_NAME);
      if (_eIsSet) {
        this.error(msg, dispatchGroup, DispatchingPackage.Literals.DISPATCH_GROUP__ITERATOR_NAME);
      }
    }
  }

  /**
   * Check if all constraints inside dispatch-groups refer to dispatch groups with unique names
   */
  @Check
  public void checkDispatchGroupNameUniqueInConstraint(final Constraint constraint) {
    ActivityDispatching activityDispatching = ((ActivityDispatching) null);
    EObject _eContainer = constraint.eContainer();
    if ((_eContainer instanceof DispatchGroup)) {
      EObject _eContainer_1 = constraint.eContainer();
      final DispatchGroup dispatchGroup = ((DispatchGroup) _eContainer_1);
      EObject _eContainer_2 = dispatchGroup.eContainer();
      activityDispatching = ((ActivityDispatching) _eContainer_2);
    } else {
      EObject _eContainer_3 = constraint.eContainer();
      if ((_eContainer_3 instanceof ActivityDispatching)) {
        EObject _eContainer_4 = constraint.eContainer();
        activityDispatching = ((ActivityDispatching) _eContainer_4);
      }
    }
    EObject _eContainer_5 = constraint.getSourceDispatch().eContainer();
    final DispatchGroup sourceDispatchGroup = ((DispatchGroup) _eContainer_5);
    EObject _eContainer_6 = constraint.getTargetDispatch().eContainer();
    final DispatchGroup targetDispatchGroup = ((DispatchGroup) _eContainer_6);
    final String sourceDispatchGroupName = sourceDispatchGroup.getName();
    final String targetDispatchGroupName = targetDispatchGroup.getName();
    final Function1<DispatchGroup, Boolean> _function = (DispatchGroup it) -> {
      String _name = it.getName();
      return Boolean.valueOf(Objects.equals(sourceDispatchGroupName, _name));
    };
    final Function1<DispatchGroup, String> _function_1 = (DispatchGroup it) -> {
      return it.getName();
    };
    final Function1<List<DispatchGroup>, Boolean> _function_2 = (List<DispatchGroup> it) -> {
      int _size = it.size();
      return Boolean.valueOf((_size > 1));
    };
    final Consumer<DispatchGroup> _function_3 = (DispatchGroup it) -> {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Dispatch group name \'");
      String _name = it.getName();
      _builder.append(_name);
      _builder.append("\' must be unique for use in constraints.");
      this.error(_builder.toString(), constraint, 
        DispatchingPackage.Literals.CONSTRAINT__SOURCE_DISPATCH, DispatchingValidator.CONSTRAINT_DISPATCH_NAME);
    };
    Iterables.<DispatchGroup>concat(IterableExtensions.<List<DispatchGroup>>filter(IterableExtensions.<String, DispatchGroup>groupBy(IterableExtensions.<DispatchGroup>filter(activityDispatching.getDispatchGroups(), _function), _function_1).values(), _function_2)).forEach(_function_3);
    final Function1<DispatchGroup, Boolean> _function_4 = (DispatchGroup it) -> {
      String _name = it.getName();
      return Boolean.valueOf(Objects.equals(targetDispatchGroupName, _name));
    };
    final Function1<DispatchGroup, String> _function_5 = (DispatchGroup it) -> {
      return it.getName();
    };
    final Function1<List<DispatchGroup>, Boolean> _function_6 = (List<DispatchGroup> it) -> {
      int _size = it.size();
      return Boolean.valueOf((_size > 1));
    };
    final Consumer<DispatchGroup> _function_7 = (DispatchGroup it) -> {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("Dispatch group name \'");
      String _name = it.getName();
      _builder.append(_name);
      _builder.append("\' must be unique for use in constraints.");
      this.error(_builder.toString(), constraint, 
        DispatchingPackage.Literals.CONSTRAINT__TARGET_DISPATCH, DispatchingValidator.CONSTRAINT_DISPATCH_NAME);
    };
    Iterables.<DispatchGroup>concat(IterableExtensions.<List<DispatchGroup>>filter(IterableExtensions.<String, DispatchGroup>groupBy(IterableExtensions.<DispatchGroup>filter(activityDispatching.getDispatchGroups(), _function_4), _function_5).values(), _function_6)).forEach(_function_7);
  }

  /**
   * This method makes sure that the upper bound of a Constraint is larger or equal than the lower bound
   */
  @Check
  public void checkConstraintBounds(final Constraint constraint) {
    JitBounds _bounds = constraint.getBounds();
    if ((_bounds instanceof TimeBounds)) {
      JitBounds _bounds_1 = constraint.getBounds();
      final TimeBounds timeBounds = ((TimeBounds) _bounds_1);
      BigDecimal _lower = timeBounds.getLower();
      BigDecimal _upper = timeBounds.getUpper();
      boolean _greaterThan = (_lower.compareTo(_upper) > 0);
      if (_greaterThan) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Time constraint between ");
        String _name = constraint.getSourceDispatch().getName();
        _builder.append(_name);
        _builder.append(".");
        String _name_1 = constraint.getSourceAction().getName();
        _builder.append(_name_1);
        _builder.append(" ");
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append("and ");
        String _name_2 = constraint.getTargetDispatch().getName();
        _builder_1.append(_name_2);
        _builder_1.append(".");
        String _name_3 = constraint.getTargetAction().getName();
        _builder_1.append(_name_3);
        _builder_1.append(" ");
        String _plus = (_builder.toString() + _builder_1);
        StringConcatenation _builder_2 = new StringConcatenation();
        _builder_2.append("must have a lower bound less than or equal to the upper bound");
        String _plus_1 = (_plus + _builder_2);
        this.error(_plus_1, constraint, ActivityPackage.Literals.TIME_BOUNDS__LOWER, DispatchingValidator.CONSTRAINT_LIMITS);
      }
    }
  }

  @Check
  public void checkParameters(final Dispatch dis) {
    final EList<Parameter> parameters = dis.getParameters();
    final List<TypeDefinition> declarations = ActivityParametersUtil.getDeclarations(dis.getActivity());
    int _size = parameters.size();
    int _size_1 = declarations.size();
    boolean _lessThan = (_size < _size_1);
    if (_lessThan) {
      final TypeDefinition dec = declarations.get(parameters.size());
      final EClass declType = ActivityParametersUtil.getExpectedType(dec, null);
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("No item specified for \'");
      String _name = dec.getName();
      _builder.append(_name);
      _builder.append(":");
      String _name_1 = declType.getName();
      _builder.append(_name_1);
      _builder.append("\' Press <ctrl> spacebar");
      this.error(_builder.toString(), dis, 
        CommonPackage.Literals.HAS_PARAMETERS__PARAMETERS, parameters.size());
    }
    int _size_2 = declarations.size();
    int _size_3 = parameters.size();
    boolean _lessThan_1 = (_size_2 < _size_3);
    if (_lessThan_1) {
      final Parameter par = parameters.get(declarations.size());
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("Unnecessary parameter specified. Please remove it");
      this.error(_builder_1.toString(), par, 
        CommonPackage.Literals.PARAMETER__REFERENCES, declarations.size());
    }
    final int size = Math.min(parameters.size(), declarations.size());
    for (int index = 0; (index < size); index++) {
      {
        final Parameter par_1 = parameters.get(index);
        TypeDefinition dec_1 = par_1.getDeclaration();
        if ((dec_1 == null)) {
          dec_1 = declarations.get(index);
        }
        boolean _isEmpty = par_1.getReferences().isEmpty();
        boolean _not = (!_isEmpty);
        if (_not) {
          EList<ParameterReference> _references = par_1.getReferences();
          ParameterReference _last = null;
          if (_references!=null) {
            _last=_references.getLast();
          }
          HasName _reference = null;
          if (_last!=null) {
            _reference=_last.getReference();
          }
          final HasName actualParType = _reference;
          boolean _matchesType = ActivityParametersUtil.matchesType(dec_1, actualParType);
          boolean _not_1 = (!_matchesType);
          if (_not_1) {
            StringConcatenation _builder_2 = new StringConcatenation();
            _builder_2.append("Parameter \'");
            String _name_2 = dec_1.getName();
            _builder_2.append(_name_2);
            _builder_2.append("\' type mismatch, expecting \'");
            String _name_3 = ActivityParametersUtil.getExpectedType(dec_1, actualParType).getName();
            _builder_2.append(_name_3);
            _builder_2.append("\' but got \'");
            String _name_4 = actualParType.eClass().getName();
            String _plus = (_name_4 + "\'");
            _builder_2.append(_plus);
            int _size_4 = par_1.getReferences().size();
            int _minus = (_size_4 - 1);
            this.error(_builder_2.toString(), par_1, 
              CommonPackage.Literals.PARAMETER__REFERENCES, _minus);
          }
        }
      }
    }
    ArrayList<Parameter> actions = CollectionLiterals.<Parameter>newArrayList();
    for (final Parameter par_1 : parameters) {
      {
        HasName _xifexpression = null;
        boolean _isEmpty = par_1.getReferences().isEmpty();
        if (_isEmpty) {
          _xifexpression = null;
        } else {
          ParameterReference _last = par_1.getReferences().getLast();
          HasName _reference = null;
          if (_last!=null) {
            _reference=_last.getReference();
          }
          _xifexpression = _reference;
        }
        final HasName actualParType = _xifexpression;
        if ((actualParType instanceof ActionType)) {
          actions.add(par_1);
        }
        if (((actualParType instanceof IResource) && (!(actualParType instanceof Event)))) {
          boolean _isEmpty_1 = actions.isEmpty();
          boolean _not = (!_isEmpty_1);
          if (_not) {
            List<Parameter> _of = List.<Parameter>of(par_1);
            Iterable<Parameter> _plus = Iterables.<Parameter>concat(actions, _of);
            for (final Parameter p : _plus) {
              StringConcatenation _builder_2 = new StringConcatenation();
              _builder_2.append("Machine \'Resource\' parameters must precede \'Action\' parameters");
              this.error(_builder_2.toString(), p, 
                CommonPackage.Literals.PARAMETER__REFERENCES);
            }
          }
        }
      }
    }
  }

  private LinkedHashMap<ResourcePeripheralKey, SymbolicPosition> indexPrerequisites(final Dispatch disp, final int index) {
    EList<LocationPrerequisite> prerequisites = disp.getActivity().getPrerequisites();
    final LinkedHashMap<ResourcePeripheralKey, SymbolicPosition> result = CollectionLiterals.<ResourcePeripheralKey, SymbolicPosition>newLinkedHashMap();
    for (final LocationPrerequisite prerequisite : prerequisites) {
      {
        IResource _actualResource = DispatchingValidator.getActualResource(disp, prerequisite.getResource());
        Peripheral _peripheral = prerequisite.getPeripheral();
        ResourcePeripheralKey key = new ResourcePeripheralKey(_actualResource, _peripheral);
        boolean _containsKey = result.containsKey(key);
        if (_containsKey) {
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("Only one location prerequisite per peripheral is allowed");
          this.error(_builder.toString(), disp, 
            DispatchingPackage.Literals.DISPATCH__ACTIVITY);
        }
        result.put(key, prerequisite.getPosition());
      }
    }
    return result;
  }

  private static IResource getActualResource(final Dispatch dispatch, final IResource resource) {
    final Function1<IResource, Boolean> _function = (IResource it) -> {
      Resource _resource = it.getResource();
      return Boolean.valueOf((_resource == resource));
    };
    final IResource result = IterableExtensions.<IResource>findFirst(Iterables.<IResource>filter(dispatch.getParameters(), IResource.class), _function);
    IResource _xifexpression = null;
    if ((result == null)) {
      _xifexpression = resource;
    } else {
      _xifexpression = result;
    }
    return _xifexpression;
  }
}
