package Map::Tube::Sydney;

$Map::Tube::Sydney::VERSION   = '1.00';
$Map::Tube::Sydney::AUTHORITY = 'cpan:EARLYBEAN';

=head1 NAME

Map::Tube::Sydney - Interface to the Sydney Rail Network.

=head1 VERSION

Version 1.00

=cut

use 5.006;
use File::ShareDir ':ALL';

use Moo;
use namespace::autoclean;

has json => (is => 'lazy');
has skip => (is => 'lazy');
with 'Map::Tube';

sub _build_json { dist_file('Map-Tube-Sydney', 'sydney-map.json') }
sub _build_skip {
    return {
        'T2'  => {
            'T125' => { 'T126' => 1,   # Redfern <-> Strathfield express
                        'T212' => 1 }, # Redfern <-> Burwood express
            'T126' => { 'T125' => 1,
                        'T127' => 1 }, # Lidcombe <-> Strathfield express
            'T127' => { 'T126' => 1 },
            'T212' => { 'T125' => 1 },
            'T218' => { 'T219' => 1 }, # Harris Park <-> Merrylands
            'T219' => { 'T218' => 1 },
        },
        'T3'  => {
            'T125' => { 'T126' => 1,   # Redfern <-> Strathfield express
                        'T212' => 1 }, # Redfern <-> Burwood express
            'T126' => { 'T125' => 1,
                        'T127' => 1 }, # Lidcombe <-> Strathfield express
            'T127' => { 'T126' => 1 },
            'T212' => { 'T125' => 1 },
        },
        'T8'   => {
            'T125' => { 'T405' => 1 }, # Redfern <-> Sydenham express
            'T405' => { 'T125' => 1 },
        },
        'T9'   => {
            'T125' => { 'T126' => 1 }, # Redfern <-> Strathfield express
            'T126' => { 'T125' => 1 },
        },
        'L2'   => {
            'T122' => { 'T203' => 1,   # Wynyard <-> Circular Quay by heavy rail
                        'T123' => 1 }, # Wynyard <-> Town Hall by heavy rail
            'T123' => { 'T122' => 1 },
            'T203' => { 'T122' => 1 },
        },
        'L3'   => {
            'T122' => { 'T203' => 1,   # Wynyard <-> Circular Quay by heavy rail
                        'T123' => 1 }, # Wynyard <-> Town Hall by heavy rail
            'T123' => { 'T122' => 1 },
            'T203' => { 'T122' => 1 },
        },
    };
}

=head1 DESCRIPTION

It currently provides functionality to find the shortest  route between  the  two
given nodes. It covers the following rail lines:

=over 2

=item * L<T1 Line|Map::Tube::Sydney::Line::T1>

=item * L<T2 Line|Map::Tube::Sydney::Line::T2>

=item * L<T3 Line|Map::Tube::Sydney::Line::T3>

=item * L<T4 Line|Map::Tube::Sydney::Line::T4>

=item * L<T5 Line|Map::Tube::Sydney::Line::T5>

=item * L<T6 Line|Map::Tube::Sydney::Line::T6>

=item * L<T7 Line|Map::Tube::Sydney::Line::T7>

=item * L<T8 Line|Map::Tube::Sydney::Line::T8>

=item * L<T9 Line|Map::Tube::Sydney::Line::T9>

=item * L<M1 Line|Map::Tube::Sydney::Line::M1>

=item * L<L1 Line|Map::Tube::Sydney::Line::L1>

=item * L<L2 Line|Map::Tube::Sydney::Line::L2>

=item * L<L3 Line|Map::Tube::Sydney::Line::L3>

=item * L<L4 Line|Map::Tube::Sydney::Line::L4>

=back

=head1 CONSTRUCTOR

The constructor DOES NOT expects parameters.This setup the default node definitions.

    use strict; use warnings;
    use Map::Tube::Sydney;

    my $tube = Map::Tube::Sydney->new;

=head1 METHODS

=head2 get_shortest_route($from, $to)

It expects C<$from> and C<$to> station name, required param. It returns an object
of type L<Map::Tube::Route>. On error it throws exception of type L<Map::Tube::Exception>.

    use strict; use warnings;
    use Map::Tube::Sydney;

    my $tube  = Map::Tube::Sydney->new;

    my $route = $tube->get_shortest_route("Roseville", "Central");

    print "Route: $route\n";

=head2 as_image($line_name)

It expects the plugin  L<Map::Tube::Plugin::Graph> to be  installed. Returns line
image  as  base64  encoded string if C<$line_name> passed in otherwise it returns
base64 encoded string of the entire map.

    use strict; use warnings;
    use MIME::Base64;
    use Map::Tube::Sydney;

    my $tube = Map::Tube::Sydney->new;
    my $map  = $tube->name;
    open(my $IMAGE, ">", "$map.png")
       or die "ERROR: Cannot open [$map.png]: $!";
    binmode($IMAGE);
    print $IMAGE decode_base64($tube->as_image);
    close($IMAGE);

The L<Sydney Tube Map|https://raw.githubusercontent.com/earlybean4/Map-Tube-Sydney/master/maps/Sydney-Rail.png>
as generated by plugin L<Map::Tube::Plugin::Graph>.

=begin html

<a href = "https://raw.githubusercontent.com/earlybean4/Map-Tube-Sydney/master/maps/Sydney-Rail.png">
<img src    = "https://raw.githubusercontent.com/earlybean4/Map-Tube-Sydney/master/maps/Sydney-Rail.png"
     alt    = "Sydney Rail Map"
     width  = "571px"
     height = "830px"/>
</a>

=end html

=head1 AUTHOR

Peter Harrison, C<< <pete at 28smith.com> >>

=head1 REPOSITORY

L<https://github.com/earlybean4/Map-Tube-Sydney>

=head1 BUGS

Please report any bugs or feature requests through the web interface at L<https://github.com/earlybean4/Map-Tube-Sydney/issues>.
I will be notified and then you'll automatically be notified of progress on your
bug as I make changes.

=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Map::Tube::Sydney

You can also look for information at:

=over 4

=item * BUG Report

L<https://github.com/earlybean4/Map-Tube-Sydney/issues>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Map-Tube-Sydney>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Map-Tube-Sydney>

=item * Search MetaCPAN

L<https://metacpan.org/dist/Map-Tube-Sydney>

=back

=head1 LICENSE AND COPYRIGHT

Copyright (C) 2025 Peter Harrison.

This program  is free software; you can  redistribute it and / or modify it under
the  terms  of the the Artistic License  (2.0). You may obtain a copy of the full
license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any  use,  modification, and distribution of the Standard or Modified Versions is
governed by this Artistic License.By using, modifying or distributing the Package,
you accept this license. Do not use, modify, or distribute the Package, if you do
not accept this license.

If your Modified Version has been derived from a Modified Version made by someone
other than you,you are nevertheless required to ensure that your Modified Version
 complies with the requirements of this license.

This  license  does  not grant you the right to use any trademark,  service mark,
tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge patent license
to make,  have made, use,  offer to sell, sell, import and otherwise transfer the
Package with respect to any patent claims licensable by the Copyright Holder that
are  necessarily  infringed  by  the  Package. If you institute patent litigation
(including  a  cross-claim  or  counterclaim) against any party alleging that the
Package constitutes direct or contributory patent infringement,then this Artistic
License to you shall terminate on the date that such litigation is filed.

Disclaimer  of  Warranty:  THE  PACKAGE  IS  PROVIDED BY THE COPYRIGHT HOLDER AND
CONTRIBUTORS  "AS IS'  AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES. THE IMPLIED
WARRANTIES    OF   MERCHANTABILITY,   FITNESS   FOR   A   PARTICULAR  PURPOSE, OR
NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY YOUR LOCAL LAW. UNLESS
REQUIRED BY LAW, NO COPYRIGHT HOLDER OR CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL,  OR CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE
OF THE PACKAGE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=cut

1; # End of Map::Tube::Sydney
